/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// rendereffect.cpp

#include "rendereffect.h"
#include "renderer.h"

#include <kmessagebox.h>

#include <iostream>
#include <math.h>



namespace KoverArtist
{


static unsigned short sMultTab[65536];
static bool sMultTabNeedInit = true;


unsigned short mult256(int a, int b)
{
   if (a>256) a=256;
   if (b>256) b=256;

   if (sMultTabNeedInit)
   {
      int x, y;
      for (x=0; x<256; ++x)
         for (y=0; y<256; ++y)
            sMultTab[(x<<8)+y] = x*y;
      sMultTabNeedInit = false;
   }

   return sMultTab[(a<<8)+b];
}



//
//  CLASS  RenderEffect
//
RenderEffect::RenderEffect(const RendererBase* aParent, const Effect& aEff,
                           int aWidth, int aHeight)
:Inherited(aWidth, aHeight)
,mParent(aParent)
,mEff(aEff)
,mWidth(aWidth)
,mHeight(aHeight)
,mBB(0, 0, mWidth, mHeight)
,mColors(256)
{
}


RenderEffect& RenderEffect::operator=(const QImage& o)
{
   Inherited::operator=(o);
   return *this;
}


RenderEffect* RenderEffect::create(const RendererBase* aParent, const Effect& aEff,
                                   int aWidth, int aHeight)
{
   switch (aEff.type())
   {
   case Effect::NONE:
      return 0;
   case Effect::OUTLINE:
      return new RenderEffectOutline(aParent, aEff, aWidth, aHeight);
   case Effect::GLOW:
      return new RenderEffectGlow(aParent, aEff, aWidth, aHeight);
   case Effect::SHADOW:
      return new RenderEffectShadow(aParent, aEff, aWidth, aHeight);
   case Effect::BOX:
      return new RenderEffectBox(aParent, aEff, aWidth, aHeight);
   case Effect::ROUNDED_BOX:
//       return new RenderEffectRoundedBox(aParent, aEff, aWidth, aHeight);
   default:
      KMessageBox::error(0, QString("Internal error: unknown text effect #%1")
                         .arg(int(aEff.type())));
      return 0;
   }
}


void RenderEffect::setBoundingBox(const QRect& aBB)
{
   mBB = aBB;
}


void RenderEffect::setup(QPainter&, QPixmap&)
{
}


QRgb RenderEffect::rgba(const QColor& aColor, int aAlpha)
{
   return qRgba(aColor.red(), aColor.green(), aColor.blue(), aAlpha);
}


QRgb RenderEffect::mixColors(QRgb c1, QRgb c2, double f, int alpha)
{
   if (f<0) f = 0;
   else if (f>1.0) f = 1.0;

   double mf = 1.0 - f;

   if (alpha<0)
      alpha = int(qAlpha(c1)*mf + qAlpha(c2)*f);

   return qRgba(int(qRed(c1)*mf + qRed(c2)*f),
	        int(qGreen(c1)*mf + qGreen(c2)*f),
	        int(qBlue(c1)*mf + qBlue(c2)*f),
	        alpha);
}


void RenderEffect::growBoundingBox(unsigned int d)
{
   int x0 = mBB.x() - d;
   int y0 = mBB.y() - d;
   int x1 = mBB.right() + d;
   int y1 = mBB.bottom() + d;

   int w = mWidth-1;
   int h = mHeight-1;

   if (x0 <= 0) x0 = 1;
   if (y0 <= 0) y0 = 1;
   if (x1 >= w) x1 = w-1;
   if (y1 >= h) y1 = h-1;

   mBB.setX(x0);
   mBB.setY(y0);
   mBB.setRight(x1);
   mBB.setBottom(y1);
}


void RenderEffect::setupColorTable(QRgb c1, QRgb c2, int tr1, int tr2, double e)
{
   int tr = tr1 - tr2;
   int i, j, r, g, b, alpha;
   bool isLinear = (fabs(1.0 - e) < 0.0001);
   double f;

   if (tr1>255) tr1 = 255;
   else if (tr1<0) tr1 = 0;

   if (tr2>tr1) tr2 = tr1;
   else if (tr2<0) tr2 = 0;

   for (i=255,j=0,f=1.0; i>=0; --i,++j,f-=0.00392157)
   {
      r = (mult256(qRed(c1),i)   + mult256(qRed(c2),j))  >> 8;
      g = (mult256(qGreen(c1),i) + mult256(qGreen(c2),j))>> 8;
      b = (mult256(qBlue(c1),i)  + mult256(qBlue(c2),j)) >> 8;

      if (isLinear) alpha = int(tr*f + tr2);
      else alpha = int(tr*pow(f, e)) + tr2;

      if (alpha<0) alpha = 0;
      else if (alpha>255) alpha = 255;
      mColors[i] = qRgba(r, g, b, alpha);
   }
}


bool RenderEffect::makeDistanceMapPixel(int aX, int aY, int aAlpha)
{
   QRgb pix = pixel(aX, aY);
   int alpha = qAlpha(pix);

   if (alpha>=aAlpha || (alpha==0 && qBlue(pix)>=aAlpha)) return false;

   setPixel(aX, aY, qRgba(0, 255, aAlpha, alpha));

   return true;
}


void RenderEffect::makeDistanceMap(int aMaxDist)
{
   if (aMaxDist<1) return;

   int x, x0=mBB.left(), x1=mBB.right();
   int y, y0=mBB.top(), y1=mBB.bottom();
   int alpha, dalpha, a, aa, aad, i;
   int alphaTab[256], alphadTab[256];
   bool found;
   QRgb pix;

   aa = 255/aMaxDist;
   if (aa<1) aa = 1;
   aad = int(aa*1.41);

   for (i=0; i<256; ++i)
   {
      alphaTab[i] = i-aa;
      if (alphaTab[i]<0) alphaTab[i] = 0;

      alphadTab[i] = i-aad;
      if (alphadTab[i]<0) alphadTab[i] = 0;
   }

   for (i=aMaxDist,a=255,found=true; i>0 && found; --i,a-=aa)
   {
      for (found=false,y=y0; y<=y1; ++y)
      {
         for (x=x0; x<=x1; ++x)
         {
            pix = pixel(x, y);
            a = qAlpha(pix);
            if (a==0) continue;
            alpha = alphaTab[a];
            dalpha = alphadTab[a];
            if (y>y0)
            {
               found |= makeDistanceMapPixel(x, y-1, alpha);
               if (dalpha>0)
               {
                  if (x>x0) found |= makeDistanceMapPixel(x-1, y-1, dalpha);
                  if (x<x1) found |= makeDistanceMapPixel(x+1, y-1, dalpha);
               }
            }
            if (y<y1)
            {
               found |= makeDistanceMapPixel(x, y+1, alpha);
               if (dalpha>0)
               {
                  if (x>x0) found |= makeDistanceMapPixel(x-1, y+1, dalpha);
                  if (x<x1) found |= makeDistanceMapPixel(x+1, y+1, dalpha);
               }
            }
            if (x>x0) found |= makeDistanceMapPixel(x-1, y, alpha);
            if (x<x1) found |= makeDistanceMapPixel(x+1, y, alpha);
         }
      }

      for (y=y0; y<=y1; ++y)
      {
         for (x=x0; x<=x1; ++x)
         {
            pix = pixel(x, y);
            alpha = qBlue(pix);
            setPixel(x, y, qRgba(qRed(pix), qGreen(pix), alpha, alpha));
         }
      }
   }
}



//
//  CLASS  RenderEffectBox
//
RenderEffectBox::RenderEffectBox(const RendererBase* aParent, const Effect& aEff,
                                 int aWidth, int aHeight)
:Inherited(aParent, aEff, aWidth, aHeight)
{
}


RenderEffectBox::~RenderEffectBox()
{
}


void RenderEffectBox::render(QPixmap /*aTextPix*/)
{
   growBoundingBox(mEff.size());

   QRgb pix1(rgba(mEff.color1(), mEff.transparency()));
   QRgb pix2(rgba(mEff.color2(), mEff.transparency()));
   int x, x0=mBB.left(), x1=mBB.right();
   int y, y0=mBB.top(), y1=mBB.bottom();
   int i, sz=(mEff.size()+1)>>1;

   if (sz>y1-y0) sz = y1-y0;
   if (sz>x1-x0) sz = x1-x0;
   if (sz<0) sz = 0;

   for (y=y0+sz; y<=y1-sz; ++y)
   {
      for (x=x0+sz; x<=x1-sz; ++x)
         setPixel(x, y, pix1);
   }

   for (i=0; i<sz; ++i)
   {
      for (x=x0; x<=x1; ++x)
      {
         setPixel(x, y0+i, pix2);
         setPixel(x, y1-i, pix2);
      }
      for (y=y0; y<=y1; ++y)
      {
         setPixel(x0+i, y, pix2);
         setPixel(x1-i, y, pix2);
      }
   }
}



//
//  CLASS  RenderEffectGlow
//
RenderEffectGlow::RenderEffectGlow(const RendererBase* aParent, const Effect& aEff,
                                   int aWidth, int aHeight)
:Inherited(aParent, aEff, aWidth, aHeight)
{
}


RenderEffectGlow::~RenderEffectGlow()
{
}


void RenderEffectGlow::setup(QPainter&, QPixmap&)
{
   setupColorTable(mEff.color1().rgb(), mEff.color2().rgb(), 255, 0,
                   (256-mEff.transparency())/128.0);
}


void RenderEffectGlow::render(QPixmap aTextPix)
{
   growBoundingBox(mEff.size());

   int x, x0=mBB.left(), x1=mBB.right();
   int y, y0=mBB.top(), y1=mBB.bottom();
   int alpha;

   *this = aTextPix.convertToImage();
   threshold(80, qRgba(0,0,0,0), qRgba(255,255,255,255));
   makeDistanceMap(mEff.size());

   for (y=y0; y<=y1; ++y)
   {
      for (x=x0; x<=x1; ++x)
      {
         alpha = qAlpha(pixel(x, y));
         if (alpha) setPixel(x, y, mColors[alpha]);
         else setPixel(x, y, 0);
      }
   }
}



//
//  CLASS  RenderEffectShadow
//
RenderEffectShadow::RenderEffectShadow(const RendererBase* aParent, const Effect& aEff,
                                       int aWidth, int aHeight)
:Inherited(aParent, aEff, aWidth, aHeight)
{
}


RenderEffectShadow::~RenderEffectShadow()
{
}


void RenderEffectShadow::setup(QPainter& aPaint, QPixmap&)
{
   int sz = mEff.size();
   aPaint.translate(int(sz*0.6), int(sz*0.6));

   setupColorTable(mEff.color1().rgb(), mEff.color2().rgb(),
                   mEff.transparency()<<1, 0, (256-mEff.transparency())/128.0);
}



//
//  CLASS  RenderEffectOutline
//
RenderEffectOutline::RenderEffectOutline(const RendererBase* aParent,
                           const Effect& aEff, int aWidth, int aHeight)
:Inherited(aParent, aEff, aWidth, aHeight)
{
}


RenderEffectOutline::~RenderEffectOutline()
{
}


void RenderEffectOutline::setup(QPainter&, QPixmap&)
{
   setupColorTable(mEff.color1().rgb(), mEff.color2().rgb(),
                   mEff.transparency(), 0, 0.2);
}



#ifdef BROKEN
//
//  CLASS  RenderEffectRoundedBox
//
RenderEffectRoundedBox::RenderEffectRoundedBox(const Renderer* aParent, double aScale,
                           const QFont& aFont, const QRect& aBB, Qt::AlignmentFlags aAlign,
			   const QString& aText, const Effect& aEff)
:RenderEffect(aParent, aScale, aFont, aBB, aAlign, aText, aEff)
,mRndX(mEff.size()*99/(mBB.width()>>1))
,mRndY(mEff.size()*99/(mBB.height()>>1))
{
}


void RenderEffectRoundedBox::setupHelper(QPainter& phlp, int aWidth, int aHeight)
{
   RenderEffect::setupHelper(phlp, aWidth, aHeight);

   growBoundingBox(mEff.size(), aWidth, aHeight);
   int trans = mEff.transparency();

   phlp.save();
   phlp.setPen(Qt::NoPen);
   phlp.setBrush(QColor(trans, trans, trans));
   phlp.drawRoundRect(mBB, mRndX, mRndY);
   phlp.restore();
}


bool RenderEffectRoundedBox::renderIt(QPainter& p, int aStep, QImage& /*aImgHelper*/)
{
   int sz=mEff.size();
   int i = aStep;
   if (i>sz) return false;

   p.save();
   p.setPen(Qt::NoPen);
   p.setBrush(mixColors(mEff.color2(), mEff.color1(), double(i)/sz));
   p.drawRoundRect(mBB.x()+i, mBB.y()+i, mBB.width()-(i<<1), mBB.height()-(i<<1),
                   mRndX, mRndY);
   p.restore();

   return true;
}

#endif /*BROKEN*/


} //namespace
