/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// image.cpp

#include "image.h"
#include "settings.h"

#include <qdom.h>
#include <qimage.h>
#include <qpainter.h>

#include <kmessagebox.h>
#include <klocale.h>
#include <kio/netaccess.h>

#include <iostream>


namespace KoverArtist
{


Image::Image()
:mUrl()
,mPosition(KoverArtist::ScaleCrop)
,mPixmap()
,mOnSides(true)
,mTransparency(255)
{
}


Image::Image(const Image& o)
:mUrl(o.mUrl)
,mPosition(o.mPosition)
,mPixmap(o.mPixmap)
,mOnSides(o.mOnSides)
,mTransparency(o.mTransparency)
{
}


Image& Image::operator=(const Image& o)
{
   if (&o!=this)
   {
      mUrl = o.mUrl;
      mPosition = o.mPosition;
      mPixmap = o.mPixmap;
      mOnSides = o.mOnSides;
      mTransparency = o.mTransparency;
   }
   return *this;
}


void Image::clear()
{
   mUrl = "";
   mPosition = KoverArtist::Scaled;
   mPixmap = QPixmap();
   mOnSides = false;
   mTransparency = 255;
}


QPixmap Image::imageToPixmapBordersRemoved(QImage& aImg)
{
   int x, y, w=aImg.width(), h=aImg.height();
   int x0=0, x1=w-1;
   bool cont;

   if (w<=0 && h<=0) return QPixmap(0, 0);

   for (x=0,cont=true; x<w && cont; ++x)
   {
      for (y=0; y<h && cont; y+=32)
         if (aImg.pixel(x,y)!=0) cont = false;
   }
   x0 = x;

   for (x=w-1,cont=true; x>x0 && cont; --x)
   {
      for (y=0; y<h && cont; y+=32)
         if (aImg.pixel(x,y)!=0) cont = false;
   }
   x1 = x;

   QPixmap pix(x1-x0, h);
   QPainter p(&pix);
   p.drawImage(0, 0, aImg, x0, 0, x1-x0, h);
   p.end();

   return pix;
}


bool Image::load(const KURL& aUrl)
{
   bool isLocal = aUrl.isLocalFile();
   QString fname;
   bool ok = true;

   mPixmap = QPixmap(0, 0);
   if (aUrl.isEmpty())
   {
      mUrl = aUrl;
      return true;
   }

   if (isLocal) fname = aUrl.path();
   else
   {
      if (!KIO::NetAccess::download(aUrl, fname, 0))
      {
         KMessageBox::error(0, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   if (Settings::instance()->removeImageBorders)
   {
      QImage img;
      ok = img.load(fname);
      if (ok) mPixmap = imageToPixmapBordersRemoved(img);
   }
   else ok = mPixmap.load(fname);

   if (ok) mUrl = aUrl;
   else KMessageBox::error(0, i18n("Cannot load image %1").arg(aUrl.prettyURL()));

   if (isLocal) KIO::NetAccess::removeTempFile(fname);
   return ok;
}


void Image::toDom(QDomDocument& aDoc, QDomElement& aElem, const QString& aProjectDir) const
{
   aElem.setAttribute("position", imagePositionAsString(mPosition));
   aElem.setAttribute("onsides", mOnSides ? "true" : "false");
   aElem.setAttribute("transparency", mTransparency);

   if (aProjectDir==mUrl.directory())
      aElem.appendChild(aDoc.createTextNode(mUrl.fileName()));
   else aElem.appendChild(aDoc.createTextNode(mUrl.url()));
}


bool Image::fromDom(const QDomElement& aElem, const QString& aProjectDir)
{
   mPosition = imagePositionFromString(aElem.attribute("position"));
   mOnSides = (aElem.attribute("onsides") == "true");

   if (aElem.hasAttribute("transparency"))
      mTransparency = aElem.attribute("transparency").toInt() & 255;
   else mTransparency = 255;

   QString fname = aElem.text().stripWhiteSpace();
   if (fname.find('/')>0) mUrl = KURL::fromPathOrURL(fname);
   else if (!fname.isEmpty()) mUrl = KURL::fromPathOrURL(aProjectDir+'/'+fname);
   else mUrl = KURL();
//    std::cout<<"Image::fromDom: "<<fname<<" =("<<aProjectDir<<")=> "<<mUrl.prettyURL()<<std::endl;

   if (mUrl.isValid()) load(mUrl);
   return true;
}


const char* Image::imagePositionAsString(ImagePosition ip)
{
   switch (ip)
   {
   case Centered: return "centered";
   case Tiled: return "tiled";
   case CenteredTiled: return "centered-tiled";
   case CenteredMaxpect: return "maxpect";
   case Scaled: return "scaled";
   case ScaleCrop: return "scale-crop";
   default: return "";
   }
}


ImagePosition Image::imagePositionFromString(const QString& aStr)
{
   if (aStr=="centered") return KoverArtist::Centered;
   if (aStr=="tiled") return KoverArtist::Tiled;
   if (aStr=="centered-tiled") return KoverArtist::CenteredTiled;
   if (aStr=="maxpect") return KoverArtist::CenteredMaxpect;
   if (aStr=="scaled") return KoverArtist::Scaled;
   if (aStr=="scale-crop") return KoverArtist::ScaleCrop;
   return KoverArtist::Scaled;
}


} //namespace
