/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "cddbquery.h"

#include "cddbchooser.h"
#include "settings.h"

#include <kio/job.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qstringlist.h>

#include <iostream>
#include <errno.h>
#include <string.h>



#ifdef ENABLE_AUDIOCD

typedef QMap<QString,QString> StringMap;


CddbQuery::CddbQuery(QWidget *aParent)
:Inherited()
,mParent(aParent)
,mJob(0)
,mDiscInfo(0)
,mDisc()
,mHello()
,mData()
,mCddbIdStr()
,mActive(false)
{
   // Fill the cddb-hello string
   const char* logName = getenv("LOGNAME");
   if (!logName || !*logName) logName = "anonymous";
   const char* hostName = getenv("HOSTNAME");
   if (!hostName || !*hostName) hostName = "localhost";
   mHello = QString("hello=%1+%2+%3+%4").arg(logName).arg(hostName)
      .arg(PACKAGE).arg(VERSION);

}


CddbQuery::~CddbQuery()
{
   if (mJob) mJob->kill(true);
   delete mDiscInfo;
}


bool CddbQuery::testBusy() const
{
   if (mActive)
   {
      KMessageBox::error(mParent, i18n("A cddb query is already active.\n"
         "Please try again afterwards."), i18n("Error")+" - KoverArtist", 0);
   }
   return mActive;
}


void CddbQuery::cddbQueryAudioCd()
{
   if (testBusy()) return;

   const KoverArtist::Settings* stg = KoverArtist::Settings::instance();
   KoverArtist::Settings::CddbAccess mode = stg->cddbAccess;

   mCddbIdStr = QString::null;
   mDisc.clear();

   mActive = true;
   if (!readCd())
   {
      mActive = false;
      return;
   }

   if (mode & KoverArtist::Settings::CDDB_LOCAL)
   {
      emit status(i18n("Scanning local CDDB files..."));
      qApp->processEvents(500);

      if (localCddbLookup())
      {
         emit status(i18n("Found: %1").arg(mDisc.title()));
         emit finished();
         mActive = false;
         return;
      }
      else emit status(i18n("No matching local CDDB files found."));
   }

   if (mode & KoverArtist::Settings::CDDB_REMOTE)
   {
      emit status(i18n("Querying CDDB server %1...")
         .arg(KoverArtist::Settings::instance()->cddbServer));

      QString cddbStr = mDiscInfo->cddbQueryString();
      cddbStr.replace(' ', '+');
      startCddbQuery("cddb+query+"+cddbStr);
   }
   else mActive = false;
}


bool CddbQuery::readCd()
{
   if (mDiscInfo) delete mDiscInfo;
   mDiscInfo = new CdInfo(KoverArtist::Settings::instance()->cdDevice);

   emit status(i18n("Scanning audio-cd..."));
   if (!mDiscInfo->open() || !mDiscInfo->readDisc())
   {
      emit status(mDiscInfo->errorText());
      return false;
   }

   mCddbIdStr.sprintf("%08lx", mDiscInfo->cddbId());
   return true;
}


void CddbQuery::startCddbQuery(const QString& aCmd)
{
   KoverArtist::Settings* stg = KoverArtist::Settings::instance();
   mData = QString::null;

   QString url = QString("%1:%2/~cddb/cddb.cgi?cmd=%3&%4&proto=3")
      .arg(stg->cddbServer).arg(stg->cddbPort).arg(aCmd).arg(mHello);

   std::cout<<"CDDB-query string: "<<url<<std::endl;

   mJob = KIO::get(url);
   mJob->setWindow(mParent);

   connect(mJob, SIGNAL(data(KIO::Job*, const QByteArray&)),
           this, SLOT(cddbDataArrived(KIO::Job*, const QByteArray&)));
   connect(mJob, SIGNAL(result( KIO::Job*)), this, SLOT(cddbResult(KIO::Job*)));
}


void CddbQuery::cddbResult(KIO::Job* aJob)
{
   std::cout<<"CDDB-query job ended"<<std::endl;

   if (aJob->error())
   {
      KMessageBox::error(mParent, aJob->errorString(),
         i18n("Error")+" - KoverArtist", 0);
   }

   if (!mData.isEmpty() && mData.right(5)!="\r\n.\r\n")
      KMessageBox::error(mParent, mData, i18n("Error")+" - KoverArtist");

   mJob = 0;
   emit finished();
   mActive = false;
}


QString CddbQuery::handleInexactMatch(const QStringList& aLines) const
{
   CddbChooser dlg(aLines, i18n("Multiple Matches Found"),
                   i18n("Multiple database entries match.\nPlease select the right one."),
                   mParent);

   if (dlg.exec()!=QDialog::Accepted) return QString::null;
   return dlg.currentItem();
}


void CddbQuery::cddbDataArrived(KIO::Job* aJob, const QByteArray& aData)
{
   mData += aData;

   int code = 0;
   int idx = mData.find(' ');
   if (idx>0) code = mData.left(idx).toInt();

   if ((code==211 || code==210) && mData.right(5)!="\r\n.\r\n")
      return;

   QStringList res, lines;
   splitString(lines, mData, "\r\n");
   QString str, line0=lines[0];
   bool done = false;

   std::cout<<"CDDB-query reply: "<<line0<<std::endl;

   switch (code)
   {
   case 200: // Exact match for cddb query found
      splitString(res, line0, " ", 4);
      emit status(i18n("Fetching details..."));
      cddbFetchDetails(res[1], res[2], res[3]);
      break;

   case 210: // CDDB database entry
      emit status(i18n("Found: %1").arg(mDisc.title()));
      splitString(res, line0, " ", 4);
      storeCddbFile(res[1], res[2], mData);
      parseCddbEntry(lines);
      done = true;
      break;

   case 211: // Inexact match, let the user choose the right one
      lines.pop_front();
      lines.pop_back();
      if (lines.count()>1) str = handleInexactMatch(lines);
      else str = lines.front();
      if (str.isEmpty())
      {
         done = true;
         break;
      }
      splitString(res, str, " ", 3);
      emit status(i18n("Fetching details..."));
      cddbFetchDetails(res[0], res[1], res[2]);
      break;

   case 0:
      return;

   default: // Unknown command
      emit status(i18n("Unknown CDDB reply: %1").arg(line0));
      done = true;
      break;
   }

   if (mJob==aJob)
   {
      mJob->kill();
      mJob = 0;
   }

   if (done)
   {
      mJob = 0;
      emit finished();
      mActive = false;
      mData = QString::null;
   }
}


void CddbQuery::cddbFetchDetails(const QString& aGenre, const QString& aCddbId,
                                 const QString& aTitle)
{
   mDisc.clear();
   mDisc.setTitle(aTitle);
   startCddbQuery("cddb+read+"+aGenre+"+"+aCddbId);
}


void CddbQuery::parseCddbEntry(const QStringList& aLines)
{
   QStringList::const_iterator it;
   QString k;
   int idx, i;

   mDisc.clear();

   QStringList& entries = mDisc.entries();
   for (it=aLines.begin(), i=1; it!=aLines.end(); ++it)
   {
      const QString& line = *it;
      idx = line.find('=');
      if (idx>0)
      {
         k = line.left(6);
         if (k=="TTITLE")
         {
            k.sprintf("%2d. ", i++);
            entries.push_back(k + line.mid(idx+1));
         }
         else if (k=="DTITLE") mDisc.setTitle(line.mid(idx+1));
      }
   }
}


void CddbQuery::splitString(QStringList& aResult, const QString& aStr,
                            const QString& aSep, int aElems) const
{
   int beg, end, cnt, len=aStr.length();
   int sepLen=aSep.length();

   aResult.clear();
   for (beg=0,cnt=1,end=-1; (aElems<0||cnt<aElems)&&beg<len; beg=end+sepLen, ++cnt)
   {
      end = aStr.find(aSep, beg);
      if (end<0) end = len;
      if (end>=0 && end-beg>0)
         aResult.push_back(aStr.mid(beg, end-beg));
   }

   if (beg<len)
      aResult.push_back(aStr.mid(beg));
}


bool CddbQuery::canDecode(const KURL& aUrl)
{
   QChar ch;
   int i;

   // Test if the url could be a cddb entry
   QString fname = aUrl.fileName();
   for (i=fname.length()-1; i>=0; --i)
   {
      ch = fname[i].lower();
      if (!ch.isDigit() && (ch<'a' || ch>'f')) break;
   }
   if (i<0) return true;

   // Test if the url could be a cddb result page
   if (aUrl.queryItem("cat").isEmpty() || aUrl.queryItem("id").isEmpty())
      return false;

   return true;
}


bool CddbQuery::fromUrl(const KURL& aUrl)
{
   if (testBusy()) return false;

   QString cat = aUrl.queryItem("cat");
   QString id = aUrl.queryItem("id");
   bool ok = true;

   if (!cat.isEmpty() && !id.isEmpty())
   {
      mActive = true;
      cddbFetchDetails(cat, id, QString::null);
   }
   else
   {
      QString fname;
      bool isLocal = aUrl.isLocalFile();
      if (isLocal) fname = aUrl.path();
      else
      {
         if (!KIO::NetAccess::download(aUrl, fname, mParent))
         {
            KMessageBox::error(mParent, KIO::NetAccess::lastErrorString());
            return false;
         }
      }

      QFile in(fname);
      if (!in.open(IO_ReadOnly))
      {
         KMessageBox::error(mParent, i18n("Cannot read temporary file: %1")
            .arg(in.errorString()));
         ok = false;
      }
      else
      {
         static const int dataMax=8193;
         char data[dataMax];
         int len;
         mData = QString::null;
         while (!in.atEnd())
         {
            len = in.readBlock(data, dataMax-1);
            if (len<=0) break;
            data[len] = '\0';
            mData += data;
         }
         in.close();

         mActive = true;
         QStringList lines;
         splitString(lines, mData, mData.find("\r\n")>0 ? "\r\n" : "\n");
         parseCddbEntry(lines);
         emit finished();
         mActive = false;
      }

      if (isLocal) KIO::NetAccess::removeTempFile(fname);
   }

   return ok;
}


void CddbQuery::localCddbLookupFile(const QString& aPath, StringMap& aResults)
{
   QFile fin(aPath);
   QTextStream in(&fin);
   in.setEncoding(QTextStream::Locale);

   if (!fin.open(IO_ReadOnly))
   {
      KMessageBox::error(mParent, i18n("Cannot read %1: %2").arg(fin.errorString()),
         i18n("Error")+" - KoverArtist", 0);
      return;
   }

   QString data = in.read();
   if (data.isEmpty()) return;

   int beg = data.find("\nDTITLE=");
   int end = data.find("\n", beg+1);
   if (beg<0 || end<beg)
   {
      std::cerr<<"Malformed cddb file "<<aPath<<": contains no DTITLE line"<<std::endl;
      return;
   }

   QString title = data.mid(beg+8, end-beg-8).stripWhiteSpace();
   aResults[title] = data;
}


void CddbQuery::localCddbLookupDir(const QString& aPath, StringMap& aResults)
{
   QDir dir(aPath, QString::null, QDir::Unsorted, QDir::Files|QDir::Dirs);
   if (!dir.exists()) return;

   const QFileInfoList *entries = dir.entryInfoList();
   QFileInfo *entry;
   QString path, fname;

   QPtrListIterator<QFileInfo> it(*entries);
   for (; (entry=*it)!=0; ++it)
   {
      path = entry->filePath();
      fname = entry->fileName();

      if (entry->isDir())
      {
         if (path.length()<1000 && fname[0]!='.')
            localCddbLookupDir(path, aResults);
      }
      else if (entry->isFile())
      {
         if (fname==mCddbIdStr)
            localCddbLookupFile(path, aResults);
      }
   }
}


bool CddbQuery::localCddbLookup()
{
   const KoverArtist::Settings* stg = KoverArtist::Settings::instance();
   const QStringList& lst = stg->cddbDirs;
   StringMap matches;

   for (QStringList::const_iterator it=lst.begin(); it!=lst.end(); ++it)
      localCddbLookupDir(*it, matches);

   int num = matches.count();
   if (num<=0) return false;

   QString key;
   if (num==1) key = matches.begin().key();
   else key = handleInexactMatch(matches.keys());

   if (key.isEmpty()) return false;

   QStringList lines;
   const QString& data = matches[key];
   splitString(lines, data, data.find("\r\n")>0 ? "\r\n" : "\n");
   parseCddbEntry(lines);

   return true;
}


bool CddbQuery::mkdirPath(const QString& aPath) const
{
   QDir d(aPath);
   QStringList dirs(QStringList::split(QDir::separator(), d.absPath()));
   d.setPath("/");

   for (QStringList::Iterator it=dirs.begin(); it!=dirs.end(); ++it)
   {
      if (!d.exists(*it))
      {
         if (!d.mkdir(*it))
         {
            KMessageBox::error(mParent,
               i18n("Cannot create directory %1/%2 for cddb file storage: %3")
                  .arg(d.path()).arg(*it).arg(strerror(errno)),
               i18n("Error")+" - KoverArtist", 0);
            return false;
         }
      }
      if (!d.cd(*it))
      {
         KMessageBox::error(mParent, i18n("Cannot enter directory %1: %2")
            .arg(d.path()).arg(strerror(errno)),
            i18n("Error")+" - KoverArtist", 0);
         return false;
      }
   }
   return true;
}


void CddbQuery::storeCddbFile(const QString& aGenre, const QString& aCddbId,
                              const QString& aData) const
{
   KoverArtist::Settings* stg = KoverArtist::Settings::instance();
   if (!stg->cddbCacheFiles || aData.isEmpty()) return;

   QString path = stg->cddbDirs.front()+'/'+aGenre;
   if (!QDir(path).exists() && !mkdirPath(path))
      return;
   path += '/'+aCddbId;

   std::cout<<"Writing cddb file "<<path<<std::endl;

   QFile fout(path);
   if (!fout.open(IO_WriteOnly))
   {
      KMessageBox::error(mParent, i18n("Cannot write %1: %2")
         .arg(path).arg(fout.errorString()),
         i18n("Error")+" - KoverArtist", 0);
      return;
   }

   QTextStream out(&fout);
   out<<aData;

   fout.close();
}


#include "cddbquery.moc"

#endif /*ENABLE_AUDIOCD*/
