/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#ifndef KITAIMGMANAGER_H
#define KITAIMGMANAGER_H

#include "event.h"

#include <kurl.h>

#include <qobject.h>
#include <qdict.h>
#include <qmutex.h>
#include <qevent.h>
#include <qstringlist.h>

struct IMGDAT
{
    int code;
    bool mosaic;
    int width;
    int height;
    unsigned int size;
    KURL datURL;
};


/*----------------------------------------------------------*/

namespace Kita
{
    class LoaderData;

    class ImgManager : public QObject
    {
        Q_OBJECT

        static ImgManager* instance;
        static QMutex m_mutex;
        QDict< IMGDAT > m_imgDatDict;
        QWidget* m_mainwidget;
        QStringList m_urlList;

    public:
        ImgManager( QWidget* mainwidget );
        ~ImgManager();

        static void setup( QWidget* mainwidget );
        static void deleteInstance();
        static ImgManager* getInstance();
        static bool isImgFile( const KURL& url, bool use_mimetype );
        static bool isAnimationGIF( const KURL& url );
        static bool isBMP( const KURL& url );

        static bool load( const KURL& url, const KURL& datURL );
        static void stop( const KURL& url );
        static QString getPath( const KURL& url );
        static bool isLoadingNow( const KURL& url );
        static bool deleteCache( const KURL& url, QWidget* parent );
        static bool copyCache( const KURL& url, QWidget* parent );

        static bool mosaic( const KURL& url );
        static void setMosaic( const KURL& url, bool status );
        static int code( const KURL& url );
        static int width( const KURL& url );
        static int height( const KURL& url );
        static unsigned int size( const KURL& url );
        static QPixmap icon( const KURL& url );
        static KURL datURL( const KURL& url );

    private:
        bool isImgFilePrivate( const KURL& url, bool use_mimetype );
        bool loadPrivate( const KURL& url, const KURL& datURL );
        bool deleteCachePrivate( const KURL& url, QWidget* parent );
        bool copyCachePrivate( const KURL& url, QWidget* parent );
        void setMosaicPrivate( const KURL& url, bool status );

        void createImgDat( const KURL& url, int code );
        IMGDAT* getImgDat( const KURL& url );
        void deleteImgDat( const KURL& url );
        void getSize( const KURL& url );
        bool cacheExists( const KURL& url );

    private slots:
        void slotData( const Kita::LoaderData&, const QByteArray& );
        void slotResult( const Kita::LoaderData& );

    protected:
        virtual void customEvent( QCustomEvent * e );

    signals:
        void receiveImgData( const KURL&, unsigned int, unsigned int );
        void finishImgLoad( const KURL& );
        void cacheDeleted( const KURL& );
    };


    /* finish event */
    class EmitFinishEvent : public QCustomEvent
    {
        KURL m_url;

    public:
        EmitFinishEvent( KURL url ) : QCustomEvent( EVENT_EmitFinigh ), m_url( url ) {}

        KURL url() const { return m_url; }
    };

}

#endif
