/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 22-11-2006
 * Description : Implements a list of photographs.
 *
 * Copyright 2005,2007 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#include "photo.h"

#include <klocale.h>
#include <kicontheme.h>
#include <kiconloader.h>
#include <kio/previewjob.h>

#include <qfile.h>
#include <qwmatrix.h>
#include <qfileinfo.h>

#include "exif.h"


namespace
{
    /** Helper function to generate a DOM Element for the given DOM document
     *  and append it to the children of the given node.
     *  @param doc the DOM document
     *  @param node the parent we are going to be appended to
     *  @param tag our tag name
     *  @param value our value
     */
    QDomElement addElement( QDomDocument &doc, QDomNode &node, const QString &tag,
        const QString &value = QString::null )
    {
        QDomElement el = doc.createElement( tag );
        node.appendChild( el );
        if ( !value.isNull() )
        {
            QDomText txt = doc.createTextNode( value );
            el.appendChild( txt );
        }
        return el;
    }
}

Photo::Photo() : QObject()
  , m_URL()
  , m_exposed(TRUE)
  , m_family(FALSE)
  , m_friends(FALSE)
  , m_rotation(0)
  , m_desc("")
  , m_size(i18n("Original"))
  , m_title("")
  , m_license("All Rights Reserved")
  , m_photoset(i18n("<photostream only>"))
  , m_tags()
  , m_selected(FALSE)
  , m_preview(SmallIcon("camera", KIcon::SizeHuge, KIcon::DefaultState))
{
}

Photo::Photo(const KURL& url) : QObject()
  , m_URL(url)
  , m_exposed(TRUE)
  , m_family(FALSE)
  , m_friends(FALSE)
  , m_rotation(0)
  , m_desc(EXIF(url.path()).userComment())
  , m_size(i18n("Original"))
  , m_title(QFileInfo(url.filename()).baseName(TRUE))
  , m_license("All Rights Reserved")
  , m_photoset(i18n("<photostream only>"))
  , m_tags()
  , m_selected(FALSE)
  , m_preview(SmallIcon("camera", KIcon::SizeHuge, KIcon::DefaultState))
{
}

Photo::Photo(const Photo& photo) : QObject()
  , m_URL(photo.URL())
  , m_exposed(photo.exposed())
  , m_family(photo.family())
  , m_friends(photo.friends())
  , m_rotation(photo.rotation())
  , m_desc(photo.description())
  , m_size(photo.size())
  , m_title(photo.title())
  , m_license(photo.license())
  , m_photoset(photo.photoset())
  , m_tags(photo.tags())
  , m_selected(photo.selected())
  , m_preview(photo.preview())
{
}

Photo::~Photo()
{
}

bool Photo::operator==(const Photo& other) const
{
    if( other.m_URL == m_URL )
        return TRUE;

    return FALSE;
}

void Photo::rotate()
{
    if( m_rotation == 0 )
        m_rotation = 270;
    else
        m_rotation -= 90;

    QWMatrix matrix;
    matrix = matrix.rotate(-90);
    m_preview = m_preview.xForm(matrix);
    emit update(PHOTO_PREVIEW | PHOTO_ROTATION);
}

void Photo::preview(const QPixmap& pm)
{
    m_preview = pm;

    // Ensure the preview gets the existing rotation applied
    if(m_rotation)
    {
        QWMatrix matrix;
        matrix = matrix.rotate(m_rotation);
        m_preview = m_preview.xForm(matrix);
    }

    emit update(PHOTO_PREVIEW);
}

void Photo::URL(const KURL& u)
{
    if( m_URL != u )
    {
        m_URL = u;
        emit update(PHOTO_URL);
    }
}

void Photo::exposed(bool b)
{
    if( m_exposed != b )
    {
        m_exposed = b;
        emit update(PHOTO_PERMISSIONS);
    }
}

void Photo::family(bool b)
{
    if( m_family != b )
    {
        m_family = b;
        emit update(PHOTO_PERMISSIONS);
    }
}

void Photo::friends(bool b)
{
    if( m_friends != b )
    {
        m_friends = b;
        emit update(PHOTO_PERMISSIONS);
    }
}

void Photo::rotation(uint r)
{
    // Ensure rotation is restricted to 0-360 degrees
    if( m_rotation != (r % 360) )
    {
        int diff = (r % 360) - m_rotation;
        m_rotation = r % 360;

        // rotate the preview accordingly
        QWMatrix matrix;
        matrix = matrix.rotate(diff);
        m_preview = m_preview.xForm(matrix);

        emit update(PHOTO_PREVIEW | PHOTO_ROTATION);
    }
}

void Photo::description(const QString& d)
{
    if( m_desc != d )
    {
        m_desc = d;
        emit update(PHOTO_DESCRIPTION);
    }
}

void Photo::size(const QString & s)
{
    // Ensure it is one of the valid sizings
    if( m_size != s )
    {
        m_size = s;
        emit update(PHOTO_SIZE);
    }
}

void Photo::title(const QString& t)
{
    if( m_title != t )
    {
        m_title = t;
        emit update(PHOTO_TITLE);
    }
}

void Photo::license(const QString& l)
{
    if( m_license != l )
    {
        m_license = l;
        emit update(PHOTO_LICENSE);
    }
}

void Photo::photoset(const QString& ps)
{
    if( m_photoset != ps )
    {
        m_photoset = ps;
        emit update(PHOTO_PHOTOSET);
    }
}

void Photo::tags(const QStringList& t)
{
    if( m_tags != t )
    {
        m_tags = t;
        emit update(PHOTO_TAGS);
    }
}

void Photo::selected(bool b)
{
    if( m_selected != b )
    {
        m_selected = b;
        emit update(PHOTO_SELECTION);
    }
}

bool Photo::isPortrait( )
{
    if( m_preview.width() < m_preview.height() )
    {
        return TRUE;
    }
    return FALSE;
}

void Photo::save2DOMDoc(QDomDocument& doc, QDomElement& el)
{
    QDomElement p = addElement(doc, el, "photo");
    p.setAttribute("url", URL().url());

    addElement(doc, p, "exposed", exposed() ? "1" : "0");
    addElement(doc, p, "family", family() ? "1" : "0");
    addElement(doc, p, "friends", friends() ? "1" : "0");
    addElement(doc, p, "rotation", QString::number(rotation()));
    addElement(doc, p, "description", description());
    addElement(doc, p, "size", size());
    addElement(doc, p, "title", title());
    addElement(doc, p, "license", license());
    addElement(doc, p, "photoset", photoset());

    QDomElement ts = addElement(doc, p, "tags");
    for( QStringList::iterator it = m_tags.begin(); it != m_tags.end(); ++it )
    {
        addElement(doc, ts, "tag", *it);
    }
}

void Photo::restoreFromDOM(QDomElement& e)
{
    QDomElement v;

    URL(KURL(e.attribute("url")));

    v = e.namedItem("exposed").toElement();
    if( !v.isNull() ) exposed(v.text() == "0" ? FALSE : TRUE);
    v = e.namedItem("family").toElement();
    if( !v.isNull() ) family(v.text() == "0" ? FALSE : TRUE);
    v = e.namedItem("friends").toElement();
    if( !v.isNull() ) friends(v.text() == "0" ? FALSE : TRUE);
    v = e.namedItem("rotation").toElement();
    if( !v.isNull() ) rotation(v.text().toUInt());
    v = e.namedItem("description").toElement();
    if( !v.isNull() ) description(v.text());
    v = e.namedItem("size").toElement();
    if( !v.isNull() ) size(v.text());
    v = e.namedItem("title").toElement();
    if( !v.isNull() ) title(v.text());
    v = e.namedItem("license").toElement();
    if( !v.isNull() ) license(v.text());
    v = e.namedItem("photoset").toElement();
    if( !v.isNull() ) photoset(v.text());

    v = e.namedItem("tags").toElement();
    if( !v.isNull() )
    {
        QStringList ts;
        for( QDomNode n = v.firstChild(); !n.isNull(); n = n.nextSibling() )
        {
            if( n.isElement() && n.toElement().tagName() == "tag" )
                ts += n.toElement().text();
        }
        tags(ts);
    }
}


#include "photo.moc"

