/*
 * IIIMF-Canna, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 *
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org> 
 * Copyright (C) 2002 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Shingo Akagaki <dora@momonga-linux.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: canna.c,v 1.19.2.7 2003/08/04 09:27:29 famao Exp $ */

/* vi:set ts=4 sw=4: */


/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
#ifdef	WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <canna/jrkanji.h>
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <assert.h>

#include "im_util.h"
#include "im_aux.h"

static char canna_le_init_filename[] = ".canna";

#define CANNA_COMMIT_STRING_BUFSIZE (8192 * 2)
#define EUC_JP_SS2 0x8E
#define EUC_JP_SS3 0x8F
#define CANNA_NEXT_CHAR(p) 		\
  (((*p) == 0) ? (p) :			\
   (((*p) < 0x80) ? ((p) + 1) :		\
    (((*p) == EUC_JP_SS3) ? ((p) + 3) :	\
     ((p) + 2))))
#define CANNA_GUIDELINE_DELIMITER_P(p)	\
  (((*p) == ' ') || ((*p) == '\t') || 	\
   (((*p) == 0xA1) && ((p[1]) == 0xA1)))

	
static Bool    if_canna_OpenIF(iml_if_t *);
static Bool    if_canna_CloseIF(iml_if_t *);
static Bool    if_canna_GetIFValue(iml_if_t *, IMArgList, int);
static Bool    if_canna_SetIFValue(iml_if_t *, IMArgList, int);
static Bool    if_canna_OpenDesktop(iml_desktop_t *, IMArgList, int);
static Bool    if_canna_CloseDesktop(iml_desktop_t *);
static Bool    if_canna_CreateSC(iml_session_t *, IMArgList, int);
static Bool    if_canna_DestroySC(iml_session_t *);
static Bool    if_canna_GetSCValue(iml_session_t *, IMArgList, int);
static Bool    if_canna_SetSCValue(iml_session_t *, IMArgList, int);
static IMText  *if_canna_ResetSC(iml_session_t *);
static void    if_canna_SetSCFocus(iml_session_t *);
static void    if_canna_UnsetSCFocus(iml_session_t *);
static void    if_canna_SendEvent(iml_session_t *, IMInputEvent *);

/* IF Method */
if_methods_t canna_methods = {
    if_canna_OpenIF,
    if_canna_CloseIF,
    if_canna_GetIFValue,
    if_canna_SetIFValue,

    if_canna_OpenDesktop,
    if_canna_CloseDesktop,

    if_canna_CreateSC,
    if_canna_DestroySC,
    if_canna_GetSCValue,
    if_canna_SetSCValue,
    if_canna_ResetSC,
    if_canna_SetSCFocus,
    if_canna_UnsetSCFocus,
    if_canna_SendEvent,
};

UTFCHAR lename_string[] = {0x304B, 0x3093, 0x306A, 0x4C, 0x45, 0x0};
UTFCHAR jahrn_string[] = {0x65E5, 0x672C, 0x8A9E, 0x0};
UTFCHAR lookup_choice_title[] = {0x5019, 0x88DC, 0x9078, 0x629E, 0x0};

static IMLEName lename = {
    "canna", lename_string	/* LE id, HRN */
};

static IMLocale locales[] = {
    {"ja", jahrn_string},	/* locale id, HRN */
    {NULL, NULL},
};

/*
  Desktop data:
 */

#if 0
typedef struct
{
} CannaLEDesktop;
#endif

/*
  Session data:
*/

static int canna_context_id_counter = 1;

typedef struct _CannaDesktopData CannaDesktopData;
typedef struct _CannaSessionData CannaSessionData;

struct _CannaDesktopData
{
	int drop_priv;
};

#define SET_SESSION_FLAGS(X,Y) ((canna_session_data(X))->flags |= Y)
#define UNSET_SESSION_FLAGS(X,Y) ((canna_session_data(X))->flags &= ~(Y))
#define SESSION_FLAGS_IS_SET(X,Y) ((canna_session_data(X))->flags & Y)

typedef enum   {
	PALETTE = 1 << 0
}CannaLEFlags;

struct _CannaSessionData
{
    int context_id;
    jrKanjiStatusWithValue ksv;
    int flags;
    int conversion_start;
    int conversion_mode;
};

static void canna_change_mode(iml_session_t *s,int id);
static Bool canna_drop_privilege (const char *username);

CannaSessionData*
canna_session_data(iml_session_t *s)
{
    return (CannaSessionData*)(s->specific_data);
}

int
canna_session_context(iml_session_t *s)
{
    return ((CannaSessionData*)(s->specific_data))->context_id;
}

jrKanjiStatusWithValue*
canna_session_status(iml_session_t *s)
{
    return &((CannaSessionData*)(s->specific_data))->ksv;
}

/*
  LEIF operations.
*/

static void
send_commit(iml_session_t *s, IMText *p, int executep)
{
    iml_inst *lp;

    lp = s->If->m->iml_make_commit_inst(s, p);
    s->If->m->iml_execute(s, &lp);
    return;
}

/*
  Canna operations.
*/

static char*
canna_init_filename(char *user)
{
    char *buf;
    int ipsize;
    struct passwd *pw;

    if (!user)
	    return NULL;
    setpwent ();
    if ((pw = getpwnam(user)) == NULL) {
	    endpwent ();
	    return NULL;
    }
    ipsize = strlen(pw->pw_dir);
    buf = (char*) malloc((ipsize + 2) * sizeof(char)
			 + sizeof(canna_le_init_filename));
    if (ipsize < 1) return NULL;
    strcpy (buf, pw->pw_dir);
    buf[ipsize] = '/';
    buf[ipsize + 1] = '\0';
    strcat(buf, canna_le_init_filename);

    endpwent();

    /* check whether the file is readable. */
    if (access (buf, R_OK) != 0) {
	    free (buf);
	    return NULL;
    }

    fprintf (stderr, "%s\n", buf);
    return buf;
}

Bool
canna_init(char *user)
{
    char **warning = NULL;
    char *init_filename;        

    init_filename = canna_init_filename(user);
    if (init_filename) {
	jrKanjiControl(0, KC_SETINITFILENAME, init_filename);
	free(init_filename);
    }
    jrKanjiControl(0, KC_INITIALIZE, (char *) &warning);

    if (warning) {
	char          **p;

	for (p = warning; *p; p++)
	    fprintf(stderr, "htt: canna.so: %s\n", *p);

	return False;
    }

    jrKanjiControl(0, KC_SETAPPNAME, (char *) "iiimf-canna");
    /* undefined keys is sent to an application. */
    jrKanjiControl (0, KC_SETUNDEFKEYFUNCTION, (char*)kc_through);

#if 1
    /* set user info */
    if (user) {
	    jrUserInfoStruct info;
	    memset (&info, 0, sizeof (info));
	    info.uname = user;
	    jrKanjiControl (0, KC_SETUSERINFO, (char *)&info);
    }
#endif
    canna_context_id_counter = 1;

    return True;
}

IMText*
canna_commit_string(iml_session_t *s, char *buf)
{
    IMText *p;
    int len = strlen(buf);
    
    p = im_string_to_IMText(s, 1, &len, (unsigned char**) &buf, NULL);
    return p;
}

IMText*
canna_kakutei(iml_session_t *s)
{
    jrKanjiStatusWithValue *pksv;

    pksv = canna_session_status(s);
    jrKanjiControl(canna_session_context(s), KC_KAKUTEI, (char*) pksv);
    return canna_commit_string(s, pksv->buffer);
}

static void
canna_status_draw_off (iml_session_t *s)
{
    iml_inst *lp;
    iml_inst *rrv = NULL;
    IMText *p;
    unsigned char *str = "Canna(OFF)";
    int len = strlen(str);
    int ft1 = IM_Feedback_Normal;

    p = im_string_to_IMText(s, 1, &len, &str, &ft1);
    if (!IS_REGION_ACTIVE (s, STATUS)) {
            lp = s->If->m->iml_make_status_start_inst (s);
            s->If->m->iml_link_inst_tail (&rrv, lp);
    }
    lp = s->If->m->iml_make_status_draw_inst (s, p);
    s->If->m->iml_link_inst_tail (&rrv, lp);
    s->If->m->iml_execute (s, &lp);

    im_aux_mode_change (s,CANNA_MODE_AlphaMode);
    return;
}

void
canna_status_draw(iml_session_t *s)
{
    iml_inst *lp;
    iml_inst *rrv = NULL;
    IMText *p;
    CannaSessionData *pcls = canna_session_data(s);
    char *currentMode;

    jrKanjiStatusWithValue *pksv;
    pksv = canna_session_status(s);
    if (pcls->conversion_start == False) {
        canna_status_draw_off (s);
        return;
    }

    currentMode = alloca(jrKanjiControl(canna_session_context(s),KC_QUERYMAXMODESTR,0));
    jrKanjiControl(canna_session_context(s),KC_QUERYMODE,currentMode);

    {
	/* Create IMText with feedback. */
	int len;
	unsigned char *str;
	int ft1;
	str = currentMode;
	len = strlen(str);
	ft1 = IM_Feedback_Normal;
	p = im_string_to_IMText(s, 1, &len, &str, &ft1);
    }

    if (!IS_REGION_ACTIVE (s, STATUS)) {
        lp = s->If->m->iml_make_status_start_inst(s);
        s->If->m->iml_link_inst_tail(&rrv, lp);
    }
    lp = s->If->m->iml_make_status_draw_inst(s, p);
    s->If->m->iml_link_inst_tail(&rrv, lp);
    
    s->If->m->iml_execute(s, &rrv);


    im_aux_statusline_draw (s,currentMode);
}

static void
canna_status_done (iml_session_t *s)
{
    iml_inst *lp;
    if (IS_REGION_ACTIVE(s, STATUS)) {
            lp = s->If->m->iml_make_status_done_inst (s);
            s->If->m->iml_execute (s, &lp);
    }
    return;
}

void
canna_preedit_draw(iml_session_t *s)
{
    iml_inst *lp;
    iml_inst *rrv = NULL;
    IMText *p;
    jrKanjiStatus *pks = canna_session_status(s)->ks;

    /* When KanjiStatus is uninited, return immediately. */
    if (!pks->echoStr) return;

    if (!IS_REGION_ACTIVE (s, PREEDIT)) {
        lp = s->If->m->iml_make_preedit_start_inst(s);
        s->If->m->iml_link_inst_tail(&rrv, lp);
    }

    {
	/* Create IMText with feedbacks.  */
	int nb[3], fts[3];
	unsigned char *strs[3];
	nb[0] = pks->revPos;
	nb[1] = pks->revLen;
	nb[2] = pks->length - nb[0] - nb[1];
	fts[0] = IM_Feedback_Input;
	fts[1] = IM_Feedback_Strong;
	fts[2] = IM_Feedback_Input;
	strs[0] = pks->echoStr;
	strs[1] = strs[0] + pks->revPos;
	strs[2] = strs[1] + pks->revLen;
	p = im_string_to_IMText(s, 3, nb, strs, fts);
    }

    lp = s->If->m->iml_make_preedit_draw_inst(s, p);
    s->If->m->iml_link_inst_tail(&rrv, lp);
    s->If->m->iml_execute(s, &rrv);
}

void
canna_preedit_done(iml_session_t *s)
{
    if (IS_REGION_ACTIVE(s, PREEDIT)) {
	iml_inst *lp;
        lp = s->If->m->iml_make_preedit_done_inst(s);
	s->If->m->iml_execute(s, &lp);
    }
}


/*
  Caution!!!
  This part assumes the structure of guidline given by canna UI
  library.  It parses guideline string without any protocol on it.
  Therefore, it may not work with the future version of canna
  UI library!
*/
Bool
canna_parse_guideline(iml_session_t *s, int *pnum,
		      unsigned char ***psegs,
		      int **pnb,
		      int *pcurrent)
{
#if 1
    char linestr[1024];
#endif
    jrKanjiStatus *pks = canna_session_status(s)->ks;
    unsigned char *str = pks->gline.line;
    unsigned char *p, *st;
    int i, idx, tot, delimiterp;

#if 0
    fprintf (stderr, "%s\n", str);
#endif
    tot = 0;
    for (p = str, st = NULL;*p;p = CANNA_NEXT_CHAR(p)) {
	delimiterp = CANNA_GUIDELINE_DELIMITER_P(p);
	if (st && delimiterp) {
	    tot++;
	    st = NULL;
	}else if (!st && !delimiterp) {
	    st = p;
	}
    }
    *pnum = tot;
    *pcurrent = 0;
    *psegs = (unsigned char**) malloc(sizeof(unsigned char*) * tot * 2);
    *pnb = (int*) malloc(sizeof(int) * tot * 2);
    for (p = str, i = 0, idx = 0, st = NULL;
	 (idx < tot);p = CANNA_NEXT_CHAR(p)) {
	delimiterp = CANNA_GUIDELINE_DELIMITER_P(p);
	if (st && delimiterp) {
	    /* the size of the value */
	    (*pnb)[i] = (p - st);
	    i++;
	    idx++;
	    st = NULL;
	}else if (!st && !delimiterp) {
	    /* label */
	    (*psegs)[i] = st = p;
	    (*pnb)[i] = CANNA_NEXT_CHAR(p) - p;
	    i++;
	    if (pks->gline.revPos == (p - str))
		*pcurrent = idx;
	    /* value */
	    (*psegs)[i] = CANNA_NEXT_CHAR(p);
	}
    }
#if 1
    for (i = 0;i < (tot * 2);i++) {
	memcpy(linestr, (*psegs)[i], (*pnb)[i]);
	linestr[(*pnb)[i]] = '\0';
	fprintf(stderr, "Seg(%d):%s\n", i, linestr);
    }
#endif
    return True;
}

void
canna_start_lookup_choice(iml_session_t *s,
			  iml_inst **prrv, int num)
{
    if (!IS_REGION_ACTIVE(s, LOOKUP)) {
	iml_inst *lp;
	IMLookupStartCallbackStruct *start;
        start = ((IMLookupStartCallbackStruct *)
		 s->If->m->iml_new(s, sizeof(IMLookupStartCallbackStruct)));
	start->whoIsMaster = IMIsMaster;
        start->IMPreference = (LayoutInfo *) s->If->m->iml_new(s, sizeof(LayoutInfo));
	memset(start->IMPreference, 0, sizeof(LayoutInfo));

        start->IMPreference->choice_per_window = num;
        start->IMPreference->ncolumns = 1;
        start->IMPreference->nrows = num;
        start->IMPreference->drawUpDirection = DrawUpHorizontally;
        start->IMPreference->whoOwnsLabel = IMOwnsLabel;
        start->CBPreference = NULL;

        lp = s->If->m->iml_make_lookup_start_inst(s, start);
	s->If->m->iml_link_inst_tail(prrv, lp);
    }
}

void
canna_show_lookup_choice(iml_session_t *s)
{
    int num;
    iml_inst *lp;
    iml_inst *rrv = NULL;
    IMLookupDrawCallbackStruct *draw;
    jrKanjiStatus *pks = canna_session_status(s)->ks;

    /* When KanjiStatus is uninited, return immediately. */
    if (!pks->gline.line) return;

    draw = ((IMLookupDrawCallbackStruct *)
	    s->If->m->iml_new(s, sizeof(IMLookupDrawCallbackStruct)));
    memset(draw, 0, sizeof(IMLookupDrawCallbackStruct));
    draw->title = im_UTFCHAR_to_IMText(s, lookup_choice_title);

    /* set choices */
    {
	int i, cid;
	unsigned char **ps;
	int *segs;
        IMText *pvt;
        IMText *plt;
	int max_len = 0;
	
	if (!canna_parse_guideline(s, &num, &ps, &segs, &cid))
	    return;

	draw->index_of_first_candidate = 0;
	draw->index_of_last_candidate = num - 1;
	draw->n_choices = num;
	draw->choices = ((IMChoiceObject *)
			 s->If->m->iml_new(s, num * sizeof(IMChoiceObject)));
	memset(draw->choices, 0, num * sizeof(IMChoiceObject));
	draw->index_of_current_candidate = cid;

	for (cid = 0, i = 0;i < num;i++) {
	    plt = draw->choices[i].label
		= im_string_to_IMText(s, 1, &segs[cid], &ps[cid], NULL);
	    cid++;
	    pvt = draw->choices[i].value
		= im_string_to_IMText(s, 1, &segs[cid], &ps[cid], NULL);
	    cid++;
	    if (max_len < pvt->char_length)
		max_len = pvt->char_length;
	    if (max_len < plt->char_length)
		max_len = plt->char_length;
	}
	free(ps);
	free(segs);
	draw->max_len = max_len;
#if 0
	fprintf(stderr, "draw->index_of_first_candidate=%x\n",
		draw->index_of_first_candidate);
	fprintf(stderr, "draw->index_of_last_candidate=%x\n",
		draw->index_of_last_candidate);
	fprintf(stderr, "draw->n_choices=%x\n",
		draw->n_choices);
	fprintf(stderr, "draw->choices=%x\n",
		draw->choices);
	fprintf(stderr, "draw->choices->label=%x\n",
		draw->choices->label);
	fprintf(stderr, "draw->max_len=%x\n", max_len);
	fprintf(stderr, "draw->index_of_current_candidate=%x\n",
		draw->index_of_current_candidate);
#endif
    }
    canna_start_lookup_choice(s, &rrv, num);
    lp = s->If->m->iml_make_lookup_draw_inst(s, draw);
    s->If->m->iml_link_inst_tail(&rrv, lp);
    s->If->m->iml_execute(s, &rrv);
}

void
canna_lookup_choice_done(iml_session_t *s)
{
    if (IS_REGION_ACTIVE (s, LOOKUP)) {
	iml_inst *lp;
        lp = s->If->m->iml_make_lookup_done_inst(s);
	s->If->m->iml_execute(s, &lp);
    }
}

static void
canna_start_conversion (iml_session_t *s)
{
        iml_inst *lp;
        lp = s->If->m->iml_make_start_conversion_inst (s);
        s->If->m->iml_execute (s, &lp);
        return;
}

static void
canna_end_conversion (iml_session_t *s)
{
        iml_inst *lp;
        lp = s->If->m->iml_make_end_conversion_inst (s);
        s->If->m->iml_execute (s, &lp);
        return;
}

void
canna_make_conversion_off(iml_session_t *s)
{
    CannaSessionData *pcls = canna_session_data (s);
    if (pcls->conversion_start == True) {
        pcls->conversion_start = False;

        canna_lookup_choice_done(s);
        canna_preedit_done(s);
        canna_status_draw(s);
        canna_status_done(s);
    }
}

void
canna_make_conversion_on(iml_session_t  *s)
{
    CannaSessionData *pcls = canna_session_data (s);
    if (pcls->conversion_start == False) {
        pcls->conversion_start = True;
        if (pcls->conversion_mode == CANNA_MODE_AlphaMode) {
            canna_change_mode (s,CANNA_MODE_HenkanMode);
            pcls->conversion_mode = CANNA_MODE_HenkanMode;
        }
        im_aux_mode_change (s,pcls->conversion_mode);
        canna_status_draw(s);
    }
}

int
canna_translate_keyevent(iml_session_t *s,IMKeyListEvent *kev)
{
    IMKeyEventStruct *k = (IMKeyEventStruct *) kev->keylist;
    fprintf(stderr,"iml_session_t() keycode=%x,keychar=%x, state=%x\n",
	   k->keyCode, k->keyChar, k->modifier);

    switch(k->keyCode) {
    case IM_VK_UP:
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Up;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Up;
	}
	return  CANNA_KEY_Up;

    case IM_VK_DOWN:
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Down;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Down;
	}
	return  CANNA_KEY_Down;

    case IM_VK_LEFT:
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Left;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Left;
	}
	return  CANNA_KEY_Left;

    case IM_VK_RIGHT:
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Right;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Right;
	}
	return  CANNA_KEY_Right;

    case IM_VK_INSERT:
	return CANNA_KEY_Insert;

    case IM_VK_PAGE_UP:
	return CANNA_KEY_Rolldown;

    case IM_VK_PAGE_DOWN:
	return CANNA_KEY_Rollup;

    case IM_VK_HOME:
	return CANNA_KEY_Home;

    case IM_VK_HELP:
	return CANNA_KEY_Help;

    case IM_VK_F1:
    case IM_VK_F2:
    case IM_VK_F3:
    case IM_VK_F4:
    case IM_VK_F5:
    case IM_VK_F6:
    case IM_VK_F7:
    case IM_VK_F8:
    case IM_VK_F9:
    case IM_VK_F10:
    case IM_VK_F11:
    case IM_VK_F12:
	/* CANNA_KEY_PF1 */
	return CANNA_KEY_F1 + k->keyCode - IM_VK_F1;

    case IM_VK_CONVERT: /* XFER */
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Xfer;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Xfer;
	}
	return CANNA_KEY_Xfer;

    case IM_VK_NONCONVERT: /* NFER */
	if (k->modifier & IM_CTRL_MASK) {
	    return CANNA_KEY_Cntrl_Nfer;
	} else if (k->modifier & IM_SHIFT_MASK) {
	    return CANNA_KEY_Shift_Nfer;
	}
	return CANNA_KEY_Nfer;

    case IM_VK_ENTER:
	return 0x0D; /* C-m */

    case IM_VK_BACK_SPACE:
	return IM_VK_BACK_SPACE;

    case IM_VK_SPACE:
	if ((k->modifier & (IM_SHIFT_MASK)) | (k->modifier & (IM_CTRL_MASK)))
            return IM_VK_O - IM_VK_A + 1; /* C-o .. ba-tari- */
//            return CANNA_KEY_Shift_Space;

    default:
        if ( IM_VK_A <= k->keyChar  && k->keyChar <= IM_VK_Z ){
	    if (k->modifier & IM_CTRL_MASK)
                return k->keyChar - IM_VK_A + 1; /* 1 == C-a */
        }
        if ( k->keyChar == IM_VK_AT ){
	    if (k->modifier & IM_CTRL_MASK)
                return 0; /* 0 == C-@ */
        }

	if ((k->keyChar > 0)
	    && (k->keyChar < 0xFFFF)) {
	    /* Should we translate it to EUC? */
	    return k->keyChar;
	}
    }

    fprintf(stderr, "translation failed:keycode=%x,keychar=%x, state=%x\n",
	    k->keyCode, k->keyChar, k->modifier);
    return 0xff;
}

Bool
canna_process_kanji_string (iml_session_t *s, int len, char* buf, jrKanjiStatus *ks)
{
    Bool ret = True;
    if (len > 0) {
        if (len == 1 && (ks->info & KanjiThroughInfo) &&
            (((buf[0] & 0x7f) < (unsigned)0x20 &&
               buf[0] != '\r' && buf[0] != '\t') ||
               buf[0] == '\177')) {
	    ret = False;
        } else {
            IMText *p;
            ks->info &= ~KanjiThroughInfo;
            buf[len] = '\0';
            p = canna_commit_string(s, buf);
            send_commit(s, p, 1);
        }
    }

    if (ks->length >= 0) 
        canna_preedit_draw(s);

    if (ks->info & KanjiGLineInfo) {
        if (ks->gline.length > 0)
            canna_show_lookup_choice(s);
        else
            canna_lookup_choice_done(s);
    }

    if (ks->info & KanjiModeInfo) {
        char modeinfo[4];

        jrKanjiControl(canna_session_context(s), KC_SETMODEINFOSTYLE, (char *)1);
        jrKanjiControl(canna_session_context(s), KC_QUERYMODE, (char *)modeinfo);
        jrKanjiControl(canna_session_context(s), KC_SETMODEINFOSTYLE, (char *)0);
        if (modeinfo[0] - '@' == CANNA_MODE_AlphaMode) {
            CannaSessionData *pcls = canna_session_data (s);
            canna_make_conversion_off(s);
            canna_end_conversion(s);
            pcls->conversion_mode = CANNA_MODE_AlphaMode;
        }
        canna_status_draw(s);
    }
    return ret;
}

void
canna_process_keyevent(iml_session_t *s, IMKeyListEvent *kev)
{
    int ch;
    iml_inst *lp;

    ch = canna_translate_keyevent(s,kev);

    if (ch != 0xff) {
	int size, n;
	char buf[CANNA_COMMIT_STRING_BUFSIZE + 1];

	jrKanjiStatus *pks;
	pks = canna_session_status(s)->ks;
	size = CANNA_COMMIT_STRING_BUFSIZE;
	n = jrKanjiString(canna_session_context(s),
			  ch, buf, size, pks);
        if (canna_process_kanji_string (s, n,buf,pks))
            return;
    }
    /* I don't process this keyevent.  Return it. */
    lp = s->If->m->iml_make_keypress_inst(s, ((IMKeyEventStruct *)
	  kev->keylist));
    s->If->m->iml_execute(s, &lp);
    return;
}

/*
  IF offer.
*/

void
if_GetIfInfo(IMArgList args, int num_args)
{
    int i;
    for (i = 0; i < num_args; i++, args++) {
        switch (args->id) {
	case IF_VERSION:
	    args->value = (IMArgVal) "1.2";
	    break;
	case IF_METHOD_TABLE:
	    args->value = (IMArgVal) &canna_methods;
	    break;
	case IF_LE_NAME:
	    args->value = (IMArgVal) &lename;
	    break;
	case IF_SUPPORTED_LOCALES:
	    args->value = (IMArgVal) &locales;
	    break;
	case IF_SUPPORTED_OBJECTS:
            args->value = (IMArgVal) im_aux_get_objects ();
	    break;
	case IF_NEED_THREAD_LOCK:
	    args->value = (IMArgVal) True;
	    break;
	default:
	    break;
	}
    }
}

/*
  IFs
*/

static Bool
if_canna_OpenIF(iml_if_t *If)
{

    return im_util_setup();
}

static Bool
if_canna_CloseIF(iml_if_t *If)
{
    return True;
}

static Bool
if_canna_GetIFValue(iml_if_t *If, IMArgList args, int num_args)
{
    return True;
}

static Bool
if_canna_SetIFValue(iml_if_t *If, IMArgList args, int num_args)
{
    return True;
}

static Bool
if_canna_OpenDesktop(iml_desktop_t * desktop,
		     IMArgList args,
		     int num_args)
{
    CannaDesktopData *d;
    fprintf (stderr, "Opendesktop\n");
    d = malloc (sizeof (CannaDesktopData));
    memset (d, 0, sizeof (CannaDesktopData));
    fprintf (stderr, "%p\n", desktop);
    d->drop_priv = canna_drop_privilege (desktop->user_name);
    desktop->specific_data = (void *)d;
    fprintf (stderr, "Opendesktop %s\n", desktop->user_name);
    if (canna_init (d->drop_priv ? desktop->user_name: NULL)) {
    fprintf (stderr, "%p\n", desktop);
	if (!jrKanjiControl(0, KC_QUERYCONNECTION, (char *) 0)) {
	    fprintf(stderr,
		"htt: canna.so: Unable to connect with canna server.\n");
	    return False;
	}
    }
    fprintf (stderr, "Opendesktop\n");
    return True;
}

static Bool
if_canna_CloseDesktop(iml_desktop_t * desktop)
{
    CannaDesktopData *d = (CannaDesktopData *)desktop->specific_data;
    free (d);
    jrKanjiControl(0, KC_FINALIZE, (char *) 0);
    return True;
}

static Bool
if_canna_CreateSC(iml_session_t *s, IMArgList args, int num_args)
{
    CannaSessionData *pcls = (CannaSessionData*) malloc(sizeof(CannaSessionData));
    jrKanjiStatus *pks = (jrKanjiStatus*) malloc(sizeof(jrKanjiStatus));
    unsigned char *buf;
    buf = (unsigned char *) malloc(CANNA_COMMIT_STRING_BUFSIZE);
    if ((!pcls) || (!pks) || (!buf)) return False;
    pcls->ksv.ks = pks;
    pcls->ksv.buffer = buf;
    buf[0] = '\0';
    pcls->ksv.bytes_buffer = CANNA_COMMIT_STRING_BUFSIZE;
    pcls->context_id = canna_context_id_counter++;
    pcls->conversion_start = False;
    pcls->conversion_mode = CANNA_MODE_AlphaMode;
    pcls->flags = 0;
    /* Init jrKanjiStatus variable with 0. */
    memset(pks, 0, sizeof(jrKanjiStatus));

    s->specific_data = (void*) pcls;
    return True;
}

static Bool
if_canna_DestroySC(iml_session_t *s)
{
    CannaSessionData *pcls;
    pcls = canna_session_data(s);
    jrKanjiControl(canna_session_context(s),
		   KC_CLOSEUICONTEXT,
		   (char*) canna_session_status(s));
    free(pcls->ksv.buffer);
    free(pcls->ksv.ks);
    free(pcls);
    return True;
}

static IMText*
if_canna_ResetSC(iml_session_t *s)
{
    iml_inst *lp;
    IMText *p;

    /* erase preedit. */
    lp = s->If->m->iml_make_preedit_erase_inst(s);
    s->If->m->iml_execute(s, &lp);

    /* fix the current string. (kakutei) */
    p = canna_kakutei(s);

    if (p->char_length) return p;
    return (IMText*) NULL;
}

static void
canna_change_mode(iml_session_t *s,int id)
{
    CannaSessionData *pcls = canna_session_data(s);
    jrKanjiStatusWithValue *pksv;
    pksv = canna_session_status(s);

    pksv->val = id;
    jrKanjiControl(canna_session_context(s),
      		   KC_CHANGEMODE, (char*) pksv);
    pcls->conversion_mode = id;
}

static void
canna_aux_mode_add_id(iml_session_t *s,int id)
{
    jrKanjiStatusWithValue *pksv;
    canna_change_mode(s,id);
    pksv = canna_session_status(s);
    im_aux_menu_add (s,id,pksv->ks->mode);
}

static void
aux_mode_change_cb (iml_session_t *s,int id)
{
    if (id == CANNA_MODE_AlphaMode){
        canna_change_mode (s,CANNA_MODE_AlphaMode);
        canna_make_conversion_off (s);
        canna_end_conversion (s);
    } else {
        canna_change_mode (s,id);
        canna_status_draw (s);
        im_aux_mode_change (s,id);
    }
}

static void
aux_code_defined_cb (iml_session_t *s,IMText* str)
{
    send_commit (s,str,1);
}

static Bool
if_canna_SetSCValue(iml_session_t *s, IMArgList args, int num)
{
    int i;
    IMArg *p = args;

    for (i = 0; i < num; i++, p++) {
        switch (p->id) {
	case SC_TRIGGER_ON_NOTIFY:
	    canna_make_conversion_on(s);
	    break;
	case SC_TRIGGER_OFF_NOTIFY:
	    canna_make_conversion_off(s);
	    break;
	case SC_REALIZE:
            if (s->desktop->session_count == 1) {
                    IMAuxCallbacks callbacks = {aux_mode_change_cb,aux_code_defined_cb,NULL};

                    im_aux_setup (s,&callbacks);

            	    im_aux_menu_add (s,CANNA_MODE_AlphaMode,"OFF");
                    canna_aux_mode_add_id (s,CANNA_MODE_HenkanMode);
                    canna_aux_mode_add_id (s,CANNA_MODE_ZenHiraKakuteiMode);
                    canna_aux_mode_add_id (s,CANNA_MODE_ZenKataKakuteiMode);
                    canna_aux_mode_add_id (s,CANNA_MODE_HanKataKakuteiMode);
                    canna_aux_mode_add_id (s,CANNA_MODE_ZenAlphaKakuteiMode);
                    canna_aux_mode_add_id (s,CANNA_MODE_HanAlphaKakuteiMode);
                    canna_change_mode(s,CANNA_MODE_AlphaMode);
            }
	    break;
	case SC_LOOKUP_LABELTYPE:
	    break;
	default:
	    break;
	}
    }

    return True;
}

static Bool
if_canna_GetSCValue(iml_session_t *s, IMArgList args, int num_args)
{
    int i;
    IMArg *p = args;

    /* Canna uses at least LATIN, HIRAGANA, KATAKANA,
       and KANJI scripts.
       That's all to it? */
    static int charsubset[] = {
        67,			/* LATIN */
        47,			/* HIRAGANA */
        48,			/* KATAKANA */
        71,			/* KANJI */
        0
    };

    for (i = 0; i < num_args; i++, p++) {
        switch (p->id) {
            case SC_SUPPORTED_CHARACTER_SUBSETS:
                /* specify CHARACTER_SUBSETS */
                p->value = (IMArgVal) charsubset;
                break;
            default:
                break;
            }
    }
    return True;
}

static void
if_canna_SetSCFocus(iml_session_t *s)
{
    canna_status_draw(s);
}

static void
if_canna_UnsetSCFocus(iml_session_t *s)
{
    /* do nothing. */
}

static void
if_canna_SendEvent(iml_session_t *s, IMInputEvent *ev)
{
    if (ev) {
	switch (ev->type) {
	case IM_EventKeyList:
	    canna_process_keyevent(s, (IMKeyListEvent*) ev);
	    break;
	case IM_EventAux:
            im_aux_process_event (s,((IMAuxEvent *) ev)->aux);
            break;
	case IM_EventString:
	case IM_EventText:
	default:
	    break;
	}
    }
    return;
}

static Bool
canna_drop_privilege (const char *username)
{
    struct passwd *pw;
    uid_t uid;
    gid_t gid;
    if (!username)
        goto error;
    if ((pw = getpwnam (username)) == NULL) {
        goto error;
    } else {
        uid = pw->pw_uid;
        gid = pw->pw_gid;
        if (uid < 500)
            goto error;
    }
    if (setregid (gid, gid) < 0) {
#ifdef DEBUG
        printf ("setregid fail %s\n", strerror (errno));
#endif
        return False;
    }
    if (setreuid (uid, uid) < 0) {
#ifdef DEBUG
        printf ("setreuid fail %s\n", strerror (errno));
#endif
        return False;
    }
    return True;

error:
    if ((pw = getpwnam ("nobody")) != NULL) { 
        gid = pw->pw_gid;
        uid = pw->pw_uid;
        if (setregid (gid, gid) < 0) {
#ifdef DEBUG
            printf ("setregid fail %s\n", strerror (errno));
#endif
		}
		if (setreuid (uid, uid) < 0) {
#ifdef DEBUG
			printf ("setreuid fail %s\n", strerror (errno));
#endif
		}
	} else {
		/* not reached */
		assert (0);
	}
	return False;
}
