/*
 * IIIMF-Canna, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Shingo Akagaki <dora@momonga-linux.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: codetable.c,v 1.8.2.1 2003/08/01 13:47:53 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <string.h>
#include <wchar.h>
#include <stdio.h>
#include <stdlib.h>
#include <X11/Xmd.h>
#include "canna_xaux.h"
#include "support.h"
#include "codetable.h"
#include "codetable_eucjp.h"
#include "config.h"

#ifndef AUX_CODE_DEBUG
#include "xaux_common.h"
#include "xaux_ext_common.h"
#endif

extern void canna_aux_palette_send_engine (gint int_count, gint *integers, 
		gint str_count, gchar **strings);

static void for_each (GtkWidget *widget, gpointer user_data);
static gboolean conv_table_delete (GtkWidget *widget, gpointer user_data);
static void next_button_clicked (GtkWidget *widget, gpointer user_data);
static void prev_button_clicked (GtkWidget *widget, gpointer user_data);
static void close_clicked (GtkWidget *widget, gpointer user_data);
static void table_clicked (GtkWidget *widget, gpointer user_data);
static void code_button_clicked (GtkWidget *widget, gpointer user_data);
static GdkFilterReturn ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data);
static CodeTable *lookup_table(void);

static GdkFilterReturn
ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data)
{
	XEvent *xevent;
	xevent = (XEvent *) gdk_xevent;
	if (xevent->type == ClientMessage) {
#ifndef AUX_CODE_DEBUG
		xaux_ext_process_client_message (GDK_DISPLAY (), (XClientMessageEvent *) xevent);
#endif
		return GDK_FILTER_REMOVE;
	} else if (xevent->type == KeyRelease) {
		return GDK_FILTER_REMOVE;
	}
	return GDK_FILTER_CONTINUE;
}

static gboolean
conv_table_delete (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *button;
	CodeTable *table = (CodeTable*) gtk_object_get_data (
			GTK_OBJECT (widget), "cur_table");
	/* Free */
	if (table)
		g_free (table);
	button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (widget), "next_button");
	if (button)
		gtk_widget_set_sensitive (button, FALSE);
	button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (widget), "prev_button");
	if (button)
		gtk_widget_set_sensitive (button, FALSE);
	gtk_object_remove_data (GTK_OBJECT (widget), "cur_table");
	gtk_object_remove_data (GTK_OBJECT (widget), "cur_page");
	gtk_widget_hide (widget);
	return TRUE;
}

static void
next_button_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *toplevel;
	GtkWidget *prev_button;
	CodeTable *table;
	GtkWidget *t;
	gint cur_page;
	gint store;

	toplevel = gtk_widget_get_toplevel (widget);
	table = (CodeTable *) gtk_object_get_data (GTK_OBJECT (toplevel), "cur_table");
	t = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (toplevel), "table");
	cur_page = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (toplevel),
				"cur_page"));
	cur_page++;

	if (!table)
		return;
	store = table->start;
	table->start += (0xff + 1) * (cur_page - 1);
	gtk_container_foreach (GTK_CONTAINER (t), for_each, table);
	table->start = store;
	if (table->page == cur_page) {
		gtk_widget_set_sensitive (widget, FALSE);
	}
	/* re-store */
	gtk_object_set_data (GTK_OBJECT (toplevel), "cur_page", GINT_TO_POINTER (cur_page));
	gtk_object_set_data (GTK_OBJECT (toplevel), "cur_table", (gpointer) table);

	prev_button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (toplevel),
			"prev_button");
	if (!prev_button)
		return;
	gtk_widget_set_sensitive (prev_button, TRUE);
	return;
}

static void
prev_button_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *toplevel;
	GtkWidget *next_button;
	CodeTable *table;
	GtkWidget *t;
	gint cur_page;
	gint store;

	/* get toplevel */
	toplevel = gtk_widget_get_toplevel (widget);
	/* get table */
	table = (CodeTable*) gtk_object_get_data (GTK_OBJECT (toplevel), "cur_table");
	/* get t */
	t = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (toplevel), "table");
	/* get cur page */
	cur_page = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (toplevel),
				"cur_page"));
	/* decrement it */
	cur_page--;
	if (!table)
		return;
	store = table->start;
	table->start += (0xff + 1) * (cur_page - 1);
	gtk_container_foreach (GTK_CONTAINER (t), for_each , table);
	table->start = store;
	if (cur_page == 1) {
		/* we don't have prev page any more */
		gtk_widget_set_sensitive (widget, FALSE);
	}
	/* re-store */
	gtk_object_set_data (GTK_OBJECT (toplevel), "cur_page", GINT_TO_POINTER (cur_page));
	gtk_object_set_data (GTK_OBJECT (toplevel), "cur_table", (gpointer) table);

	next_button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (toplevel),
			"next_button");
	if (!next_button)
		return;
	gtk_widget_set_sensitive (next_button, TRUE);
	return;
}

static void
close_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *toplevel;
	toplevel = gtk_widget_get_toplevel (widget);
	conv_table_delete (toplevel, NULL);
	return;
}

static void
table_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *child; /* Label */
	gchar *label;
#ifndef AUX_CODE_DEBUG
	gchar *send_str[1];
	gint  send_int[1];
#endif

	child = GTK_BIN (widget)->child;
	if (!child)
		return;
	gtk_label_get (GTK_LABEL (child), &label);
#ifdef AUX_CODE_DEBUG
	if (label)
		g_message ("%s\n",label);
#else
	if (label && *label) {
#ifdef USE_GTK2
		send_str[0] = g_locale_from_utf8 (label, -1, NULL, NULL, NULL);
#else
		send_str[0] = g_strdup (label);
#endif
		send_int[0] = AUX_CODE_DEFINED;
		canna_aux_palette_send_engine (1, send_int, 1, send_str);
		g_free (send_str[0]);
	}
#endif
	return;
}

static void
for_each (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *child; /* Label */
	guchar *text;
	wchar_t wc;
	int result;
	CodeTable *table = (CodeTable *) user_data;
#ifdef USE_GTK2
	gchar *ustr;
#endif
	child = GTK_BIN (widget)->child;
	if (!child)
		return;
	if (!table)
		return;
	if (table->start > 0xffff) {
		text = g_strdup_printf ("%c%c%c", (table->start >> 16) & 0xff, 
				(table->start >> 8) & 0xff, table->start & 0xff);
	} else {
		text = g_strdup_printf ("%c%c", (table->start >> 8) & 0xff,
				table->start & 0xff);
	}
	table->start += 1;
	/* try to conver to wchar_t */
	result = mbtowc (&wc, text, MB_CUR_MAX);
	if (result > 0) {
#ifdef AUX_CODE_DEBUG
		printf ("%s\n",text);
#endif
#ifdef USE_GTK2
		ustr = g_locale_to_utf8 (text, -1, NULL, NULL, NULL);
		gtk_label_set_text (GTK_LABEL (child), ustr);
		g_free (ustr);
#else
		gtk_label_set_text (GTK_LABEL (child), text);
#endif
	} else {
		gtk_label_set_text (GTK_LABEL (child), NULL);
	}
	g_free (text);
}

static void
code_button_clicked (GtkWidget *widget, gpointer user_data)
{
	CodeTable *table = (CodeTable*) user_data;
	CodeTable *copy;
	GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
	GtkWidget *code_table_window;
	GtkWidget *button;
	GtkWidget *t;
	if (!table)
		return;
	code_table_window = gtk_object_get_data (GTK_OBJECT (toplevel), "code_table_window");
	if (!code_table_window)
		code_table_window = canna_aux_code_table_new ("FUGA");
	gtk_object_set_data (GTK_OBJECT (toplevel), "code_table_window", (gpointer) code_table_window);
	t = gtk_object_get_data (GTK_OBJECT (code_table_window), "table");
	if (!t)
		return;
	copy = g_new0 (CodeTable,1);
	memcpy (copy, table, sizeof (CodeTable));
	gtk_container_foreach (GTK_CONTAINER (t), for_each, copy);
	copy->start = table->start;
//	g_free (copy);

	if (table->page > 1) {
		button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (code_table_window), "next_button");
		if (!button)
			return;
		gtk_widget_set_sensitive (button, TRUE);
		gtk_object_set_data (GTK_OBJECT (code_table_window), "cur_page", GINT_TO_POINTER (1));
		gtk_object_set_data (GTK_OBJECT (code_table_window), "cur_table", (gpointer) copy);
	} else {
		button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (code_table_window), "next_button");
		gtk_widget_set_sensitive (button, FALSE);
		button = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (code_table_window), "prev_button");
		gtk_widget_set_sensitive (button, FALSE);
		g_free (copy);
		gtk_object_remove_data (GTK_OBJECT (code_table_window), "cur_page");
		gtk_object_remove_data (GTK_OBJECT (code_table_window), "cur_table");
	}

	gtk_widget_show_all (code_table_window);
	gtk_widget_hide (toplevel);
	return;
}

GtkWidget *
canna_aux_code_table_new (const gchar *label)
{
	GtkWidget *toplevel;
	GtkWidget *frame;
	GtkWidget *table;
	GtkWidget *button;
	GtkWidget *vbox;
	GtkWidget *buttonbox;
	XWMHints hints;
	int x,y;

	toplevel = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_position (GTK_WINDOW (toplevel),
			GTK_WIN_POS_MOUSE);

	gtk_container_set_border_width (GTK_CONTAINER (toplevel), 3);
	gtk_window_set_policy (GTK_WINDOW (toplevel), FALSE, FALSE, FALSE);
	gtk_window_set_title (GTK_WINDOW (toplevel), _("Code Table"));

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (toplevel), vbox);

	frame = gtk_frame_new (label);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

	table = gtk_table_new (8, 16, TRUE);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_container_set_border_width (GTK_CONTAINER (table), 2);
	for ( y = 8 ; y > 0 ; y--) {
		for ( x = 16; x > 0; x--) {
			button = gtk_button_new_with_label ("");
			gtk_table_attach_defaults (GTK_TABLE (table), button, x - 1, x, y - 1, y);
			gtk_signal_connect (GTK_OBJECT (button), "clicked",
					GTK_SIGNAL_FUNC (table_clicked), NULL);
		}
	}
	gtk_object_set_data (GTK_OBJECT (toplevel), "table", (gpointer)table);
	gtk_widget_show_all (frame);

	buttonbox = gtk_hbutton_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (buttonbox), 5);
	gtk_box_pack_start (GTK_BOX (vbox), buttonbox, FALSE, FALSE, 0);
//	gtk_button_box_set_layout (GTK_BUTTON_BOX (buttonbox), GTK_BUTTONBOX_SPREAD);
//
	button = gtk_button_new_with_label (_("<<"));
	gtk_container_add (GTK_CONTAINER (buttonbox), button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_object_set_data (GTK_OBJECT (toplevel), "prev_button", (gpointer) button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked", 
			GTK_SIGNAL_FUNC (prev_button_clicked), NULL);

	button = gtk_button_new_with_label (_(">>"));
	gtk_container_add (GTK_CONTAINER (buttonbox), button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_object_set_data (GTK_OBJECT (toplevel), "next_button", (gpointer) button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked", 
			GTK_SIGNAL_FUNC (next_button_clicked), NULL);

	button = gtk_button_new_with_label (_("Close"));
	gtk_container_add (GTK_CONTAINER (buttonbox), button);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (close_clicked), NULL);

	/* do not delete */
	gtk_signal_connect (GTK_OBJECT (toplevel), "delete_event", 
			GTK_SIGNAL_FUNC (conv_table_delete), NULL);
	gtk_widget_realize (toplevel);
	gdk_window_add_filter (toplevel->window, ext_event_handler, 0);
	/* Set Hints */
	memset (&hints, 0, sizeof (XWMHints));
	hints.flags = InputHint;
	hints.input = False;
	XSetWMHints (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (toplevel->window), &hints);
	return toplevel;
}

static CodeTable*
lookup_table (void)
{
	return &euc_jp[0];
}

GtkWidget *
canna_aux_code_new (void)
{
	GtkWidget *toplevel;
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *button;
	XWMHints hints;
	CodeTable *find_table;
	gint i = 0;

	
	toplevel = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW(toplevel), _("Code Table"));
	gtk_window_set_position (GTK_WINDOW (toplevel),
			GTK_WIN_POS_MOUSE);
	gtk_window_set_policy (GTK_WINDOW (toplevel), FALSE, FALSE, FALSE);

	frame = gtk_frame_new (_("Code Table"));
	gtk_widget_show (frame);
	gtk_container_add (GTK_CONTAINER (toplevel), frame);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	find_table = lookup_table();
	if (find_table) {
		for (; find_table[i].label != NULL ; i++) {
			button = gtk_button_new_with_label (_(find_table[i].label));
			gtk_widget_show (button);
			gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
			gtk_signal_connect (GTK_OBJECT (button), "clicked",
					GTK_SIGNAL_FUNC (code_button_clicked), (gpointer) &find_table[i]);
		}
	}

	/* do not destroy */
	gtk_signal_connect (GTK_OBJECT (toplevel), "delete_event",
			GTK_SIGNAL_FUNC (gtk_widget_hide_on_delete), NULL);
	gtk_widget_realize (toplevel);
	/* set hints */
	memset (&hints, 0, sizeof (XWMHints));
	hints.flags = InputHint;
	hints.input = False;
	XSetWMHints (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (toplevel->window), &hints);
	return toplevel;
}

#ifdef AUX_CODE_DEBUG
int 
main (int argc, char *argv[])
{
	GtkWidget *toplevel;
	/* force to EUC-JP */
#ifdef HAVE_SETENV
	setenv ("LC_ALL","ja_JP.EUC-JP",TRUE);
#elif HAVE_PUTENV
	putenv ("LC_ALL=ja_JP.EUC-JP");
#endif
	gtk_set_locale ();

#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef USE_GTK2
	bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif
	textdomain (PACKAGE); 
#endif
	gtk_init (&argc, &argv);
	toplevel = canna_aux_code_new ();
	gtk_widget_show (toplevel);
	gtk_main ();
	return 0;
}
#endif
