/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#if defined(ENABLE_NLS) && defined(HAVE_GETTEXT)
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif

const int istep=5000;

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef ENABLE_NLS
#include <locale.h>
#endif


#if HAVE_LINUX_CDROM_H

#include <linux/cdrom.h>

static int fd;

int CDOpen(char* dev){
  fd=open(dev,O_RDONLY);
  return (fd>=0);
};

int CDClose(){
  if (fd>=0) close(fd);
};

int CDReadMSF(int min,int sec,int frame,char* buf){
  struct cdrom_msf *msf;
  msf=(struct cdrom_msf*)buf;
  msf->cdmsf_min0=min;
  msf->cdmsf_sec0=sec;
  msf->cdmsf_frame0=frame;
  msf->cdmsf_min1=min;
  msf->cdmsf_sec1=sec;
  msf->cdmsf_frame1=frame;
  return ioctl(fd,CDROMREADRAW,msf);
}

int CDReadLBA(int lba,char* buf){
  int min,sec,frame;
  lba+=150;
  frame=lba%75;
  lba/=75;
  min=lba/60;
  sec=lba%60;
  return CDReadMSF(min,sec,frame,buf);
}
#endif

/* For FreeBSD */
#ifdef HAVE_SYS_CDIO_H
#include <sys/cdio.h>

static int fd=-1;
static struct ioc_read_audio cdio;

int CDOpen(char* dev){
  fd=open(dev,O_RDONLY);
  cdio.buffer=malloc(2352*256);
  return (fd>=0);
};

int CDClose(){
  if (fd>=0) {
    close(fd);
    free(cdio.buffer);
    fd=-1;
  } else {
    fprintf(stderr,_("CDIO: CD close error\n"));
  }
  return 1;
};

int CDReadMSF(int min,int sec,int frame,char* buf){
  int r;
  cdio.address_format=CD_MSF_FORMAT;
  cdio.address.msf.minute=min;
  cdio.address.msf.second=sec;
  cdio.address.msf.frame=frame;
  cdio.nframes=1;
  r=ioctl(fd,CDIOCREADAUDIO,&cdio);
  if (r<0) {
    fprintf(stderr,"CDIO: CD ioctl error (code = %d)\n",errno);
  } else {
    memcpy(buf,cdio.buffer,100);
  }
  return r>=0;
}

int CDReadLBA(int lba,char* buf){
  int min,sec,frame;
  lba+=150;
  frame=lba%75;
  lba/=75;
  min=lba/60;
  sec=lba%60;
  return CDReadMSF(min,sec,frame,buf);
}
#endif

/* For IRIX */
#ifdef HAVE_DMEDIA_CDAUDIO_H
#include <sys/types.h>
#include <dmedia/cdaudio.h>

CDPLAYER* cd=NULL;
CDSTATUS status;
CDFRAME buf[12];

int CDOpen(char* dev){
  cd=CDopen(dev,"r");
  return cd!=NULL;
}

int CDClose(void){
  if (cd!=NULL) CDclose(cd);
  cd=NULL;
  return 1;
}

int CDReadLBA(int lba,char* bufptr){
  static int start=-1;
  int lba2,lba3;
  if (cd==NULL) return 0;
  lba3=lba % 12;
  lba2=lba-lba3;
  if (start<0 || lba2!=start){
    CDseekblock(cd,lba2);
    CDreadda(cd,buf,12);
  }
  memcpy(bufptr,buf+lba3,2304);
}
#endif

int main(int argc,char** argv){
  char buf0[3000];
  unsigned char* buf=(unsigned char*)buf0;
  int secs;
  int fst;
  int lba,lba2;
  int offst;
  int i,j;

#ifdef ENABLE_NLS
  textdomain(PACKAGE);
  setlocale(LC_ALL,"");
#endif

  fprintf(stderr,_("Read EB-XA sound data from CD-ROM version %s\n"),VERSION);
  fprintf(stderr,_("Copyright(c) 2000 : T. Nemoto\n"));
  if (argc!=4){
    fprintf(stderr,"\n");
    fprintf(stderr,_("Usage: %s <CD DEVICE> <sound file> <output file>\n"),
	    argv[0]);
    fprintf(stderr,_("    ex: %s /dev/cdrom /cdrom/crger/sound ./sound\n"),
	    argv[0]);
    exit(1);
  }
  
  if (!CDOpen(argv[1])){
    fprintf(stderr,_("Device Open Error : %s\n"),argv[1]);
    exit(1);
  }

  lba=-1;
  offst=-1;
  for(i=0;i<1000 && lba<0;i++){
    CDReadLBA(i,buf0);
    for(j=0;j<1024;j++){
      if (buf0[j]=='S' && buf0[j+1]=='O' && buf0[j+2]=='U' && 
	  buf0[j+3]=='N' && buf0[j+4]=='D'){
	lba=i;
	offst=j;
      }
    }
  }
  if (lba>=0){
    if (offst>0x20){
      offst-=0x1b;
      fst=(((long)buf[offst])<<24)+
	(((long)buf[offst+1])<<16)+
	(((long)buf[offst+2])<<8)+
	(((long)buf[offst+3]));
      printf(_("Found: LBA=%d, Offset=%d\n"),lba,offst);
      printf(_("        FILE START = (LBA) %d\n"),fst);
    }
  }
  if (argc>1){
    struct stat st;
    if (lstat(argv[2],&st)==0){
      secs=st.st_size/2048;
      fprintf(stderr,_("        FILE LENGTH= %d sectors / %d bytes (%d)\n"),
	     secs, secs*0x900,secs*0x800);
    }
  }
  lba2=-1;
  for(i=0;i<350000;i+=istep){
    CDReadLBA(i,buf0);
    if (buf0[16+3]!=4) continue;
    printf("%d\n",i);
    {
      int lbamin=i-istep;
      int lbamax=i;
      int lbamid=(lbamax+lbamin)/2;
      int lastmid=-1;
      while(lbamin!=lbamax){
	CDReadLBA(lbamid,buf0);
	if (buf0[16+3]!=4) lbamin=lbamid; else lbamax=lbamid;
	lbamid=(lbamin+lbamax)/2;
	if (lastmid==lbamid) {
	  lbamid++;
	  lbamin++;
	}
	lastmid=lbamid;
      }
      lba2=lbamid;
    }
    break;
  }
  fprintf(stderr,
	  _("Start from dir=%d\nStart from SecterHeader=%d\nLength=%d\n"),
	  fst,lba2,secs);
  if (fst==lba2){
    FILE* ofp=fopen(argv[3],"w");
    for(i=0;i<secs;i++){
      CDReadLBA(fst+i,buf0);
      fwrite(buf0+16+8,1,2304,ofp);
    }
    fclose(ofp);
  } else {
    fprintf(stderr,_("File Header Detection Error! \n"),
	    _("Please send the previous message to author.\n"));
  }
  CDClose();
}
