/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef GRUB_URL_REQUESTER_H
#define GRUB_URL_REQUESTER_H

//KDE
#include <kurlcompletion.h>
#include <kurlrequester.h>

//GRUBPathCompletion
/**
 * @short GRUB-like completion of file names.
 *
 * This class does GRUB-like completion of file-names.
 * It turns the root of a file to the standard GRUB device
 * naming scheme.
 */
class GRUBPathCompletion : public KUrlCompletion
{
Q_OBJECT
public:
	/**
	* Finds completions to the given text.
	* The first match is returned and emitted in the signal match().
	* @param text the text to complete
	* @return the first match, or QString() if not found
	*/
	QString makeCompletion( const QString &text );
};

//GRUBUrlRequester
/**
 * @short A KUrlRequester customised for GRUB usage.
 *
 * Since GRUB doesn't understand the mountpoints as used in
 * a Linux system, but instead uses its own naming scheme
 * (usually based on the contents of /boot/grub/device.map),
 * this KUrlRequester automatically turns the URL which
 * was selected by the user into a GRUB-compatible. The reverse
 * functionality is provided, too (from a GRUB-compatible file
 * name find the real file).
 *
 * The following 2 signals should be handled:
 *
 * pathChanged( const QString & )
 * pathCleared()
 *
 * @li pathChanged is emitted when the user selects a file using the
 * file dialog or when the user manually changes the lineedit's contents.
 *
 * @li pathCleared is emitted when the user clears the lineedit's
 * contents using the arrow-clear button.
 */
class GRUBUrlRequester : public KUrlRequester
{
Q_OBJECT
public:
	/**
	* Default constructor.
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	GRUBUrlRequester( QWidget *parent = 0 );

	/**
	* @return the associated KLineEdit root editor.
	*
	* For more info check @see setAssociatedRootEditor().
	*/
	inline const KLineEdit* associatedRootEditor() const { return m_editor; };
	/**
	* You may associate this widget with a KLineEdit widget
	* which will contain a custom root.
	*
	* This is useful because GRUB has a useful function:
	* if you somehow set the root (eg. in an entry), then if
	* a file you want to specify resides in that root,
	* there is no need to sepcify the (grub) root in front
	* of the filename.
	*
	* Consider this example, where we have the following entry:
	*
	* @code
	* title Dummy Entry
	* root (hd0,0)
	* kernel (hd0,0)/boot/kernel
	* @endcode
	*
	* Since the root is already specified and the file
	* /boot/kernel resides in it, there is no need for the
	* (hd0,0) prefix and thus the entry would be rewritten:
	*
	* @code
	* title Dummy Entry
	* root (hd0,0)
	* kernel /boot/kernel
	* @endcode
	*/
	inline void setAssociatedRootEditor( KLineEdit *editor ) { m_editor = editor; };
signals:
	/**
	* Emitted when the user selects a file using the file dialog 
	* or when the user manually changes the lineedit's contents.
	*/
	void pathChanged( const QString & );
	/**
	* Emitted when the user clears the line edit's
	* contents using the arrow-clear button.
	*/
	void pathCleared();
private slots:
	/**
	* Used before the file dialog is going to open in
	* order to convert the GRUB path found in the lineedit
	* into a generic one.
	*/
	void openFileDialog( KUrlRequester * );
	/**
	* Converts the text which was entered by the user into a
	* GRUB-compatible. Triggered when the user changes the
	* current text by manually editing the line edit.
	*
	* @param text the current text.
	*/
	void returnPressed( const QString &text );
	/**
	* Converts the URL which was selected by the user into a
	* GRUB-compatible. Triggered when the user changes the
	* selected URL using the file dialog.
	*
	* @param url the selected URL.
	*/
	void urlSelected( const KUrl &url );
private:
	KLineEdit *m_editor;
};

#endif
