/* $USAGI: ipmip6.c,v 1.13 2003/10/22 10:00:40 nakam Exp $ */

/*
 * A frontend for mip6/xfrm like iproute2 command
 */
/*
 * Authors:
 *	Noriaki TAKAMIYA @USAGI
 *	Masahide NAKAMURA @USAGI
 */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#define _GNU_SOURCE
#include <getopt.h>

#include <time.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <asm/types.h>

#include <net/if.h>

#include <linux/xfrm.h>

#include <glue.h>
#include <mip6nl.h>
#include <mip6log.h>
#include "ipmip6.h"

#define CMD_NAME	"ipmip6"
#define CMD_VERSION	"0.0.1"

static const char opt_str[] = "v";

static const struct option opt_table[] = {
	{"verbose", 0, 0, 'v'},

	{"example", 0, 0, 0},
	{"version", 0, 0, 0},
	{"help", 0, 0, 0},

	{0, 0, 0, 0}
};

static const char *opt_arg_name[] = {
	NULL,

	NULL,
	NULL,
	NULL,

	NULL
};

static const char *opt_desc[] = {
	"verbose output",

	"show example",
	"show version",
	"show this help",

	NULL
};

static int is_verbose_mode = 0; /* XXX: quick hacking... */
static int is_flush = 0; /* XXX: quick hacking... */
static int error_code = 0; /* XXX: quick hacking... */

static void version()
{
	printf("%s %s\n", CMD_NAME, CMD_VERSION);
}

static void help()
{
	char opt_buf[1024];
	char lopt_buf[1024];
	int i;

	/* XXX: global options */
	printf("Usage: %s [options]\n", CMD_NAME);

	for (i = 0; ; i ++) {
		if (opt_table[i].name == 0)
			break;

		if (opt_table[i].val != 0)
			sprintf(opt_buf, "-%c,", (char)opt_table[i].val);
		else
			sprintf(opt_buf, "   ");

		if (opt_table[i].has_arg == 1)
			sprintf(lopt_buf, "--%s=<%s>",
				opt_table[i].name, opt_arg_name[i]);
		else
			sprintf(lopt_buf, "--%s", opt_table[i].name);

		printf("  %s %-*s %s\n",
			opt_buf,
			21, /* format-length for long option name */
			lopt_buf,
			opt_desc[i]);
	}

#if 0
	/* XXX: basic syntax */
	printf("Usage: %s OBJECT COMMAND [COMMAND_ARG...]\n", CMD_NAME);
	printf("OBJECT := [ policy | state ]\n");
	printf("\n");
#endif
	/* XXX: each syntax by object */
	printf("Usage: %s policy { add | change } DADDR SADDR dir DIR TMPL\n", CMD_NAME);
	printf("          [ proto PROTO ] [ dev DEV ] [ priority PRIORITY] [ lookup LOOKUP ] \n");
	printf("       %s policy { del | get } DADDR SADDR dir DIR [proto PROTO] [dev DEV]\n", CMD_NAME);
	printf("       %s policy { flush }\n", CMD_NAME);

	printf("Usage: %s state { add | change } DADDR SADDR EXT [ proto PROTO ] [ dev DEV ]\n", CMD_NAME);
	printf("       %s state { del | get } EXT [ proto PROTO ] [ dev DEV ]\n", CMD_NAME);
	printf("       %s state { flush }\n", CMD_NAME);

	printf("DADDR - destination address(\"any\" ok)\n");
	printf("SADDR - source address(\"any\" ok)\n");

	printf("DIR := [ in | out | fwd ]\n");
	printf("TMPL := [ EXT [ EXT ...] ](max count=%d)\n", TMPL_MAX);
	printf("PROTO := [ any | mh | # ](default=any)\n");
	printf("DEV - device name(default=none)\n");

	printf("EXT := [ [ dopt | rt | ip6 ] TUNNEL ]\n");
 	printf("TUNNEL := DADDR SADDR\n");
	printf("PRIORITY - priority value(default=0)\n");
	printf("LOOKUP := [ none | any | src | all ](default=none)\n");
}

static void usage()
{
#if 0
	printf("Try `%s --help` for more information.\n", CMD_NAME);
#else
	help();
#endif
}


static void example()
{
	printf("Example:\n");
	printf("\n");
	printf("\n");
	printf("\n");
	printf("\n");
	printf("\n");
	printf("\n");
}

static int in6_addr_get(struct in6_addr *addr, const char *str)
{
	struct addrinfo hints, *res = NULL;
	int ret;

	if (strcmp(str, "any") == 0) {
		memcpy(addr, &in6addr_any, sizeof(addr));
		goto fin;
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET6;
	hints.ai_socktype = SOCK_RAW;
	hints.ai_flags = AI_PASSIVE;

	ret = getaddrinfo(str, NULL, &hints, &res);
	if (ret != 0) {
		printf("getaddrinfo(\"%s\", ...):%s\n", str, gai_strerror(ret));
		goto fin;
	}
	if (res->ai_addrlen != sizeof(struct sockaddr_in6)) {
		printf("getaddrinfo(\"%s\", ...) contains invalid length:%d\n",
		       str, res->ai_addrlen);
		goto fin;
	}
	memcpy(addr,
	       &((struct sockaddr_in6 *)(res->ai_addr))->sin6_addr,
	       sizeof(struct in6_addr));

 fin:
	if (res)
		freeaddrinfo(res);
	return ret;
}

static const char *in6_straddr(const struct in6_addr *addr)
{
	static char str[INET6_ADDRSTRLEN];
	struct sockaddr_in6 sa;
	int ret;

	if (memcmp(addr, &in6addr_any, sizeof(*addr)) == 0) {
		strcpy(str, "(any)");
		return str;
	}

	memset(&sa, 0, sizeof(sa));
	sa.sin6_family = PF_INET6;
	memcpy(&sa.sin6_addr, addr, sizeof(sa.sin6_addr));

	ret = getnameinfo((struct sockaddr *)&sa, sizeof(sa), str, sizeof(str), NULL, 0, NI_NUMERICHOST);
	if (ret != 0) {
		__eprintf("getnameinfo():%s\n", gai_strerror(ret));
		return NULL;
	}

	return str;
}

static const char *strproto(__u8 proto)
{
	static char str[32];

	switch (proto) {
	case 0:
		strcpy(str, "any");
		break;
	case IPPROTO_IPV6:
		strcpy(str, "ip6");
		break;
	case IPPROTO_DSTOPTS:
		strcpy(str, "dopt");
		break;
	case IPPROTO_ROUTING:
		strcpy(str, "rt");
		break;
	case IPPROTO_MOBILITY:
		strcpy(str, "mh");
		break;
	default:
		sprintf(str, "(%d)", proto);
		break;
	}

	return str;
}

static const char *strpolicy_dir(__u8 dir)
{
	static char str[32];

	switch (dir) {
	case XFRM_POLICY_IN:
		strcpy(str, "in");
		break;
	case XFRM_POLICY_OUT:
		strcpy(str, "out");
		break;
	case XFRM_POLICY_FWD:
		strcpy(str, "fwd");
		break;
	default:
		sprintf(str, "(%d)", dir);
		break;
	}

	return str;
}

static const char *strpolicy_action(__u8 action)
{
	static char str[32];

	switch (action) {
	case XFRM_POLICY_ALLOW:
		strcpy(str, "allow");
		break;
	case XFRM_POLICY_BLOCK:
		strcpy(str, "block");
		break;
	default:
		sprintf(str, "(%d)", action);
		break;
	}

	return str;
}

static const char *strpolicy_flags(__u8 flags)
{
	static char str[32];


#if 0
	memset(str, '\0', sizeof(str));
	if (flags & XFRM_POLICY_LOCALOK)
		strcpy(str, "localok");
#endif

	if (flags == 0)
		strcpy(str, "none");
	else if ((flags & XFRM_LOOKUP_ALLOW_ANY) &&
		 (flags & XFRM_LOOKUP_PRIORITY_SRC))
		strcpy(str, "all");
	else if (flags & XFRM_LOOKUP_ALLOW_ANY)
		strcpy(str, "any");
	else if (flags & XFRM_LOOKUP_PRIORITY_SRC)
		strcpy(str, "src");
	else
		sprintf(str, "(%d)", flags);

	return str;
}

static const char *strlimit(__u64 limit)
{
	static char str[32];
	if (limit == XFRM_INF)
		strcpy(str, "(INF)");
	else
		sprintf(str, "%llu", limit);

	return str;
}

static const char *strtime(__u64 time)
{
	static char str[32];
	struct tm *tp;
	time_t t;

	if (time == 0) {
		strcpy(str, "(undefined)");
	} else {
		/* XXX: treat time in the same manner of xfrm_user.c, xfrm_state.c */
		t = (long)time;
		tp = localtime(&t);

		sprintf(str, "%04d/%02d/%02d %02d:%02d:%02d",
			tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday,
			tp->tm_hour, tp->tm_min, tp->tm_sec);
	}

	return str;
}

static void dump_selector(const struct xfrm_selector *sel, const char *prefix)
{
	if (prefix)
		printf(prefix);
	printf("%s", in6_straddr((struct in6_addr *)&sel->daddr.a6));
	if (sel->prefixlen_d)
		printf("/%d", sel->prefixlen_d);
	printf(" ");

	printf("%s", in6_straddr((struct in6_addr *)&sel->saddr.a6));
	if (sel->prefixlen_s)
		printf("/%d", sel->prefixlen_s);
	printf(" ");

	printf("proto=%s", strproto(sel->proto));
	printf(" ");

	printf("ifindex=%d", sel->ifindex);
	if (sel->ifindex > 0) {
		char name[IF_NAMESIZE];
		memset(name, '\0', sizeof(name));
		if_indextoname(sel->ifindex, name);
		printf("(%s)", name);
	}

	printf("\n");
}

static void dump_lifetime(const struct xfrm_lifetime_cfg *cfg, const struct xfrm_lifetime_cur *cur,
			  const char *prefix)
{
	if (cfg) {
		if (prefix)
			printf(prefix);
		printf("lifetime config:\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("limit: ");
		printf("soft=");
		printf(strlimit(cfg->soft_byte_limit));
		printf(" bytes, hard=");
		printf(strlimit(cfg->hard_byte_limit));
		printf(" bytes\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("limit: ");
		printf("soft=");
		printf(strlimit(cfg->soft_packet_limit));
		printf(" packets, hard=");
		printf(strlimit(cfg->hard_packet_limit));
		printf(" packets\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("expire add: ");
		printf("soft=");
		printf("%llu", cfg->soft_add_expires_seconds);
		printf(" sec, hard=");
		printf("%llu", cfg->hard_add_expires_seconds);
		printf(" sec\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("expire use: ");
		printf("soft=");
		printf("%llu", cfg->soft_use_expires_seconds);
		printf(" sec, hard=");
		printf("%llu", cfg->hard_use_expires_seconds);
		printf(" sec\n");
	}
	if (cur) {
		if (prefix)
			printf(prefix);
		printf("lifetime current:\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("%llu bytes, ", cur->bytes);
		printf("%llu packets\n", cur->packets);
		if (prefix)
			printf(prefix);
		printf("  ");
		printf("add %s ", strtime(cur->add_time));
		printf("use %s", strtime(cur->use_time));
		printf("\n");
	}
}

static void dump_stats(const struct xfrm_stats *stats, const char *prefix)
{
	if (prefix)
		printf(prefix);
	printf("stats:\n");

	if (prefix)
		printf(prefix);
	printf("  ");
	printf("replay-window=%d ", stats->replay_window);
	printf("replay=%d ", stats->replay);
	printf("failed=%d", stats->integrity_failed);
	printf("\n");
}

static void dump_policy(const struct xfrm_userpolicy_info *pol)
{
	dump_selector(&pol->sel, NULL);

	printf("\t");
	printf("priority=%d ", pol->priority);
	printf("index=%d ", pol->index);
	/*printf("family=%d ", pol->family);*/

	printf("dir=%s ", strpolicy_dir(pol->dir));
	printf("action=%s ", strpolicy_action(pol->action));
	printf("lookup=%s", strpolicy_flags(pol->flags));

	printf("\n");

	if (is_verbose_mode)
		dump_lifetime(&pol->lft, &pol->curlft, "\t");
}

static void dump_state(const struct xfrm_usersa_info *sa)
{
	dump_selector(&sa->sel, NULL);

	/* xfrm_id */
	printf("\t");
	printf("ext=%s", strproto(sa->id.proto));

	switch (sa->id.proto) {
	case IPPROTO_ROUTING:
	case IPPROTO_DSTOPTS:
	case IPPROTO_IPV6:
	default:
		break;
	}
	printf(" ");

	printf("mode=");
	if (sa->mode) {
		printf("tunnel");
	} else {
		printf("transport");
	}
	printf("\n");

	printf("\t");
	printf("dst=%s", in6_straddr((struct in6_addr *)&sa->id.daddr.a6));
	printf(" ");
	printf("src=%s", in6_straddr((struct in6_addr *)&sa->saddr.a6));
	printf("\n");

	if (is_verbose_mode) {

		printf("\t");
		printf("spi=0x%08u", sa->id.spi);
		printf(" ");

		printf("seq=0x%08u ", sa->seq);
		/*printf("family=%d\n", sa->family);*/
		printf("reqid=%d ", sa->reqid);
		printf("replay-window=%d", sa->replay_window);

		printf("\n");

		/* xfrm_lifetime_cfg/cur */
		dump_lifetime(&sa->lft, &sa->curlft, "\t");

		/* xfrm_stats */
		dump_stats(&sa->stats, "\t");

		printf("\n");
	}
}

static int del_policy(const struct mip6nl_parms *mp,
		      const struct xfrm_userpolicy_info *pol)
{
	struct mip6nl_parms mp2;
	memset(&mp2, 0, sizeof(mp2));
	/*mip6nl_init(&mp2, mp->sock);*/
	mp2 = *mp; /* XXX */
	mp2.type = XFRM_MSG_DELPOLICY;

	memcpy(&mp2.sel.daddr, &pol->sel.daddr, sizeof(mp2.sel.daddr));
	memcpy(&mp2.sel.saddr, &pol->sel.saddr, sizeof(mp2.sel.saddr));
	mp2.pol.index = pol->index;
	mp2.sel.proto = pol->sel.proto;

	return mip6nl_talk(&mp2);
}

static int del_state(const struct mip6nl_parms *mp,
		     const struct xfrm_usersa_info *sa)
{
	struct mip6nl_parms mp2;
	memset(&mp2, 0, sizeof(mp2));
	/*mip6nl_init(&mp2, mp->sock);*/
	mp2 = *mp; /* XXX */
	mp2.type = XFRM_MSG_DELSA;
	memcpy(&mp2.sel.daddr, &sa->sel.daddr, sizeof(mp2.sel.daddr));
	memcpy(&mp2.sel.saddr, &sa->sel.saddr, sizeof(mp2.sel.saddr));
	mp2.sel.proto = sa->sel.proto;
	mp2.id.ext.proto = sa->id.proto;
	memcpy(&mp2.id.ext.daddr, &sa->id.daddr, sizeof(mp2.id.ext.daddr));
	memcpy(&mp2.id.ext.saddr, &sa->saddr, sizeof(mp2.id.ext.saddr));

	return mip6nl_talk(&mp2);
}

static int mip_probe_handler_policy(const struct mip6nl_parms *mp,
				    const struct xfrm_userpolicy_info *pol)
{
	if (is_flush) {
		/* return del_policy(mp, pol); */
		del_policy(mp, pol); /* XXX: ignore each error when flushing */
		return 0;
	} else {
		dump_policy(pol);
		return 0;
	}
}

static int mip_probe_handler_state(const struct mip6nl_parms *mp,
				   const struct xfrm_usersa_info *sa)
{
	if (is_flush) {
		/* return del_state(mp, sa); */
		del_state(mp, sa); /* XXX: ignore each error when flushing */
		return 0;
	} else {
		dump_state(sa);
		return 0;
	}
}

static void mip_error_handler(const struct mip6nl_parms *mp,
			      const struct nlmsghdr *hdr, int error)
{
	error_code = error;
}

int main(int argc, char **argv)
{
	int opt;
	char *object;
	char *daddr_p = NULL;
	char *saddr_p = NULL;
	int cur_ind;
	int err = EINVAL;
	struct mip6nl_parms parms;
	int ext_ind;

	memset(&parms, 0, sizeof(parms));

	if (mip6nl_open(&parms) < 0){
		printf("failed to open netlink socket: %s\n", strerror(errno));
		goto usage_exit;
	}

	while (1) {
		/*int this_option_optind = optind ? optind : 1;*/
		int option_index = 0;

		opt = getopt_long(argc, argv, opt_str,
				  opt_table, &option_index);
		if (opt == -1)
			break;

		switch (opt) {
		case 0: /* option has no short one */
			if (strcmp(opt_table[option_index].name, "example") == 0) {
				example();
				exit(0);
			} else if (strcmp(opt_table[option_index].name, "version") == 0) {
				version();
				exit(0);
			} else if (strcmp(opt_table[option_index].name, "help") == 0) {
				help();
				exit(0);
			} else
				goto usage_exit;

		case 'v':
			is_verbose_mode = 1;
			break;
		case '?':
			break;
		default:
			printf("unknown opt=%d\n", opt);
			goto usage_exit;
		}
	}

	cur_ind = optind;

	if (cur_ind >= argc)
		goto usage_exit;

	/* OBJECT, COMMAND */
	object = argv[cur_ind ++];

	if (strcmp(object, "policy") == 0) {
		char *act;

		if (cur_ind == argc) {
			parms.type = XFRM_MSG_GETPOLICY;
			parms.flag |= NLM_F_DUMP;
			goto talk;
		} else if (cur_ind > argc)
			goto usage_exit;

		act = argv[cur_ind ++];

		if (strcmp(act, "add") == 0) {
			parms.type = XFRM_MSG_NEWPOLICY;
		} else if (strcmp(act, "del") == 0) {
			parms.type = XFRM_MSG_DELPOLICY;
		} else if (strcmp(act, "get") == 0) {
			parms.type = XFRM_MSG_GETPOLICY;
		} else if (strcmp(act, "change") == 0) {
			parms.type = XFRM_MSG_UPDPOLICY;
		} else if (strcmp(act, "flush") == 0) {
			parms.type = XFRM_MSG_GETPOLICY;
			parms.flag |= NLM_F_DUMP;
			is_flush = 1;
			goto talk;
		} else {
			printf("invalid arguments: %s: unknown command.\n", act);
			goto usage_exit;
		}

	} else if (strcmp(object, "state") == 0) {
		char *act;

		if (cur_ind == argc) {
			parms.type = XFRM_MSG_GETSA;
			parms.flag |= NLM_F_DUMP;
			goto talk;
		} else if (cur_ind > argc)
			goto usage_exit;

		act = argv[cur_ind ++];

		if (strcmp(act, "add") == 0) {
			parms.type = XFRM_MSG_NEWSA;
		} else if (strcmp(act, "del") == 0) {
			parms.type = XFRM_MSG_DELSA;
		} else if (strcmp(act, "get") == 0) {
			parms.type = XFRM_MSG_GETSA;
		} else if (strcmp(act, "change") == 0) {
			parms.type = XFRM_MSG_UPDSA;
		} else if (strcmp(act, "flush") == 0) {
			parms.type = XFRM_MSG_GETSA;
			parms.flag |= NLM_F_DUMP;
			is_flush = 1;
			goto talk;
		} else {
			printf("invalid arguments: %s: unknown command.\n", act);
			goto usage_exit;
		}

	} else {
		printf("invalid argument: %s: uknown object.\n", object);
		goto usage_exit;
	}

	if (parms.type == XFRM_MSG_DELSA ||
	    parms.type == XFRM_MSG_GETSA) {
		;
	} else {
		/* DADDR */
		if (cur_ind >= argc)
			goto usage_exit;

		daddr_p = argv[cur_ind ++];
		if (in6_addr_get(&parms.sel.daddr, daddr_p) < 0)
			goto usage_exit;

		/* SADDR */
		if (cur_ind >= argc)
			goto usage_exit;

		saddr_p = argv[cur_ind ++];
		if (in6_addr_get(&parms.sel.saddr, saddr_p) < 0)
			goto usage_exit;
	}

	if (parms.type == XFRM_MSG_NEWPOLICY ||
	    parms.type == XFRM_MSG_DELPOLICY ||
	    parms.type == XFRM_MSG_GETPOLICY ||
	    parms.type == XFRM_MSG_UPDPOLICY) {
		/* DIR */
		char *name = argv[cur_ind ++];
		char *dir_p = NULL;

		if (cur_ind >= argc)
			goto usage_exit;

		if (strcmp(name, "dir") != 0) {
			printf("dir is missing\n");
			goto usage_exit;
		}

		if (cur_ind >= argc) {
			printf("missing arguments: %s\n", name);
			goto usage_exit;
		}
		dir_p = argv[cur_ind ++];

		if (strcmp(dir_p, "in") == 0)
			parms.pol.dir = XFRM_POLICY_IN;
		else if (strcmp(dir_p, "out") == 0)
			parms.pol.dir = XFRM_POLICY_OUT;
		else if (strcmp(dir_p, "fwd") == 0)
			parms.pol.dir = XFRM_POLICY_FWD;
		else
			goto usage_exit;
	}

#if 0
	/* {NEW/DEL/GET/UPD}POLICY may not need argument any more */
	if (cur_ind == argc) { /* nothing after that */
		if (parms.type == XFRM_MSG_NEWPOLICY ||
		    parms.type == XFRM_MSG_DELPOLICY ||
		    parms.type == XFRM_MSG_GETPOLICY ||
		    parms.type == XFRM_MSG_UPDPOLICY)
			goto talk;
		else
			goto usage_exit;
	} else { /* anything after that */
		if (parms.type == XFRM_MSG_DELPOLICY ||
		    parms.type == XFRM_MSG_GETPOLICY)
			goto usage_exit;
	}
#endif

	/* TMPL/EXT */
	ext_ind = 0;
	while (cur_ind < argc) {
		char *name;
		struct mip6nl_parms_id *id = NULL;

		if (cur_ind >= argc)
			goto usage_exit;
		name = argv[cur_ind ++];

		if (ext_ind >= TMPL_MAX) {
			printf("too many EXT.\n");
			goto usage_exit;
		}

		/* DOPT */
		if (strcmp(name, "dopt") == 0) {
			id = &parms.id.tmpls[ext_ind ++];
			id->proto = IPPROTO_DSTOPTS;

		/* RT */
		} else if (strcmp(name, "rt") == 0) {
			id = &parms.id.tmpls[ext_ind ++];
			id->proto = IPPROTO_ROUTING;

		/* IP6 */
		} else if (strcmp(name, "ip6") == 0) {
			id = &parms.id.tmpls[ext_ind ++];
			id->proto = IPPROTO_IPV6;

		} else {
			/* back track */
			cur_ind --;
			/* exit the loop here */
			break;
		}

		assert(id != NULL);

		/* TUNNEL */
		if (cur_ind >= argc) {
			printf("missing arguments: %s\n", name);
			goto usage_exit;
		} else if (cur_ind + 2 > argc) {
			printf("missing arguments: TUNNEL needs two args.\n");
			goto usage_exit;
		} else if (cur_ind + 2 <= argc) {
			char *daddr_p = argv[cur_ind ++];
			char *saddr_p = argv[cur_ind ++];

			if (in6_addr_get((struct in6_addr *)&id->daddr,
					 daddr_p) < 0)
				goto usage_exit;
			if (in6_addr_get((struct in6_addr *)&id->saddr,
					 saddr_p) < 0)
				goto usage_exit;
		}

	}
	if (parms.type == XFRM_MSG_NEWPOLICY ||
	    parms.type == XFRM_MSG_UPDPOLICY) {
		; /* valid type */
	} else if(parms.type == XFRM_MSG_NEWSA ||
		  parms.type == XFRM_MSG_DELSA ||
		  parms.type == XFRM_MSG_GETSA ||
		  parms.type == XFRM_MSG_UPDSA) {
		; /* valid type */
		if (ext_ind == 0) {
			printf("missing arguments: EXT needed with that type.\n");
			goto usage_exit;
		} else if (ext_ind > 1) {
			printf("duplicated arguments: EXT is allowed only once with that type.\n");
			goto usage_exit;
		}
	} else {
		if (ext_ind > 0) {
			printf("invalid arguments: can't specify EXT with that type.\n");
			goto usage_exit;
		}
	}

	/* arguments that are not always required  */
	while (cur_ind < argc) {
		char *name;

		if (cur_ind >= argc)
			goto usage_exit;
		name = argv[cur_ind ++];

		/* PROTO */
		if (strcmp(name, "proto") == 0) {
			char *proto_p = NULL;

			if (cur_ind >= argc) {
				printf("missing arguments: %s\n", name);
				goto usage_exit;
			}
			proto_p = argv[cur_ind ++];

			if (strcmp(proto_p, "mh") == 0)
				parms.sel.proto = IPPROTO_MOBILITY;
			else if (strcmp(proto_p, "any") == 0)
				parms.sel.proto = 0;
			else {
				/* XXX: currently no check to parse number.*/
				parms.sel.proto = (__u8)atoi(proto_p);
			}

		/* DEV */
		} else if (strcmp(name, "dev") == 0) {
			char *dev_p = NULL;
			int ifindex = 0;

			if (cur_ind >= argc) {
				printf("missing arguments: %s\n", name);
				goto usage_exit;
			}
			dev_p = argv[cur_ind ++];

			ifindex = if_nametoindex(dev_p);
			if (ifindex <= 0) {
				printf("if_nametoindex(\"%s\"): %s\n", dev_p, strerror(errno));
				goto usage_exit;
			}
			parms.sel.ifindex = ifindex;

		/* PRIORITY */
		} else if (strcmp(name, "priority") == 0) {
			char *priority_p = NULL;

			if (parms.type == XFRM_MSG_NEWPOLICY ||
			    parms.type == XFRM_MSG_UPDPOLICY) {
				; /* valid type */
			} else {
				printf("invalid arguments: %s: can't specify with that type.\n", name);
				goto usage_exit;
			}

			if (cur_ind >= argc) {
				printf("missing arguments: %s\n", name);
				goto usage_exit;
			}
			priority_p = argv[cur_ind ++];

			/* XXX: currently no check to parse number.*/
			parms.pol.priority = (__u32)atol(priority_p);

		/* LOOKUP */
		} else if (strcmp(name, "lookup") == 0) {
			char *lookup_p = NULL;

			if (parms.type == XFRM_MSG_NEWPOLICY ||
			    parms.type == XFRM_MSG_UPDPOLICY) {
				; /* valid type */
			} else {
				printf("invalid arguments: %s: can't specify with that type.\n", name);
				goto usage_exit;
			}

			if (cur_ind >= argc) {
				printf("missing arguments: %s\n", name);
				goto usage_exit;
			}
			lookup_p = argv[cur_ind ++];

			if (strcmp(lookup_p, "none") == 0)
				parms.pol.flags = 0;
			else if (strcmp(lookup_p, "any") == 0)
				parms.pol.flags = XFRM_LOOKUP_ALLOW_ANY;
			else if (strcmp(lookup_p, "src") == 0)
				parms.pol.flags = XFRM_LOOKUP_PRIORITY_SRC;
			else if (strcmp(lookup_p, "all") == 0)
				parms.pol.flags = (XFRM_LOOKUP_ALLOW_ANY |
						   XFRM_LOOKUP_PRIORITY_SRC);
			else {
				printf("invalid argument: %s: unknown.\n", lookup_p);
				goto usage_exit;
			}
		} else {
			printf("invalid arguments: %s: unknown command.\n", name);
			goto usage_exit;
		}
	}

	if (cur_ind != argc) {
		printf("too many arguments\n");
		goto usage_exit;
	}

 talk:
	/* XXX: */
	parms.error_handler = mip_error_handler;
	parms.probe_handler_policy = mip_probe_handler_policy;
	parms.probe_handler_state = mip_probe_handler_state;

	err = mip6nl_talk(&parms);

	mip6nl_close(&parms);

	if (err != 0)
		return (err > 0) ? err : -err;
	else
		return error_code;
		
 usage_exit:
	usage();
	exit((err > 0) ? err : -err);
}
