/* Copyright (c) 1999 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/time.h>
#include "nis_db.h"
#include "db_private.h"

db_result *
db_remove_entry (const char *table_name, u_int numattrs,
		 const nis_attr *attrs)
{
  return __remove_entry (table_name, numattrs, attrs, TRUE);
}

db_result *
__remove_entry (const char *table_name, u_int numattrs,
		const nis_attr *attrs, bool_t write_log)
{
  table_t *tobj;
  entry_t *eobj, *eold;
  db_result *res;
  struct timeval t_start, t_end;
  u_int i, j;
  u_int32_t *hashes = NULL;
  u_int len;
  xdb_logaction action;
  XDR xdrs;

  gettimeofday (&t_start, NULL);

  if ((res = calloc (1, sizeof (db_result))) == NULL)
    return NULL;

  res->status = DB_SUCCESS;
  if (table_name == NULL || strlen (table_name) == 0)
    {
      res->status = DB_BADTABLE;
      goto bailout;
    }

  tobj = __get_table (table_name);
  if (tobj == NULL)
    {
      res->status = DB_BADTABLE;
      goto bailout;
    }

  len = tobj->object->ta_cols.ta_cols_len;
  hashes = calloc (sizeof (uint32_t), len);

  /* Calculate the hashes for every given attribute. Fields for not
     given table column names will be zero. */
  for (i = 0; i < numattrs; ++i)
    for (j = 0; j < len; ++j)
      {
        if (strcmp (attrs[i].zattr_ndx,
                    tobj->object->ta_cols.ta_cols_val[j].tc_name) == 0)
          hashes[j] = __nis_hash (attrs[i].zattr_val.zattr_val_val,
                                  attrs[i].zattr_val.zattr_val_len);
      }

  eold = NULL;
  eobj = tobj->entry;

  while (eobj != NULL)
    {
      u_int found;

      if (len != eobj->hashes_len)
        abort ();

      found = 0;

      for (i = 0; i < len; ++i)
        if (hashes[i] && (hashes[i] == eobj->hashes[i]))
          ++found;

      if (found == numattrs)
        {
          res->objects.objects_len++;
          res->objects.objects_val =
            realloc (res->objects.objects_val,
                     res->objects.objects_len * sizeof (entry_obj *));
          if (res->objects.objects_val == NULL)
            {
              res->status = DB_MEMORY_LIMIT;
              goto bailout;
            }
          res->objects.objects_val[res->objects.objects_len - 1] =
	    eobj->object;
          if (res->objects.objects_val[res->objects.objects_len - 1] == NULL)
            {
              res->status = DB_MEMORY_LIMIT;
              goto bailout;
            }
	  if (eold == NULL)
	    tobj->entry = tobj->entry->next;
	  else
	    eold->next = eobj->next;
	  free (eobj->hashes);
	  free (eobj);
	  if (eold == NULL)
	    eobj = NULL;
	  else
	    eobj = eold->next;
        }
      else
	{
	  eold = eobj;
	  eobj = eobj->next;
	}
    }

  if (write_log)
    {
      res->status = __open_log (tobj);
      if (res->status != DB_SUCCESS)
	goto bailout;

      action = LOG_DELETE;

      xdrstdio_create (&xdrs, tobj->fp_log, XDR_ENCODE);
      if (!xdr_xdb_logaction (&xdrs, &action))
	{
	  xdr_destroy (&xdrs);
	  fclose (tobj->fp_log);
	  tobj->fp_log = NULL;
	  res->status = DB_INTERNAL_ERROR;
	  goto bailout;
	}

      if (!xdr_u_int (&xdrs, &numattrs))
	{
	  xdr_destroy (&xdrs);
	  fclose (tobj->fp_log);
	  tobj->fp_log = NULL;
	  res->status = DB_INTERNAL_ERROR;
	  goto bailout;
	}

      for (i = 0; i < numattrs; ++i)
	if (!__nisdb_xdr_attr (&xdrs, &attrs[i]))
	  {
	    xdr_destroy (&xdrs);
	    fclose (tobj->fp_log);
	    tobj->fp_log = NULL;
	    res->status = DB_INTERNAL_ERROR;
	    goto bailout;
	  }

      xdr_destroy (&xdrs);
    }

 bailout:
  if (hashes)
    free (hashes);
  gettimeofday (&t_end, NULL);
  res->ticks += (t_end.tv_usec - t_start.tv_usec) +
    ((t_end.tv_sec - t_start.tv_sec) * 1000000);

  return res;
}
