/*
 * $Id: swflib.hxx,v 1.1 2002/02/15 23:44:28 skavish Exp $
 *
 * ==========================================================================
 *
 * The JGenerator Software License, Version 1.0
 *
 * Copyright (c) 2000 Dmitry Skavish (skavish@usa.net). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *    "This product includes software developed by Dmitry Skavish
 *     (skavish@usa.net, http://www.flashgap.com/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The name "The JGenerator" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact skavish@usa.net.
 *
 * 5. Products derived from this software may not be called "The JGenerator"
 *    nor may "The JGenerator" appear in their names without prior written
 *    permission of Dmitry Skavish.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL DMITRY SKAVISH OR THE OTHER
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SWFLIB_H
#define SWFLIB_H

#include "geom.hxx"
#include "vector.hxx"

typedef signed char sbyte;
typedef unsigned char byte;

/**
 * Returns maximum element of array.
 *
 * @param arr    array to be searched for maximum
 * @param size   size of the array
 * @return maximim element of the array
 */
int getMax( int arr[], int size );

/**
 * Returns maximum of absolute values of two numbers.
 *
 * @param a      specified number
 * @param b      specified number
 * @return maximum of two numbers
 */
int getMax( int a, int b );

/**
 * Returns maximum of absolute values of three numbers.
 *
 * @param a      specified number
 * @param b      specified number
 * @param c      specified number
 * @return maximum of three numbers
 */
int getMax( int a, int b, int c );

/**
 * Returns maximum of absolute values of four numbers.
 *
 * @param a      specified number
 * @param b      specified number
 * @param c      specified number
 * @param d      specified number
 * @return maximum of four specified numbers
 */
int getMax( int a, int b, int c, int d );

/**
 * Returns number of significant bits for signed integer.
 *
 * @param v      signed integer
 * @return number of significant bits for signed integer
 */
int getMinBitsS( int v );

/**
 * Returns number of significant bits for unsigned integer.
 *
 * @param v      unsigned integer
 * @return number of significant bits for unsigned integer
 */
int getMinBitsU( unsigned int v );

/* --------------------------------------------------------------------------
 * Double, float, fixed, twips conversion helpers
 * -------------------------------------------------------------------------- */

#define FRACTION_DBL ((double) 0x10000)
#define FRACTION_FLT ((float) 0x10000)

inline double fixed2double( int value ) {
    return (double)value / FRACTION_DBL;
}

inline float fixed2float( int value ) {
    return (float)value / FRACTION_FLT;
}

inline int double2fixed( double value ) {
    return (int) (value * FRACTION_DBL);
}

inline int float2fixed( float value ) {
    return (int) (value * FRACTION_FLT);
}

inline double twips2double( int value ) {
    return (double)value / 20.0;
}

inline float twips2float( int value ) {
    return (float)value / 20.0f;
}

inline int double2twips( double value ) {
    return (int) (value * 20.0);
}

inline int float2twips( float value ) {
    return (int) (value * 20.0f);
}


/**
 * FlashBuffer class
 */

class FlashBuffer {
private:
    int bitBuf;
    int bitPos;
    byte* buf;
    int capacity;
    int pos;
    int size;

public:

    FlashBuffer( int capacity ) {
        init(new byte[capacity], capacity, 0, 0);
    }

    FlashBuffer( byte* buf, int size ) {
        init(buf, size, 0, size);
    }

    FlashBuffer( byte* buf, int pos, int size ) {
        init(buf, size, pos, size);
    }

    FlashBuffer( byte* buf, int capacity, int pos, int size ) {
        init(buf, capacity, pos, size);
    }

    void init( byte* buf, int capacity, int pos, int size ) {
        this->buf = buf;
        this->pos = pos;
        this->size = size;
        this->capacity = capacity;
        bitBuf = 0;
        bitPos = 0;
    }

    ~FlashBuffer() {
        if( buf != NULL ) delete [] buf;
    }

    void ensureCapacity( int cap );
    FlashBuffer* getCopy();

    int getPos() { return pos; }

    int getSize() { return size; }

    void setPos( int pos ) {
        this->pos = pos;
        if( pos > size ) size = pos;
    }

    void incPos() {
        if( ++pos > size ) size = pos;
    }

    void setSize( int size ) {
        this->size = size;
    }

    void skip( int inc ) {
        setPos( pos + inc );
    }

    byte* getBuf() {
        return buf;
    }

    void writeByteAt( int b, int pos ) {
        buf[pos] = (byte) b;
    }

    void writeWordAt( int b, int pos ) {
        buf[pos] = (byte) b;
        buf[pos+1] = (byte) (b>>8);
    }

    void writeDWordAt( int b, int pos ) {
        buf[pos] = (byte) b;
        buf[pos+1] = (byte) (b>>8);
        buf[pos+2] = (byte) (b>>16);
        buf[pos+3] = (byte) (b>>24);
    }

    void writeByte( int b ) {
        ensureCapacity( pos );
        buf[pos] = (byte) b;
        incPos();
    }

    void writeWord( int b ) {
        ensureCapacity( pos+1 );
        buf[pos] = (byte) b;
        buf[pos+1] = (byte) (b>>8);
        setPos( pos+2 );
    }

    void writeDWord( int b ) {
        ensureCapacity( pos+3 );
        buf[pos] = (byte) b;
        buf[pos+1] = (byte) (b>>8);
        buf[pos+2] = (byte) (b>>16);
        buf[pos+3] = (byte) (b>>24);
        setPos( pos+4 );
    }

    void writeFOB( FlashBuffer* fob ) {
        writeFOB(*fob);
    }

    void writeFOB( FlashBuffer& fob ) {
        int len = fob.getSize();
        ensureCapacity( pos+len );
        memcpy(buf+pos, fob.getBuf(), len);
        setPos( pos+len );
    }

    void writeArray( byte b[], int off, int len ) {
        ensureCapacity( pos+len );
        memcpy(buf+pos, b+off, len);
        setPos( pos+len );
    }

    void writeStringZ( char s[] );

    void writeStringL( char s[] );

    void writeTag( int tagCode, int tagSize ) {
        if( tagSize >= 0x3f ) {
            writeLongTag( tagCode, tagSize );
        } else {
            writeWord( (tagCode<<6) | tagSize );
        }
    }

    void writeLongTag( int tagCode, int tagSize ) {
        writeWord( (tagCode<<6) | 0x3f );
        writeDWord( tagSize );
    }

    void writeShortTagAt( int tagCode, int tagSize, int pos ) {
        writeWordAt( (tagCode<<6) | tagSize, pos );
    }

    void writeLongTagAt( int tagCode, int tagSize, int pos ) {
        writeWordAt( (tagCode<<6) | 0x3f, pos );
        writeDWordAt( tagSize, pos+2 );
    }

    void writeBit( int b ) {
        writeBits( b, 1 );
    }

    void writeBool( bool b ) {
        writeBits( b?1:0, 1 );
    }

    void writeBits( int v, int len );

    void flushBits() {
        if( bitPos != 0 ) {
            int bb = bitBuf << (8-bitPos);
            writeByte( bb );
        }
        bitBuf = 0;
        bitPos = 0;
    }

    void initBits() {
        bitBuf = 0;
        bitPos = 0;
    }

    void write( AffineTransform& m );
    void write( AffineTransform* m ) { write(*m); }
    void write( Rectangle2D& r );
    void write( Rectangle2D* r ) { write(*r); }

};


/**
 * FlashItem
 */
class FlashItem {

public:

    /**
     * Writes object to flash buffer
     *
     * @param fob    flash buffer to write
     */
    virtual void write( FlashBuffer& fob ) =0;

    virtual bool isStyleChange() { return false; }

};

class ItemsVector : public Vector<FlashItem*>, public FlashItem {

public:

    ItemsVector( int capacity ) : Vector<FlashItem*>(capacity) {}

    /**
     * Writes object to flash buffer
     *
     * @param fob    flash buffer to write
     */
    void write(FlashBuffer &fob) {
        int sz = size();
        for( int i=0; i<sz; i++ ) {
            elementAt(i)->write(fob);
        }
    }

};

/**
 * StyleChangeRecord
 */
class StyleChangeRecord : public FlashItem {

#define NEW_STYLES 0x10
#define LINESTYLE  0x08
#define FILLSTYLE1 0x04
#define FILLSTYLE0 0x02
#define MOVETO     0x01

private:
    int flags;
    int deltaX;
    int deltaY;
    int fillStyle0;
    int fillStyle1;
    int lineStyle;

public:

    StyleChangeRecord() {
        flags = 0;
        fillStyle0 = fillStyle1 = lineStyle = 0;
    }

    int getFlags() {
        return flags;
    }

    void setFlags( int flags ) {
        this->flags = flags;
    }

    void addFlags( int flags ) {
        this->flags |= flags;
    }

    int getDeltaX() {
        return deltaX;
    }

    void setDeltaX( int deltaX ) {
        this->deltaX = deltaX;
    }

    int getDeltaY() {
        return deltaY;
    }

    void setDeltaY( int deltaY ) {
        this->deltaY = deltaY;
    }

    int getFillStyle0() {
        return fillStyle0;
    }

    void setFillStyle0( int fillStyle0 ) {
        this->fillStyle0 = fillStyle0;
    }

    int getFillStyle1() {
        return fillStyle1;
    }

    void setFillStyle1( int fillStyle1 ) {
        this->fillStyle1 = fillStyle1;
    }

    int getLineStyle() {
        return lineStyle;
    }

    void setLineStyle( int lineStyle ) {
        this->lineStyle = lineStyle;
    }

    void write( FlashBuffer& fob ) {
        write(fob, 1, 0);
    }

    void write( FlashBuffer& fob, int nFillBits, int nLineBits ) {
        fob.writeBits(flags, 6);
        if( (flags&MOVETO) != 0 ) {
            int nBits = getMinBitsS( getMax(deltaX, deltaY) );
            fob.writeBits(nBits, 5);
            fob.writeBits(deltaX, nBits);
            fob.writeBits(deltaY, nBits);
        }
        if( (flags&FILLSTYLE0) != 0 ) {
            fob.writeBits(fillStyle0, nFillBits);
        }
        if( (flags&FILLSTYLE1) != 0 ) {
            fob.writeBits(fillStyle1, nFillBits);
        }
        if( (flags&LINESTYLE) != 0 ) {
            fob.writeBits(lineStyle, nLineBits);
        }
    }

    virtual bool isStyleChange() { return true; }
};

/**
 * Straight-edge record.
 * <P>
 * The straight-edge record stores the edge as an X-Y delta.
 * The delta is added to the current drawing position,
 * and this becomes the new drawing position.
 * The edge is rendered between the old and new drawing positions.
 * <P>
 * Straight-edge records support three types of line:
 * <OL>
 * <LI>General lines.
 * <LI>Horizontal lines.
 * <LI>Vertical lines.
 * </OL>
 * General lines store both X & Y deltas, the horizontal and vertical
 * lines store only the X delta and Y delta respectively.
 *
 * @author Dmitry Skavish
 */
class StrightEdgeRecord : public FlashItem {

#define GENERAL_LINE 0
#define VERT_LINE    1
#define HORIZ_LINE   2

    int type;         // type of this record
    int deltaX;       // delta X
    int deltaY;       // delta Y

public:

    StrightEdgeRecord() {
    }

    int getType() {
        return type;
    }

    void setType( int type ) {
        this->type = type;
    }

    int getDeltaX() {
        return deltaX;
    }

    void setDeltaX( int deltaX ) {
        this->deltaX = deltaX;
    }

    int getDeltaY() {
        return deltaY;
    }

    void setDeltaY( int deltaY ) {
        this->deltaY = deltaY;
    }

    /**
     * Creates general line
     *
     * @param deltaX new X delta
     * @param deltaY new Y delta
     * @return general line
     */
    static StrightEdgeRecord* newLine( int deltaX, int deltaY ) {
        StrightEdgeRecord* sr = new StrightEdgeRecord();
        sr->setType( GENERAL_LINE );
        sr->setDeltaX( deltaX );
        sr->setDeltaY( deltaY );
        return sr;
    }

    /**
     * Creates horizontal line
     *
     * @param deltaX new X delta
     * @return horizontal line
     */
    static StrightEdgeRecord* newHLine( int deltaX ) {
        StrightEdgeRecord* sr = new StrightEdgeRecord();
        sr->setType( HORIZ_LINE );
        sr->setDeltaX( deltaX );
        return sr;
    }

    /**
     * Creates vertical line
     *
     * @param deltaY new Y delta
     * @return vertical line
     */
    static StrightEdgeRecord* newVLine( int deltaY ) {
        StrightEdgeRecord* sr = new StrightEdgeRecord();
        sr->setType( VERT_LINE );
        sr->setDeltaY( deltaY );
        return sr;
    }

    void write( FlashBuffer& fob ) {
        fob.writeBits(0x3, 2);
        switch( type ) {
            case GENERAL_LINE: {
                int nBits = getMinBitsS( getMax(deltaX, deltaY) );
                if( nBits < 3 ) nBits = 3;
                fob.writeBits(nBits-2, 4);
                fob.writeBit(1);
                fob.writeBits(deltaX, nBits);
                fob.writeBits(deltaY, nBits);
                break;
            }
            case VERT_LINE: {
                int nBits = getMinBitsS( deltaY );
                if( nBits < 3 ) nBits = 3;
                fob.writeBits(nBits-2, 4);
                fob.writeBit(0);
                fob.writeBit(1);
                fob.writeBits(deltaY, nBits);
                break;
            }
            case HORIZ_LINE: {
                int nBits = getMinBitsS( deltaX );
                if( nBits < 3 ) nBits = 3;
                fob.writeBits(nBits-2, 4);
                fob.writeBit(0);
                fob.writeBit(0);
                fob.writeBits(deltaX, nBits);
                break;
            }
        }
    }

};

/**
 * CurveEdge record.
 * <P>
 * SWF differs from most vector file formats by using Quadratic Bezier
 * curves rather than Cubic Bezier curves. PostScript uses Cubic Beziers,
 * as do most drawing applications, such as Illustrator, FreeHand and Corel Draw.
 * SWF uses Quadratic Bezier curves because they can be stored more compactly,
 * and can be rendered more efficiently.
 * <P>
 * A Quadratic Bezier curve has 3 points.  Two on-curve anchor points, and one off-curve
 * control point. A Cubic Bezier curve has 4 points.  Two on-curve anchor points,
 * and two off-curve control points.
 * <P>
 * The curved-edge record stores the edge as two X-Y deltas.
 * The three points that define the Quadratic Bezier are calculated like this:
 * <UL>
 * <LI>The first anchor point is the current drawing position.
 * <LI>The control point is the current drawing position + ControlDelta.
 * <LI>The last anchor point is the current drawing position + ControlDelta + AnchorDelta.
 * </UL>
 * The last anchor point becomes the current drawing position.
 *
 * @author Dmitry Skavish
 */
class CurvedEdgeRecord : public FlashItem {

    int controlDeltaX;
    int controlDeltaY;
    int anchorDeltaX;
    int anchorDeltaY;

public:

    CurvedEdgeRecord() {}

    CurvedEdgeRecord( int controlDeltaX, int controlDeltaY, int anchorDeltaX, int anchorDeltaY ) {
        setControlDeltaX(controlDeltaX);
        setControlDeltaY(controlDeltaY);
        setAnchorDeltaX(anchorDeltaX);
        setAnchorDeltaY(anchorDeltaY);
    }

    int getControlDeltaX() {
        return controlDeltaX;
    }
    void setControlDeltaX( int controlDeltaX ) {
        this->controlDeltaX = controlDeltaX;
    }

    int getControlDeltaY() {
        return controlDeltaY;
    }
    void setControlDeltaY( int controlDeltaY ) {
        this->controlDeltaY = controlDeltaY;
    }

    int getAnchorDeltaX() {
        return anchorDeltaX;
    }
    void setAnchorDeltaX( int anchorDeltaX ) {
        this->anchorDeltaX = anchorDeltaX;
    }

    int getAnchorDeltaY() {
        return anchorDeltaY;
    }
    void setAnchorDeltaY( int anchorDeltaY ) {
        this->anchorDeltaY = anchorDeltaY;
    }

    void write( FlashBuffer& fob ) {
        fob.writeBits(0x2, 2);
        int nBits = getMinBitsS( getMax(controlDeltaX, controlDeltaY, anchorDeltaX, anchorDeltaY) );
        if( nBits < 3 ) nBits = 3;
        fob.writeBits(nBits-2, 4);
        fob.writeBits(controlDeltaX, nBits);
        fob.writeBits(controlDeltaY, nBits);
        fob.writeBits(anchorDeltaX, nBits);
        fob.writeBits(anchorDeltaY, nBits);
    }

};

/**
 * Shape
 */
class Shape : public FlashItem {

private:

    ItemsVector shape_records;

    // last drawing positions, used to keep track of drawing pen
    int last_x;
    int last_y;

public:

    Shape() : shape_records(ItemsVector(30)), last_x(2000000000), last_y(2000000000) {}

    /**
     * Returns vector of shape records.
     *
     * @return vector of shape records
     */
    ItemsVector* getShapeRecords() {
        return &shape_records;
    }

    /**
     * Draws curve record.<p>
     * All coordinates are in twixels.
     *
     * @param cx     X control point
     * @param cy     Y control point
     * @param ax     X anchor point
     * @param ay     Y anchor point
     */
    void drawCurveTo( int cx, int cy, int ax, int ay ) {
        shape_records.addElement( new CurvedEdgeRecord(cx-last_x, cy-last_y, ax-cx, ay-cy) );
        last_x = ax;
        last_y = ay;
    }

    /**
     * Draws curve record.<p>
     * All coordinates are in twixels.
     *
     * @param ax1    X anchor point 1
     * @param ay1    Y anchor point 1
     * @param cx     X control point
     * @param cy     Y control point
     * @param ax2    X anchor point 2
     * @param ay2    Y anchor point 2
     */
    void drawCurve( int ax1, int ay1, int cx, int cy, int ax2, int ay2 ) {
        movePenTo(ax1, ay1);
        drawCurveTo(cx, cy, ax2, ay2);
    }

    /**
     * Draws curve record.<p>
     * All coordinates are in twixels.
     *
     * @param anchor0 first anchor point
     * @param control control point
     * @param anchor1 second anchor point
     */
    void drawCurve( Point2D& anchor1, Point2D& control, Point2D& anchor2 ) {
        drawCurve( (int) anchor1.getX(), (int) anchor1.getY(),
                   (int) control.getX(), (int) control.getY(),
                   (int) anchor2.getX(), (int) anchor2.getY() );
    }

    /**
     * Draws a straight line from current position to the specified one.<p>
     * All coordinates are in twixels.
     *
     * @param x      X of end of line
     * @param y      Y of end of line
     */
    void drawLineTo( int x, int y ) {
        int deltaX = x-last_x;
        int deltaY = y-last_y;
        if( deltaX == 0 ) {
            if( deltaY == 0 ) return;
            shape_records.addElement( StrightEdgeRecord::newVLine(deltaY) );
        } else if( deltaY == 0 ) {
            shape_records.addElement( StrightEdgeRecord::newHLine(deltaX) );
        } else {
            shape_records.addElement( StrightEdgeRecord::newLine(deltaX,deltaY) );
        }
        last_x = x;
        last_y = y;
    }

    /**
     * Draws a straight line from current position to the specified one.<p>
     * All coordinates are in twixels.
     *
     * @param p1     end of line
     */
    void drawLineTo( Point2D& p1 ) {
        drawLineTo( (int) p1.getX(), (int) p1.getY() );
    }

    /**
     * Draws a straight line specified by two points.
     * <P>
     * All coordinates are in twixels.
     *
     * @param x1     X of the beginning of the line
     * @param y1     Y of the beginning of the line
     * @param x2     X of the end of the line
     * @param y2     Y of the end of the line
     */
    void drawLine( int x1, int y1, int x2, int y2 ) {
        movePenTo( x1, y1 );
        drawLineTo( x2, y2 );
    }

    /**
     * Draws a straight line specified by two points.
     * <P>
     * All coordinates are in twixels.
     *
     * @param p0     first point
     * @param p1     second point
     */
    void drawLine( Point2D& p0, Point2D& p1 ) {
        drawLine( (int) p0.getX(), (int) p0.getY(), (int) p1.getX(), (int) p1.getY() );
    }

    /**
     * Draws a rectangle specified by its top-left corner and width and height
     * <p>
     * All coordinates are in twixels.
     *
     * @param x      x coordinates of top-left corner of the rectangle
     * @param y      y coordinates of top-left corner of the rectangle
     * @param width  width of the rectangle
     * @param height height of the rectangle
     */
    void drawRectangle( int x, int y, int width, int height ) {
        movePenTo( x, y );
        drawLineTo( x+width, y );
        drawLineTo( x+width, y+height );
        drawLineTo( x, y+height );
        drawLineTo( x, y );
    }

    /**
     * Draws a rectangle specified by {@link java.awt.geom.Rectangle2D}
     * <p>
     * All coordinates are in twixels.
     *
     * @param r      specified rectangle
     */
    void drawRectangle( Rectangle2D& r ) {
        drawRectangle( (int) r.getX(), (int) r.getY(), (int) r.getWidth(), (int) r.getHeight() );
    }

    /**
     * Moves pen to the specified position.<p>
     * All coordinates are ABSOLUTE and are in twixels.
     *
     * @param x      new current X
     * @param y      new current Y
     */
    void movePenTo( int x, int y ) {
        if( last_x != x || last_y != y ) {
            StyleChangeRecord* sc = getStyleChange();
            sc->addFlags( MOVETO );
            sc->setDeltaX(x);
            sc->setDeltaY(y);
            last_x = x;
            last_y = y;
        }
    }

    /**
     * Moves pen to the specified point.<p>
     * All coordinates are ABSOLUTE and are in twixels!
     *
     * @param p     new pen position
     */
    void movePenTo( Point2D& p ) {
        movePenTo( (int) p.getX(), (int) p.getY() );
    }

    /**
     * Add style change record
     *
     * @param scr    add this record
     */
    StyleChangeRecord* addStyleChangeRecord( StyleChangeRecord* scr ) {
        if( shape_records.size() == 0 ) {
            scr->addFlags(FILLSTYLE1|LINESTYLE);
            scr->setFillStyle1(1);
        }
        shape_records.addElement(scr);
        return scr;
    }

    /**
     * Add style change record
     */
    StyleChangeRecord* addStyleChangeRecord() {
        return addStyleChangeRecord(new StyleChangeRecord());
    }

    StyleChangeRecord* getStyleChange() {
        if( shape_records.size() > 0 ) {
            FlashItem* item = (FlashItem*) shape_records.elementAt( shape_records.size()-1 );
            if( item->isStyleChange() ) return (StyleChangeRecord*) item;
        }
        return addStyleChangeRecord();
    }

    /**
     * Writes this shape records into specified buffer, writes end of records and flush the bits.
     * Number of fill and line style bits is considered to be 0 (zero).
     *
     * @param fob    buffer
     */
    void write( FlashBuffer& fob ) {
        //printf("draw shape: records.size()=%d,", shape_records.size());
        fob.writeByte(0x10);   // fill and line bits
        fob.initBits();
        shape_records.write(fob);
        fob.writeBits(0, 6);
        fob.flushBits();
    }
};

void writeFlashFile( char * fft_file, FlashBuffer* font_fob );

#endif


