# -*- ruby -*-
# $Id: tml.rb,v 1.25 2004/01/22 06:16:25 tommy Exp $
#
# Copyright (C) 2003 TOMITA Masahiro
# tommy@tmtm.org
#

TML_VERSION = "0.4.2"

$var_dir = "/var/spool/tml"
$smtp_server = "localhost"
$smtp_port = "smtp"
$lock_timeout = 5*60		# 5min
$max_lock_time = 30*60		# 30min
$auto_aliases = true
$newaliases = nil		# Ex. for postfix "postalias #{TML::etc_dir}/aliases"
$strip_received = false
$logging = true

$ml_dir = $etc_dir = $deleted_dir = $log_file = nil

$default_not_member = false
$default_subject_mlname = true
$default_counter_width = 5
$default_remote_command = false
$default_remote_admin = false
$default_spool = true
$default_web_interface = 0
$default_need_confirm_command = "subscribe,admin"
$default_allowed_command = "all"
$default_anonymous_command = "subscribe,confirm"
$default_admin_command = "all"
$default_logging = true
$default_replyto_type = "ml"

$access_priv = [
  :permit, :permit_post, :permit_subscribe,
  :reject, :reject_post, :reject_subscribe,
]

class TML

  class Error < StandardError
    def initialize(id, msg)
      TML::log msg
      @id, @msg = id, msg
      super msg
    end
    attr_reader :id, :msg
  end

  class Lock
    def Lock::init(lock_f)
      File::open(lock_f, "w").close
    end

    def Lock::callback(lock_f, orig_lock_f)
      proc do
	if File::exist? lock_f then
	  TML::log "unlock: #{lock_f}"
	  File::rename lock_f, orig_lock_f
	end
      end
    end

    def initialize(lock_f)
      st = Time::now
      loop do
	begin
	  @lock_f = lock_f+"."+Time::now.to_i.to_s
	  @orig_lock_f = lock_f
	  TML::log "lock: #{@lock_f}"
	  File::rename lock_f, @lock_f
	  ObjectSpace::define_finalizer(self, Lock::callback(@lock_f, lock_f))
	  break
	rescue Errno::ENOENT
	  if st+$lock_timeout < Time::now then
	    raise Error::new(:ETIMEDOUT, "lock timed out")
	  end
	  Dir::glob(lock_f+".*") do |f|
	    if f.split(/\./)[-1].to_i < Time::now.to_i-$max_lock_time then
	      TML::log "#{f}: max_lock_time(#{$max_lock_time}) exceeded"
	      File::rename f, lock_f rescue nil
	      retry
	    end
	  end
	  sleep 1
	end
      end
    end

    def unlock()
      TML::log "unlock: #{@lock_f}"
      File::rename @lock_f, @orig_lock_f
    end
  end

  class Attr
    def initialize(name, type)
      @name, @type = name, type
      @value = nil
    end

    attr_reader :name, :type, :value

    def set(value)
      case @type
      when :Integer
	if value.kind_of? Integer then
	  @value = value
	elsif value.kind_of? String then
	  unless value =~ /^\d+$/ then
	    raise Error::new(:EINVAL, "#{value}: integer value required")
	  end
	  @value = value.to_i
	else
	  raise Error::new(:EINVAL, "#{value}: invalid type")
	end
      when :Boolean
	if value.kind_of? TrueClass or value.kind_of? FalseClass then
	  @value = value
	elsif value.kind_of? NilClass then
	  @value = false
	elsif value.kind_of? String then
	  case value.downcase
	  when "true", "yes"
	    @value = true
	  when "false", "no"
	    @value = false
	  else
	    raise Error::new(:EINVAL, "#{value}: invalid value")
	  end
	else
	  raise Error::new(:EINVAL, "#{value}: invalid type")
	end
      when :String
	@value = value
      else
	raise "#{value}: invalid type"
      end
    end
  end

  class Passwd

    def initialize()
      @pwd_f = "#{TML::etc_dir}/passwd"
      Lock::new "#{TML::etc_dir}/lock"
      @list = {}
      unless File::exist? @pwd_f then
	File::open(@pwd_f, File::WRONLY|File::CREAT, 0600).close
      end
      begin
        @list = Marshal::load File::open(@pwd_f){|f| f.read}
      rescue
        IO::foreach(@pwd_f) do |l|
          next if l[0] == ?\# or l =~ /^\s*$/
          addr, pw = l.chomp.split(/\t+/)
          @list[addr] = pw
        end
      end
    end

    def have?(addr)
      @list.key? addr
    end

    def list()
      @list.keys.sort
    end

    def add(addr, pw)
      if @list.key? addr then
	raise TML::Error::new(:EEXIST, "#{addr}: already exist")
      end
      @list[addr] = pw
      flush
      TML::log "add password: #{addr}"
    end

    def bye(addr)
      unless @list.key? addr then
	raise TML::Error::new(:ENOENT, "#{addr}: not found")
      end
      @list.delete addr
      flush
      TML::log "delete password: #{addr}"
    end

    def password(addr, pw)
      unless @list.key? addr then
	raise TML::Error::new(:ENOENT, "#{addr}: not found")
      end
      @list[addr] = pw
      flush
      TML::log "set password: #{addr}"
    end

    def auth(addr, pw)
      unless @list[addr] == pw then
	TML::log "auth failed: #{addr}"
	return false
      end
      true
    end

    def flush()
      File::open("#{@pwd_f}.new", File::WRONLY|File::CREAT, 0600) do |f|
        f.write @list.map{|addr,pw|"#{addr}\t#{pw}\n"}.join
      end
      File::rename(@pwd_f, "#{@pwd_f}.bak")
      File::rename("#{@pwd_f}.new", @pwd_f)
    end

  end

  @@cache = {}
  @@attributes = {
    :sender		=> :String,
    :overview		=> :String,
    :not_member		=> :Boolean,
    :subject_mlname	=> :Boolean,
    :counter_width	=> :Integer,
    :remote_command	=> :Boolean,
    :remote_admin	=> :Boolean,
    :spool		=> :Boolean,
    :web_interface	=> :Integer,
    :need_confirm_command => :String,
    :allowed_command	=> :String,
    :anonymous_command	=> :String,
    :admin_command	=> :String,
    :logging		=> :Boolean,
    :replyto_type	=> :String,
  }

  def TML::ml_dir()
    $ml_dir || "#{$var_dir}/ml"
  end

  def TML::etc_dir()
    $etc_dir || "#{$var_dir}/etc"
  end

  def TML::deleted_dir()
    $deleted_dir || "#{$var_dir}/deleted"
  end

  def TML::log_file()
    $log_file || "#{$var_dir}/log"
  end

  def TML::log(msg)
    return unless $logging
    File::open(TML::log_file, "a") do |f|
      f.puts "#{TML::log_date} #{File::basename $0}[#{$$}] #{msg}"
    end
  end

  def TML::log_date()
    Time::now.strftime("%Y-%m-%d %H:%M:%S")
  end

  def TML::mkdir_p(dir)
    dirs = dir.split(/\/+/)
    if dirs[0] == "" then dirs[0] = "/" end
    1.upto(dirs.size) do |i|
      d = dirs[0,i].join("/")
      next if File::exist? d
      Dir::mkdir d
    end
  end

  def TML::list()
    ret = []
    Dir::foreach(TML::ml_dir) do |f|
      next if f[0] == ?. or f[0] == ?-
      ret << f
    end
    ret
  end

  def TML::create(mlname, admin)
    if TML::exist? mlname then
      raise TML::Error::new(:EEXIST, "#{mlname}: already exist")
    end
    TML::log "create: #{mlname}"
    TML::mkdir_p "#{TML::ml_dir}/#{mlname}"
    TML::mkdir_p "#{TML::ml_dir}/#{mlname}/tmp"
    File::open("#{TML::ml_dir}/#{mlname}/conf", File::WRONLY|File::CREAT|File::EXCL, 0644).close
    Lock::init "#{TML::ml_dir}/#{mlname}/lock"
    tml = TML::new mlname
    tml.attr[:sender].set "#{mlname}-admin@#{$domain}"
    tml.attr[:overview].set nil
    tml.attr[:not_member].set $default_not_member
    tml.attr[:subject_mlname].set $default_subject_mlname
    tml.attr[:counter_width].set $default_counter_width
    tml.attr[:remote_admin].set $default_remote_admin
    tml.attr[:spool].set $default_spool
    tml.attr[:web_interface].set $default_web_interface
    tml.attr[:remote_command].set $default_remote_command
    tml.attr[:need_confirm_command].set $default_need_confirm_command
    tml.attr[:allowed_command].set $default_allowed_command
    tml.attr[:anonymous_command].set $default_anonymous_command
    tml.attr[:admin_command].set $default_admin_command
    tml.attr[:logging].set $default_logging
    tml.attr[:replyto_type].set $default_replyto_type
    tml.write_conf
    tml.add_admin admin
    tml.add_alias if $auto_aliases
    tml
  end

  def TML::drop(n)
    ml = TML::new n
    ml.del_alias if $auto_aliases
    ml.close
    TML::log "drop: #{n}"
    unless TML::exist? n then
      raise TML::Error::new(:ENOENT, "#{n}: not exist")
    end
    ts = Time::now.strftime("%Y%m%d%H%M%S")
    File::rename("#{TML::ml_dir}/#{n}", "#{TML::deleted_dir}/#{n}-#{ts}")
  end

  def TML::new(n)
    if @@cache.has_key? n then
      return @@cache[n]
    end
    tml = super n
    @@cache[n] = tml
    tml
  end

  def TML::[](n)
    TML::new n
  end

  def TML::exist?(n)
    File::exist? "#{TML::ml_dir}/#{n}/conf"
  end

  def initialize(n)
    unless TML::exist? n then
      err :ENOENT, "#{n}: not exist"
    end
    @dir = "#{TML::ml_dir}/#{n}"
    @name = n
    @conf = "#{@dir}/conf"
    @cnt_f = "#{@dir}/counter"
    @admin_f = "#{@dir}/admin"
    @member_f = "#{@dir}/member"
    @access_f = "#{@dir}/access"
    @children_f = "#{@dir}/children"
    @parents_f = "#{@dir}/parents"
    @summary_f = "#{@dir}/summary"
    @spool_d = "#{@dir}/spool"
    @counter = nil
    @admins = nil
    @members = nil
    @children = nil
    @parents = nil
    @attr = {}
    @@attributes.each do |n,t|
      @attr[n] = Attr::new(n, t)
    end
    @lock_f = Lock::new "#{@dir}/lock"
    read_conf

    # clear tmp file
    Dir::glob("#{@dir}/tmp/*") do |f|
      File::unlink f if File::mtime(f) < Time::now-24*60*60
    end
  end

  def close()
    @lock_f.unlock
    @name = @conf = @cnt_f = @member_f = @children_f = @parents_f = nil
    @counter = @members = @children = @parents = nil
  end

  attr_reader :name, :attr

  def log(msg)
    TML::log "#{@name}: #{msg}"
    return unless @attr[:logging]
    File::open("#{@dir}/log", "a") do |f|
      f.puts "#{TML::log_date} #{File::basename $0}[#{$$}] #{msg}"
    end
  end

  # Counter

  def counter()
    return @counter if @counter
    if File::exist? @cnt_f then
      @counter = File::open(@cnt_f, "r") do |f| f.read.to_i end
    else
      File::open(@cnt_f, "w") do |f| f.puts 0 end
      @counter = 0
    end
  end

  def counter=(cnt)
    File::open("#{@cnt_f}.new", "w") do |f| f.puts cnt end
    File::rename("#{@cnt_f}.new", @cnt_f)
    @counter = cnt
    log "set counter: #{cnt}"
  end

  def countup()
    counter
    @counter += 1
    File::open("#{@cnt_f}.new", "w") do |f| f.puts @counter end
    File::rename("#{@cnt_f}.new", @cnt_f)
    @counter
  end

  # Member

  def members()
    return @members.keys.sort if @members
    @members = {}
    unless File::exist? @member_f then
      return []
    end
    begin
      @members = Marshal::load File::open(@member_f){|f| f.read}
    rescue
      IO::foreach(@member_f) do |l|
        next if l[0] == ?\# or l =~ /^\s*$/
        addr = l.chomp
        @members[addr] = true
      end
    end
    @members.keys.sort
  end

  def all_members()
    m = []
    m += members
    children.each do |c|
      m += c.all_members
    end
    m.uniq
  end

  def have?(addr)
    members.include? addr
  end

  def add(*addrs)
    m = {}
    addrs.flatten.each do |addr|
      if self.have? addr then
	err :EEXIST, "#{addr}: already exist"
      end
      m[addr] = true
    end
    @members.update m
    flush_member
    log "add member: #{addrs.join(",")}"
  end

  def bye(*addrs)
    addrs.flatten.each do |addr|
      unless self.have? addr then
	err :ENOENT, "#{addr}: not exist"
      end
    end
    addrs.each do |addr|
      @members.delete addr
    end
    flush_member
    log "bye member: #{addrs.join(",")}"
  end

  def flush_member()
    File::open("#{@member_f}.new", "w") do |f|
      f.write @members.map{|addr,|"#{addr}\n"}.join
    end
    File::rename(@member_f, "#{@member_f}.bak") if File::exist? @member_f
    File::rename("#{@member_f}.new", @member_f)
  end

  # ML tree

  def children()
    return @children if @children
    unless File::exist? @children_f then
      @children = []
      return @children
    end
    begin
      @children = Marshal::load(File::open(@children_f){|f| f.read}).map{|r| TML::new(r)}
    rescue
      @children = []
      IO::foreach(@children_f) do |l|
        next if l[0] == ?\# or l =~ /^\s*$/
        @children << TML::new(l.chomp)
      end
    end
    @children
  end

  def addml(ml)
    ml = TML::new ml if ml.kind_of? String
    if children.include? ml then
      err :EEXIST, "#{ml.name}: already exist"
    end
    @children << ml
    flush_children
    ml.add_parent self
    log "add child: #{ml.name}"
  end

  def byeml(ml)
    ml = TML::new ml if ml.kind_of? String
    unless children.include? ml then
      err :ENOENT, "#{ml.name}: not exist"
    end
    @children.delete ml
    flush_children
    ml.bye_parent self
    log "bye child: #{ml.name}"
  end

  def flush_children()
    File::open("#{@children_f}.new", "w") do |f|
      f.write @children.map{|c|"#{c.name}\n"}.join
    end
    File::rename(@children_f, "#{@children_f}.bak") if File::exist? @children_f
    File::rename("#{@children_f}.new", @children_f)
  end

  def parents()
    @parents if @parents
    unless File::exist? @parents_f then
      @parents = []
      return @parents
    end
    begin
      @parents = Marshal::load(File::open(@parents_f){|f|f.read}).map{|r| TML::new(r)}
    rescue
      @parents = []
      IO::foreach(@parents_f) do |l|
        next if l[0] == ?\# or l =~ /^\s*$/
        @parents << TML::new(l.chomp)
      end
    end
    @parents
  end

  def add_parent(ml)
    if parents.include? ml then
      err :EEXIST, "#{ml.name}: already exist"
    end
    @parents << ml
    flush_parents
  end

  def bye_parent(ml)
    unless parents.include? ml then
      err :ENOENT, "#{ml.name}: not exist"
    end
    @parents.delete ml
    flush_parents
  end

  def flush_parents()
    File::open("#{@parents_f}.new", "w") do |f|
      f.write @parents.map{|c|"#{c.name}\n"}.join
    end
    File::rename(@parents_f, "#{@parents_f}.bak") if File::exist? @parents_f
    File::rename("#{@parents_f}.new", @parents_f)
  end

  # Administrator

  def admins()
    return @admins if @admins
    @admins = []
    unless File::exist? @admin_f then
      return @admins
    end
    IO::foreach(@admin_f) do |l|
      next if l[0] == ?\# or l =~ /^\s*$/
      @admins << l.chomp
    end
    @admins
  end

  def add_admin(addr)
    if admins.include? addr then
      err :EEXIST, "#{addr}: already exist"
    end
    @admins << addr
    File::open(@admin_f, "a") do |f| f.puts "#{addr}" end
    log "add admin: #{addr}"
  end

  def bye_admin(addr)
    unless admins.include? addr then
      err :ENOENT, "#{addr}: not exist"
    end
    if admins.size == 1 then
      err :EBUSY, "one admin is needed at least "
    end
    @admins.delete addr
    File::open("#{@admin_f}.new", "w") do |f|
      IO::foreach(@admin_f) do |line|
	next if line.chomp == addr
	f.puts line
      end
    end
    File::rename(@admin_f, "#{@admin_f}.bak")
    File::rename("#{@admin_f}.new", @admin_f)
    log "bye admin: #{addr}"
  end

  # Access

  def access()
    return @access if @access
    @access = []
    if File::exist? @access_f then
      af = @access_f
    elsif File::exist? TML::etc_dir+"/access" then
      af = TML::etc_dir+"/access"
    else
      return @access
    end
    IO::foreach(af) do |l|
      next if l[0] == ?\# or l =~ /^\s*$/
      re, priv = l.chomp.split(/\t+/, 2)
      unless $access_priv.include? priv.intern then
	raise Error::new(:EINVAL, "#{priv}: unknown privilege")
      end
      @access << [Regexp::new(re), priv.intern]
    end
    @access
  end

  def check_subscribe(mail)
    access.each do |re, priv|
      if mail =~ re then
	if priv == :permit or priv == :permit_subscribe then
	  return true
	end
	if priv == :reject or priv == :reject_subscribe then
	  log "reject to subscribe: #{mail}"
	  return false
	end
      end
    end
    true
  end

  def check_post(mail)
    access.each do |re, priv|
      if mail =~ re then
	if priv == :permit or priv == :permit_post then
	  return true
	end
	if priv == :reject or priv == :reject_post then
	  log "reject to post: #{mail}"
	  return false
	end
      end
    end
    true
  end

  # Attribute

  def set_attr(attr, value)
    log "set attribute: #{attr}=#{value}"
    attr = attr.intern if attr.kind_of? String
    unless @@attributes.key? attr then
      err :EINVAL, "#{attr}: invalid argument"
    end
    @attr[attr].set value
    write_conf
  end

  def get_attr(attr=nil)
    if attr then
      attr = attr.intern if attr.kind_of? String
      unless @@attributes.key? attr then
	err :EINVAL, "#{attr}: unkonwn attribute"
      end
      return @attr[attr].value
    else
      ret = {}
      @attr.each do |n, a|
	ret[n] = a.value
      end
      return ret
    end
  end

  def read_conf()
    IO::foreach(@conf) do |line|
      line = line.chomp.sub(/#.*$/,"")
      next if line =~ /\A\s*\Z/
      var, val = line.strip.split(/\s+/, 2)
      id = var.downcase.intern
      unless @@attributes.key? id then
	warn "unknown parameter: #{var}"
      else
	@attr[id].set val
      end
    end
  end

  def write_conf()
    log "write config"
    File::open("#{@conf}.new", File::WRONLY|File::CREAT, 0600) do |f|
      @attr.each do |k,v|
	f.puts "#{k}\t#{v.value}"
      end
    end
    File::rename(@conf, "#{@conf}.bak") if File::exist? @conf
    File::rename("#{@conf}.new", @conf)
  end

  # Alias

  def add_alias()
    log "add alias"
    lock = Lock::new "#{TML::etc_dir}/lock"
    file = "#{TML::etc_dir}/aliases"
    File::open(file, "a") do |f|
      f.puts <<EOS
#{@name}: "|#{$tml_dir}/tml #{@name}"
#{@name}-admin: :include:#{@dir}/admin
#{@name}-ctl: "|#{$tml_dir}/tmlctl #{@name}"
EOS
    end
    system $newaliases if $newaliases
    lock.unlock
  end

  def del_alias()
    log "delete alias"
    lock = Lock::new "#{TML::etc_dir}/lock"
    file = "#{TML::etc_dir}/aliases"
    File::open(file+".new", "w") do |f|
      IO::foreach(file) do |line|
	next if line =~ /^#{Regexp::escape @name}(-ctl|-admin)?:/
	f.puts line
      end
    end
    File::rename(file+".new", file)
    system $newaliases if $newaliases
    lock.unlock
  end

  def tmpfile(name)
    return "#{@dir}/tmp/#{name}"
  end

  def template(name)
    if File::exist? "#{@dir}/templates/#{name}" then
      return File::open("#{@dir}/templtates/#{name}"){|f| f.read}
    elsif File::exist? "#{TML::ml_dir}/.etc/templates/#{name}" then
      return File::open("#{TML::ml_dir}/.etc/templates/#{name}"){|f| f.read}
    elsif File::exist? "#{$tml_dir}/templates/#{name}" then
      return File::open("#{$tml_dir}/templates/#{name}"){|f| f.read}
    else
      return nil
    end
  end

  def spool(cnt, msg)
    log "spool ##{cnt}"
    TML::mkdir_p @spool_d unless File::directory? @spool_d
    File::open("#{@spool_d}/#{cnt}", "w") do |f|
      f.print msg
    end
  end

  def get(s, e=nil)
    m = []
    if e == nil or s > e then
      m << s if File::exist? "#{@spool_d}/#{s}"
    else
      s.upto(e) do |i|
	m << i if File::exist? "#{@spool_d}/#{i}" 
      end
    end
    m
  end

  def archive(msg)
    a = "/tmp/messages-#{$$}.tar.bz2"
    system "cd #{@spool_d}; tar cf - #{msg.join(" ")} | bzip2 > #{a}"
    if $? != 0 then
      err :EINVAL, "archive error"
    end
    m = [File::open(a){|f|f.read}].pack("m")
    File::unlink a
    m
  end

  def add_summary(n, sub)
    log "add summary: #{n} #{sub}"
    File::open(@summary_f, "a") do |f|
      f.puts "#{n} #{sub}"
    end
  end

  def summary_all()
    return "" unless File::exist? @summary_f
    return File::open(@summary_f){|f| f.read}
  end

  def summary(s, e=nil)
    return "" unless File::exist? @summary_f
    hash = {}
    File::open(@summary_f){|f| f.read}.split(/\n/).each do |l|
      n, sub = l.split(/ /,2)
      hash[n.to_i] = sub
    end
    if e == nil or s > e then
      return hash.key?(s) ? hash[s]+"\n" : ""
    end
    ret = ""
    s.upto(e) do |i|
      ret << hash[i]+"\n" if hash.key? i
    end
    ret
  end

  def summary_last(n)
    summary_all.split(/\n/)[-n,n].join("\n")+"\n"
  end

  def send_msg(from, to, template, hash={})
    require "net/smtp"
    hash[:from] = from
    hash[:to] = to
    hash[:mlname] = self.name
    hash[:adminaddr] = "#{self.name}-admin@#{$domain}"
    hash[:ctladdr] = "#{self.name}-ctl@#{$domain}"
    msg = template(template)
    msg.gsub!(/%\{(\w+)\}/){hash[$1.intern]}
    date = Time.now.strftime("Date: %a, %d %b %Y %H:%M:%S %z")+"\n"
    msgid = "Message-Id: <#{Time.now.to_i}.#{$$}.#{self.name}.tml@#{$domain}>\n"
    msg = date + msgid + msg
    Net::SMTP.start($smtp_server, $smtp_port) do |smtp|
      smtp.send_mail(msg, get_attr(:sender), to)
    end
  end

  private

  def err(errid, msg)
    raise TML::Error::new(errid, msg)
  end

  def warn(msg)
    $stderr.puts "warning: "+msg
  end

end
