/**************************************************
 libtimemachine.c -- Copyright(c) 1999,2003 Jiro SEKIBA <jir@hello.to>
 **************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <dlfcn.h>

#include <sys/time.h>
#include <errno.h>

#define LIBC "libc.so.6"   	/* for time, gettimeofday */
#define LIBRT "librt.so.1" 	/* for clock_gettime */

static time_t get_drift()
{
  char *tm_date = NULL;
  time_t tm = 0;
  char **error=NULL;

  tm_date = getenv("TM_DATE");

  if(tm_date != NULL)
  {
    tm = strtol(tm_date,error,0);
    if(error)
    {
      tm = 0;
    }
  }
  return tm;
}

void *open_lib_syscall(char *lib, void **handle, char *syscall)
{
  void *syscallp;
  char *err=NULL;

  *handle = dlopen(lib,RTLD_LAZY);
  if(!handle)
  {
#if DEBUG
    fputs(dlerror(),stderr);
    fprintf(stderr,"\nCan't open %s\n",lib);
#endif    
    return NULL;
  }
  
  syscallp = dlsym(*handle,syscall);
  if((err = dlerror()) != NULL)
  {
#if DEBUG
    fputs(err,stderr);
    fprintf(stderr,"\nCan't open %s()\n",syscall);
#endif
    return NULL;
  }
  return syscallp;

}
      
static void *open_syscall(void **handle, char *syscall)
{
  void *ret=NULL;

  ret = open_lib_syscall(LIBC,handle,syscall);
  if(ret != NULL)
      return ret;

  if(*handle)
      dlclose(*handle);

  ret = open_lib_syscall(LIBRT,handle,syscall);
  if(ret != NULL)
      return ret;
  return NULL;
}

time_t time(time_t *argt)
{
  time_t tm;
  time_t rt = 0;
  void *handle = NULL;
  time_t (*syscall_time)(time_t *t) = NULL;

  tm = get_drift();
  
  syscall_time = open_syscall(&handle,"time");
  if(syscall_time == NULL)
  {
    errno = EINVAL;
    return (time_t)-1;
  }
  
  rt = (*syscall_time)(NULL);
  tm += rt;
  dlclose(handle);
  
  if(argt != NULL)
    *argt = tm;
  
  return tm;
}

int gettimeofday(struct timeval *tv, struct timezone *tz)
{
  long tm = 0;
  struct timeval tm_tv;
  struct timezone tm_tz;
  int rt = 0;
  void *handle = NULL;
  int (*syscall_gettimeofday)(struct timeval *tv, struct timezone *tz) = NULL;

  tm = (long)get_drift();

  syscall_gettimeofday = open_syscall(&handle,"gettimeofday");
  if(syscall_gettimeofday == NULL)
  {
    errno = EINVAL;
    return -1;
  }
  
  rt = (*syscall_gettimeofday)(&tm_tv,&tm_tz);
  tm_tv.tv_sec += tm;
  dlclose(handle);

  if(tv != NULL)
      *tv = tm_tv;
  if(tz != NULL)
      *tz = tm_tz;
  
  return rt;
}

int clock_gettime(clockid_t clk_id, struct timespec *tp)
{
  time_t tm = 0;
  struct timespec tm_tp;
  int rt = 0;
  void *handle = NULL;
  int (*syscall_clock_gettime)(clockid_t, struct timespec *) = NULL;

  tm = get_drift();

  syscall_clock_gettime = open_syscall(&handle,"clock_gettime");
  if(syscall_clock_gettime == NULL)
  {
    errno = EINVAL;
    return -1;
  }

  rt = (*syscall_clock_gettime)(clk_id,&tm_tp);
  dlclose(handle);

      /* linux has clock_gettime syscall.  However,  some version of *
       * clock_gettime in librt calls gettimeofday(2) internaly, and *
       * some calls syscall directly.                                *
       * To avoid adding drift time twice, need checking             */
  if(time(NULL) != tm_tp.tv_sec) 
      tm_tp.tv_sec += tm;        
				 
  if(tp != NULL)
      *tp = tm_tp;
  
  return rt;
}
