/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 * Copyright (C) 1999-2004 Hiroyuki Yamamoto
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>

#include "intl.h"
#include "procheader.h"
#include "filter.h"
#include "folder.h"
#include "utils.h"

typedef enum
{
	FLT_O_CONTAIN	= 1 << 0,
	FLT_O_CASE_SENS	= 1 << 1,
	FLT_O_REGEX	= 1 << 2
} FilterOldFlag;


FolderItem *filter_get_dest_folder(GSList *fltlist, const gchar *file)
{
	static FolderItem *dummy = NULL;
	FolderItem *dest_folder = NULL;
	GSList *hlist, *cur;
	FilterRule *rule;
	FilterAction *action;

	g_return_val_if_fail(file != NULL, NULL);
	if (!fltlist) return NULL;

	hlist = procheader_get_header_list_from_file(file);
	if (!hlist) return NULL;

	for (cur = fltlist; cur != NULL; cur = cur->next) {
		rule = (FilterRule *)cur->data;
		if (filter_match_rule(rule, hlist))
			break;
		rule = NULL;
	}

	procheader_header_list_destroy(hlist);

	if (!rule) return NULL;

	action = filter_rule_get_last_action(rule);
	if (!action) return NULL;
	if (action->type == FLT_ACTION_MOVE) {
		dest_folder = folder_find_item_from_identifier
			(action->dest_folder);
		debug_print("filter_get_dest_folder(): dest_folder = %s\n",
			    action->dest_folder);
	} else if (action->type == FLT_ACTION_NOT_RECEIVE) {
		if (!dummy) {
			dummy = folder_item_new(NULL, NULL);
			dummy->path =
				g_strdup(FILTER_NOT_RECEIVE);
		}
		dest_folder = dummy;
		debug_print("filter_get_dest_folder(): don't receive\n");
	}

	return dest_folder;
}

static gboolean strfind(const gchar *haystack, const gchar *needle)
{
	return strstr(haystack, needle) != NULL ? TRUE : FALSE;
}

static gboolean strcasefind(const gchar *haystack, const gchar *needle)
{
	return strcasestr(haystack, needle) != NULL ? TRUE : FALSE;
}

static gboolean strmatch_regex(const gchar *haystack, const gchar *needle)
{
	gint ret = 0;
	regex_t preg;
	regmatch_t pmatch[1];

	ret = regcomp(&preg, needle, REG_EXTENDED);
	if (ret != 0) return FALSE;

	ret = regexec(&preg, haystack, 1, pmatch, 0);
	regfree(&preg);

	if (ret == REG_NOMATCH) return FALSE;

	if (pmatch[0].rm_so != -1)
		return TRUE;
	else
		return FALSE;
}

gboolean filter_match_rule(FilterRule *rule, GSList *hlist)
{
	FilterCond *cond;
	Header *header;
	GSList *cur, *hcur;
	gboolean matched;

	g_return_val_if_fail(rule->cond_list != NULL, FALSE);
	g_return_val_if_fail(rule->action_list != NULL, FALSE);

	if (rule->bool_op == FLT_AND) {
		for (cur = rule->cond_list; cur != NULL; cur = cur->next) {
			cond = (FilterCond *)cur->data;
			matched = FALSE;

			for (hcur = hlist; hcur != NULL; hcur = hcur->next) {
				header = (Header *)hcur->data;

				if (!strcasecmp(header->name, cond->header)) {
					if (!cond->body ||
					    cond->match_func(header->body,
							     cond->body)) {
						matched = TRUE;
						break;
					}
				}
			}

			if (FLT_IS_NOT_MATCH(cond->match_flag))
				matched = !matched;

			if (matched == FALSE)
				return FALSE;
		}

		return TRUE;
	} else if (rule->bool_op == FLT_OR) {
		for (cur = rule->cond_list; cur != NULL; cur = cur->next) {
			cond = (FilterCond *)cur->data;
			matched = FALSE;

			for (hcur = hlist; hcur != NULL; hcur = hcur->next) {
				header = (Header *)hcur->data;

				if (!strcasecmp(header->name, cond->header)) {
					if (!cond->body ||
					    cond->match_func(header->body,
							     cond->body)) {
						matched = TRUE;
						break;
					}
				}
			}

			if (FLT_IS_NOT_MATCH(cond->match_flag))
				matched = !matched;

			if (matched == TRUE)
				return TRUE;
		}

		return FALSE;
	}

	return FALSE;
}

gchar *filter_get_str(FilterRule *rule)
{
	gchar *str;
	FilterCond *cond1, *cond2;
	FilterAction *action;
	gint flag1 = 0, flag2 = 0;

	cond1 = (FilterCond *)rule->cond_list->data;
	if (rule->cond_list->next)
		cond2 = (FilterCond *)rule->cond_list->next->data;
	else
		cond2 = NULL;

	/* new -> old flag conversion */
	switch (cond1->match_type) {
	case FLT_CONTAIN:
	case FLT_EQUAL:
		flag1 = FLT_IS_NOT_MATCH(cond1->match_flag) ? 0 : FLT_O_CONTAIN;
		if (FLT_IS_CASE_SENS(cond1->match_flag))
			flag1 |= FLT_O_CASE_SENS;
		break;
	case FLT_REGEX:
		flag1 = FLT_O_REGEX; break;
	default:
		break;
	}
	if (cond2) {
		switch (cond2->match_type) {
		case FLT_CONTAIN:
		case FLT_EQUAL:
			flag2 = FLT_IS_NOT_MATCH(cond2->match_flag) ? 0 : FLT_O_CONTAIN;
			if (FLT_IS_CASE_SENS(cond2->match_flag))
				flag2 |= FLT_O_CASE_SENS;
			break;
		case FLT_REGEX:
			flag2 = FLT_O_REGEX; break;
		default:
			break;
		}
	} else
		flag2 = FLT_O_CONTAIN;

	action = filter_rule_get_last_action(rule);

	str = g_strdup_printf
		("%s\t%s\t%c\t%s\t%s\t%s\t%d\t%d\t%c",
		 cond1->header, cond1->body ? cond1->body : "",
		 (cond2 && cond2->header) ?
		 	(rule->bool_op == FLT_AND ? '&' : '|') : ' ',
		 (cond2 && cond2->header) ? cond2->header : "",
		 (cond2 && cond2->body) ? cond2->body : "",
		 action->dest_folder ? action->dest_folder : "",
		 flag1, flag2,
		 action->type == FLT_ACTION_MOVE    ? 'm' :
		 action->type == FLT_ACTION_NOT_RECEIVE ? 'n' :
		 action->type == FLT_ACTION_DELETE  ? 'd' : ' ');

	return str;
}

#define PARSE_ONE_PARAM(p, srcp) \
{ \
	p = strchr(srcp, '\t'); \
	if (!p) return NULL; \
	else \
		*p++ = '\0'; \
}

FilterRule *filter_read_str(const gchar *str)
{
	FilterRule *rule;
	FilterCond *cond;
	gint flag;
	FilterMatchType match_type;
	FilterMatchFlag match_flag;
	FilterAction *action;
	gchar *tmp;
	gchar *name1, *body1, *op, *name2, *body2, *dest;
	gchar *flag1 = NULL, *flag2 = NULL, *action1 = NULL;

	Xstrdup_a(tmp, str, return NULL);

	name1 = tmp;
	PARSE_ONE_PARAM(body1, name1);
	PARSE_ONE_PARAM(op, body1);
	PARSE_ONE_PARAM(name2, op);
	PARSE_ONE_PARAM(body2, name2);
	PARSE_ONE_PARAM(dest, body2);
	if (strchr(dest, '\t')) {
		gchar *p;

		PARSE_ONE_PARAM(flag1, dest);
		PARSE_ONE_PARAM(flag2, flag1);
		PARSE_ONE_PARAM(action1, flag2);
		if ((p = strchr(action1, '\t'))) *p = '\0';
	}

	rule = g_new0(FilterRule, 1);

	if (*name1) {
		if (flag1)
			flag = (FilterOldFlag)strtoul(flag1, NULL, 10);
		else
			flag = FLT_O_CONTAIN;
		match_type = FLT_CONTAIN;
		match_flag = 0;
		if (flag & FLT_O_REGEX)
			match_type = FLT_REGEX;
		else if (!(flag & FLT_O_CONTAIN))
			match_flag = FLT_NOT_MATCH;
		if (flag & FLT_O_CASE_SENS)
			match_flag |= FLT_CASE_SENS;
		cond = filter_cond_new(name1, body1, match_type, match_flag);
		rule->cond_list = g_slist_append(rule->cond_list, cond);
	}
	if (*name2) {
		if (flag2)
			flag = (FilterOldFlag)strtoul(flag2, NULL, 10);
		else
			flag = FLT_O_CONTAIN;
		match_type = FLT_CONTAIN;
		match_flag = 0;
		if (flag & FLT_O_REGEX)
			match_type = FLT_REGEX;
		else if (!(flag & FLT_O_CONTAIN))
			match_flag = FLT_NOT_MATCH;
		if (flag & FLT_O_CASE_SENS)
			match_flag |= FLT_CASE_SENS;
		cond = filter_cond_new(name2, body2, match_type, match_flag);
		rule->cond_list = g_slist_append(rule->cond_list, cond);
	}
	rule->bool_op = (*op == '|') ? FLT_OR : FLT_AND;

	action = g_new0(FilterAction, 1);
	action->dest_folder = *dest ? g_strdup(dest) : NULL;
	if (!strcmp2(dest, FILTER_NOT_RECEIVE))
		action->type = FLT_ACTION_NOT_RECEIVE;
	else
		action->type = FLT_ACTION_MOVE;
	if (action1) {
		switch (*action1) {
		case 'm': action->type = FLT_ACTION_MOVE;		break;
		case 'n': action->type = FLT_ACTION_NOT_RECEIVE;	break;
		case 'd': action->type = FLT_ACTION_DELETE;		break;
		default:  g_warning("Invalid action: `%c'\n", *action1);
		}
	}
	rule->action_list = g_slist_append(rule->action_list, action);

	return rule;
}

FilterCond *filter_cond_new(const gchar *header, const gchar *body,
			    FilterMatchType type, FilterMatchFlag flag)
{
	FilterCond *cond;

	cond = g_new0(FilterCond, 1);
	cond->header = g_strdup(header);
	cond->body = (body && *body) ? g_strdup(body) : NULL;
	cond->match_type = type;
	cond->match_flag = flag;

	if (type == FLT_REGEX)
		cond->match_func = strmatch_regex;
	else if (FLT_IS_CASE_SENS(flag))
		cond->match_func = strfind;
	else
		cond->match_func = strcasefind;

	return cond;
}

gboolean filter_rule_is_equal(FilterRule *rule1, FilterRule *rule2)
{
	gchar *str1, *str2;
	gboolean result;

	g_return_val_if_fail(rule1 != NULL, FALSE);
	g_return_val_if_fail(rule2 != NULL, FALSE);

	str1 = filter_get_str(rule1);
	str2 = filter_get_str(rule2);

	result = (strcmp(str1, str2) == 0);

	g_free(str1);
	g_free(str2);

	return result;
}

void filter_rule_rename_dest_path(FilterRule *rule, const gchar *old_path,
				  const gchar *new_path)
{
	FilterAction *action;
	gchar *base;
	gchar *dest_path;
	gint oldpathlen;

	action = filter_rule_get_last_action(rule);
	oldpathlen = strlen(old_path);
	if (action->dest_folder &&
	    !strncmp(old_path, action->dest_folder, oldpathlen)) {
		base = action->dest_folder + oldpathlen;
		while (*base == G_DIR_SEPARATOR) base++;
		if (*base == '\0')
			dest_path = g_strdup(new_path);
		else
			dest_path = g_strconcat(new_path,
						G_DIR_SEPARATOR_S,
						base, NULL);
		g_free(action->dest_folder);
		action->dest_folder = dest_path;
	}
}

gboolean filter_rule_match_dest_path(FilterRule *rule, const gchar *path)
{
	FilterAction *action;

	action = filter_rule_get_last_action(rule);
	return (action->dest_folder &&
		!strncmp(path, action->dest_folder, strlen(path)));
}

FilterAction *filter_rule_get_last_action(FilterRule *rule)
{
	GSList *list;

	list = g_slist_last(rule->action_list);
	if (list)
		return (FilterAction *)list->data;

	return NULL;
}

static void filter_cond_free(FilterCond *cond)
{
	g_free(cond->header);
	g_free(cond->body);
	g_free(cond);
}

static void filter_action_free(FilterAction *action)
{
	g_free(action->dest_folder);
	g_free(action);
}

void filter_rule_free(FilterRule *rule)
{
	GSList *cur;

	if (!rule) return;

	for (cur = rule->cond_list; cur != NULL; cur = cur->next)
		filter_cond_free((FilterCond *)cur->data);
	g_slist_free(rule->cond_list);

	for (cur = rule->action_list; cur != NULL; cur = cur->next)
		filter_action_free((FilterAction *)cur->data);
	g_slist_free(rule->action_list);

	g_free(rule);
}
