#!/usr/bin/python
#
#    SnortWatch 0.7    -  Copyright (c) 2000 Yves Perrenoud
#    ==============
#
# This application will allow monitoring of the snort alert file in a
# convenient GUI fashion and will allow to dig in to events at a glance.
#
# Currently still in early revisions, you can expect some additions in the
# near future (and feel free to make suggestions).
#
# In terms of the glade interface, if you were to make some usefull
# modifications, feel free to send me a copy so I can have a look (I'm no GUI
# expert so any improvement is welcome), if I like it, I'll either use it
# as the default or offer it as an extra option in the standard distribution.
#
#
# This software is subject to the GNU General Public License (GPL) and for
# details, please consult it at http://www.gnu.org/copyleft/gpl.html

import gtk, GDK, libglade
import re, StringIO, os.path, sys, threading, socket, string

# some key variable definitions

gladefilename = os.path.dirname(sys.argv[0]) + '/snortwatch.glade'

alertfilename = '/var/log/snort.alert'

snortdir = '/var/log/snort'

textfont = '-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1'

sumalert = ''
sortcol = 0
sorttype = gtk.SORT_DESCENDING

# some normal function definitions

def stringtoipv4(stringip):
    shiftable = 24
    hexip = 0
    for elem in string.split(stringip, '.'):
        hexip = hexip + (int(elem) << shiftable)
        shiftable = shiftable - 8
    return hexip

def isinsubnet(network, mask, ipaddr):
    if stringtoipv4(network) == stringtoipv4(ipaddr) & (0xffffffff << 32 - int(mask)):
        return 1
    else:
        return 0

def refreshdns(widget):
    if libglade.get_widget_tree(widget).get_widget('dns_reverse_resolve').active:
        dns_reverse_resolve(libglade.get_widget_tree(widget).get_widget('dns_reverse_resolve'))

def addalertrows(filehandle, clist, highlight):
    global dnsready

    alertlistlock.acquire()
    clist.freeze()

    totalert = ''

    line = 1
    while line:
        line = filehandle.readline()

        if not (re.match('\s*$', line) or re.match('\[\*\*\]', line)):
            totalert = totalert + line

            m = re.match('([0-9.:/-]+)\s*$', line)
            if m:
                alerttime = m.group(1)

                m = re.match('spp_portscan:\s+(.+?\s+from\s+([0-9.]+).*)', alerttype)
                if m:
                    alerttype = m.group(1)
                    alertip = m.group(2)
                    alertproto = 'Unknown'
                    alertdestport = ''
                    srctuple = ()

                    if isinsubnet(homenet, homemask, alertip):
                        alertdir = 'Out'
                    else:
                        alertdir = 'In'

                continue

            m = re.match('([0-9.:/-]+)\s+([0-9.:]+)[ >-]+([0-9.:]+)', line)
            if m:
                alerttime = m.group(1)
                alertsrcip = m.group(2)
                alertdestip = m.group(3)

                srctuple = string.split(alertsrcip, ':')
                desttuple = string.split(alertdestip, ':')

                if isinsubnet(homenet, homemask, srctuple[0]):
                    alertip = desttuple[0]
                    alertdir = 'Out'
                else:
                    alertip = alertsrcip
                    alertdir = 'In'

                if len(desttuple) > 1:
                    alertdestport = ":%s" % (desttuple[1])
                else:
                    alertdestport = ''

                continue

            m = re.match('(TCP|UDP|ICMP)', line)
            if m:
                alertproto = m.group(1)
                continue
        else:
            if totalert:
                row = clist.append((alerttime, alertdir, alertip, "%s%s" % (alertproto, alertdestport), alerttype))
                if len(srctuple) > 1:
                    clist.set_row_data(row, [totalert, alertip, '', srctuple[1]])
                else:
                    clist.set_row_data(row, [totalert, alertip, '', ''])
                if highlight:
                    clist.set_background(row, clist.get_colormap().alloc(highlight))
                threadevent.set()

            m = re.match('\[\*\*\]\s*(.*?)\s*\[\*\*\]', line)
            if m:
                totalert = line
                alerttype = m.group(1)

    clist.thaw()

    try:
        clist.moveto(row)
    except:
        pass

    alertlistlock.release()

    if highlight:
        dnsready = 0

def fillclist(filehandle, clist, highlight=None):
    global sumalert

    totalert = ''

    line = 1
    while line:
        line = filehandle.readline()
        if not line:
            continue

        if re.match('\s*$', line):
            if not sumalert:
                continue
            else:
                totalert = totalert + sumalert
                sumalert = ''
        else:
            sumalert = sumalert + line

    addalertrows(StringIO.StringIO(totalert), clist, highlight)

def prepareforalerts(filename, clist):
    global dnsready

    alertlistlock.acquire()
    clist.freeze()
    clist.clear()
    clist.thaw()
    alertlistlock.release()

    dnsready = 0
    alertfile = open(filename)
    fillclist(alertfile, clist)
    gtk.timeout_add(1000, checknewdata, alertfile, clist)

def parsehomenet(string):
    global homenet, homemask

    m = re.match('([0-9.]+?)/([0-9]+)$', string)
    if m:
        homenet = m.group(1)
        homemask = m.group(2)
        return 1
    else:
        return 0

# the gtk callback functions

def mainwindow_delete(*args):
    gtk.mainquit()

def alertlist_select_row(clist, row, col, event):
    alertlistlock.acquire()
    alerttext = libglade.get_widget_tree(clist).get_widget('alerttext')
    alerttext.freeze()
    alerttext.backward_delete(alerttext.get_length())

    if libglade.get_widget_tree(clist).get_widget('detailed_info').active:
        rowdata = clist.get_row_data(row)

        m = re.match('([0-9.]+):?(.*?)$', rowdata[1])
        if m:
            srcip = m.group(1)
            srcport = rowdata[3]

        m = re.match('([A-Za-z]+):?([0-9]*)$', clist.get_text(row, 3))
        if m:
            proto = m.group(1)
            destport = m.group(2)

        if proto == 'UDP':
            detailedfile = ( 'UDP:%s-%s' % (destport, srcport),
                             'UDP:%s-%s' % (srcport, destport) )
        elif proto == 'TCP':
            detailedfile = ( 'TCP:%s-%s' % (destport, srcport),
                             'TCP:%s-%s' % (srcport, destport) )
        elif proto == 'ICMP':
            icmptype = re.search('\s*([A-Z ]+)$', StringIO.StringIO(clist.get_row_data(row)[0]).readlines()[3]).group(1)
            icmptype = re.sub(' ', '_', icmptype)
            icmptype = re.sub('SOURCE', 'SRC', icmptype)
            icmptype = re.sub('EXCEEDED', 'EXCEED', icmptype)
            detailedfile = ( 'ICMP_%s' % (icmptype), )
        else:
            detailedfile = ()

        detailedpath = []
        for file in detailedfile:
            detailedpath.append('%s/%s/%s' % (snortdir, srcip, file))

        text = ''

        for filename in detailedpath:
            if os.path.exists(filename):
                glu = open(filename)
                for line in open(filename).readlines():
                    if re.match('\s*$', line):
                        if not text:
                            continue
                        else:
                            if re.search("^%s" % (clist.get_text(row, 0)), text, re.M):
                                break
                            text = ''
                    else:
                        text = text + line

        if not text:
            text = "Couldn't find any detailled log file!"
    else:
        text = clist.get_row_data(row)[0]

    alerttext.insert(gtk.load_font(textfont), None, None, text)
    alerttext.thaw()
    alertlistlock.release()

def alertlist_click_column(clist, col):
    global sortcol, sorttype

    alertlistlock.acquire()

    if sortcol == col:
        if sorttype == gtk.SORT_DESCENDING:
            sorttype = gtk.SORT_ASCENDING
        else:
            sorttype = gtk.SORT_DESCENDING

        clist.set_sort_type(sorttype)

    clist.set_sort_column(col)
    sortcol = col
    clist.sort()

    alertlistlock.release()

def dns_reverse_resolve(menuitem):
    alertlistlock.acquire()
    clist = libglade.get_widget_tree(menuitem).get_widget('alertlist')

    clist.freeze()

    row = 0
    while 1:
        rowdata = clist.get_row_data(row)
        if not rowdata:
            break

        if menuitem.active and rowdata[2]:
            clist.set_text(row, 2, rowdata[2])
        else:
            clist.set_text(row, 2, rowdata[1])
        row = row + 1

    clist.thaw()
    alertlistlock.release()

def open_fileselection(menuitem):
    libglade.get_widget_tree(menuitem).get_widget('fileselection').show()

def filesel_ok(widget):
    wtree = libglade.get_widget_tree(widget)
    wtree.get_widget('homenet').set_text('%s/%s' % (homenet, homemask))
    wtree.get_widget('fileselection').hide()
    wtree.get_widget('netdialog').show()

def filesel_cancel(widget):
    libglade.get_widget_tree(widget).get_widget('fileselection').hide()

def netdialog_ok(widget):
    wtree = libglade.get_widget_tree(widget)

    if not parsehomenet(wtree.get_widget('homenet').get_text()):
        wtree.get_widget('netdialog_status').set_text('Incorrect Syntax!')
    else:
        detailed_info = wtree.get_widget('detailed_info')
        if detailed_info.active:
            detailed_info.activate()
        wtree.get_widget('netdialog').hide()
        prepareforalerts(wtree.get_widget('fileselection').get_filename(),
                         wtree.get_widget('alertlist'))

def showabout(widget):
    libglade.get_widget_tree(widget).get_widget('aboutdialog').show()

def hideabout(widget):
    libglade.get_widget_tree(widget).get_widget('aboutdialog').hide()

def checknewdata(handle, clist):
    global dnsready

    if dnsready == 1:
        refreshdns(clist)
        dnsready = 2

    newalerts = handle.read()
    if newalerts:
        fillclist(StringIO.StringIO(newalerts), clist, 'OrangeRed')

    return 1

def resolveips(clist):
    global dnsready

    dnsdict = {}

    row = 0
    while 1:
        threadevent.wait()
        alertlistlock.acquire()
        rowdata = clist.get_row_data(row)
        if rowdata:
            if not rowdata[2]:
                m = re.match('([0-9.]+)(.*?)$', rowdata[1])
                if m:
                    ipaddress = m.group(1)
                    port = m.group(2)
                    alertlistlock.release()
                    try:
                        try:
                            hostname = dnsdict[ipaddress]
                        except KeyError:
                            hostname = socket.gethostbyaddr(ipaddress)[0]
                            dnsdict[ipaddress] = hostname
                    except socket.error:
                        hostname = ipaddress

                    alertlistlock.acquire()
                    rowdata[2] = "%s%s" % (hostname, port)
                    alertlistlock.release()
            else:
                alertlistlock.release()
            row = row + 1
        else:
            if not dnsready:
                dnsready = 1

            alertlistlock.release()
            row = 0
            threadevent.clear()

signals = { 'mainwindow_delete': mainwindow_delete,
            'on_alertlist_select_row': alertlist_select_row,
            'on_alertlist_click_column': alertlist_click_column,
            'on_dns_reverse_resolve_activate': dns_reverse_resolve,
            'on_open_activate': open_fileselection,
            'on_about_activate': showabout,
            'on_filesel_ok_clicked': filesel_ok,
            'on_filesel_cancel_clicked': filesel_cancel,
            'on_aboutdialog_ok_clicked': hideabout,
            'on_netdialog_ok_clicked': netdialog_ok,
            }

# The main code

if len(sys.argv) < 2:
    print "usage: %s <homenet>/<mask> [<alert_filename>]"
    sys.exit(1)

if not parsehomenet(sys.argv[1]):
    print "Couldn't parse you string %s as <homenet>/<mask>" % (sys.argv[1])
    sys.exit(1)

if len(sys.argv) > 2:
    alertfilename = sys.argv[2]

if os.path.exists(gladefilename):
    wtree = libglade.GladeXML(gladefilename)
else:
    print "Can't locate %s, this is essential!" % (gladefilename)
    sys.exit(1)

threadevent = threading.Event()
alertlistlock = threading.Lock()

alertlist = wtree.get_widget('alertlist')
wtree.signal_autoconnect(signals)
alertlist.set_sort_type(sorttype)
alertlist.set_auto_sort(gtk.TRUE)

prepareforalerts(alertfilename, alertlist)

# TEMPORARY HACK FOR MISSING LIBGLADE FEATURE
wtree.get_widget('mainpain').set_position(190)

dnsthread = threading.Thread(target=resolveips, args=(alertlist,))
dnsthread.setDaemon(1)
dnsthread.start()

gtk.mainloop()
