/*
 * Copyright (c) 1997, 98, 2000, 01  
 *    Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, stdlib.h)
 *   AC_CHECK_FUNCS(inet_pton strchr)
 *
 *   and HAVE_STRUCT_IN6_ADDR check.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include "dummyin6.h"

#include "permission.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif /* not HAVE_MEMCPY */

#ifndef HAVE_INET_PTON
#ifdef __STDC__
int inet_pton(int, const char *, void *);
#else
int inet_pton();
#endif
#endif

#ifdef __STDC__
static void ipv6_net_mask_length_to_address(int, struct in6_addr *);
static const char *strnopbrk(const char *, const char *);
#else
static void ipv6_net_mask_length_to_address();
static const char *strnopbrk();
#endif


/*
 * Prefix of IPv4 mapped IPv6 address.
 */
static const unsigned char ipv4_mapped_address_prefix[] = {
    0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0xff
};


/*
 * Initialize `permission'.
 */
void
initialize_permission(permission)
    Permission *permission;
{
    permission->next = NULL;
}


/*
 * Free all nodes in `permission'.
 */
void
finalize_permission(permission)
    Permission *permission;
{
    Permission *permission_p;
    Permission *next;

    permission_p = permission->next;
    while (permission_p != NULL) {
	next = permission_p->next;
	if (permission_p->host_name != NULL)
	    free(permission_p->host_name);
	if (permission_p->scope_name != NULL)
	    free(permission_p->scope_name);
	free(permission_p);
	permission_p = next;
    }

    permission->next = NULL;
}


/*
 * Add `pattern' to the list `permission'.
 *
 * If `permission' is a valid IPv4/IPv6 prefix or address, it is recognized
 * as it is.  Otherwise `pattern' is recognized as a host name.
 *
 * The function returns 0 upon successful, -1 otherwise.
 */
int
add_permission(permission, pattern)
    Permission *permission;
    const char *pattern;
{
    Permission *permission_p = NULL;
    char *copied_pattern = NULL;
    struct in_addr ipv4_address;
    int net_mask_length = 0;
    unsigned char mask_buffer[16];
    char *percent;
    char *slash;
    int i;

    /*
     * Allocate memories for a new node and a host name.
     */
    permission_p = (Permission *)malloc(sizeof(Permission));
    if (permission_p == NULL) {
	syslog(LOG_ERR, "memory exhausted");
	goto failed;
    }
    permission_p->not = 0;
    permission_p->host_name = NULL;
    permission_p->scope_name = NULL;
    memcpy(&permission_p->address, &in6addr_any, sizeof(struct in6_addr));
    ipv6_net_mask_length_to_address(128, &permission_p->net_mask);

    /*
     * Allocate memories for duplication of `pattern'.
     */
    copied_pattern = (char *)malloc(strlen(pattern) + 1);
    if (copied_pattern == NULL) {
	syslog(LOG_ERR, "memory exhausted");
	goto failed;
    }
    if (*pattern == '!')
	strcpy(copied_pattern, pattern + 1);
    else
	strcpy(copied_pattern, pattern);

    /*
     * Set <NOT> flag.
     */
    if (*pattern == '!')
	permission_p->not = 1;

    /*
     * Get net mask length if exists.
     */
    slash = strchr(copied_pattern, '/');
    if (slash != NULL) {
	char *p = slash + 1;

	while ('0' <= *p && *p <= '9')
	    p++;

	if (*p == '\0' && p != slash + 1) {
	    net_mask_length = atoi(slash + 1);
	    *slash = '\0';
	} else {
	    syslog(LOG_ERR, "invalid netmask: %s", slash);
	    goto failed;
	}
    }

    /*
     * Get scope name if exists.
     */
    percent = strchr(copied_pattern, '%');
    if (percent != NULL)
	*percent = '\0';

    /*
     * Parse the given pattern.
     */
    if (strchr(copied_pattern, ':') != NULL) {
	/*
	 * The pattern seems to be an IPv6 address.
	 */
	if (inet_pton(AF_INET6, copied_pattern, &permission_p->address) != 1) {
	    syslog(LOG_ERR, "invalid IPv6 adderss: %s", copied_pattern);
	    goto failed;
	}

	if (slash == NULL)
	    net_mask_length = 128;
	if (net_mask_length < 0 || 128 < net_mask_length) {
	    syslog(LOG_ERR, "invalid netmask: %d", net_mask_length);
	    goto failed;
	}
	ipv6_net_mask_length_to_address(net_mask_length,
	    &permission_p->net_mask);

	if (percent != NULL) {
	    memmove(copied_pattern, percent + 1, strlen(percent + 1) + 1);
	    permission_p->scope_name = copied_pattern;
	}

    } else if (strnopbrk(copied_pattern, "0123456789.") == NULL) {
	/*
	 * The pattern seems to be an IPv4 address.
	 */
	if (inet_pton(AF_INET, copied_pattern, &ipv4_address) != 1) {
	    syslog(LOG_ERR, "invalid IPv4 adderss: %s", copied_pattern);
	    goto failed;
	}
	memcpy(permission_p->address.s6_addr, ipv4_mapped_address_prefix, 12);
	memcpy((unsigned char *)permission_p->address.s6_addr + 12,
	    &ipv4_address.s_addr, 4);

	if (slash == NULL)
	    net_mask_length = 32;
	if (net_mask_length < 0 || 32 < net_mask_length) {
	    syslog(LOG_ERR, "invalid netmask: %d", net_mask_length);
	    goto failed;
	}
	ipv6_net_mask_length_to_address(net_mask_length + 96,
	    &permission_p->net_mask);

	if (percent != NULL) {
	    syslog(LOG_ERR, "IPv4 adderss with scope is not permitted: %s",
		pattern);
	    goto failed;
	}

    } else {
	/*
	 * The pattern seems to be a host name.
	 */
	if (slash != NULL)
	    *slash = '/';
	if (percent != NULL)
	    *percent = '/';
	permission_p->host_name = copied_pattern;
    }

    /*
     * Check address/netmask pair, if the given pattern is address.
     */
    if (permission_p->host_name == NULL) {
	for (i = 0; i < 16; i++) {
	    mask_buffer[i]
		= *((unsigned char *)permission_p->address.s6_addr + i)
		& *((unsigned char *)permission_p->net_mask.s6_addr + i);
	}
	if (memcmp(mask_buffer, &permission_p->address.s6_addr, 16) != 0) {
	    syslog(LOG_ERR, "address/netmask mismatch: %s", pattern);
	    goto failed;
	}
    }

    /*
     * Insert the new node into the permission list.
     */
    permission_p->next = permission->next;
    permission->next = permission_p;

    if (permission_p->host_name == NULL && permission_p->scope_name == NULL)
	free(copied_pattern);

    return 0;

    /*
     * If an error occurs...
     */
  failed:
    if (permission_p != NULL)
	free(permission_p);
    if (copied_pattern != NULL)
	free(copied_pattern);
    return -1;
}


/*
 * Convert IPv6 netmask length to IPv6 address.
 * For example, 60 is converted to FFFF:FFFF:FFFF:FFF0::.
 */
static void
ipv6_net_mask_length_to_address(length, address)
    int length;
    struct in6_addr *address;
{
    static const unsigned char mod8_masks[]
	= {0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe};
    unsigned char *address_p = (unsigned char *)address->s6_addr;
    int rest_length = length;

    memset(address->s6_addr, 0, 16);

    while (8 <= rest_length) {
	*address_p++ = 0xff;
	rest_length -= 8;
    }

    if (0 < rest_length)
	*address_p = mod8_masks[rest_length];
}


/*
 * strnopbrk() is similar to strpbrk(), but it locates the string `s'
 * the first occurrence of any character *NOT* listed in `charset'.
 */
static const char *
strnopbrk(string, charset)
    const char *string;
    const char *charset;
{
    unsigned char table[256];	/* assumes CHAR_BITS is 8 */
    const unsigned char *string_p = (const unsigned char *)string;
    const unsigned char *charset_p = (const unsigned char *)charset;

    memset(table, 0, sizeof(table));

    while (*charset_p != '\0')
	table[*charset_p++]  = 1;

    while (*string_p != '\0') {
	if (!table[*string_p])
	    return (const char *)string_p;
	string_p++;
    }

    return NULL;
}


/*
 * Examine access permission to `host_name' and `address'.
 * If access is permitted, 1 is returned.  Otherwise 0 is returned.
 */
int
test_permission(permission, host_name, address_string, function)
    const Permission *permission;
    const char *host_name;
    const char *address_string;
#ifdef __STDC__
    int (*function)(const char *, const char *);
#else
    int (*function)();
#endif
{
    const Permission *permission_p;
    int result = 0;
    struct in6_addr ipv6_address;
    struct in_addr ipv4_address;
    char address_buffer[INET6_ADDRSTRLEN];
    size_t address_length;
    const char *scope_name;
    unsigned char mask_buffer[16];
    int i;

    /*
     * Eliminate IPv6 scope name from `address_string'.
     */
    scope_name = strchr(address_string, '%');
    if (scope_name == NULL) {
	address_length = strlen(address_string);
	scope_name = "";
    } else {
	address_length = scope_name - address_string;
	scope_name++;
    }
    if (address_length + 1 > INET6_ADDRSTRLEN)
	return 0;

    memcpy(address_buffer, address_string, address_length);
    *(address_buffer + address_length) = '\0';

    /*
     * Convert `address_string' to binary format address.
     */
    if (inet_pton(AF_INET6, address_buffer, &ipv6_address) == 1) {
	/* nothing to be done */;
    } else if (inet_pton(AF_INET, address_buffer, &ipv4_address) == 1) {
	memcpy(ipv6_address.s6_addr, ipv4_mapped_address_prefix, 12);
	memcpy((unsigned char *)ipv6_address.s6_addr + 12,
	    &ipv4_address.s_addr, 4);
    } else {
	return 0;
    }

    /*
     * Test.
     */
    for (permission_p = permission->next; permission_p != NULL;
	 permission_p = permission_p->next) {
	if (permission_p->host_name != NULL) {
	    if (function(permission_p->host_name, host_name))
		result = permission_p->not ? 0 : 1;
	} else {
	    for (i = 0; i < 16; i++) {
		mask_buffer[i] = *((unsigned char *)ipv6_address.s6_addr + i)
		    & *((unsigned char *)permission_p->net_mask.s6_addr + i);
	    }
	    if (memcmp(mask_buffer, permission_p->address.s6_addr, 16) == 0) {
		if (permission_p->scope_name == NULL
		    || function(permission_p->scope_name, scope_name)) {
		    result = permission_p->not ? 0 : 1;
		}
	    }
	}
    }

    return result;
}


/*
 * Count the entires of the permission list, and return it.
 */
int
count_permission(permission)
    const Permission *permission;
{
    const Permission *permission_p;
    int count = 0;

    for (permission_p = permission->next; permission_p != NULL;
	 permission_p = permission_p->next)
	count++;

    return count;
}


/*
 * Test program.
 */
#ifdef TEST
int streq(s1, s2)
    const char *s1;
    const char *s2;
{
    return (strcmp(s1, s2) == 0);
}

int
main(argc, argv)
    int argc;
    char *argv[];
{
    Permission permission;
    FILE *file;
    char buffer[512];
    char *newline;

    if (argc != 2) {
	fprintf(stderr, "Usage: %s access-list-file\n", argv[0]);
	return 1;
    }

    initialize_permission(&permission);
    set_fakelog_mode(FAKELOG_TO_STDERR);

    file = fopen(argv[1], "r");
    if (file == NULL) {
	fprintf(stderr, "%s: failed to open the file: %s\n", argv[0], argv[1]);
	return 1;
    }

    while (fgets(buffer, sizeof(buffer), file) != NULL) {
	newline = strchr(buffer, '\n');
	if (newline != NULL)
	    *newline = '\0';
	if (add_permission(&permission, buffer) < 0) {
	    fprintf(stderr, "%s: failed to add permission to the list: %s\n",
		argv[0], buffer);
	    fclose(file);
	    return 1;
	}
    }
    fclose(file);

    while (fgets(buffer, sizeof(buffer), stdin) != NULL) {
	newline = strchr(buffer, '\n');
	if (newline != NULL)
	    *newline = '\0';
	if (test_permission(&permission, "?", buffer, streq))
	    printf("allowed\n");
	else
	    printf("denied\n");
    }

    return 0;
}

#endif /* TEST */
