/* mainpageseq.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "mainpageseq.h"
#include "interface_cb.h"

static void main_page_seq_class_init (MainPageSeqClass * class);
static void main_page_seq_init (MainPageSeq * wid);
static void main_page_seq_finalize (GObject *object);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

typedef struct {
	ConfManager *conf;
	DbSequence *seq;
	void *query;		/* to be a Query object */
}
Row_Data;

/*
 * static functions 
 */
static void selection_made (GnomeDbGrid * wid, gint row, gpointer data);
static void selection_unmade (GnomeDbGrid * wid, gpointer data);
static void remove_seq_cb (GObject   * obj, gpointer data);
static void main_page_seq_add_cb (GObject   * obj, DbSequence * seq,
				  gpointer data);
static void main_page_seq_drop_cb (GObject   * obj, DbSequence * seq,
				   gpointer data);
static void main_page_db_updated_cb (Database * db, MainPageSeq * mps);
static void main_page_seq_conn_close_cb (GObject   * obj, MainPageSeq * mps);
static void database_added_cb (ConfManager *conf, Database *db, MainPageSeq * mps);
static void database_removed_cb (ConfManager *conf, Database *db, MainPageSeq * mps);

guint
main_page_seq_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MainPageSeqClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) main_page_seq_class_init,
			NULL,
			NULL,
			sizeof (MainPageSeq),
			0,
			(GInstanceInitFunc) main_page_seq_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "MainPageSeq", &info, 0);
	}
	return type;
}

static void
main_page_seq_class_init (MainPageSeqClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	object_class->finalize = main_page_seq_finalize;
}



static void
main_page_seq_init (MainPageSeq * wid)
{
	GtkWidget *bb;
	gint i;
	GdaDataModel *model;

	/* Toolbar */
	bb = gnome_db_new_toolbar_widget (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD / 2);
	gtk_widget_show (bb);

	wid->new_seq = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_NEW,
						 _("Create new sequence"), NULL,
						 NULL, NULL, -1); /* FIXME */
	wid->remove_seq = gtk_toolbar_insert_stock (GTK_TOOLBAR (bb), GTK_STOCK_DELETE,
						    _("Drop selected sequence"), NULL,
						    GTK_SIGNAL_FUNC (remove_seq_cb), wid, -1);

	/* sequences list */
	model = gda_data_model_array_new (3);
	gda_data_model_set_column_title (model, 0, _("Sequences"));
	gda_data_model_set_column_title (model, 1, _("Owner"));
	gda_data_model_set_column_title (model, 2, _("Comments"));
	wid->list = gnome_db_grid_new_with_model (model);
	gtk_box_pack_start (GTK_BOX (wid), wid->list, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (wid->list);
	g_signal_connect (G_OBJECT (wid->list), "row_selected",
			    G_CALLBACK (selection_made), wid);
	g_signal_connect (G_OBJECT (wid->list), "selection_cleared",
			    G_CALLBACK (selection_unmade), wid);

	wid->sel_row = -1;

	g_object_unref (G_OBJECT (model));
}


GtkWidget *
main_page_seq_new (ConfManager * conf)
{
	GObject   *obj;
	MainPageSeq *wid;

	obj = g_object_new (MAIN_PAGE_SEQ_TYPE, NULL);
	wid = MAIN_PAGE_SEQ (obj);
	wid->conf = conf;

	g_signal_connect (G_OBJECT (conf), "database_added",
			  G_CALLBACK (database_added_cb), wid);

	g_signal_connect (G_OBJECT (conf), "database_removed",
			  G_CALLBACK (database_removed_cb), wid);


	gtk_widget_set_sensitive (wid->remove_seq, FALSE);
	gtk_widget_set_sensitive (wid->new_seq, FALSE);
	conf_manager_register_sensitive_on_connect (wid->conf,
						    GTK_WIDGET (wid->new_seq));

	g_signal_connect (G_OBJECT (conf->srv), "conn_closed",
			  G_CALLBACK (main_page_seq_conn_close_cb), obj);

	return GTK_WIDGET (obj);
}

static void main_page_seq_finalize (GObject *object)
{
	parent_class->finalize (object);
}

static void database_added_cb (ConfManager *conf, Database *db, MainPageSeq * mps)
{
	g_signal_connect (G_OBJECT (conf->db), "seq_created",
			  G_CALLBACK (main_page_seq_add_cb),
			  mps);
	g_signal_connect (G_OBJECT (conf->db), "seq_dropped",
			  G_CALLBACK (main_page_seq_drop_cb),
			  mps);
	g_signal_connect (G_OBJECT (conf->db), "updated",
			  G_CALLBACK (main_page_db_updated_cb),
			  mps);
}

static void database_removed_cb (ConfManager *conf, Database *db, MainPageSeq * mps)
{
	/* nothing to do about it */
}


static void
selection_made (GnomeDbGrid * wid, gint row, gpointer data)
{
	MainPageSeq *mps = MAIN_PAGE_SEQ (data);

	mps->sel_row = row;
	gtk_widget_set_sensitive (mps->remove_seq, TRUE);
}

static void
selection_unmade (GnomeDbGrid * wid, gpointer data)
{
	MainPageSeq *mps = MAIN_PAGE_SEQ (data);

	mps->sel_row = -1;
	gtk_widget_set_sensitive (mps->remove_seq, FALSE);
}


static void remove_seq_answer_cb (gint reply, GObject   * obj);
static void
remove_seq_cb (GObject   * obj, gpointer data)
{
	Row_Data *rdata = NULL;
	MainPageSeq *wid = MAIN_PAGE_SEQ (data);
	gchar *txt;

	if (wid->sel_row >= 0)
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), wid->sel_row);
	if (rdata) {
		txt = g_strdup_printf (_("Do you really want to remove\n"
					 "the sequence '%s'?"),
				       rdata->seq->name);
		g_object_set_data (obj, "conf", wid->conf);
		g_object_set_data (obj, "seq", rdata->seq);
		gnome_question_dialog (txt,
				       (GnomeReplyCallback)
				       remove_seq_answer_cb, obj);
		g_free (txt);
	}
}

static void
remove_seq_answer_cb (gint reply, GObject   * obj)
{
	ConfManager *conf;
	DbSequence *seq;
	gchar *txt;

	conf = (ConfManager *) g_object_get_data (obj, "conf");
	seq = DB_SEQUENCE (g_object_get_data (obj, "seq"));

	if (reply == 0) {
		ServerResultset *rs;
		txt = g_strdup_printf ("DROP sequence %s", seq->name);
		rs = server_access_do_query (conf->srv, txt, SERVER_ACCESS_QUERY_SQL);
		if (rs) {
			sql_mem_update_cb (NULL, conf);
			g_object_unref (G_OBJECT (rs));
		}
		g_free (txt);
	}
}



/* this CB is intended to be connected to the "seq_created" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
   - data is the recepting MainPageSeq object 
*/
static void
main_page_seq_add_cb (GObject   * obj, DbSequence * seq, gpointer data)
{
	MainPageSeq *wid = MAIN_PAGE_SEQ (data);
	GList *value_list = NULL;
	Row_Data *rdata;

	value_list = g_list_append (value_list, gda_value_new_string (seq->name));
	value_list = g_list_append (value_list, gda_value_new_string (seq->owner));
	value_list = g_list_append (value_list, gda_value_new_string (seq->comments));
	gda_data_model_append_row (gnome_db_grid_get_model (GNOME_DB_GRID (wid->list)), value_list);

	rdata = (Row_Data *) g_malloc (sizeof (Row_Data));
	rdata->conf = wid->conf;
	rdata->seq = seq;
	rdata->query = NULL;
	gnome_db_grid_set_row_data (
		GNOME_DB_GRID (wid->list),
		gda_data_model_get_n_rows (gnome_db_grid_get_model (GNOME_DB_GRID (wid->list))) - 1,
		rdata);

	/* free memory */
	g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
	g_list_free (value_list);
}

/* this CB is intended to be connected to the "seq_dropped" signal of the 
   Database objects it represents.  So:
   - obj is a Database object
   - data is the recepting MainPageSeq object 
*/
static void
main_page_seq_drop_cb (GObject   * obj, DbSequence * seq, gpointer data)
{
	MainPageSeq *wid = MAIN_PAGE_SEQ (data);
	gchar *str;
	gint i = 0;
	gboolean found = FALSE;
	Row_Data *rdata;
	GdaDataModel *model;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (wid->list));

	while ((i < gda_data_model_get_n_rows (model)) && !found) {
		str = gda_value_stringify (gda_data_model_get_value_at (model, 0, i));
		if (strcmp (str, seq->name) == 0) {
			found = TRUE;
			rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (wid->list), i);
			g_free (rdata);
			gda_data_model_remove_row (model,
						   gda_data_model_get_row (model, i));

			if (wid->sel_row == i) {
				wid->sel_row = -1;
				gtk_widget_set_sensitive (wid->remove_seq,
							  FALSE);
			}
		}
		i++;

		g_free (str);
	}
}

static void
main_page_db_updated_cb (Database * db, MainPageSeq * mps)
{
	gint row;
	Row_Data *rd;
	GdaDataModel *model;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (mps->list));

	for (row = 0; row < gda_data_model_get_n_rows (model); row++) {
		rd = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mps->list), row);
		gda_value_set_string (gda_data_model_get_value_at (model, 1, row), rd->seq->owner);
		gda_data_model_row_updated (model, row);
	}
}


static void
main_page_seq_conn_close_cb (GObject   * obj, MainPageSeq * mps)
{
	Row_Data *rdata;
	GdaDataModel *model;
	gint i, nb;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (mps->list));
	nb = gda_data_model_get_n_rows (model);

	for (i = 0; i < nb; i++) {
		rdata = (Row_Data *) gnome_db_grid_get_row_data (GNOME_DB_GRID (mps->list), 0);
		g_free (rdata);
		gda_data_model_remove_row (model, gda_data_model_get_row (model, i));
	}
	gtk_widget_set_sensitive (mps->remove_seq, FALSE);
	gtk_widget_set_sensitive (mps->new_seq, FALSE);
	mps->sel_row = -1;
}
