/* item-selector.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "item-selector.h"
#include "marshal.h"

static void item_selector_class_init (ItemSelectorClass * class);
static void item_selector_init (ItemSelector * wid);
static void item_selector_finalize (GObject   * object);
static void item_selector_changed_cb (GtkWidget * wid, ItemSelector * is);

enum
{
	SELECTION_CHANGED,
	LAST_SIGNAL
};

static gint item_selector_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

struct _ItemSelectorPrivate 
{
	ItemSelectorDisplayMode mode;

	GtkWidget              *combo;
	GtkWidget              *option_menu;

	GObject                *manager;
	gboolean                manager_is_weak_refed;
	ItemSelectorFunc        data_function;

	/* those 2 lists are always synchronised */
	GSList                 *main_list;
	GList                  *strings_list;

	gint                    changed_signal_handler;	
};


guint
item_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (ItemSelectorClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) item_selector_class_init,
			NULL,
			NULL,
			sizeof (ItemSelector),
			0,
			(GInstanceInitFunc) item_selector_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "ItemSelector", &info, 0);
	}

	return type;
}

static void
item_selector_class_init (ItemSelectorClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	item_selector_signals[SELECTION_CHANGED] =
		g_signal_new ("selection_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (ItemSelectorClass, selection_changed),
			      NULL, NULL,
			      marshal_VOID__POINTER, G_TYPE_NONE, 1,
			      G_TYPE_POINTER);

	class->selection_changed = NULL;
	object_class->finalize = item_selector_finalize;
}

static void
item_selector_init (ItemSelector * wid)
{
	wid->priv = g_new0 (ItemSelectorPrivate, 1);
	
	wid->priv->mode = 0;
	wid->priv->combo = NULL;
	wid->priv->option_menu = NULL;
	wid->priv->strings_list = NULL;

	wid->priv->manager = NULL;
	wid->priv->manager_is_weak_refed = FALSE;
	wid->priv->data_function = NULL;

	wid->priv->main_list = NULL;
	wid->priv->changed_signal_handler = -1;
}

GtkWidget *
item_selector_new (ItemSelectorDisplayMode mode)
{
	GObject   *obj;
	ItemSelector *wid;

	obj = g_object_new (ITEM_SELECTOR_TYPE, NULL);
	wid = ITEM_SELECTOR (obj);

	wid->priv->mode = mode;

	switch (mode) {
	case ITEM_SELECTOR_MENU:
		wid->priv->option_menu = gtk_option_menu_new ();
		wid->priv->changed_signal_handler =
			g_signal_connect (G_OBJECT (wid->priv->option_menu), "changed",
					  G_CALLBACK (item_selector_changed_cb), wid);
		gtk_box_pack_start (GTK_BOX (wid), wid->priv->option_menu, TRUE, TRUE, 0);
		gtk_widget_show (wid->priv->option_menu);
		break;
	case ITEM_SELECTOR_COMBO:
		wid->priv->combo = gtk_combo_new ();
		gtk_combo_set_use_arrows (GTK_COMBO (wid->priv->combo), TRUE);
		gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (wid->priv->combo)->entry), FALSE);
		wid->priv->changed_signal_handler =
			g_signal_connect (G_OBJECT (GTK_COMBO (wid->priv->combo)->entry), "changed",
					  G_CALLBACK (item_selector_changed_cb), wid);
		gtk_box_pack_start (GTK_BOX (wid), wid->priv->combo, TRUE, TRUE, 0);
		gtk_widget_show (wid->priv->combo);
		break;
	default:
		g_warning ("item_selector_new (%d): unrecognized mode\n", mode);
	}

	return GTK_WIDGET (obj);
}

static void manager_new_obj (GObject *manager, GObject *new_obj, ItemSelector *is);
static void manager_del_obj (GObject *manager, GObject *new_obj, ItemSelector *is);
static void manager_weak_notify (ItemSelector *is, GObject *manager);
static void
item_selector_finalize (GObject   * object)
{
	ItemSelector *is;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_ITEM_SELECTOR (object));

	is = ITEM_SELECTOR (object);
	
	if (is->priv) {
		GList *listiter;

		if (is->priv->main_list)
			g_slist_free (is->priv->main_list);

		if (is->priv->strings_list) {
			listiter = is->priv->strings_list;
			while (listiter) {
				g_free (listiter->data);
				listiter = g_list_next (listiter);
			}
			g_list_free (is->priv->strings_list);
		}

		if (is->priv->manager_is_weak_refed) {
			g_object_weak_unref (is->priv->manager, (GWeakNotify) manager_weak_notify, is);
			g_signal_handlers_disconnect_by_func (G_OBJECT (is->priv->manager), 
							      G_CALLBACK (manager_new_obj), is);
			g_signal_handlers_disconnect_by_func (G_OBJECT (is->priv->manager), 
							      G_CALLBACK (manager_del_obj), is);
		}

		g_free (is->priv);
		is->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

/* DOES free the new_list list */
static void display_update (ItemSelector * is, GList *new_list, gint selitem);
void
item_selector_set_static_content_offset (ItemSelector * is, GSList * list, gint offset)
{
	GList *nlist=NULL;
	gpointer *selection;
	gint selitem = -1;

	g_return_if_fail (is && IS_ITEM_SELECTOR (is));

	selection = item_selector_get_selection (is);
	if (is->priv->main_list) {
		g_slist_free (is->priv->main_list);
		is->priv->main_list = NULL;
	}
	
	/* preparing the strings list */
	if (list) {
		GSList *ptr, *lastptr;
		gchar **laststr = NULL, **str, *cptr;
		gchar *strinc;
		gint i, item;

		item = 0;
		is->priv->main_list = g_slist_copy (list);
		
		ptr = lastptr = list;
		i = 0;
		while (ptr) {
			if (ptr->data == NULL) {
				ptr = g_slist_next (ptr);
				strinc = g_strdup (_("NONE"));
				nlist = g_list_append (nlist, strinc);
				lastptr = ptr;
				laststr = NULL;
				item++;
			}
			else {
				cptr = (char *) (ptr->data);
				str = (gchar **) (cptr + offset);
				
				if ((lastptr != ptr) && laststr
				    && (!strcmp (*str, *laststr)))
					i++;
				else
					i = 0;
				strinc = g_strdup_printf ("%-*s", i + (gint) strlen (*str), *str);
				nlist = g_list_append (nlist, strinc);
				lastptr = ptr;
				laststr = str;
				
				if (ptr->data == selection)	/* set the selection to this item */
					selitem = item;
				ptr = g_slist_next (ptr);
				item++;
			}
		}
	}
	else 
		nlist = g_list_append (NULL, g_strdup (_("NONE")));


	/* widget update */
	display_update (is, nlist, selitem);
}

void real_item_selector_set_static_content_func (ItemSelector * is, GSList * list, 
						 ItemSelectorFunc func, gboolean sort);

void
item_selector_set_static_content_func (ItemSelector * is, GSList * list, ItemSelectorFunc func) {
	real_item_selector_set_static_content_func (is, list, func, FALSE);
}

static gint find_insert_pos (GList *list, const gchar *str);
void
real_item_selector_set_static_content_func (ItemSelector * is, GSList * list, 
					    ItemSelectorFunc func, gboolean sort)
{
	GList *nlist=NULL;
	gpointer *selection;
	gint selitem = -1;

	g_return_if_fail (is && IS_ITEM_SELECTOR (is));

	selection = item_selector_get_selection (is);
	if (is->priv->main_list) {
		g_slist_free (is->priv->main_list);
		is->priv->main_list = NULL;
	}
	
	/* preparing the strings list */
	if (list) {
		GSList *ptr;
		gint i, item, pos = -1;
		gchar *strinc;

		item = 0;
		selitem = -1;
			
		ptr = list;
		i = 0;
		while (ptr) {
			strinc = (*func)((gpointer) ptr->data);
			
			if (sort) 
				pos = find_insert_pos (nlist, strinc);

			if (pos >= 0) {
				is->priv->main_list = g_slist_insert (is->priv->main_list, ptr->data, pos);
				nlist = g_list_insert (nlist, strinc, pos);
			}
			else {
				is->priv->main_list = g_slist_append (is->priv->main_list, ptr->data);
				nlist = g_list_append (nlist, strinc);
			}
			
			if (ptr->data == selection)	/* set the selection to this item */
				selitem = item;
			ptr = g_slist_next (ptr);
			item++;
		}
	}
	else 
		nlist = g_list_append (NULL, g_strdup (_("NONE")));


	/* widget update */
	display_update (is, nlist, selitem);
}

static gint
find_insert_pos (GList *list, const gchar *str)
{
	gint pos = 0;

	if (list) {
		GList *listiter, *sibling = NULL;

		listiter = list;
		while (listiter && !sibling) {
			if (strcmp (str, (gchar *)(listiter->data)) < 0)
				sibling = listiter;
			else {
				listiter = g_list_next (listiter);
				pos++;
			}
		}
	}

	return pos;
}

static void
display_update (ItemSelector * is, GList *new_list, gint selitem)
{
	GtkWidget *menu;
	GList *listiter;

	/* free the previous strings, if any */
	if (is->priv->strings_list) {
		listiter = is->priv->strings_list;
		while (listiter) {
			g_free (listiter->data);
			listiter = g_list_next (listiter);
		}
		g_list_free (is->priv->strings_list);
		is->priv->strings_list = NULL;
	}

	/* widget update */
	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		gtk_option_menu_remove_menu (GTK_OPTION_MENU (is->priv->option_menu));

		menu = gtk_menu_new ();
		listiter = new_list;
		while (listiter) {
			GtkWidget *mitem;
			mitem = gtk_menu_item_new_with_label ((gchar *)(listiter->data));
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
			gtk_widget_show (mitem);
			listiter = g_list_next (listiter);
		}

		g_signal_handler_block (G_OBJECT (is->priv->option_menu),
					is->priv->changed_signal_handler);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (is->priv->option_menu), menu);
		if (selitem >= 0) 
			gtk_option_menu_set_history (GTK_OPTION_MENU (is->priv->option_menu), selitem);
		g_signal_handler_unblock (G_OBJECT (is->priv->option_menu),
					  is->priv->changed_signal_handler);
		break;
	case ITEM_SELECTOR_COMBO: 		
		g_signal_handler_block (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					is->priv->changed_signal_handler);
		gtk_combo_set_popdown_strings (GTK_COMBO (is->priv->combo), new_list);
		if (selitem >= 0)
			gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), selitem);
		g_signal_handler_unblock (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					  is->priv->changed_signal_handler);	
		break;
	default:
		g_warning ("item_selector_set_static_content_offset (): mode %d unrecognized\n", 
			   is->priv->mode);
		break;
	}

	is->priv->strings_list = new_list;
}

static gint 
item_selector_get_selection_num (ItemSelector * is) 
{
	gint pos = -1;

	g_return_val_if_fail (is && IS_ITEM_SELECTOR (is), pos);

	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		pos = gtk_option_menu_get_history (GTK_OPTION_MENU (is->priv->option_menu));
		break;
	case ITEM_SELECTOR_COMBO:
		if (GTK_LIST (GTK_COMBO (is->priv->combo)->list)->selection) 
			pos = g_list_index (GTK_LIST (GTK_COMBO (is->priv->combo)->list)->children,
					    GTK_LIST (GTK_COMBO (is->priv->combo)->list)->selection->data);
		break;
	default:
		g_warning ("item_selector_get_selection_num (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
	return pos;
}

gpointer
item_selector_get_selection (ItemSelector * is)
{
	gint pos;
	gpointer retval = NULL;

	g_return_val_if_fail (is && IS_ITEM_SELECTOR (is), NULL);

	pos = item_selector_get_selection_num (is);

	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		if (pos >= 0) 
			retval = g_slist_nth_data (is->priv->main_list, pos);
		break;
	case ITEM_SELECTOR_COMBO:
		if (pos >= 0) 
			retval = g_slist_nth_data (is->priv->main_list, pos);
		break;
	default:
		g_warning ("item_selector_get_selection (): mode %d unrecognized\n", is->priv->mode);
		break;
	}

	return retval;
}

void
item_selector_set_selection_txt (ItemSelector * is, const gchar * item)
{
	gint pos = -1;
	GList *list;

	g_return_if_fail (is && IS_ITEM_SELECTOR (is));

	list = is->priv->strings_list;
	while (list && (pos==-1)) {
		if (!strcmp ((gchar *)(list->data), item))
			pos = g_list_position (is->priv->strings_list, list);
		list = g_list_next (list);
	}


	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		if (pos >= 0) {
			g_signal_handler_block (G_OBJECT (is->priv->option_menu),
						is->priv->changed_signal_handler);
			gtk_option_menu_set_history (GTK_OPTION_MENU (is->priv->option_menu), pos);
			g_signal_handler_unblock (G_OBJECT (is->priv->option_menu),
						  is->priv->changed_signal_handler);
		}
		
		break;
	case ITEM_SELECTOR_COMBO:
		if (pos >= 0 ) {
			g_signal_handler_block (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
						is->priv->changed_signal_handler);
			gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (is->priv->combo)->entry), item);
			gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), pos);
			g_signal_handler_unblock (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
						  is->priv->changed_signal_handler);
		}
		break;
	default:
		g_warning ("item_selector_set_selection_txt (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
}

void
item_selector_set_selection_num (ItemSelector * is, gint num)
{
	g_return_if_fail (is && IS_ITEM_SELECTOR (is));
	g_return_if_fail (num >= 0);

	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		g_signal_handler_block (G_OBJECT (is->priv->option_menu),
					is->priv->changed_signal_handler);
		gtk_option_menu_set_history (GTK_OPTION_MENU (is->priv->option_menu), num);
		g_signal_handler_unblock (G_OBJECT (is->priv->option_menu),
					  is->priv->changed_signal_handler);
		break;
	case ITEM_SELECTOR_COMBO:
		g_signal_handler_block (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					is->priv->changed_signal_handler);
		gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), num);
		g_signal_handler_unblock (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					  is->priv->changed_signal_handler);
		break;
	default:
		g_warning ("item_selector_set_selection_num (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
}

void
item_selector_set_selection_ptr (ItemSelector * is, gpointer ptr)
{
	gint num;

	g_return_if_fail (is && IS_ITEM_SELECTOR (is));

	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		if (g_slist_find (is->priv->main_list, ptr)) {
			g_signal_handler_block (G_OBJECT (is->priv->option_menu),
						is->priv->changed_signal_handler);
			num = g_slist_index (is->priv->main_list, ptr);
			gtk_option_menu_set_history (GTK_OPTION_MENU (is->priv->option_menu), num);
			g_signal_handler_unblock (G_OBJECT (is->priv->option_menu),
						  is->priv->changed_signal_handler);
		}
		break;
	case ITEM_SELECTOR_COMBO:
		if (g_slist_find (is->priv->main_list, ptr)) {
			g_signal_handler_block (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					is->priv->changed_signal_handler);
			num = g_slist_index (is->priv->main_list, ptr);
			gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), num);
			g_signal_handler_unblock (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
						  is->priv->changed_signal_handler);
		}
		break;
	default:
		g_warning ("item_selector_set_selection_ptr (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
}

static void
item_selector_changed_cb (GtkWidget * wid, ItemSelector * is)
{
#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from item_selector_changed_cb\n");
#endif
	g_signal_emit (G_OBJECT (is), item_selector_signals[SELECTION_CHANGED], 0,
		       item_selector_get_selection (is));
#ifdef debug_signal
	g_print (">> 'SELECTION_CHANGED' from item_selector_changed_cb\n");
#endif
}





/*
 * Same as static functions except that there is a "manager" object from which we
 * get the creation and destruction of objects  (=> the displayed items MUST be GObjects)
 */
void 
item_selector_set_dyn_content (ItemSelector * is, GObject *manager,
			       const gchar *creation_signal_name,
			       const gchar *destroy_signal_name,
			       GSList * initial_list, ItemSelectorFunc func)
{
	g_return_if_fail (is && IS_ITEM_SELECTOR (is));
	g_return_if_fail (manager && G_IS_OBJECT (manager));
	g_return_if_fail (creation_signal_name && destroy_signal_name);

	real_item_selector_set_static_content_func (is, initial_list, func, TRUE);

	g_signal_connect (G_OBJECT (manager), creation_signal_name,
			  G_CALLBACK (manager_new_obj), is);
	g_signal_connect (G_OBJECT (manager), destroy_signal_name,
			  G_CALLBACK (manager_del_obj), is);

	g_object_weak_ref (manager, (GWeakNotify) manager_weak_notify, is);
	is->priv->manager_is_weak_refed = TRUE;
	is->priv->manager = manager;
	is->priv->data_function = func;
}

static void 
manager_weak_notify (ItemSelector *is, GObject *manager)
{
	is->priv->manager_is_weak_refed = FALSE;
	item_selector_set_static_content_func (is, NULL, is->priv->data_function);
}

static void 
manager_new_obj (GObject *manager, GObject *new_obj, ItemSelector *is)
{
	gchar *str;
	GtkWidget *menu, *mitem;
	gint pos = 0, selitem = 0;

	/* find the accurate position */
	str = (*is->priv->data_function) (new_obj);
	pos = find_insert_pos (is->priv->strings_list, str);

	/* update the strings_list list */
	is->priv->strings_list = g_list_insert (is->priv->strings_list, str, pos);

	/* insert at that position */
	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (is->priv->option_menu));
		mitem = gtk_menu_item_new_with_label (str);
		gtk_widget_show (mitem);
		gtk_menu_shell_insert (GTK_MENU_SHELL (menu), mitem, pos);
		break;
	case ITEM_SELECTOR_COMBO:
		selitem = item_selector_get_selection_num (is);
		g_signal_handler_block (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					is->priv->changed_signal_handler);
		gtk_combo_set_popdown_strings (GTK_COMBO (is->priv->combo), is->priv->strings_list);
		if (selitem >= 0)
			gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), selitem);
		g_signal_handler_unblock (G_OBJECT (GTK_COMBO (is->priv->combo)->entry),
					  is->priv->changed_signal_handler);	
		break;
	default:
		g_warning ("manager_new_object (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
}

static void
manager_del_obj (GObject *manager, GObject *new_obj, ItemSelector *is)
{
	gchar *str;
	gint pos = -1, selitem;
	GList *list;
	GSList *slist;

	str = (*is->priv->data_function) (new_obj);

	/* position of the deleted object */
	list = is->priv->strings_list;
	while (list && (pos==-1)) {
		if (!strcmp ((gchar *)(list->data), str))
			pos = g_list_position (is->priv->strings_list, list);
		list = g_list_next (list);
	}

	/* position of the current selection */
	selitem = item_selector_get_selection_num (is);

	switch (is->priv->mode) {
	case ITEM_SELECTOR_MENU:
		if (pos >= 0) {
			slist = g_slist_nth (is->priv->main_list, pos);
			is->priv->main_list = g_slist_delete_link (is->priv->main_list, slist);
			slist = g_slist_copy (is->priv->main_list);

			real_item_selector_set_static_content_func (is, slist,
								    is->priv->data_function, TRUE);
			g_slist_free (slist);

			if (pos == selitem) 
				gtk_option_menu_set_history (GTK_OPTION_MENU (is->priv->option_menu), 0);
		}
		break;
	case ITEM_SELECTOR_COMBO:
		if (pos >= 0) {
			slist = g_slist_nth (is->priv->main_list, pos);
			is->priv->main_list = g_slist_delete_link (is->priv->main_list, slist);
			slist = g_slist_copy (is->priv->main_list);

			real_item_selector_set_static_content_func (is, slist,
								    is->priv->data_function, TRUE);
			g_slist_free (slist);

			if (pos == selitem) 
				gtk_list_select_item (GTK_LIST (GTK_COMBO (is->priv->combo)->list), 0);
		}
		break;
	default:
		g_warning ("item_selector_del_object (): mode %d unrecognized\n", is->priv->mode);
		break;
	}
}
