%module MeCab
%include exception.i
%include arg.i
%newobject getSurface;
%newobject toString;
%newobject format;

%{
#undef do_open
#undef do_close
#include "mecab.h"

#define _PATH_ERROR "Path::check(): access violation"
#define _NODE_ERROR "Node::check(): access violation"

  namespace SWIG_MeCab {
    
    class Node;

    class Path {
    private:	
      mecab_path_t *path;
      char *str;
      char **ptr;
      inline void check () { if (!path || *ptr != str) throw _PATH_ERROR; }
    public:
      Path (mecab_path_t *_path, char *_str, char **_ptr): path (_path), str (_str), ptr (_ptr) {};
      Path (): path (0), str(0), ptr(0) {};
      ~Path() {};
      inline bool hasPath () 
      {
	if (*ptr != str) throw _PATH_ERROR;
	return path ? true : false;
      }
      Node getNode (); 
      inline unsigned int getCost () { check(); return path->cost; }
      inline Path next ()            { check(); Path p (path->next, str, ptr); return p; }
    };

    class Node {

    private:
      mecab_node_t *node;
      char *str;
      char **ptr; // pointer to str;
      inline void check () { if (!node || *ptr != str) throw _NODE_ERROR; }
    public:
      Node (mecab_node_t *_node, char *_str, char **_ptr): node(_node), str(_str), ptr(_ptr) {}
      Node (): node(0), str(0) {}
      ~Node() {};
      inline bool hasNode () 
      {
	if (*ptr != str) throw _NODE_ERROR;
	return node ?  true : false; 
      } 
      inline bool isBest () { check(); return (node->next || node->length == 0)  ? true : false; }
      inline Node prev   () { check(); Node n (node->prev,  str, ptr); return n; }
      inline Node next   () { check(); Node n (node->next,  str, ptr); return n; }

      inline char* toString ()
      {
         check ();
	 unsigned int flen = strlen (node->feature);
	 char *s = new char [node->length + flen + 2];
	 memcpy (s, node->surface, node->length);
	 s[node->length] = '\t';
	 memcpy (s + node->length + 1, node->feature, flen);
	 s[node->length + flen + 1] = '\0';
	 return s;
      }
      
      inline char* getSurface () 
      {
	check();
	char *s = new char [node->length + 1];
	memcpy (s, node->surface, node->length);
	s[node->length] = '\0';
   	return s;
      };

      inline Path getPath ()                   { check(); Path p (node->path, str, ptr); return p; }
      inline char* getFeature ()               { check(); return node->feature; };
      inline unsigned int getLength ()         { check(); return node->length; }
      inline unsigned int getBegin ()          { check(); return node->surface - str; }
      inline unsigned int getEnd ()            { check(); return node->surface - str + node->end; }
      inline unsigned int getID ()             { check(); return node->id; }
      inline unsigned int getRcAttr2 ()        { check(); return node->token->rcAttr2; }
      inline unsigned int getRcAttr1 ()        { check(); return node->token->rcAttr1; }
      inline unsigned int getLcAttr ()         { check(); return node->token->lcAttr;  }
      inline unsigned int getPOSID ()          { check(); return node->token->posid;   }
      inline unsigned int getTotalCost ()      { check(); return node->cost; }
      inline unsigned int getConnectionCost () { check(); return node->cost -node->prev->cost - node->token->cost; }
      inline unsigned int getWordCost ()       { check(); return node->token-> cost; }
      inline unsigned int getNodeCost ()       { check(); return node->cost - node->prev->cost; }
    };

    Node Path::getNode () { check(); Node n (path->node, str, ptr); return n; };

    class Tagger {

    private:
      mecab_t* m;
      char *str;
      
    public:
      Tagger (int argc, char *argv[]): m(0), str(0)
      { 
	if (! (m = mecab_new (argc, argv))) throw mecab_strerror (m); 
      }

      ~Tagger ()
      { 
	delete [] str; str = 0;
	mecab_destroy (m); m = 0;
      }

      Node parseToNode (char *p)
      {
	delete [] str;
	str = new char [strlen(p)+1] ;
	strcpy (str, p);
	Node node (mecab_sparse_tonode (m, str), str, &str);
	return node; 
      }

      inline char* parseToString (char *p) 
      { 
	char *r = mecab_sparse_tostr (m, p);
	if (!r) throw mecab_strerror (m);
	return r;
      }

      inline char* parse (char *p) { return parseToString (p); } // alias
      inline int   lock   ()       { return mecab_lock (m); }
      inline int   unlock ()       { return mecab_unlock (m); }
    };
  }
%}

%exception {
  try { $action }
  catch (char *e) { SWIG_exception (SWIG_RuntimeError, e); }
  catch (const char *e) { SWIG_exception (SWIG_RuntimeError, (char*)e); }
}

namespace SWIG_MeCab {

  class Path;

  class Node {
  public:
    int hasNode ();
    int isBest();
    Node prev ();
    Node next ();
    char* toString ();
    char* getSurface ();
    char* getFeature ();
    Path         getPath ();
    unsigned int getLength ();
    unsigned int getBegin ();
    unsigned int getEnd ();
    unsigned int getID ();
    unsigned int getRcAttr2 ();
    unsigned int getRcAttr1 ();
    unsigned int getLcAttr ();
    unsigned int getPOSID ();
    unsigned int getTotalCost ();
    unsigned int getConnectionCost ();
    unsigned int getWordCost ();
    unsigned int getNodeCost ();
  };

  class Path {
  public:
    bool hasPath ();
    Node getNode ();
    unsigned int getCost ();
    Path next ();
  };

  class Tagger {
  public:
    Tagger (int argc, char *argv[]);
    ~Tagger ();
    char* parse (char *);
    char* parseToString (char *);
    Node parseToNode (char *);
    int lock ();
    int unlock ();
  };
}

