/*
 MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

 $Id: print.cpp,v 1.17 2003/01/24 17:13:54 taku-ku Exp $;

 Copyright (C) 2001-2002  Taku Kudo <taku-ku@is.aist-nara.ac.jp>
 All rights reserved.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later verjsion.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; if not, write to the
 Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 Boston, MA 02111-1307, USA.
*/
#include <stdexcept>
#include <cstdio>
#include <cstring>
#include <fstream>
#include <iostream>
#include "common.h"
#include "mecab.h"

namespace MeCab {

  inline static char getEscapedChar (const char p) 
  {
    switch (p) {
    case '0':  return '\0'; break;
    case 'a':  return '\a'; break;
    case 'b':  return '\b'; break;
    case 't':  return '\t'; break;
    case 'n':  return '\n'; break;
    case 'v':  return '\v'; break;
    case 'f':  return '\f'; break;
    case 'r':  return '\r'; break;
    case 's':  return ' ';  break; // space	     
    case '\\': return '\\'; break;
    default: throw std::runtime_error ("format error \\");
    }

    return '\0'; // never be here
  }

  std::ostream& writeNode (const char *p, char *ibuf, Node *node, std::ostream& os) 
  {
    char buf  [1024];
    char *ptr [64];
    unsigned int psize = 0;

    for (; *p; p++) {
      switch (*p) {

      default: os << *p; break;

      case '\\': os << getEscapedChar (*++p); break;

      case '%': { // macros
	switch (*++p) {
	default: throw std::runtime_error (std::string ("unkonwn meta char ") + *p);
	case 'S': os.write (ibuf, strlen(ibuf)); break;  // input sentence
	case 'L': os << strlen(ibuf); break;  // sentence length
	case 'm': os.write (node->surface, node->length); break; // morph
	case 'h': os << node->token->posid; break; // Part-Of-Speech ID
	case '%': os << '%'; break;           // %
	case 'c': os << (int)(node->token->cost); break; // word cost  
	case 'H': os << node->feature; break; // feature
	case 'p': { 
	  switch (*++p) {
	  default: throw std::runtime_error ("[iseCwcnblh] is required after %p");
	  case 'i': os << node->id; break; // node id
	  case 's': os << (int)(node->surface - ibuf); break; // start position
	  case 'e': os << (int)(node->surface - ibuf + node->end); break; // end position
	  case 'C': os << (int)(node->cost - node->prev->cost - node->token->cost); // connection cost
	  case 'w': os << (int)node->token->cost; break; // word cost
	  case 'c': os << (int)(node->cost); break; // best cost
	  case 'n': os << (int)(node->cost - node->prev->cost); break; // node cost
	  case 'b': os << (node->next ? '*' : ' '); break; // * if best path, ' 'otherwise 
	  case 'l': os << node->length; break; // length of morph
	  case 'h': { // Hidden Layer ID
	    switch (*++p) {
	    default: throw std::runtime_error ("[012] is required after %ph");
	    case '0': os << node->token->lcAttr;  break;  // current
	    case '1': os << node->token->rcAttr1; break;  // prev
	    case '2': os << node->token->rcAttr2; break;  // prev-prev
	    }
	  } break;

	  case 'p': {
	    char mode = *++p;
	    char sep = *++p;
	    if (sep == '\\') sep = getEscapedChar (*++p);
	    if (!node->path) throw std::runtime_error ("no path information, use -a option");
	    for (Path *path = node->path; path; path = path->next) {
	      if (path != node->path) os << sep;
	      switch (mode) {
	      case 'i': os << path->node->id; break;
	      case 'c': os << path->cost; break;
	      default: throw std::runtime_error ("[ic] is required after %pp"); // connection cost
	      }
	    }
	  } break; 

	  }
	} break;

	case 'F':
	case 'f': {
	  if (! psize) {
	    strncpy (buf, node->feature, 1024);
	    ptr[psize++] = buf;
	    for (char *p = buf; *p;  ++p) { // split
	      if (',' == *p) { 
		*p = '\0'; 
		ptr[psize++] = p + 1; 
	      }
	    }
	  } 

	  // separator
	  char separator = '\t'; // default separator
	  if (*p == 'F') { // change separator
	    if (*++p == '\\') separator = getEscapedChar (*++p); 
	    else separator = *p;
	  }

	  if (*++p !='[') throw std::runtime_error ("cannot find '['");
	  unsigned int n = 0;
	  bool sep = false;
	  bool isfil = false;
	  p++;

	  for (;; ++p) {
	    switch (*p) {
	    case '0': case '1': case '2': case '3': case '4': 
	    case '5': case '6': case '7': case '8': case '9':
	      n = 10 * n + (*p - '0');
	      break;
	    case ',': case ']':
	      if (n >= psize) throw std::runtime_error ("given index is out of range");
	      isfil = (ptr[n][0] != '*');
	      if (isfil) { if (sep) os << separator; os << ptr[n]; } 
	      if (*p == ']') goto last;
	      sep = isfil;
	      n = 0;
	      break;
	    default:
	      throw std::runtime_error ("cannot find ']'");
	      break;
	    }
	  }
	} last: break;
	} // end switch 
      } break; // end case '%'
      } // end switch
    }

    return os;
  }
}
