/***************************************************************************
                           playlist.cpp  
                           -------------------
    begin                : Die Apr 22 2003
    revision             : $Revision: 1.59 $
    last modified        : $Date: 2004/05/10 09:11:00 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by J. Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/



#include <krandomsequence.h>
#include <kapplication.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <kurldrag.h>
#include <klocale.h>
#include <kurlrequesterdlg.h>
#include <kfiledialog.h>
#include <kdirselectdialog.h>
#include <kmimetype.h>
#include <kstandarddirs.h>
#include <kio/netaccess.h>
#include <kprinter.h>
#include <kmessagebox.h>
#include <kprogress.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kfilemetainfo.h>
#include <kstatusbar.h>
#include <klineedit.h>
#include <kdebug.h>

#include <qdir.h>
#include <qlabel.h>
#include <qclipboard.h>
#include <qwhatsthis.h>
#include <qdragobject.h>
#include <qstringlist.h>
#include <qdom.h>
#include <qxml.h>
#include <qmutex.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qregexp.h>
#include <qheader.h>


#include <stdlib.h>
#include <pwd.h>

#include "http.h"
#include "convert.h"
#include "playlist.h"
#include "playlist.moc"



PlayList::PlayList(const xine_t* const xine, QWidget *videoWindow, QWidget* parent, const char *name, bool needGUI )
 : KMainWindow( parent , name ), videoWin( videoWindow ), withGUI( needGUI), sortAscending( true ), xineEngine( (xine_t*)xine ), currentEntry( NULL ),
   currentChild( NULL ), fileFilter( QString::null ), metaInfoString( QString::null )
{
  random = false; endless = false; metaOnLoading = true;
  playTime = 0; playTimeVisible = 0; countVisible = 0; searchSelection = false;
  list = new UrlListView(this);

  if (!withGUI) return; /* konqueror-plugin don't need menus,etc */

/************************/
  
  setAcceptDrops(true);
 
  list->setHScrollBarMode(KListView::AlwaysOff); 
  list->setItemMargin(1);
  list->setMargin(2);
  list->setSelectionMode(QListView::Extended);
  list->addColumn("");
  list->addColumn(i18n("Title"));
  list->addColumn(i18n("Length"));
  list->addColumn("");
  list->addColumn("");
  list->setShowToolTips(true);

  list->setColumnWidthMode(0, QListView::Manual);
  list->setColumnWidthMode(1, QListView::Manual);
  list->setColumnWidthMode(2, QListView::Manual);
  list->setColumnWidthMode(3, QListView::Manual);
  list->setColumnWidthMode(4, QListView::Manual);
  list->setResizeMode(QListView::NoColumn);

  list->setDragEnabled(true);
  list->setAcceptDrops(true);
  list->setDropVisualizer(true);
  list->setItemsMovable(true);
  list->setSorting(-1);
  list->setItemsRenameable(true);
  list->setRenameable(1);
  list->setAllColumnsShowFocus(true);

  QWhatsThis::add(list, i18n("Playlist: Add entries via menu or use drag and drop. Drag and drop is supported from Konqueror, Mozilla, ..."));

  setCentralWidget(list);

  InitActions();
  InitMenus();

  toolBar()->setCloseMode(KToolBar::Undocked); /** show close button if toolbar undocked **/
  
  connect(list, SIGNAL(dropped(QDropEvent*, QListViewItem*)), this, SLOT(slotDropEvent(QDropEvent*, QListViewItem*)));
  connect(list, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotPlayDirect(QListViewItem*)));
  connect(list, SIGNAL(returnPressed(QListViewItem*)), this, SLOT(slotPlayDirect(QListViewItem*)));
  connect(list, SIGNAL(signalCut()), this, SLOT(slotCut()));
  connect(list, SIGNAL(signalCopy()), this, SLOT(slotCopy()));
  connect(list, SIGNAL(signalPaste()), this, SLOT(slotPaste()));
  connect(list, SIGNAL(signalPlayItem(QListViewItem*)), this, SLOT(slotPlayDirect(QListViewItem*)));
  connect(list->header(), SIGNAL(clicked(int)), this, SLOT(slotSort(int)));

 /** notice if toolbar visiblity is changed **/
  connect(toolBar(), SIGNAL(visibilityChanged(bool)), this, SLOT(slotToolbarVisibilityChanged(bool)));
  
  isCurrentEntry = KGlobal::iconLoader()->loadIcon("player_play", KIcon::Small);
  infoPixmap = KGlobal::iconLoader()->loadIcon("info", KIcon::Small);
  cdPixmap = KGlobal::iconLoader()->loadIcon("mime_track", KIcon::Small);
  bookmarkPixmap = KGlobal::iconLoader()->loadIcon("bookmark", KIcon::Small);

  setCaption(i18n("Kaffeine Playlist"));
  statusBar()->insertItem( i18n("Entries: %1, Playtime: %2  (Total: %3, %4)").arg("0").arg("0:00:00").arg("0").arg("0:00:00"), 0 );
}



PlayList::~PlayList()
{  
  delete list;
}


/**** init actions and menus ***/

void PlayList::InitActions()
{
  actionCollection = new KActionCollection(this);

  fileNew = KStdAction::openNew(this, SLOT(slotClearList()), actionCollection);
  fileOpenPlaylist = KStdAction::open(this, SLOT(slotOpenPlaylist()), actionCollection);
  fileSavePlaylist = KStdAction::save(this, SLOT(slotTrySaveCurrentPlaylist()), actionCollection);
  fileSavePlaylistAs = KStdAction::saveAs(this, SLOT(slotSavePlaylist()), actionCollection);
  
  fileExportHtml   = new KAction(i18n("&Html"),"html", 0, this, SLOT(slotExportHtml()), actionCollection, "exporthtml");
  fileImportPls    = new KAction(i18n("&pls (ShoutCast Playlist)"), "txt", 0, this, SLOT(slotImportPls()), actionCollection, "importpls");
  fileImportNoatun = new KAction(i18n("&Noatun Playlist"),"noatun", 0, this, SLOT(slotImportNoatun()), actionCollection, "importnoatun");
  fileImportM3U    = new KAction(i18n("&m3u Playlist"),"txt", 0, this, SLOT(slotImportM3U()), actionCollection, "importm3u");
  fileImportAsx    = new KAction(i18n("&asx (Windows Media Metafile)"),"html", 0, this, SLOT(slotImportAsx()), actionCollection, "importasx");

  fileClose        = KStdAction::close(this, SLOT(hide()), actionCollection);
  filePrint        = KStdAction::print(this, SLOT(slotPrint()), actionCollection);
  fileQuit         = KStdAction::quit(this, SIGNAL(signalQuit()), actionCollection);
  
  addFiles   = new KAction(i18n("&Add File(s)"), "video", Qt::CTRL|Qt::Key_F, this, SLOT(slotAddFiles()), actionCollection, "addfiles");
  addUrl     = new KAction(i18n("Add &URL"), "package_network", Qt::CTRL|Qt::Key_U, this, SLOT(slotAddUrl()), actionCollection, "addurl");
  addDir     = new KAction(i18n("Add &Directory"), "folder_video", Qt::CTRL|Qt::Key_D, this, SLOT(slotAddDir()), actionCollection, "adddir");
 
  editSort =  new KAction(i18n("Sort by titles"), NULL, Qt::Key_S, this, SLOT(slotSortByTitle()), actionCollection, "sortbytitle");
  editCut = KStdAction::cut(this, SLOT(slotCut()), actionCollection);
  editPaste = KStdAction::paste(this, SLOT(slotPaste()), actionCollection);
  editCopy = KStdAction::copy(this, SLOT(slotCopy()), actionCollection);
  
  viewToolbar = KStdAction::showToolbar(this, SLOT(slotViewToolbar()), actionCollection, "showToolbar");

 /** unplugged actions **/
  new KAction(i18n("Play/Pause"), NULL, Qt::Key_Space, this, SIGNAL(signalPlayPause()), actionCollection, "toggleplayplause");
  new KAction(i18n("Stop"), NULL, Qt::Key_Backspace, this, SIGNAL(signalStop()), actionCollection, "stopplayback");
  new KAction(i18n("Delete Selected Entries"), NULL, Qt::Key_Delete, this, SLOT(slotRemoveSelected()), actionCollection, "deleteselected");
  new KAction(i18n("Activate Main Window"), NULL, Qt::Key_P, this, SLOT(slotSetVideoWinActive()), actionCollection, "activatemainwin");
}


void PlayList::InitMenus()
{
  fileMenu = new KPopupMenu(this);
  addMenu  = new KPopupMenu(this);
  editMenu = new KPopupMenu(this);
  importMenu = new KPopupMenu(this);
  exportMenu = new KPopupMenu(this);
  viewMenu = new KPopupMenu(this);

  fileExportHtml->plug(exportMenu);

  fileImportNoatun->plug(importMenu);
  fileImportM3U->plug(importMenu);
  fileImportPls->plug(importMenu);
  fileImportAsx->plug(importMenu);

  fileNew->plug(fileMenu);
  fileNew->plug(toolBar());
  fileOpenPlaylist->plug(fileMenu);
  fileOpenPlaylist->plug(toolBar());
  fileSavePlaylist->plug(fileMenu);
  fileSavePlaylist->plug(toolBar());
  fileSavePlaylistAs->plug(fileMenu);
 
  fileMenu->insertSeparator();
  fileMenu->insertItem( KGlobal::iconLoader()->loadIconSet("fileimport", KIcon::Small),
                        i18n("&Import..."), importMenu);
  fileMenu->insertItem( KGlobal::iconLoader()->loadIconSet("fileexport", KIcon::Small),
                        i18n("&Export to"), exportMenu);
  filePrint->plug(fileMenu);
  fileMenu->insertSeparator();

  fileClose->plug(fileMenu);
  fileQuit->plug(fileMenu);
  toolBar()->insertLineSeparator();

  addDir->plug(addMenu);
  addDir->plug(toolBar());
  addFiles->plug(addMenu);
  addFiles->plug(toolBar());
  addUrl->plug(addMenu);
  addUrl->plug(toolBar());
  toolBar()->insertLineSeparator();

  editSort->plug(editMenu);
  editCut->plug(editMenu);
  editCut->plug(toolBar());
  editCopy->plug(editMenu);
  editCopy->plug(toolBar());
  editPaste->plug(editMenu);
  editPaste->plug(toolBar());

  viewToolbar->plug(viewMenu);
  
  menuBar()->insertItem(i18n("&File"), fileMenu);
  menuBar()->insertItem(i18n("&Add"), addMenu);
  menuBar()->insertItem(i18n("&Edit"), editMenu);
  menuBar()->insertItem(i18n("&View"), viewMenu);
  
  toolBar()->insertSeparator();
  toolBar()->insertLined("", 123, SIGNAL(textChanged(const QString&)), this, SLOT(slotFindText(const QString&)),
                       true, i18n("Find Expression in Titles"));
  toolBar()->getLined(123)->insert( i18n("Search") );                     
  toolBar()->alignItemRight(123);
}


void PlayList::closeEvent(QCloseEvent*)
{
  hide();
}


void PlayList::SaveToolbarSettings(KConfig* config)
{
  toolBar()->saveSettings(config, "PlaylistToolBar");
}

    
void PlayList::LoadToolbarSettings(KConfig* config)
{
  toolBar()->applySettings(config, "PlaylistToolBar");
  if (toolBar()->isHidden()) viewToolbar->setChecked(false);
}
  

/******************************************
 *      get the urls from playlist
 ******************************************/

QListViewItem* PlayList::GetCurrent()
{
  if (random)
  {
    if (currentRandomListEntry == -1) return NULL;
    SetCurrentEntry(randomList.at(currentRandomListEntry));
    return currentEntry;
  }  

  if (!currentEntry)
    if (list->childCount()>0)
    {
      if (list->firstChild()->isVisible())
         SetCurrentEntry(list->firstChild());
        else
        {  
          if (list->firstChild()->itemBelow())
            SetCurrentEntry(list->firstChild()->itemBelow());
           else
            return NULL;
        }   
    }
    else
     return NULL;  

   return currentEntry;
}


QListViewItem* PlayList::GetNext()
{
  if (!currentEntry) return GetCurrent();

  if (random)
  {
  if ((currentRandomListEntry+1) < (int)randomList.count())
      currentRandomListEntry += 1;
     else
     {
       if (endless)
         currentRandomListEntry = 0;
        else
         return NULL;
     }
    SetCurrentEntry(randomList.at(currentRandomListEntry));
    return currentEntry;     
   }

  QListViewItem* tmpItem;
  tmpItem = currentEntry->itemBelow();
  
  if (tmpItem)
  {
    SetCurrentEntry(tmpItem);
    return currentEntry;
  }
  else
  {
    if (endless)
      {
        SetCurrentEntry(list->firstChild());
        return currentEntry;
      }  
     else
      return NULL;
  }  
}


QListViewItem* PlayList::GetPrevious() 
{
  if (!currentEntry) return GetCurrent();

  if (random)
  {
    if (currentRandomListEntry > 0)
      currentRandomListEntry -= 1;
     else
     {
       if (endless)
         currentRandomListEntry = randomList.count()-1;
        else
         return NULL;
     }
    SetCurrentEntry(randomList.at(currentRandomListEntry));
    return currentEntry;
   }

  QListViewItem* tmpItem;
  tmpItem = currentEntry->itemAbove();

  if (tmpItem)
  {
    SetCurrentEntry(tmpItem);
    return currentEntry;
  }
  else
  {
    if (endless)
     {
       SetCurrentEntry( GetLast() ); 
       return currentEntry;
     }  
     else
      return NULL;
  }  
}


QListViewItem* PlayList::GetLast()
{
  return list->lastItem();
}


/********* set current entry (with play icon) ****************/

void PlayList::SetCurrentEntry(QListViewItem* item)
{
  if (currentEntry)
    currentEntry->setPixmap(1, QPixmap());
  
  item->setPixmap(1, isCurrentEntry);
  currentEntry = item;
  currentChild = NULL;
  list->setCurrentItem(currentEntry);

  list->ensureVisible(10, list->itemPos(currentEntry), 10, 30);
}


/***************************************************
 *          create new items                       *
 ***************************************************/            

QListViewItem* PlayList::CreateItem(QListViewItem* after, QString url, QString mime, QString title,
   QListViewItem* parent, QString length, QString info, QStringList subtitles)
{
  PlaylistItem* tmp = NULL;
        
  if (list->findItem(url, 4))
  {
    kdDebug(555) << "Skipped (still exists): " << url << endl;
    return NULL;   
  }  

  if (title.isNull())
  {
    KURL kurl(url);
  
    if (kurl.isLocalFile())
       title = kurl.fileName();
     else
       title = kurl.prettyURL();
   }

   if (mime.isNull())
   {
     KMimeType::Ptr getMime = KMimeType::findByURL(url);
     mime = getMime->name();
   }
     
   if (parent)
   {
     tmp = new PlaylistItem(dynamic_cast<KListViewItem *>(parent), dynamic_cast<KListViewItem *>(after), url, mime, title, length, info, subtitles);
     if (!tmp) return NULL;
   }
   else
   {
     tmp = new PlaylistItem(list, dynamic_cast<KListViewItem *>(after), url, mime, title, length, info, subtitles);
     if (!tmp) return NULL;
   }  

   if (withGUI)
   {
     if (mime == "autoplay-plugin")
       tmp->setPixmap(0, cdPixmap);
      else
       tmp->setPixmap(0, KMimeType::mimeType(mime)->pixmap(KIcon::Small));
      
     tmp->setPixmap(4, bookmarkPixmap);
     if (!info.isEmpty()) tmp->setPixmap(3, infoPixmap);

     if (!parent)
     {
       if (tmp->length().contains(':'))
         playTime += timeStringToMs( tmp->length() );
   
       if (searchSelection)
       {
         QString text = toolBar()->getLined(123)->text(); 
         if ( !(tmp->title().contains(text, false)) && (!tmp->url().contains(text, false)) )
         {
           tmp->setVisible( false );
         }
       }

       if (tmp->isVisible())
       {
         if (tmp->length().contains(':'))
           playTimeVisible += timeStringToMs( tmp->length() );
         countVisible++;  
       }
     }
   }
     
   return tmp;
}

 

/*****************************************
  add urls to playlist
*****************************************/  

void PlayList::AddAutoplayUrls(const QStringList& urls, QListViewItem* after)
{
  QListViewItem* tmp = NULL;
  QString title, length;
  KProgressDialog* progress = NULL;
  bool CDDBEntry = true;
  
  for (uint i = 0; i < urls.count(); i++)
  {
    if ( (metaOnLoading) && (urls[i].contains("cdda")) && (CDDBEntry) )
    {
      if (!progress)
      {
        progress = new KProgressDialog( this, "cddbprogress", QString::null, i18n("Looking for CDDB entries...") );
        progress->progressBar()->setTotalSteps( urls.count() );
        progress->show();
      }
      title = QString::null;
      length = QString::null;
      GetMetaFromXine( urls[i], title, length );
      if ( title.isNull() )
      {
        kdDebug(555) << "No CDDB entry found" << endl;  
        CDDBEntry = false;
      }
      if (progress->wasCancelled())
        break;
      progress->progressBar()->setProgress( i+1 );
      KApplication::kApplication()->processEvents();  
    }    
    tmp = CreateItem(after, urls[i], "autoplay-plugin", title, NULL, length);
    if (tmp) after = tmp;
  }

  if (progress)
    delete progress;
  if (random) CreateRandomList();
  UpdateStatusBar();
}


void PlayList::Add(const KURL& kurl, QListViewItem* after)
{
  Add(KURL::List(kurl), after);
}


void PlayList::Add(const KURL::List& kurlList, QListViewItem* after)
{
  KURL::List urls( kurlList );
  QListViewItem* tmp = NULL;

  QString ext;
  QString fname;
  QString title, length;
  QString subtitleURL;
  QString url;
  QStringList subs;
  KURL tmpKURL;
  
  KProgressDialog* progress = NULL;
  if (withGUI)
  {
    progress = new KProgressDialog( this, "importprogress", QString::null, i18n("Importing media resources...") );
    progress->progressBar()->setTotalSteps( urls.count() );
    progress->show();
  }  

  for (uint i = 0; i < urls.count(); i++)
  {

    if (urls[i].isLocalFile())
      url = urls[i].path();
     else
      url = urls[i].prettyURL();

   // kdDebug(555) << "path: " << urls[i].path() << endl;
   // kdDebug(555) << "prettyURL: " << urls[i].prettyURL() << endl;
      
    /********** check if autoplay plugin ************/

     fname = urls[i].fileName();
     uint a = 0;
     while (a < autoplayList.count())
     {
       if (fname == autoplayList[a])
       {
         if (withGUI)
           delete progress;
         emit signalAutoplayPlugin(fname, after);
         return;    
       }
       a += 2;
     }    

     /*********** determine extension and mime type ************/

     ext = urls[i].fileName();
     ext = ext.remove( 0 , ext.findRev('.')+1 ).lower();
    // kdDebug(555) << "Extension: " << ext << endl;
     
     KMimeType::Ptr mime = KMimeType::findByURL( urls[i].path() ); /* works only with path() (without protocol)
                                                                      e.g. http://www.somafm.com/indipop.pls
                                                                      path: /indipop.pls */
     //kdDebug(555) << "Mime: " << mime->name() << endl;
     /*** check for kaffeine/noatun/pls/asx/m3u playlist ***/
            
     if ((mime->name() == "text/plain") || (mime->name() == "text/xml") || (mime->name() == "application/x-kaffeine")
         || (mime->name() == "audio/x-scpls") || (mime->name() == "audio/x-mpegurl")
         || (ext == "asx") || (ext == "asf") || (ext == "wvx") ) /* windows meta files */
     {
       kdDebug(555) << "Check for kaffeine/noatun/m3u/pls/asx playlist\n";
       QString localFile;
       if (KIO::NetAccess::download( urls[i], localFile ))
       {
         QFile file(localFile);
         file.open(IO_ReadOnly);

         QTextStream stream(&file);
         QString firstLine = stream.readLine();
         QString secondLine = stream.readLine();
         file.close();
          
         if (secondLine.contains("kaffeine", false))
         {
           kdDebug(555) << "Try loading kaffeine playlist\n";
           if (!LoadPlaylist(localFile, after))
              kdDebug(555) << "No playlist: " << url << endl;
           continue;
         }
         if (secondLine.contains("noatun", false))
         {
           kdDebug(555) << "Try loading noatun playlist\n";
           if (!LoadNoatunPlaylist(localFile, after))
             kdDebug(555) << "No playlist: " << url << endl;
           continue;
         }
         if (firstLine.contains("asx", false))
         {
           kdDebug(555) << "Try loading asx playlist\n";
           if (!LoadAsxPlaylist(localFile, after))
              kdDebug(555) << "No playlist: " << url << endl;
           continue;
         }
           
         if (firstLine.contains("[playlist]", false))
         {
           kdDebug(555) << "Try loading pls playlist\n";
           if (!LoadPlsPlaylist(localFile, after))
              kdDebug(555) << "No playlist: " << url << endl;
           continue;
         } 
          
         if (ext == "m3u")  //indentify by extension
         {
           kdDebug(555) << "Try loadng m3u playlist\n";
           if (!LoadM3UPlaylist(localFile, after))
              kdDebug(555) << "No playlist: " << url << endl;
           continue;
         }
       }
     }

     /* check for ram playlist */
     if ( (ext == "ra") || (ext == "rm") || (ext == "ram") || (ext == "lsc") || (ext == "pl") )
     {
       if ( LoadRamPlaylist( urls[i], after ) )
         continue;
     }    

     title = QString::null;
     length = QString::null;  

     /******** some special processing for local files! *******/

     if (urls[i].isLocalFile()) 
     {
       /**** directory? ****/
       if (mime->name() == "inode/directory")
       {
         kdDebug(555) << "Add Directory: " << url << endl;

         KURL::List kurlList;
         QDir d( url );
         uint x;

         /* subdirs */
         QStringList entryList = d.entryList( QDir::Dirs, QDir::Name );
         for (x = 0; x < entryList.count(); x++)
         {
           if (entryList[x][0] != '.')
           {
             tmpKURL.setPath( url + "/" + entryList[x] );
             urls.append( tmpKURL );
           }
         }

         /* media files */
         entryList = d.entryList( fileFilter, QDir::Files, QDir::Name );
         for (x = 0; x < entryList.count(); x++)
         {
           tmpKURL.setPath( url + "/" + entryList[x] );
           urls.append( tmpKURL );
         }

         if (withGUI)
           progress->progressBar()->setTotalSteps( urls.count() );
         continue; /* dont add directory to playlist */
       }

       /*** get meta tags ****/
       if ( (metaOnLoading) && (withGUI) )
         GetMetaInfo( url, mime->name(), title, length );
       
       /* Get all suported subs in movie dir, clear out
        * those starting with a different name than the movie,
        * prompt the user to select a sub
        */
       if (mime->name().contains("video"))
       {
         subtitleURL = QString::null;
         QDir *d = new QDir(url.section('/',0,-2));
         QString filename = url.section('/',-1,-1);
         subs = QStringList();
       //Do a case insensitive search for subs
         QStringList dirListing = d->entryList(QDir::Files|QDir::NoSymLinks); //cache directory listing
         bool matches = false;
 
         for(QStringList::Iterator it = dirListing.begin(); it != dirListing.end(); ++it )
         {
           if(StartsWith(*it, filename.section('.',0,-2), false)) //If filenames (without extensions) match
           {
             if( EndsWith(*it, ".srt", false) || EndsWith(*it, ".ssa", false) || 
                 EndsWith(*it, ".txt", false) || EndsWith(*it, ".smi", false) || 
                 EndsWith(*it, ".asc", false) || EndsWith(*it, ".sub", false) )
               matches = true;
    
             if(matches)
              subs.append(*it); //This is a subtitle for our movie
              matches = false;
            }
          }
 
          subs.prepend(i18n("(no subtitles)"));
          subs.append(i18n("Other subtitle..."));
          if((subs.size() >= 3))
          {
            SubtitleChooser *sc = new SubtitleChooser(subs, filename, this);
   
THERE:      if(sc->exec() == QDialog::Accepted)
            { 
              if((sc->getSelection() != i18n("(no subtitles)")) && (sc->getSelection() != i18n("Other subtitle...")))
               subtitleURL = d->path()+"/"+sc->getSelection();
              if((sc->getSelection() == i18n("Other subtitle...")))
              {
                subtitleURL = KFileDialog::getOpenURL(d->path(), i18n("*.smi *.srt *.sub *.txt *.ssa *.asc|Subtitle Files\n*.*|All Files"), 0, i18n("Select a subtitle file")).path();
                if(subtitleURL == "")
                {
                  subtitleURL = QString::null;
                  goto THERE;
                }
                else //The user has selected another sub, so add this to the list
                {
                  subs.clear();
                  subs.append(subtitleURL);
                }
              }
            }
            delete sc;
          }
          subs.remove(i18n("(no subtitles)"));
          subs.remove(i18n("Other subtitle..."));
        //Add the full path to the subtitle name
          for (unsigned int i = 0; i < subs.size(); i++ )
            if(!subs[i].startsWith(d->path()))    
             subs[i] = d->path()+"/"+subs[i];

          if (!subtitleURL.isNull())
          {
            kdDebug(555) << "Use subtitle file: " << subtitleURL << " for: " << url << endl;
            url = url + "#subtitle:" + subtitleURL;
          }  
        }  
     } /* if localFile() */

     /* urls from audiocd kio-slave */
     if (urls[i].protocol() == "audiocd")
     {
       QString audioTrack = QString::number( urls[i].fileName().remove( QRegExp("\\D" ) ).left( 2 ).toUInt() );
       url = audioTrack.prepend( "cdda:/" );
     }  

     /* cdda:/ urls */
     if ( (metaOnLoading) && (withGUI) && (url.left(4) == "cdda") )
       GetMetaFromXine( url, title, length );

     tmp = CreateItem( after, url, mime->name(), title, NULL, length, QString::null, subs );
     
     if (tmp)
       after = tmp;

     if (withGUI)
     {
       if ( progress->wasCancelled() )
         break;
       progress->progressBar()->setProgress( i+1 );
       KApplication::kApplication()->processEvents();
     }  
  }

  if (withGUI)
    delete progress;
  if (random) CreateRandomList();
  UpdateStatusBar();
}




/********* insert child urls (ram, etc) ***********/

void PlayList::slotAddMrl(const QString& mrl)
{
  if (mrl.isEmpty()) return;
  currentEntry->setOpen(true);

  QListViewItem* tmp = CreateItem(currentChild, mrl, QString::null, QString::null, currentEntry);
  if (tmp) currentChild = tmp;
  
  currentEntry->setText(3, i18n(" Contents %1 URL(s)").arg(currentEntry->childCount()));
  if (random) CreateRandomList();
  UpdateStatusBar();
}

/*********/



void PlayList::slotPlayDirect(QListViewItem* item)
{
  if (!item) return;

  SetCurrentEntry(item);
  if (random)
    currentRandomListEntry = randomList.find(item);
  emit signalPlayDirect();
}  
  

void PlayList::SetEndless(bool e)
{
  endless = e;
}


void PlayList::SetRandomPlay(bool r)
{
  random = r;
  if (random) CreateRandomList();
}


void PlayList::CreateRandomList()
{
  randomList.clear();
  currentRandomListEntry = 0;

  QListViewItem* tmpItem = NULL;
  tmpItem = list->firstChild();
  if ( (tmpItem) && (!tmpItem->isVisible()) )
    tmpItem = tmpItem->itemBelow();

  while (tmpItem)
  {
    randomList.append(tmpItem);
    tmpItem = tmpItem->itemBelow();
  }

 if (!(randomList.count()>0))
 {
   currentRandomListEntry = -1;
   return;
 }  
 
 KRandomSequence r(KApplication::random());
 r.randomize(&randomList);
}


void PlayList::slotClearList()
{
  list->clear();
  randomList.clear();
  playTime = 0;
  playTimeVisible = 0;
  countVisible = 0;
  if (searchSelection)
  {
    toolBar()->getLined(123)->clear();
    searchSelection = false;
  }  
  UpdateStatusBar();
  currentEntry = NULL;
  currentRandomListEntry = -1;
}


void PlayList::slotDropEvent(QDropEvent* dev, QListViewItem* after)
{
   KURL::List urls;

   if (KURLDrag::decode(dev, urls))
   {
     Add(urls, after);
   }
   else
   if (strcmp(dev->format(), "text/x-moz-url") == 0)    /* for mozilla drops */
   {
     QByteArray data = dev->encodedData("text/plain");
     QString md(data);

     Add(md, after);
   } 
}


void PlayList::slotSetAlternateColor( const QColor& col )
{
  list->setAlternateBackground( col );
  altCol = col;
  list->triggerUpdate();
}


void PlayList::slotSetMetaInfoString(const QString& mi)
{
  metaInfoString = mi;
}  


void PlayList::slotRefreshMetaInfo()
{
  kdDebug(555) << "Refreshing titles" << endl;
  PlaylistItem* tmp = NULL;
  QString title, length;
  KProgressDialog* progress = new KProgressDialog( this, "metaprogress", QString::null, i18n("Refreshing titles...") );
  progress->progressBar()->setTotalSteps( list->childCount() );
  progress->show();
  uint i = 0;
  
  QListViewItemIterator it( list );
  while ( it.current() )
  {
    tmp = dynamic_cast<PlaylistItem *>(it.current());
    title = QString::null;
    GetMetaInfo( tmp->url(), tmp->mime(), title, length );
    if (!title.isNull())
    {
      tmp->setTitle(title);
    }
    
    if (progress->wasCancelled())
      break;
    progress->progressBar()->setProgress( ++i );
    KApplication::kApplication()->processEvents();
    ++it;
  }

  delete progress;
}


void PlayList::GetMetaInfo( const QString& url, const QString& mimeName, QString& title, QString& length )
{
  KFileMetaInfo* metaInfo = NULL;
  KFileMetaInfoGroup metaGroup;
  uint m, n;
  QString metaAlbum, metaArtist, metaTitle = QString::null, metaTrack;

  metaInfo = new KFileMetaInfo( url, mimeName );
  QStringList groups = metaInfo->groups();
  QStringList keys;
  
  for (m = 0; m < groups.count(); m++)
  {
    metaGroup = metaInfo->group( groups[m] );
    QStringList keys = metaGroup.keys();
    for (n = 0; n < keys.count(); n++)
    {
      if (keys[n] == "Length")
      {
        length = msToTimeString( metaGroup.item( keys[n] ).value().toUInt() * 1000 ); /* msec */
      }
      if (keys[n] == "Title")
        metaTitle =  metaGroup.item( keys[n] ).value().toString();
      if (keys[n] == "Artist")
        metaArtist =  metaGroup.item( keys[n] ).value().toString();
      if (keys[n] == "Album")
        metaAlbum =  metaGroup.item( keys[n] ).value().toString();
      if (keys[n] == "Tracknumber")
        metaTrack =  metaGroup.item( keys[n] ).value().toString();
    }
  }
  
  if ( (!metaTitle.isNull()) && (!metaTitle.isEmpty()) )
  {
     title = metaInfoString;
     title.replace( "artist", metaArtist.simplifyWhiteSpace() );
     title.replace( "title",  metaTitle.simplifyWhiteSpace() );
     title.replace( "album",  metaAlbum.simplifyWhiteSpace() );
     title.replace( "track",  metaTrack.simplifyWhiteSpace() );
   }
   else /* let xine try to do the job */
   {
      GetMetaFromXine( url, title, length );
   }
   
   delete metaInfo;
}

  
void PlayList::GetMetaFromXine( const QString& url, QString& title, QString& length )
{
  if (!xineEngine)
    return;
  QString metaTitle = QString::null, metaArtist, metaAlbum, metaTrack;
  xine_stream_t* xineStreamForMeta = xine_stream_new( xineEngine, NULL, NULL );
  if (!xine_open( xineStreamForMeta, url ))
    return;

  metaTitle = QString::fromUtf8( xine_get_meta_info(xineStreamForMeta, XINE_META_INFO_TITLE) );
  if ( (!metaTitle.isNull()) && (!metaTitle.isEmpty()) )  //usable meta info
  {
     metaArtist = QString::fromUtf8(xine_get_meta_info(xineStreamForMeta, XINE_META_INFO_ARTIST));
     metaAlbum = QString::fromUtf8(xine_get_meta_info(xineStreamForMeta, XINE_META_INFO_ALBUM));
     metaTrack = ""; /* not implemented in xine-lib */
     title = metaInfoString;
     title.replace( "artist", metaArtist.simplifyWhiteSpace() );
     title.replace( "title",  metaTitle.simplifyWhiteSpace() );
     title.replace( "album",  metaAlbum.simplifyWhiteSpace() );
     title.replace( "track",  metaTrack );
  }
  
  int pos, time, len;
  bool ok = xine_get_pos_length(xineStreamForMeta, &pos, &time, &len);
  if ( (ok) && (len > 0) )
  {
    length = msToTimeString(len);
  }  
  xine_close( xineStreamForMeta );
  xine_dispose( xineStreamForMeta );
}


/************** add files, url or directory **************/

void PlayList::slotAddFiles()
{
  KURL::List urlList;

  urlList = KFileDialog::getOpenURLs(":kaffeinePL_AddFiles", fileFilter + "|" + i18n("Supported Media Formats") + "\n*.*|" + i18n("All Files"), 0, i18n("Add File(s)"));

  if (urlList.count())
    Add( urlList, GetLast() );
}


void PlayList::slotAddUrl()
{

 KURL kurl = KURLRequesterDlg::getURL(QString::null, 0, i18n("Add URL"));

 if (kurl.isValid()) 
    Add( kurl, GetLast() );

}


void PlayList::slotAddDir()
{
  KURL path = KDirSelectDialog::selectDirectory(":kaffeinePL_AddDir", false, 0, i18n("Add Directory"));

  if (path.isValid())
    Add( path, GetLast() );

}

/**** show/hide toolbar ****/

void PlayList::slotToolbarVisibilityChanged(bool visibility)
{
  if (visibility)
    viewToolbar->setChecked(true);
   else
    viewToolbar->setChecked(false);
}     


void PlayList::slotViewToolbar()
{
  if (viewToolbar->isChecked())
     toolBar()->show();
   else
     toolBar()->hide();
}     


void PlayList::slotSetVideoWinActive()
{
  videoWin->raise();
  videoWin->setActiveWindow();
}
        

/************ sort playlist ******************/

void PlayList::slotSortByTitle()
{
  emit slotSort( 1 );
}  


void PlayList::slotSort( int section )
{
  list->setSorting( section, sortAscending );
  list->sort();
  list->setSorting(-1);
  sortAscending = !sortAscending;
}  


/***  remove items ***/

void PlayList::slotRemoveSelected()
{
  QPtrList<QListViewItem> selected;

  if (currentEntry)
    if (currentEntry->isSelected())
     {
        currentEntry = NULL;
        currentRandomListEntry = -1;
     }   

  selected = list->selectedItems();
  PlaylistItem* item = NULL;
  PlaylistItem* child = NULL;

  for(uint i = 0; i<selected.count(); i++)
  {
   // kdDebug(555) << "Remove " << selected.at(i)->text(4) << "\n";
    item = dynamic_cast<PlaylistItem *>(selected.at(i));
    if (item->length().contains(':'))
    {
      playTime -= timeStringToMs(item->length());
      playTimeVisible -= timeStringToMs(item->length());
    }  
    if (!item->parent())
      countVisible--;
    if (item->childCount())
    {
      child = dynamic_cast<PlaylistItem*>(item->firstChild());
      while (child)
      {
        if (child->length().contains(':'))
        {
          playTime -= timeStringToMs(child->length());
          playTimeVisible -= timeStringToMs(child->length());
        }
        selected.remove( child );
        child = dynamic_cast<PlaylistItem*>(child->nextSibling());
      }      
    }
    delete selected.at(i);
  } 
 
  if (random) CreateRandomList();
  UpdateStatusBar();
}


void PlayList::UpdateStatusBar()
{
  if (withGUI)
  {
    statusBar()->changeItem( i18n("Entries: %1, Playtime: %2  (Total: %3, %4)").arg(QString::number(countVisible)).arg(msToTimeString(playTimeVisible)).arg(QString::number(list->childCount())).arg(msToTimeString(playTime)), 0 );
  }
}    


/**************************************************
 *            load xml playlist
 **************************************************/         

class MyXMLParser : public QXmlDefaultHandler
{

public:
  PlayList *playList;
  QListViewItem* tmp;
  QListViewItem* after;
  bool isKaffeinePlaylist;
  

MyXMLParser(PlayList *pl): playList(pl), tmp(NULL), after(NULL), isKaffeinePlaylist(false)
{
}

bool startElement(const QString&, const QString&,
            const QString &qname, const QXmlAttributes &att)
{
                                                 
  if (qname == "playlist")
  if (att.value("client") == "kaffeine")
   {
     isKaffeinePlaylist = true;
     return true;
   } 
   else
   {
     return false;  
   }

  
  if (qname != "entry") return true;
  
  QStringList subs = QStringList();
  
  if(att.value("subs") != "" && att.value("subs") != QString::null)
    subs =  QStringList::split("&",att.value("subs"),false);
  
  tmp = playList->CreateItem(after, att.value("url"), att.value("mime"), att.value("title"), NULL,
                   att.value("length"), att.value("info"), subs);
  if (tmp) after = tmp;
   
  return true;
}

};


bool PlayList::LoadPlaylist(const QString& pl, QListViewItem* after)
{
  QFile file(pl);
  if (!file.open(IO_ReadOnly)) return false;
 
  QXmlInputSource source(&file);
  QXmlSimpleReader reader;

  MyXMLParser parser(this);
  parser.after = after;
  reader.setContentHandler(&parser);
  reader.parse(source);
  file.close();

  return parser.isKaffeinePlaylist; 
}


void PlayList::slotOpenPlaylist()
{
  
  QString path = KFileDialog::getOpenFileName(":kaffeinePL_OpenPlaylist", i18n("*.kaffeine|Kaffeine Playlists\n*.*|All Files"), 0, i18n("Open Playlist"));

  if (path.isEmpty()) return;
  
  slotClearList();

  if (!LoadPlaylist(path, NULL))
  {
    KMessageBox::sorry(this, i18n("No Kaffeine Playlist!"));
    lastPlaylist = QString();
    return;
  }

  if (random) CreateRandomList();
  UpdateStatusBar();

  lastPlaylist = path;
  list->SetCleared(false);
  
}


/****************************************************
 *              load noatun playlist
 ****************************************************/           

class NoatunXMLParser : public QXmlDefaultHandler
{

public:
  PlayList *playList;
  QListViewItem* tmp;
  QListViewItem* after;
  bool isNoatunPlaylist;


  
NoatunXMLParser(PlayList *pl): playList(pl), tmp(NULL), after(NULL), isNoatunPlaylist(false)
{
}

bool startElement(const QString&, const QString &,
            const QString &qname, const QXmlAttributes &att)
 {
                                                 
   if (qname == "playlist")
   if (att.value("client") == "noatun")
   {
    isNoatunPlaylist = true;
    return true;
   }
   else
   {
     return false;
   }

  if (qname != "item") return true;

  QString timeString = QString::null;
  QString title = QString::null;

  if (!att.value("title").isNull())
  {
    title = playList->metaInfoString;
    title.replace( "artist", att.value("author") );
    title.replace( "title", att.value("title") );
    title.replace( "album", att.value("album") );
    title.replace( "track", att.value("track") );
  }  
  
  bool ok;
  int time = 0;
  time = att.value("length").toInt(&ok); 
  
  if ((ok) && (time > 0))
  {
    timeString = msToTimeString(time);
  }
  
  tmp = playList->CreateItem(after, att.value("url"), QString::null, title, NULL, timeString);

  if (tmp) after = tmp;
  
  return true;
 }

};


bool PlayList::LoadNoatunPlaylist(const QString& pl, QListViewItem* after)
{
  QFile file(pl);
  if (!file.open(IO_ReadOnly)) return false;

  QXmlInputSource source(&file);
  QXmlSimpleReader reader;

  NoatunXMLParser parser(this);
  parser.after = after;
  reader.setContentHandler(&parser);
  reader.parse(source);
  file.close();

  return parser.isNoatunPlaylist;
}


void PlayList::slotImportNoatun()
{
  QString path = KFileDialog::getOpenFileName(":kaffeinePL_ImportNoatun", QString::null, 0, i18n("Import Noatun Playlist"));

  if (path.isEmpty()) return;

  slotClearList();
  
  if (!LoadNoatunPlaylist(path, NULL))
  {
    KMessageBox::sorry(this, i18n("No Noatun Playlist!"));
    return;
  }

  if (random) CreateRandomList();
  UpdateStatusBar();
}


/****************************************************
 *              load ram  playlist
 ****************************************************/

bool PlayList::LoadRamPlaylist( const KURL& kurl, QListViewItem* after)
{
  Q_ULONG result;
  KURL::List kurlList;
  char buf[10], mime[256];

  kdDebug(555) << "Check for ram playlist: " << kurl.prettyURL() << endl;
  
  if (kurl.isLocalFile())
  {
    QFile file( kurl.path() );
    if (!file.open(IO_ReadOnly))
    {
       kdError(555) << "Can't open " << kurl.path() << endl;
       return false;
    }   
    result = file.readBlock( buf, 4 );
    file.close();
    if (result != 4)
    {
      kdError(555) << "Can't read " << kurl.path() << endl;
      return false;
    }
  }
  else
  {
    if ( kurl.protocol() == "http" )
    {

      result = http_peek( kurl.url(), 4, buf, mime );

      if (result <= 0)
      {
        kdError(555) << "Can't open " << kurl.prettyURL() << endl;
        return false;
      }
    }
    else
      return false;
  }    
  
  if (buf[0]=='.' && buf[1]=='R' && buf[2]=='M' && buf[3]=='F')
  {
    kdDebug(555) << "This is a real media file" << endl;
    return false;
  }

  kdDebug(555) << "Seems to be a ram playlist" << endl;

  QString localFile, url;
  if (KIO::NetAccess::download( kurl, localFile ))
  {
    QFile plFile( localFile );
    if (!plFile.open(IO_ReadOnly)) return false;
    QTextStream stream( &plFile );

    while (!stream.eof())
    {
      url = stream.readLine();
      
      if (url[0] == '#') continue; /* ignore comments */
      if (url == "--stop--") break; /* stop line */

      if ( (url.left(7) == "rtsp://") || (url.left(6) == "pnm://") || (url.left(7) == "http://") )
       kurlList.append( url );
    }
  }

  Add( kurlList, after );
  
  return true;
}


/**********************************************************************************
 *                             load asx playlist                                  *
 *   spec: http://msdn.microsoft.com/library/en-us/wmplay/mmp_sdk/asxelement.asp  *
 **********************************************************************************/

bool PlayList::LoadAsxPlaylist(const QString& pl, QListViewItem* after)
{
  QFile file(pl);
  if (!file.open(IO_ReadOnly)) return false;

  QDomDocument doc;
  doc.setContent(&file);
  QDomElement root = doc.documentElement();

  QString url;
  QString title;

  QListViewItem* tmp = NULL;

  if (root.nodeName().lower() != "asx") return false;

  QDomNode node = root.firstChild();
  QDomNode subNode;
  QDomElement element;
  KURL::List kurlList;

  while (!node.isNull())
  {
    url = QString::null;
    title = QString::null;
    if (node.nodeName().lower() == "entry")
    {
      subNode = node.firstChild();
      while (!subNode.isNull())
      {
        if ((subNode.nodeName().lower() == "ref") && (subNode.isElement()) && (url.isNull()))
        {
          element = subNode.toElement();
          if (element.hasAttribute("href"))
             url = element.attribute("href");
          if (element.hasAttribute("HREF"))
             url = element.attribute("HREF");
          if (element.hasAttribute("Href"))
             url = element.attribute("Href");
          if (element.hasAttribute("HRef"))
             url = element.attribute("HRef");     
             
         }

        if ((subNode.nodeName().lower() == "title") && (subNode.isElement()))
         {
           title = subNode.toElement().text();
           if (title.isEmpty())
             title = QString::null;
         }
        
        subNode = subNode.nextSibling();
      }

      if (!url.isNull())
      {
        tmp = CreateItem(after, url, QString::null, title, NULL, QString::null, QString::null);
        if (tmp) after = tmp;
      }  
      
    }
    node = node.nextSibling();
  }

  file.close();
  
  return true;
}


void PlayList::slotImportAsx()
{
  QString path = KFileDialog::getOpenFileName(":kaffeinePL_ImportAsx",
                 i18n("*.asx *.ASX|ASX-Files\n*.*|All Files"), 0, i18n("Import Windows Media Metafile"));

  if (path.isEmpty()) return;

  slotClearList();

  if (!LoadAsxPlaylist(path, NULL))
  {
    KMessageBox::sorry(this, i18n("No ASX File: %1").arg(path));
    return;
  }

  if (random) CreateRandomList();  
  UpdateStatusBar();
}



/******************************************************
 *                  load m3u playlist
 ******************************************************/

bool PlayList::LoadM3UPlaylist(const QString& pl, QListViewItem* after)
{
  QFile file(pl);
  if (!file.open(IO_ReadOnly)) return false;
  QTextStream stream(&file);

//  if (stream.readLine().upper() != "#EXTM3U") return false;
  QString url;
  int time;
  bool ok;
  QString timeString;
  QString title;
  KURL kUrl;
  QListViewItem* tmp = NULL;
  bool foundAnyUrl = false;
  KURL plurl (pl);
  plurl.setFileName ("");

  while (!stream.eof())
  {
    url        = stream.readLine();
    time       = 0;
    title      = QString::null;
    timeString = QString::null;
    if (url.left(1) == "#")
    {
      if (url.left(7).upper() == "#EXTINF")
      {
          url  = url.remove(0,8);
          time = url.section(",", 0,0).toInt(&ok);
          if ((ok) && (time > 0))
	  {
	    timeString = msToTimeString(time*1000); //msec
	  }

          title = url.section(",",1,1);
          url   = stream.readLine();
      } else {
          continue;
      }
    }
    url.replace ('\\', '/');
    kUrl = KURL (plurl, url);
    if (kUrl.isValid())
    {
      if (kUrl.isLocalFile())
        url = kUrl.path();
       else
        url = kUrl.prettyURL(); 

      foundAnyUrl = true;
      tmp         = CreateItem(after, url, QString::null, title, NULL, timeString, QString::null);
      if (tmp)
      {
        after = tmp;
      }
    }
    else
      kdDebug(555) << "M3U: Not valid: " << kUrl.prettyURL() << endl;
  }
  file.close();
  if (foundAnyUrl)
    return true;
   else
    return false;
}


void PlayList::slotImportM3U()
{
  QString path = KFileDialog::getOpenFileName(":kaffeinePL_ImportM3U",
                 i18n("*.m3u *.M3U|M3U-Files\n*.*|All Files"), 0, i18n("Import m3u Playlist"));

  if (path.isEmpty()) return;               

  slotClearList();

  if (!LoadM3UPlaylist(path, NULL))
  {
    KMessageBox::sorry(this, i18n("No M3U File: %1").arg(path));
    return;
  }

  if (random) CreateRandomList();    
  UpdateStatusBar();
}



/***************************************************
 *               load pls playlist
 ***************************************************/    

bool PlayList::LoadPlsPlaylist(const QString& pl, QListViewItem* after)
{
  QFile file(pl);
  if (!file.open(IO_ReadOnly)) return false;

  QListViewItem* tmp = NULL;
  
  QTextStream stream(&file);

  if (stream.readLine().upper() != "[PLAYLIST]") return false;
  
  bool ok;
  QString numEntrys = stream.readLine().remove(0,16);
  int entrys = 0;
  entrys = numEntrys.toInt(&ok);
  if (!(ok) || !(entrys>0)) return false;

  QString* titles = new QString[entrys];
  QString* files = new QString[entrys];
  QString* length = new QString[entrys];

  for (int i=0; i<entrys; i++)
  {
    titles[i] = QString::null;
    files[i] = QString::null;
    length[i] = QString::null;
  }  
  
  QString line;
  int num;
  int time;
  

  while (!stream.eof())
  {
    line = stream.readLine();
   
    if (line.left(4).lower() == "file")
    {
      line = line.remove(0,4);

      num = line.left(line.find("=")).toInt(&ok);
      if ((!ok) && (num>entrys)) continue;

      files[num-1] = line.remove(0, line.find("=") + 1);

      continue;
    }

    if (line.left(5).lower() == "title")
    {
      line = line.remove(0,5);

      num = line.left(line.find("=")).toInt(&ok);
      if ((!ok) && (num>entrys)) continue;

      titles[num-1] = line.remove(0, line.find("=") + 1);

      continue;
    }

    if (line.left(6).lower() == "length")
    {
      line = line.remove(0,6);

      num = line.left(line.find("=")).toInt(&ok);
      if ((!ok) && (num>entrys)) continue;

      time = 0;
      time = line.remove(0, line.find("=") + 1).toInt(&ok);
      if ( !(ok) || !(time > 0) ) continue;

      length[num-1] = msToTimeString(time*1000);
    }  
    
  }

  file.close();

  for (int i=0; i<entrys; i++)
  {
    if (files[i].isNull()) continue;
    tmp = CreateItem(after, files[i], QString::null, titles[i], NULL, length[0]);
    if (tmp) after = tmp;
  }  

  delete [] titles;
  delete [] files;
  delete [] length;

  return true;
}  

void PlayList::slotImportPls()
{
  QString path = KFileDialog::getOpenFileName(":kaffeinePL_ImportPls",
                  i18n("*.pls *. PLS|PLS-Files\n*.*|All Files"), 0, i18n("Import ShoutCast Playlist"));

  if (path.isEmpty()) return;
                  
  slotClearList();
  
  if (!LoadPlsPlaylist(path, NULL))
    KMessageBox::sorry(this, i18n("No PLS File: %1").arg(path));
  
}


/******************************************
 *         save xml playlist
 ******************************************/
 
bool PlayList::SavePlaylist(const QString& pl)
{
  QDomDocument doc("playlist");
  doc.setContent(QString("<!DOCTYPE XMLPlaylist>"));
  QDomElement root = doc.createElement("playlist");
  root.setAttribute("client", "kaffeine");
  doc.appendChild(root);

  QDomElement entry;
  PlaylistItem * tmp = NULL;

  QListViewItemIterator it( list );
  while ( it.current() )
  {
    tmp = dynamic_cast<PlaylistItem *>(it.current());
    if (!tmp->parent())   //dont save sub-urls
    {
      entry = doc.createElement("entry");

      entry.setAttribute("title", tmp->title());
      entry.setAttribute("url", tmp->url());
      entry.setAttribute("mime", tmp->mime());
      entry.setAttribute("length", tmp->length());
      entry.setAttribute("info", tmp->info());
     
      if(!(tmp->subtitles().isEmpty()))
      {
        QString subList;
        for(unsigned int i=0; i<tmp->subtitles().count(); i++)
         subList += tmp->subtitles()[i] + "&";  
 
         entry.setAttribute("subs", subList); 
      }
      root.appendChild(entry);
    }

    ++it;
  }


  QFile file(pl);
  if (!file.open(IO_WriteOnly)) return false;
  QTextStream stream(&file);
  stream.setEncoding(QTextStream::UnicodeUTF8);

  stream << doc.toString();

  file.close();
  
  return true;
}


void PlayList::slotSavePlaylist()
{
  QString startPath = ":kaffeinePL_SavePlaylist";
  if (!lastPlaylist.isEmpty()) startPath = lastPlaylist;
  
  QString path = KFileDialog::getSaveFileName(startPath, i18n("*.kaffeine|Kaffeine Playlists\n*.*|All Files"), 0, i18n("Save Playlist"));

  if (path.isEmpty()) return;

  if ( path.right(9) != ".kaffeine" )
    path.append(".kaffeine");
  
  if (SavePlaylist(path))
  {
    lastPlaylist = path;
    list->SetCleared(false);
  }  
  else
    {
      kdDebug(555) << "Can't save Playlist\n";
      lastPlaylist = QString();
    }  
}  

/**********************/

void PlayList::slotGetMetaInfo( const QString& metaInfo)
{
  if (currentEntry){
    dynamic_cast<PlaylistItem *>(currentEntry)->setTitle(metaInfo);
  }
}
  
      
void PlayList::slotGetLengthInfo(const QString& time)
{
  if ( (currentEntry))
  { 
    PlaylistItem* tmp = dynamic_cast<PlaylistItem *>(currentEntry);
    if (!tmp->length().contains(':'))
    {
      if (time.contains(':'))
      {
        playTime += timeStringToMs(time);
        if (tmp->isVisible())
          playTimeVisible += timeStringToMs(time);
        UpdateStatusBar();
      }  
    }
    tmp->setLength(time);
  }
}


void PlayList::slotGetStreamInfo(const QString& streamInfo)
{
  if (!currentEntry) return;
  dynamic_cast<PlaylistItem *>(currentEntry)->setInfo(streamInfo);
  if (!currentEntry->pixmap(3))
  {    
    currentEntry->setPixmap(3, infoPixmap);
  }
}


void PlayList::slotFindText(const QString& text)
{
  if (text == i18n("Search")) return;

  QListViewItemIterator it( list );
  playTimeVisible = 0;
  countVisible = 0;
  PlaylistItem* tmp = NULL;
  while ( it.current() )
  {
    tmp = dynamic_cast<PlaylistItem *>(it.current());
    if ( text.isEmpty() || tmp->title().contains(text, false) || tmp->url().contains(text, false) )
    {
      tmp->setVisible( true );
      if (tmp->length().contains(':'))
        playTimeVisible += timeStringToMs(tmp->length());
      if (!tmp->parent())  
        countVisible++;
    }
    else
    {
      tmp->setVisible( false );
      if (tmp == currentEntry)
      {
        tmp->setPixmap(1, QPixmap());
        currentEntry = NULL;
        currentRandomListEntry = -1;
      }  
    }
    ++it;
  }

  if (text.isEmpty())
    searchSelection = false;
   else
    searchSelection = true; 

  if (random) CreateRandomList();
  UpdateStatusBar();
}  



void PlayList::SetLastPlaylist(const QString& pl)
{
  lastPlaylist = pl;
}

   
const QString& PlayList::GetLastPlaylist() const
{
  return lastPlaylist;
}

  
void PlayList::ReloadLastPlaylist()
{
  if (lastPlaylist.isEmpty()) return;

  LoadPlaylist(lastPlaylist, NULL);

  if (random) CreateRandomList();
  UpdateStatusBar();

  list->SetCleared(false);
}


void PlayList::slotTrySaveCurrentPlaylist()
{
  if ( (lastPlaylist.isEmpty()) || (list->GetCleared()) )
    slotSavePlaylist();
   else
    slotSaveCurrentPlaylist(); 
}
  

void PlayList::slotSaveCurrentPlaylist()
{
  if ( (lastPlaylist.isEmpty()) || (list->GetCleared()) ) return;

  kdDebug(555) << "Save current Playlist\n";
  SavePlaylist(lastPlaylist);
}



/***************** cut/copy/paste *************************/  


void PlayList::slotCut()
{
  slotCopy();
  slotRemoveSelected();
}


void PlayList::slotPaste()
{
  QPtrList<QListViewItem> selected;
  selected = list->selectedItems();
  QListViewItem* lastSelected = NULL;
  if (selected.count())
    lastSelected = selected.at( selected.count() - 1 );

  KURL::List kurlList;

  if (KURLDrag::decode( QApplication::clipboard()->data(), kurlList ))
  {
    Add( kurlList, lastSelected );
    return;
  }
 /** try to decode as text **/
  QString text;
  if (QTextDrag::decode( QApplication::clipboard()->data(), text ))
  {
    Add( text, lastSelected );
  }  
}


void PlayList::slotCopy()
{
  QPtrList<QListViewItem> selected;
  selected = list->selectedItems();

  KURL::List kurlList;
  
  for (uint i=0; i<selected.count(); i++)
  {
    kurlList.append( dynamic_cast<PlaylistItem *>(selected.at(i))->url() );
  }

  QApplication::clipboard()->setData( KURLDrag::newDrag( kurlList ) );  
}



/***********************************************************
 *                 export to html
 ***********************************************************/                  

void PlayList::slotExportHtml()
{
  bool bgcol = false;

  QString path = KFileDialog::getSaveFileName(":kaffeinePL_ExportHtml",
                                 i18n("*.html *.htm|Html-Files\n*.*|All Files"),
                                 0, i18n("Export Playlist to Html"));
  QFile file(path);
  if (!file.open(IO_WriteOnly)) return;

  if (!QFile::exists(QDir::homeDirPath() + "/.kaffeine/kmp-playlist.png") )
  {
    QStringList dirs = KGlobal::dirs()->findDirs("data","kaffeine");
    QString logoPath = dirs[0];  // global kde data path
    logoPath.append("kmp-playlist.png");
    QImage logo;
    logo.load(logoPath);
    logo.save(QDir::homeDirPath() + "/.kaffeine/kmp-playlist.png", "PNG");
  }

  /** get real name of user and date **/
  QString realName;
  QString date;

  QString uname = getenv("USER");
  if (!uname)
    uname = getenv("LOGNAME");
  if (uname)
  {
    struct passwd* pw = getpwnam(uname);
    realName = pw->pw_gecos;
  }
    
  date = KGlobal::locale()->formatDate(QDate::currentDate(), true);
  
  QTextStream stream(&file);
  stream.setEncoding(QTextStream::UnicodeUTF8);

  stream << "<!DOCTYPE public \"-//w3c//dtd html 4.01 transitional//en\"\n";
  stream << "\"http://www.w3.org/TR/html4/loose.dtd\">\n";
  stream << "<html>\n";
  stream << "<head>\n";
  stream << "<title>" << i18n("Kaffeine Playlist") << "</title>\n";
  stream << "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF8\">\n";
  stream << "<meta name=\"GENERATOR\" content=\"Kaffeine Media Player\">\n";
  stream << "</head>\n";
  stream << "<body bgcolor=\"white\" text=\"black\">\n";
  stream << "<p align=\"center\">\n";
  stream << "<img src=\"" << QDir::homeDirPath() + "/.kaffeine/kmp-playlist.png" << "\" width=\"467\" height=\"168\" hspace=\"10\" border=\"0\">\n";
  stream << "</p><p align=\"center\">" << realName << ", " << date << "</p>\n";
  stream << "<table border=\"0\" align=\"center\" cellpadding=\"2\">\n";
  stream << "<tbody>\n";

  PlaylistItem *tmp = NULL;

  QListViewItemIterator it( list );
  while ( it.current() )
  {
    tmp = dynamic_cast<PlaylistItem *>(it.current());
    
    if (!tmp->parent())   /* dont save sub-urls */
    {
      if (bgcol)
      {
        stream << "<tr bgcolor=\"#FFD18B\">\n";
        bgcol = false;
      }
      else
      {
        stream << "<tr>\n";
        bgcol = true;
      }    
      stream << "<td>" << i18n("Title") << ":<br>" << i18n("URL") << ":<br>" + i18n("Length") << ":<br>" << i18n("Stream")<<":";
      if(!tmp->subtitles().isEmpty())
         stream<<"<br>"<<i18n("Subtitles")<<":"; 
      stream << "</td>\n";

      stream << "<td>\n";
      stream << "<b>" << tmp->title() << "</b><br>\n"; //title
      stream << "<a href =\"" << tmp->url() << "\">" << tmp->text(4) << "</a><br>\n"; //url
      stream << tmp->length() << "<br>\n"; //length
      stream << tmp->info() << "<br>\n"; //info
      if(!tmp->subtitles().isEmpty())
      {
         stream<<"<br>";
         for(unsigned int i=0; i<tmp->subtitles().size(); i++)
            stream<<tmp->subtitles()[i]<<"&nbsp;&nbsp;&nbsp;&nbsp;";
      }
      stream << "</td>\n";
      
      stream << "</tr>\n";
    }

    ++it;
  }

  stream << "</tbody>\n";
  stream << "</table>\n";
  stream << "</body>\n";
  stream << "</html>\n";

  file.close();
}



/*********************************************************
 *                print playlist
 *********************************************************/     

void PlayList::slotPrint()
{
  QImage logo;

  QString destPath = QDir::homeDirPath() + "/.kaffeine/kmp-playlist.png";
  
  if (!QFile::exists(destPath))
  {
    QStringList dirs = KGlobal::dirs()->findDirs("data","kaffeine");
    QString logoPath = dirs[0];  /* global kde data path */
    logoPath.append("kmp-playlist.png");
    logo.load(logoPath);
    logo.save(destPath, "PNG");
  }
  else
  {
    logo.load(destPath);
  }    

  /**get real name of user and date **/
  QString realName;
  QString date;

  QString uname = QString::fromLocal8Bit(getenv("USER"));
  if (!uname)
    uname = getenv("LOGNAME");
  if (uname)
  {
    struct passwd* pw = getpwnam(uname);
    realName = QString::fromLocal8Bit(pw->pw_gecos);
  }

  date = KGlobal::locale()->formatDate(QDate::currentDate(), true);
  
  QFont printFont = KGlobalSettings::generalFont();
  QFont boldPrintFont = printFont;
  boldPrintFont.setBold(true);
  QFontMetrics printFontMetrics(printFont);
  
  KPrinter* printer = new KPrinter;

  if (printer->setup(this))
  {
    printer->setFullPage(false);
    printer->setCreator("Kaffeine");
    printer->setDocName("Playlist");

    QPainter* p = new QPainter;
    p->begin(printer);
    p->setFont(printFont);
    p->setTabStops(6 * printFontMetrics.width("M"));

    QPen pen;
    pen.setWidth(2);
    p->setPen(pen);
    
    QPaintDeviceMetrics met(printer);

    p->drawImage(QPoint( (met.width()-logo.width())/2, 0), logo);
           
    int y = logo.height();
    int dy;
    QString text;
    
    text = realName + ", " + date;
    dy = p->boundingRect((met.width()-printFontMetrics.width(text))/2, y,
                         met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                         text).height();
   
    p->drawText((met.width()-printFontMetrics.width(text))/2, y, met.width(),
                met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak, text);
    y = y + dy + 20;

    PlaylistItem* tmp = NULL;

    QListViewItemIterator it( list );
    while ( it.current() )
    {
      tmp = dynamic_cast<PlaylistItem *>(it.current());
      if (!tmp->parent())   //dont print sub urls
      {
        /***** title *******/
        text = tmp->title();
        dy = p->boundingRect(0, y, met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                            text).height();
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->setPen(QColor(0xFF8900));
        p->setFont(boldPrintFont);
        p->drawText(0, y, met.width(), met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak,
                    text);
        p->setFont(printFont);            
        p->setPen(QColor("black"));
        y += dy;

        /******* url ********/
        if(tmp->url().contains("#subtitle:"))
          text = i18n("URL") + ":\t" + tmp->url().section("#subtitle:",0,0);
        else 
          text = i18n("URL") + ":\t" + tmp->url();
  
        dy = p->boundingRect(0, y, met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                            text).height();
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->drawText(0, y, met.width(), met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak,
                    text);
        y += dy;
                        
        /****** length ******/
        text = i18n("Length") + ":\t" + tmp->length();
        dy = p->boundingRect(0, y, met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                             text).height();
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->drawText(0, y, met.width(), met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak,
                    text);
        y += dy;

        /******  stream info *******/
        text = i18n("Stream") + ":\t" + tmp->info();
        dy = p->boundingRect(0, y, met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                             text).height();
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->drawText(0, y, met.width(), met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak,
                    text);
        y += dy;

      /******* subtitles *********/
        QString subString = "";
        QString subinuse = QString::null;
        if(tmp->url().contains("#subtitle:"))
          subinuse = tmp->url().section("#subtitle:",-1,-1);
        for(unsigned int i = 0; i<tmp->subtitles().size(); i++)
        {
          subString += "\t"+tmp->subtitles()[i];
          if(subinuse == tmp->subtitles()[i])
          subString += "  " + i18n("(in use)");
          subString += "\n";
        }
        text = i18n("Subtitles") + ":" + subString;
        dy = p->boundingRect(0, y, met.width(), met.height(), QPainter::ExpandTabs | QPainter::WordBreak,
                             text).height();
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->drawText(0, y, met.width(), met.height()-y, QPainter::ExpandTabs | QPainter::WordBreak,
                    text);
        y += dy;

        /* separator line */
        dy = 10;
        if(y+dy > met.height())
        {
          printer->newPage();
          y = 0;
        }
        p->drawLine(0, y+4, met.width()-500, y+4);
        y += dy;
       
      }
      
     ++it;
    }

    p->end();
    delete p;
  }

  delete printer;
}


bool PlayList::EndsWith(QString s1, QString s2, bool caseSensitive )
{
    if ( s1.isNull() || s2.isNull() )
      return false;
    int startPos = s1.length() - s2.length();

    for (unsigned int i = 0; i < s2.length(); i++ ) 
    {
      if(caseSensitive)
      {
        if ( s1[startPos + i] != s2[i] )
        return false;
      }
      else
      {
        if ( s1[startPos + i].lower() != s2[i].lower() )
        return false;
      }
    }
  
    return true;
}


bool PlayList::StartsWith(QString s1, QString s2, bool caseSensitive )
{
    if ( s1.isNull() || s2.isNull() )
      return false;
    
    if(s2.length() > s1.length())
      return false;

    for (unsigned int i = 0; i < s2.length(); i++ ) 
    {
      if(caseSensitive)
      {
        if ( s1[i] != s2[i] )
        return false;
      }
      else
      {
        if ( s1[i].lower() != s2[i].lower() )
        return false;
      }
    }
  
    return true;
}


SubtitleChooser::SubtitleChooser(QStringList values, QString filename,QWidget *parent, const char * name) :
KDialogBase(parent,name,true,i18n("Select a subtitle"), Ok|Cancel, Ok, true)
{
   QVBox *page = makeVBoxMainWidget();
   QLabel *label =  new QLabel("<qt>" + i18n("Media file:") + " <b>" + filename + "</b></qt>", this);
   page->insertChild(label);
   table = new QListBox(page);
   this->setMinimumSize(300,200);
  
   table->setFocus();
   table->insertStringList(values);
   table->setSelected(0, true);
}


SubtitleChooser::~SubtitleChooser(){}


QString SubtitleChooser::getSelection()
{
  return table->selectedItem()->text();
}
