/***************************************************************************
                          kaffeine.cpp  - main widget
                             -------------------
    begin                : Don Apr 17 11:30:44 CEST 2003
    revision             : $Revision: 1.70 $
    last modified        : $Date: 2004/05/09 19:06:37 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by J. Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/



#include <kiconloader.h>
#include <kurlrequesterdlg.h>
#include <kfiledialog.h>
#include <kurldrag.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <kkeydialog.h>
#include <kwin.h>
#include <kdeversion.h>
#include <kdebug.h>

#include <qapplication.h>
#include <qinputdialog.h>
#include <qregexp.h>

#include "wizarddialog.h"
#include "screenshotpreview.h"
#include "deinterlacequality.h"
#include "kaffeine.h"
#include "kaffeine.moc"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>


KMP::KMP(QWidget *parent, const char *name, KCmdLineArgs* args) : QWidget(parent, name), DCOPObject("Kaffeine"),
                                          playList(NULL),cPanel(NULL), tRow(NULL), osDisplay(NULL),
                                          configWin(NULL), videoSettings(NULL), equalizer(NULL), systemTray(NULL), fsPanel(NULL), filterDialog(NULL),
                                          stMode(xineSubtitles), currentVisualPlugin(QString::null),
                                          cUrl(QString::null), cTitle(QString::null), bmenu(NULL)
                                      
{
  cmdLineArgs = args;

  kdDebug(555) << "Starting Kaffeine " << VERSION << "...\n";
  kdDebug(555) << "http://kaffeine.sourceforge.net\n";
 
  setCaption(i18n("Kaffeine Media Player %1").arg( VERSION ));
       
  QVBoxLayout* boxLayout = new QVBoxLayout(this);
  
  videoWin = new VideoWindow(this, "videowin", cmdLineArgs->getOption("audiodriver"), cmdLineArgs->getOption("videodriver"), cmdLineArgs->isSet("verbose"));
  
  cPanel = new ControlPanel(this, "controlpanel");
  tRow = new TextRow(this, "inforow");


/******************* first run? ************************/

  QDir homedir = QDir::home();
  if (!homedir.exists(".kaffeine"))
    {
      kdDebug(555) << "Creating /.kaffeine directory\n";
      homedir.mkdir(".kaffeine");
    }

  QString stamp =  QDir::homeDirPath() + "/.kaffeine/wizard_stamp_v0.4.2";
 
  if ( !QFile::exists( stamp ) )
  {
    WizardDialog* wizard = new WizardDialog( 0, "wizard" );
    wizard->exec();
   
    KProcess process;
    process << "touch" << stamp;
    process.start(KProcess::Block, KProcess::Stderr);
    process.clearArguments();
  }
  
/******/

  osDisplay = new SplashWidget(0, "osd");
  equalizer = new Equalizer(0, "equalizer");

  InitActions();
  InitMenu();
  InitContextMenu();
  InitFullscreenPanel();

  InitConnections();

  boxLayout->addWidget(menuBar);
  boxLayout->addWidget(videoWin);
  boxLayout->addWidget(tRow);
  boxLayout->addWidget(cPanel);

  setAcceptDrops(true);
}


KMP::~KMP()
{
  delete osDisplay;
  delete equalizer;
  delete videoSettings;
  delete playList;
  delete fsPanel;
  delete filterDialog;
}


/***********************************************
 *  FINALIZE THE GUI
 ***********************************************/

void KMP::polish()
{

  if (!(videoWin->xineRunning) || (playList)) return;
  
/*** autoplay plugins ***/

  KAction* autoPlayPlugin = NULL;
  QStringList autoPlayList;
  uint i = 0;
  QSignalMapper* mapper = new QSignalMapper(this);
  connect(mapper, SIGNAL(mapped(const QString&)), this, SLOT(slotPlayAutoplayPlugin(const QString&)));
  
  videoWin->GetAutoplayPlugins(autoPlayList);

  while(i<autoPlayList.count())
  {
   autoPlayPlugin = new KAction(autoPlayList[i], "cdrom_unmount", 0, mapper, SLOT(map()), actionCollection);
   autoPlayPlugin->plug(playMenu);
   mapper->setMapping(autoPlayPlugin, autoPlayList[i]);
   i++;
   autoPlayPlugin->setWhatsThis(autoPlayList[i]);
   i++;
  }

/*** visual plugins ***/

  QStringList visuals;
  visuals << "none";
  visualMenu->setCheckable(true);

  videoWin->GetVisualPlugins(visuals);

  for(i = 0; i < visuals.count(); i++)
   visualMenu->insertItem( visuals[i], i );
    
  connect(visualMenu, SIGNAL(activated(int)), this, SLOT(slotSetVisual(int)));
  
/** playlist  **/

  playList = new PlayList( videoWin->GetXineEngine(), this, 0, "playlist" );
  playList->SetAutoplayList(autoPlayList);

/** set import filter for directories **/

  fileFilter = videoWin->GetSupportedExtensions();
  fileFilter = fileFilter.remove("txt");
  fileFilter = "*." + fileFilter;
  fileFilter = fileFilter.replace( ' ', " *." );
  fileFilter = fileFilter + " " + fileFilter.upper();
 // kdDebug(555) << "File import filter: " << fileFilter << endl;
  playList->SetFileFilter( fileFilter );

/**** video settings ****/
  
  int hue, sat, contrast, bright, avOffset, spuOffset;

  videoWin->GetVideoSettings(hue, sat, contrast, bright, avOffset, spuOffset);
  
  videoSettings = new VideoSettings(hue, sat, contrast, bright, avOffset,
                                    spuOffset, 0, "videosettings");

                                    
/***** postprocessing *******/

  QStringList filterList;
  videoWin->GetPostprocessFilterNames( filterList );

  filterDialog = new FilterDialog( filterList, 0, "filterconfigdialog" );
                                    

/*** connections with playlist, videosettings and filterdialog ***/

  connect(filterDialog, SIGNAL(signalCreateFilter(const QString&, QWidget*)), videoWin, SLOT(slotCreateFilter(const QString&, QWidget*)));            
  connect(filterDialog, SIGNAL(signalRemoveAll()), videoWin, SLOT(slotRemoveAllFilters()));
  connect(filterDialog, SIGNAL(signalUseFilters(bool)), videoWin, SLOT(slotEnableFilters(bool)));

  connect(playList, SIGNAL(signalPlayDirect()), this, SLOT(slotPlay()));
  connect(videoWin, SIGNAL(signalNewMrlReference(const QString&)), playList, SLOT(slotAddMrl(const QString&)));
  connect(videoWin, SIGNAL(signalLengthInfo(const QString&)),
          playList, SLOT(slotGetLengthInfo(const QString&)));
  connect(videoWin, SIGNAL(signalStreamInfo(const QString&)), playList, SLOT(slotGetStreamInfo(const QString&)));
  connect(playList, SIGNAL(signalQuit()), this, SLOT(slotQuit()));
  connect(playList, SIGNAL(signalAutoplayPlugin(const QString&, QListViewItem*)), this, SLOT(slotAddAutoplayUrls(const QString&, QListViewItem*)));
  connect(playList, SIGNAL(signalPlayPause()), cPanel, SLOT(slotPlayPause()));
  connect(playList, SIGNAL(signalStop()), this, SLOT(slotStop()));
  connect(videoWin, SIGNAL(signalMetaInfo(const QString&)), playList, SLOT(slotGetMetaInfo(const QString&)));
          
  connect(videoSettings, SIGNAL(signalNewHue(int)), videoWin, SLOT(slotSetHue(int)));
  connect(videoSettings, SIGNAL(signalNewSaturation(int)), videoWin, SLOT(slotSetSaturation(int)));
  connect(videoSettings, SIGNAL(signalNewContrast(int)), videoWin, SLOT(slotSetContrast(int)));
  connect(videoSettings, SIGNAL(signalNewBrightness(int)), videoWin, SLOT(slotSetBrightness(int)));
  connect(videoSettings, SIGNAL(signalNewAVOffset(int)), videoWin, SLOT(slotSetAVOffset(int)));
  connect(videoSettings, SIGNAL(signalNewSpuOffset(int)), videoWin, SLOT(slotSetSpuOffset(int)));
  
  tRow->slotChangeText(i18n("Ready."));

/*** config stuff ***/

  config = kapp->config();
  ReadConfig();

/*** parse command line **/

  KURL::List files;

  if (cmdLineArgs->count() > 0)
  {
    for (int i = 0; i<cmdLineArgs->count(); i++)
    {
      files.append(cmdLineArgs->url(i));
    }
  }

 CmdLineArgs( files, cmdLineArgs->getOption("device"), cmdLineArgs->isSet("play"), cmdLineArgs->isSet("fullscreen"), cmdLineArgs->isSet("minimal"), cmdLineArgs->isSet("enqueue") );

}


/************************************************
 *          INITIALIZIATON
 ************************************************/

void KMP::InitConnections()
{

  connect(cPanel, SIGNAL(signalPlay()), this, SLOT(slotPlay()));
  connect(cPanel, SIGNAL(signalVolumeChanged(int)), videoWin, SLOT(slotSetVolume(int)));
  connect(videoWin, SIGNAL(signalNewInfo(const QString&)), tRow, SLOT(slotChangeText(const QString&)));
  connect(videoWin, SIGNAL(signalNewTitle(const QString&)), this, SLOT(slotGetNewTitle(const QString&)));
  connect(cPanel, SIGNAL(signalPlayNext()), this, SLOT(slotPlayNext()));
  connect(videoWin, SIGNAL(signalPlaybackFinished()), this, SLOT(slotEndOfPlayback()));
  connect(cPanel, SIGNAL(signalPlayPrevious()), this, SLOT(slotPlayPrev()));
  connect(cPanel, SIGNAL(signalStop()), this , SLOT(slotStop()));
  connect(videoWin, SIGNAL(signalHasChapters(bool)), this, SLOT(slotHasChapters(bool)));
  connect(cPanel, SIGNAL(signalPauseOn()), videoWin, SLOT(slotSpeedPause()));
  connect(cPanel, SIGNAL(signalPauseOff()), videoWin, SLOT(slotSpeedNormal()));
  //connect(videoWin, SIGNAL(signalNewChannels(const QStringList&, const QStringList&, int, int)),
    //      cPanel, SLOT(slotNewChannels(const QStringList&, const QStringList&, int ,int)));
  connect(videoWin, SIGNAL(signalNewChannels(const QStringList&, const QStringList&, int, int)),
          this, SLOT(slotNewChannels(const QStringList&, const QStringList&, int ,int)));
  connect(cPanel, SIGNAL(signalSetAudioChannel(int)), videoWin, SLOT(slotSetAudioChannel(int)));
  connect(cPanel, SIGNAL(signalSetSubtitleChannel(int)), this, SLOT(slotSetSubtitleChannel(int)));
  connect(videoWin, SIGNAL(signalNewPosition(int, const QString&)),
          cPanel, SLOT(slotNewPosition(int, const QString&)));
  connect(cPanel, SIGNAL(signalChangePosition(int)), videoWin, SLOT(slotChangePosition(int)));
  connect(cPanel, SIGNAL(signalTogglePlayMode()), videoWin, SLOT(slotTogglePlayMode()));
  connect(cPanel, SIGNAL(signalPlayList()), this, SLOT(slotShowPlayList()));
  connect(cPanel, SIGNAL(signalSaveStreamAs()), this, SLOT(slotSaveStreamAs()));

  connect(videoWin, SIGNAL(signalShowContextMenu(const QPoint&)), this, SLOT(slotShowContextMenu(const QPoint&)));
  connect(videoWin, SIGNAL(signalShowFullscreenPanel(const QPoint&)), this, SLOT(slotShowFullscreenPanel(const QPoint&)));
  connect(videoWin, SIGNAL(signalShowOSD(const QString&)), osDisplay, SLOT(slotOsdShow(const QString&)));
  connect(videoWin, SIGNAL(signalQuit()), this, SLOT(slotQuit()));
  connect(videoWin, SIGNAL(signalNewXineMessage()), this, SLOT(slotNewXineMessage()));
  connect(videoWin, SIGNAL(signalToggleFullscreen()), this, SLOT(slotToggleFullscreenChecked()));
/** equalizer **/

  connect(equalizer, SIGNAL(signalNewEq30(int)), videoWin, SLOT(slotSetEq30(int)));
  connect(equalizer, SIGNAL(signalNewEq60(int)), videoWin, SLOT(slotSetEq60(int)));
  connect(equalizer, SIGNAL(signalNewEq125(int)), videoWin, SLOT(slotSetEq125(int)));
  connect(equalizer, SIGNAL(signalNewEq250(int)), videoWin, SLOT(slotSetEq250(int)));
  connect(equalizer, SIGNAL(signalNewEq500(int)), videoWin, SLOT(slotSetEq500(int)));
  connect(equalizer, SIGNAL(signalNewEq1k(int)), videoWin, SLOT(slotSetEq1k(int)));
  connect(equalizer, SIGNAL(signalNewEq2k(int)), videoWin, SLOT(slotSetEq2k(int)));
  connect(equalizer, SIGNAL(signalNewEq4k(int)), videoWin, SLOT(slotSetEq4k(int)));
  connect(equalizer, SIGNAL(signalNewEq8k(int)), videoWin, SLOT(slotSetEq8k(int)));
  connect(equalizer, SIGNAL(signalNewEq16k(int)), videoWin, SLOT(slotSetEq16k(int)));

 /** fullscreen panel **/
  connect(videoWin, SIGNAL(signalNewPosition(int, const QString&)), fsPanel, SLOT(slotSetPosition(int, const QString&)));
  connect(videoWin, SIGNAL(signalLengthInfo(const QString&)), fsPanel, SLOT(slotSetLength(const QString&)));
  connect(videoWin, SIGNAL(signalStreamInfo(const QString&)), fsPanel, SLOT(slotSetStreamInfo(const QString&)));

  connect(fsPanel, SIGNAL(signalPlayPause()), cPanel, SLOT(slotPlayPause()));
  connect(fsPanel, SIGNAL(signalNext()), this, SLOT(slotPlayNext()));
  connect(fsPanel, SIGNAL(signalPrevious()), this, SLOT(slotPlayPrev()));
  connect(fsPanel, SIGNAL(signalStop()), this, SLOT(slotStop()));
  connect(fsPanel, SIGNAL(signalRewind()), cPanel, SLOT(slotPosMinus()));
  connect(fsPanel, SIGNAL(signalForward()), cPanel, SLOT(slotPosPlus()));
  connect(fsPanel, SIGNAL(signalVolUp()), cPanel->volSlider, SLOT(subtractStep()));
  connect(fsPanel, SIGNAL(signalVolDown()), cPanel->volSlider, SLOT(addStep()));
  connect(fsPanel, SIGNAL(signalDVDMenu()), videoWin, SLOT(slotMenu1()));
}


void KMP::InitActions()
{

  actionCollection = new KActionCollection(this);

  fileOpen = KStdAction::open(this, SLOT(slotOpenFile()), actionCollection);
  fileOpenUrl= new KAction(i18n("Open &URL"), "package_network", Qt::CTRL|Qt::Key_U, this, SLOT(slotOpenUrl()), actionCollection, "openurl");
  fileOpenRecent = KStdAction::openRecent(this, SLOT(slotPlayRecent(const KURL&)), NULL);
  fileOpenRecent->setMaxItems( 15 );
  fileSaveScreenshot = new KAction(i18n("&Save Screenshot"), "filesave", Qt::CTRL|Qt::Key_S, this, SLOT(slotScreenshot()), actionCollection, "screenshot");
  fileQuit= KStdAction::quit(this, SLOT(slotQuit()), actionCollection);

  playEndlessPlay = new KToggleAction(i18n("&Endless Play"), 0 , Qt::CTRL|Qt::Key_E, this, SLOT(slotEndlessMode()), actionCollection,"endlessplay");
  playRandomPlay = new KToggleAction(i18n("&Random Play"), 0 , Qt::CTRL|Qt::Key_R, this, SLOT(slotRandomPlay()), actionCollection,"randomplay");

  broadcastMaster = new KToggleAction(i18n("&Define As Master"), 0, 0, this, SLOT(slotToggleBroadcastMaster()), actionCollection, "broadcastmaster");
  broadcastSlave  = new KAction(i18n("&Play As Slave"), "network", 0, this, SLOT(slotBroadcastSlave()), actionCollection, "broadcastslave");

  viewPlaylist = new KAction(i18n("Playlist"), "player_eject", Qt::Key_P, this, SLOT(slotShowPlayList()), actionCollection, "showhideplaylist");
  viewFullscreen= new KToggleAction(i18n("&Fullscreen"), NULL, Qt::Key_F, this, SLOT(slotToggleFullscreen()), actionCollection, "fullscreen");
  viewMinimalMode = new KToggleAction(i18n("&Minimal Mode"), NULL, Qt::Key_M, this, SLOT(slotToggleMinimalMode()), actionCollection, "minimalmode");

  viewAutoresize = new KToggleAction(i18n("Enable Auto &Resize"), 0, Qt::Key_A, this, SLOT(slotEnableAutoresize()), actionCollection, "enableautoresize");
  viewAutoresize->setChecked( false );
  
  viewDeinterlace = new KToggleAction(i18n("&Deinterlace"), 0, Qt::Key_I, videoWin, SLOT(slotToggleDeinterlace()), actionCollection, "deinterlace");
  viewDeinterlace->setWhatsThis(i18n("Activate this for interlaced streams, some DVD's for example."));

  viewZoomIn = KStdAction::zoomIn(videoWin, SLOT(slotZoomIn()), actionCollection);
  viewZoomOut = KStdAction::zoomOut(videoWin, SLOT(slotZoomOut()), actionCollection);
  viewZoomOff = KStdAction::fitToPage(videoWin, SLOT(slotZoomOff()), actionCollection);
 
  aspectRatioAuto =new KAction(i18n("&auto"), "viewmagfit", Qt::Key_F5, videoWin, SLOT(slotAspectRatioAuto()), actionCollection, "aspectratioauto");
  aspectRatio4_3 = new KAction(i18n("&4:3"), "viewmagfit", Qt::Key_F6, videoWin, SLOT(slotAspectRatio4_3()), actionCollection, "aspectratio43");
  aspectRatio16_9 =new KAction(i18n("&16:9"), "viewmagfit", Qt::Key_F7, videoWin, SLOT(slotAspectRatio16_9()), actionCollection, "aspectratio169");
  aspectRatioSquare = new KAction(i18n("&1:1"), "viewmagfit", Qt::Key_F8, videoWin, SLOT(slotAspectRatioSquare()), actionCollection, "aspectratio12");

  configVideoControl= new KAction(i18n("&Video Settings"), "configure", Qt::Key_V, this, SLOT(slotVideoSettings()), actionCollection, "videocontrol");
  configEqualizer = new KAction(i18n("&Equalizer"), "configure", Qt::Key_E, this, SLOT(slotEqualizer()), actionCollection, "equalizer");
  configDeinterlace = new KAction(i18n("&Deinterlace Quality"), "blend", Qt::CTRL|Qt::Key_I, this, SLOT(slotShowDeinterlaceDialog()), actionCollection, "deinterlacerconfig");
  configPostFilter = new KAction(i18n("Video &Postprocessing Filters"), "filter", Qt::Key_X, this, SLOT(slotShowFilterConfig()), actionCollection, "filterconfig");
  configXine = new KAction(i18n("&xine Engine Parameters"), "edit", 0, this, SLOT(slotConfigXine()), actionCollection, "configxine");
  configKMP = KStdAction::preferences(this, SLOT(slotShowSetup()), actionCollection);
  configShortcuts = KStdAction::keyBindings(this, SLOT(slotConfigureShortcuts()), actionCollection);

/** actions for dvd menu **/

  dvdMenuToggle = new KAction(i18n("&Menu Toggle"), "view_detailed", Qt::Key_D, videoWin, SLOT(slotMenu1()), actionCollection, "dvdmenutoggle");
  dvdMenuTitle = new KAction(i18n("&Title"), NULL, 0, videoWin, SLOT(slotMenu2()), actionCollection, "dvdmenutitle");
  dvdMenuRoot = new KAction(i18n("&Root"), NULL, 0, videoWin, SLOT(slotMenu3()), actionCollection, "dvdmenuroot");
  dvdMenuSub = new KAction(i18n("&Subpicture"), NULL, 0, videoWin, SLOT(slotMenu4()), actionCollection, "dvdmenusubpicture");
  dvdMenuAudio = new KAction(i18n("&Audio"), NULL, 0, videoWin, SLOT(slotMenu5()), actionCollection, "dvdmenuaudio");
  dvdMenuAngle = new KAction(i18n("An&gle"), NULL, 0, videoWin, SLOT(slotMenu6()), actionCollection, "dvdmenuangle");
  dvdMenuPart = new KAction(i18n("&Part"), NULL, 0, videoWin, SLOT(slotMenu7()), actionCollection, "dvdmenupart");
  
/** actions for context menu **/

  controlPlayPause = new KAction(i18n("Play/Pause"), "player_play", Qt::Key_Space, cPanel, SLOT(slotPlayPause()), actionCollection, "controlplaypause");
  controlNext      = new KAction(i18n("Next Entry/Chapter"), "player_end", Qt::Key_Down, this, SLOT(slotPlayNext()), actionCollection, "controlnext");
  controlPrevious  = new KAction(i18n("Previous Entry/Chapter"), "player_start", Qt::Key_Up, this, SLOT(slotPlayPrev()), actionCollection, "controlprevious");
  controlStop      = new KAction(i18n("Stop"), "player_stop", Qt::Key_Backspace, this, SLOT(slotStop()), actionCollection, "controlstop");
  controlSaveStreamAs = new KAction(i18n("Save Stream As..."),"player_record", Qt::Key_R, this, SLOT(slotSaveStreamAs()), actionCollection, "savestreamas");

/** unplugged actions **/

  new KAction(i18n("Leave Fullscreen Mode"), NULL, Qt::Key_Escape, this, SLOT(slotEscape()), actionCollection, "leavefullscreen");
  new KAction(i18n("Increase Volume"), NULL, Qt::Key_Plus, cPanel->volSlider, SLOT(subtractStep()), actionCollection, "incvol");
  new KAction(i18n("Decrease Volume"), NULL, Qt::Key_Minus, cPanel->volSlider, SLOT(addStep()), actionCollection, "decvol");
  new KAction(i18n("Increase Stream Position"), NULL, Qt::Key_Right, cPanel, SLOT(slotPosPlus()), actionCollection, "incpos");
  new KAction(i18n("Decrease Stream Position"), NULL, Qt::Key_Left, cPanel, SLOT(slotPosMinus()), actionCollection, "decpos");
  
  new KAction(i18n("DVD Menu Left"), NULL, Qt::CTRL|Qt::Key_Left, videoWin, SLOT(slotDVDMenuLeft()), actionCollection, "dvdmenuleft");
  new KAction(i18n("DVD Menu Right"), NULL, Qt::CTRL|Qt::Key_Right, videoWin, SLOT(slotDVDMenuRight()), actionCollection, "dvdmenuright");
  new KAction(i18n("DVD Menu Up"), NULL, Qt::CTRL|Qt::Key_Up, videoWin, SLOT(slotDVDMenuUp()), actionCollection, "dvdmenuup");
  new KAction(i18n("DVD Menu Down"), NULL, Qt::CTRL|Qt::Key_Down, videoWin, SLOT(slotDVDMenuDown()), actionCollection, "dvdmenudown");
  new KAction(i18n("DVD Menu Select"), NULL, Qt::Key_Return, videoWin, SLOT(slotDVDMenuSelect()), actionCollection, "dvdmenuselect");
}


void KMP::InitMenu()
{

  KAboutApplication* kabout = new KAboutApplication(this);
  KHelpMenu* khelp = new KHelpMenu(this);
  connect(khelp, SIGNAL(showAboutApplication()), kabout, SLOT(show()));

  menuBar = new KMenuBar(this);
  menuBar->setSizePolicy ( QSizePolicy (QSizePolicy::MinimumExpanding, QSizePolicy::Fixed) );

 /** init bookmark menu **/
  bookmarkMenu = new KPopupMenu(this);
  bookmarkManager = KBookmarkManager::managerForFile(QDir::homeDirPath() + "/.kaffeine/bookmarks.xml");
  bmenu = new KBookmarkMenu(bookmarkManager, this, bookmarkMenu, NULL, true);

  fileMenu = new KPopupMenu(this);
  playMenu = new KPopupMenu(this);
  dvdMenu = new KPopupMenu(this);
  visualMenu = new KPopupMenu(this);
  broadcastMenu = new KPopupMenu(this);
  viewMenu = new KPopupMenu(this);
  aspectMenu = new KPopupMenu(this);
  configMenu = new KPopupMenu(this);
  helpMenu = khelp->menu();

  fileOpen->plug(fileMenu);
  fileOpenUrl->plug(fileMenu);
  fileOpenRecent->plug(fileMenu);
  fileMenu->insertSeparator();
  fileSaveScreenshot->plug(fileMenu);
  fileMenu->insertSeparator();
  fileQuit->plug(fileMenu);

  broadcastMaster->plug(broadcastMenu);
  broadcastSlave->plug(broadcastMenu);

  dvdMenuToggle->plug(dvdMenu);
  dvdMenu->insertSeparator();
  dvdMenuTitle->plug(dvdMenu);
  dvdMenuRoot->plug(dvdMenu);
  dvdMenuSub->plug(dvdMenu);
  dvdMenuAudio->plug(dvdMenu);
  dvdMenuAngle->plug(dvdMenu);
  dvdMenuPart->plug(dvdMenu);

  playEndlessPlay->plug(playMenu);
  playRandomPlay->plug(playMenu);
  playMenu->insertSeparator();
  playMenu->insertItem(i18n("&Network Broadcasting"), broadcastMenu);
  playMenu->insertSeparator();
  playMenu->insertItem(i18n("&DVD Menus"), dvdMenu);
  playMenu->insertSeparator();

  aspectRatioAuto->plug(aspectMenu);
  aspectRatio4_3->plug(aspectMenu);
  aspectRatio16_9->plug(aspectMenu);
  aspectRatioSquare->plug(aspectMenu);

  viewPlaylist->plug(viewMenu);
  viewFullscreen->plug(viewMenu);
  viewMinimalMode->plug(viewMenu);
  viewMenu->insertSeparator();
  viewMenu->insertItem(i18n("Audio &Visualization"), visualMenu);
  viewDeinterlace->plug(viewMenu);
  viewAutoresize->plug(viewMenu);
  viewMenu->insertSeparator();
  viewMenu->insertItem(i18n("&Aspect Ratio"), aspectMenu);
  viewMenu->insertSeparator();
  viewZoomOff->plug(viewMenu);
  viewZoomIn->plug(viewMenu);
  viewZoomOut->plug(viewMenu);

  configVideoControl->plug(configMenu);
  configEqualizer->plug(configMenu);
  configDeinterlace->plug(configMenu);
  configPostFilter->plug(configMenu);
  configMenu->insertSeparator();
  configXine->plug(configMenu);
  configMenu->insertSeparator();
  configShortcuts->plug(configMenu);
  configKMP->plug(configMenu);

  menuBar->insertItem(i18n("&File"), fileMenu);
  menuBar->insertItem(i18n("&Play"), playMenu);
  menuBar->insertItem(i18n("&View"), viewMenu);
  menuBar->insertItem(i18n("&Bookmarks"), bookmarkMenu);
  menuBar->insertItem(i18n("&Settings"), configMenu);
  menuBar->insertItem(i18n("&Help"), helpMenu);
}


/******** system tray ***************/

void KMP::InitSystemTray()
{
  systemTray = new SystemTray(this, "systemtray");

  connect(systemTray, SIGNAL(signalPlayPause()), cPanel, SLOT(slotPlayPause()));
  connect(systemTray, SIGNAL(signalNext()), this, SLOT(slotPlayNext()));
  connect(systemTray, SIGNAL(signalPrev()), this, SLOT(slotPlayPrev()));
  connect(systemTray, SIGNAL(signalStop()), this, SLOT(slotStop()));
  connect(systemTray, SIGNAL(signalPlayFiles(const KURL::List&)), this, SLOT(slotPlayFiles(const KURL::List&)));
  connect(cPanel, SIGNAL(signalPauseOn()), systemTray, SLOT(slotPauseIcon()));
  connect(cPanel, SIGNAL(signalPauseOff()), systemTray, SLOT(slotPlayIcon()));
  connect(systemTray, SIGNAL(signalShowPlaylist()), this, SLOT(slotShowPlayList()));
  connect(systemTray, SIGNAL(signalShowEqualizer()), this, SLOT(slotEqualizer()));
  connect(systemTray, SIGNAL(signalShowVideosettings()), this, SLOT(slotVideoSettings()));
  connect(systemTray, SIGNAL(signalSaveStreamAs()), this, SLOT(slotSaveStreamAs()));
  connect(systemTray, SIGNAL(quitSelected()), this, SLOT(slotQuit()));
}


/**** context menu ********/

void KMP::InitContextMenu()
{
  contextMenu = new KPopupMenu(this);

  controlMenu = new KPopupMenu(this);
  audioChannelMenu = new KPopupMenu(this);
  audioChannelMenu->setCheckable(true);
  subtitleChannelMenu = new KPopupMenu(this);
  subtitleChannelMenu->setCheckable(true);

  controlSaveStreamAs->plug(controlMenu);
  controlPlayPause->plug(controlMenu);
  controlStop->plug(controlMenu);
  controlNext->plug(controlMenu);
  controlPrevious->plug(controlMenu);

  controlMenu->insertSeparator();
  controlMenu->insertItem(i18n("Audiochannel"), audioChannelMenu);
  controlMenu->insertItem(i18n("Subtitles"), subtitleChannelMenu);

  contextMenu->insertItem(i18n("&Control"), controlMenu);
  contextMenu->insertSeparator();
  contextMenu->insertItem(i18n("&File"), fileMenu);
  contextMenu->insertItem(i18n("&Play"), playMenu);
  contextMenu->insertItem(i18n("&View"), viewMenu);
  contextMenu->insertItem(i18n("&Bookmarks"), bookmarkMenu);
  contextMenu->insertItem(i18n("&Config"), configMenu);
  contextMenu->insertItem(i18n("&Help"), helpMenu);

  connect(audioChannelMenu, SIGNAL(activated(int)), this, SLOT(slotContextMenuAudioChannel(int)));
  connect(subtitleChannelMenu, SIGNAL(activated(int)), this, SLOT(slotContextMenuSubtitleChannel(int)));
}


/******* fullscreen panel **************/

void KMP::InitFullscreenPanel()
{
  kdDebug(555) << "Init fullscreen panel\n";

  fsPanel = new FullscreenPanel(0, "fullscreenpanel");
  fsPanel->SetContextMenus( audioChannelMenu, subtitleChannelMenu );
  fsPanel->SetWidgetToActivate( this );
}  


/*********************************************
 *         save/load configuration           *
 *********************************************/
 
void KMP::SaveConfig()
{
  QFont f;
  QColor c;
  QString mstr;
 
// kdDebug(555) << "Save config\n";

  config->setGroup("General Options");
  if (!viewFullscreen->isChecked())
    config->writeEntry("KMP Geometry", size());
  config->writeEntry("Minimal Mode", viewMinimalMode->isChecked());
  config->writeEntry("PlayList Geometry", playList->size());
  config->writeEntry("PlayList visible", playList->isVisible());
  config->writeEntry("VideoSettings Geometry", videoSettings->size());
  config->writeEntry("Filter Dialog Geometry", filterDialog->size());
  config->writeEntry("Endless Mode", playEndlessPlay->isChecked());
  config->writeEntry("Random Mode", playRandomPlay->isChecked());
  config->writeEntry("Load Playlist", loadPlaylistAtStartup);
  config->writeEntry("Current Playlist:", playList->GetLastPlaylist());
  config->writeEntry("Embed in System Tray", embedSystemTray);
  config->writeEntry("Read meta on loading", playList->GetReadMetaOnLoading());
  config->writeEntry("Meta String", playList->GetMetaInfoString());
  config->writeEntry("Screensaver Timeout", videoWin->GetScreensaverTimeout());
  tRow->GetConfig(f, c);
  config->writeEntry("Alternate Background Color", playList->GetAlternateColor());
  config->writeEntry("Textrow Font", f);
  config->writeEntry("Textrow Color", c);
  config->writeEntry("Volume", cPanel->volSlider->value());
  config->writeEntry("Autoresize", viewAutoresize->isChecked());
  config->setGroup("OSD Options");
  bool u, a;
  QFont font;
  QColor color;
  int x, y, s ,t;
  osDisplay->GetConfig(u, a, s, font, color, t, x, y);
  config->writeEntry("Use OSD", u);
  config->writeEntry("Animate", a);
  config->writeEntry("Animation Speed",s );
  config->writeEntry("OSD Font", font);
  config->writeEntry("OSD Color", color);
  config->writeEntry("Display Time", t);
  config->writeEntry("X-Pos", x);
  config->writeEntry("y-Pos", y);

  config->setGroup("Broadcasting Options");
  config->writeEntry("Port", broadcastPort);
  config->writeEntry("Master Address", broadcastMasterAddress);

  config->setGroup("Visualization");
  config->writeEntry("Visual Plugin", currentVisualPlugin);

  config->setGroup("Deinterlace");
  config->writeEntry("Quality Level", currentDeinterlaceQuality);
  config->writeEntry("Config String", videoWin->GetDeinterlaceConfig());
  config->writeEntry("Enabled", viewDeinterlace->isChecked());
  
  playList->SaveToolbarSettings(config);
  equalizer->SaveValues(config);
  fileOpenRecent->saveEntries(config, "Recent Files");

  /* write the shortcuts settings */
  actionCollection->writeShortcutSettings("Main Window Shortcuts", kapp->config());
  playList->GetActionCollection()->writeShortcutSettings("Main Window Shortcuts", kapp->config());
}


void KMP::ReadConfig()
{
  QSize size;
  QFont f;
  QColor c;
  bool b;

//  kdDebug(555) << "Read config\n";

 /* general options */
  config->setGroup("General Options");
  size = QSize(450, 420);
  size = config->readSizeEntry("KMP Geometry", &size);
  resize(size);

  b = config->readBoolEntry("Minimal Mode", false);
  if (b)
  {
    cPanel->hide();
    tRow->hide();
    menuBar->hide();
    viewMinimalMode->setChecked(true);
  }  

  size = QSize(420, 420);
  size = config->readSizeEntry("PlayList Geometry", &size);
  playList->resize(size);
  b = config->readBoolEntry("PlayList visible", true);
  if (b) playList->show();

  size = QSize(350, 100);
  size = config->readSizeEntry("VideoSettings Geometry", &size);
  videoSettings->resize(size);

  size = QSize(500, 500);
  size = config->readSizeEntry("Filter Dialog Geometry", &size);
  filterDialog->resize(size);
  
  b = config->readBoolEntry("Endless Mode", false);
  playEndlessPlay->setChecked(b);
  playList->SetEndless(b);
     
  b = config->readBoolEntry("Random Mode", false);
  playRandomPlay->setChecked(b);
  playList->SetRandomPlay(b);

  loadPlaylistAtStartup = config->readBoolEntry("Load Playlist", true);
  QString pl = config->readEntry("Current Playlist:", QString());
  playList->SetLastPlaylist(pl);

  embedSystemTray = config->readBoolEntry("Embed in System Tray", true);
  if (embedSystemTray)
    InitSystemTray();

  bool metaOnLoading = config->readBoolEntry("Read meta on loading", true);
  playList->slotReadMetaOnLoading( metaOnLoading );
  
  QString metaStr = config->readEntry("Meta String", QString("artist - title (album)"));
  if (metaStr.contains("%2"))   /* old style? */
    metaStr = "artist - title (album)";
  playList->slotSetMetaInfoString( metaStr );
  videoWin->slotSetMetaString( metaStr );

  int ssTimeout = config->readNumEntry("Screensaver Timeout", 2);
  videoWin->slotSetScreensaverTimeout( ssTimeout );

  c = QColor(0xFFF5F5);
  QColor alternateColor = config->readColorEntry("Alternate Background Color", &c);
  playList->slotSetAlternateColor( alternateColor );

  f = KGlobalSettings::generalFont();
  QFont infoFont = config->readFontEntry("Textrow Font", &f);
  c = KGlobalSettings::baseColor();
  QColor infoColor = config->readColorEntry("Textrow Color", &c);
  tRow->slotSetConfig(&infoFont, &infoColor);
  int vol = config->readNumEntry("Volume", -70);
  cPanel->setVolume(-vol);
  b = config->readBoolEntry("Autoresize", false);
  viewAutoresize->setChecked(b);
  videoWin->slotEnableAutoresize(b);
  
 /* osd options */
  config->setGroup("OSD Options");
  bool osdUse, osdAnim;
  QFont osdFont;
  QColor osdColor;
  int osdX, osdY, osdAnimSpeed, osdTime;
  
  osdUse = config->readBoolEntry("Use OSD", true);
  osdAnim = config->readBoolEntry("Animate", true);
  osdAnimSpeed = config->readNumEntry("Animation Speed", 15 );
  f.setPointSize(18);
  osdFont = config->readFontEntry("OSD Font", &f);
  c = QColor(0xA00000);
  osdColor = config->readColorEntry("OSD Color", &c);
  osdTime = config->readNumEntry("Display Time", 2);
  osdX = config->readNumEntry("X-Pos", 5);
  osdY = config->readNumEntry("y-Pos", 5);

 /* broadcasting options */ 
  config->setGroup("Broadcasting Options");
  broadcastPort = config->readNumEntry("Port", 8080);
  broadcastMasterAddress = config->readEntry("Master Address", "localhost");

 /* visualization options */
  config->setGroup("Visualization");
  currentVisualPlugin = config->readEntry("Visual Plugin", "goom");
  for (uint i = 0; i < visualMenu->count(); i++)
   if ( visualMenu->text( i ) == currentVisualPlugin )
   {
     slotSetVisual( i );
     break;
   }

 /* deinterlace options */
  config->setGroup("Deinterlace");
  currentDeinterlaceQuality = config->readNumEntry("Quality Level", 4);
  QString configString = config->readEntry("Config String", DEFAULT_TVTIME_CONFIG);
  videoWin->CreateDeinterlacePlugin( configString );
  bool deinterlaceEnabled =  config->readBoolEntry("Enabled", true);
  if (deinterlaceEnabled)
  {
    viewDeinterlace->setChecked( deinterlaceEnabled );
    videoWin->slotToggleDeinterlace();
  }  
      
  playList->LoadToolbarSettings(config);
  equalizer->ReadValues(config);
  fileOpenRecent->loadEntries(config, "Recent Files");

 /* read the shortcuts settings */
  actionCollection->readShortcutSettings("Main Window Shortcuts", kapp->config());
  playList->GetActionCollection()->readShortcutSettings("Main Window Shortcuts", kapp->config());

  osDisplay->slotSetConfig(osdUse, osdAnim, osdAnimSpeed,  &osdFont, &osdColor,
                           osdTime, osdX, osdY);

 /* create settings dialog */
  configWin = new ConfigWidget(loadPlaylistAtStartup, embedSystemTray, metaOnLoading, ssTimeout, &alternateColor, &metaStr,
                  &infoFont, &infoColor,osdUse, osdAnim, osdAnimSpeed,
                  &osdFont, &osdColor, osdTime, osdX, osdY);


  connect(configWin, SIGNAL(signalNewGenConfig(bool,bool)), this, SLOT(slotGenConfig(bool,bool)));
  connect(configWin, SIGNAL(signalNewInfoRowConfig(QFont*, QColor*)), tRow, SLOT(slotSetConfig(QFont*, QColor*)));
  connect(configWin, SIGNAL(signalNewScreensaverTimeout(int)), videoWin, SLOT(slotSetScreensaverTimeout(int)));
  connect(configWin, SIGNAL(signalNewOsdConfig(bool, bool, int, QFont*, QColor*, int, int, int)),
         osDisplay, SLOT(slotSetConfig(bool, bool, int, QFont*, QColor*, int, int, int)));
  connect(configWin, SIGNAL(signalMSForPlaylist(const QString&)), playList, SLOT(slotSetMetaInfoString(const QString&)));
  connect(configWin, SIGNAL(signalMSForPlaylist(const QString&)), videoWin, SLOT(slotSetMetaString(const QString&)));
  connect(configWin, SIGNAL(signalClearRecent()), this, SLOT(slotClearRecent()));
  connect(configWin, SIGNAL(signalRefreshTitles()), playList, SLOT(slotRefreshMetaInfo()));
  connect(configWin, SIGNAL(signalReadMetaOnLoading(bool)), playList, SLOT(slotReadMetaOnLoading(bool)));
  connect(configWin, SIGNAL(signalNewAlternateColor(const QColor&)), playList, SLOT(slotSetAlternateColor(const QColor&)));
}

/******************/



void KMP::slotSetVisual(int visId)
{
  currentVisualPlugin = visualMenu->text( visId );
 // kdDebug(555) << "Use visual plugin: " << currentVisualPlugin << endl;

  for (uint i = 0; i < visualMenu->count(); i++)
   if (visualMenu->isItemChecked(i))
     visualMenu->setItemChecked(i , false);
     
  visualMenu->setItemChecked( visId, true );

  videoWin->SetVisualPlugin( currentVisualPlugin );
}  
  

void KMP::slotShowFullscreenPanel(const QPoint& pos)
{
  /* show context menu only in minmal or fullscreen mode */
  if ( (!viewFullscreen->isChecked()) && (!viewMinimalMode->isChecked()) ) return;

  if (fsPanel->isShown())
    fsPanel->hide();
   else 
    fsPanel->ShowAt( pos );
}  


void KMP::slotShowContextMenu(const QPoint& w)
{
  /* show context menu only in minmal or fullscreen mode */
  if ( (!viewFullscreen->isChecked()) && (!viewMinimalMode->isChecked()) ) return;

  if (fsPanel->isShown())
    fsPanel->hide();

  contextMenu->popup(w);
}


void KMP::slotNewChannels(const QStringList& audio, const QStringList& subtitles, int currentAudio, int currentSub)
{
  audioChannelMenu->clear();
  subtitleChannelMenu->clear();

  audioChannelMenu->insertItem(i18n("auto"), 0);
  for (uint a = 0; a<audio.count(); a++)
  {
    audioChannelMenu->insertItem( audio[a], a+1 );
  }

  audioChannelMenu->setItemChecked( currentAudio+1, true);

  subtitleChannelMenu->insertItem(i18n("off"), 0);
  for (uint s = 0; s<subtitles.count(); s++)
  {
    subtitleChannelMenu->insertItem( subtitles[s], s+1 );
  }

  subtitleChannelMenu->setItemChecked( currentSub+1, true);
  
  switch(stMode){
     case fileSubtitles:
     {
        cPanel->slotNewChannels(audio, subtitles, currentAudio, 0);
        PlaylistItem *item = dynamic_cast<PlaylistItem *>(playList->GetCurrent());
        if(!item->subtitles().isEmpty())
        {
           stMode = fileSubtitles;
	   QString subpath = QString::null;
               
           if(item->url().contains("#subtitle:"))
	     subpath = item->url().section("#subtitle:",1,2);
     
           cPanel->slotNewSubtitles(item->subtitles(), subpath);
        }
        break;
     }
     case xineSubtitles:
     {
       cPanel->slotNewChannels(audio, subtitles, currentAudio, currentSub);
       break;
     }
     default:
       break;
  }
}  


void KMP::slotContextMenuVolume(int id)
{  
  cPanel->volSlider->setValue( -(id*10) );
}


void KMP::slotContextMenuAudioChannel(int id)
{
  for (uint i = 0; i < audioChannelMenu->count(); i++)
   if (audioChannelMenu->isItemChecked(i))
     audioChannelMenu->setItemChecked(i , false);
  audioChannelMenu->setItemChecked( id, true );   

  cPanel->audioChannelCB->setCurrentItem(id);
  videoWin->slotSetAudioChannel(id);
}  


void KMP::slotContextMenuSubtitleChannel(int id)
{
  for (uint i = 0; i < subtitleChannelMenu->count(); i++)
   if (subtitleChannelMenu->isItemChecked(i))
     subtitleChannelMenu->setItemChecked(i , false);
  subtitleChannelMenu->setItemChecked( id, true );   

  cPanel->subtitleCB->setCurrentItem(id);
  videoWin->slotSetSubtitleChannel(id);
}   
  

/***********************************************
 *      command line/remote control stuff      *
 ***********************************************/
  
void KMP::LoadPlaylist()
{
  if (loadPlaylistAtStartup)
  {
    kdDebug(555) << "Load Playlist\n";
    tRow->slotChangeText(i18n("Load Playlist..."));
    playList->ReloadLastPlaylist();
    tRow->slotChangeText(i18n("Ready."));
 }
}


void KMP::CmdLineArgs(const KURL::List& files, const QString device, bool playAtStartup, bool startFullscreen, bool minimalMode, bool enqueueFiles)
{
  if (minimalMode)
  {
    viewMinimalMode->setChecked(true);
    slotToggleMinimalMode();
  }  

  if (startFullscreen)
  {
    viewFullscreen->setChecked(true);
    slotToggleFullscreen();
  }

  if (device != "do_nothing")
  {
    videoWin->SetDevice(device);
  }  

  if (files.count() > 0)
  {
    if (!enqueueFiles)
    {
      playList->slotClearList();
    }  
    QListViewItem* last = playList->GetLast();
    playList->Add(files, last);
    if (last == NULL)  /* !enqueueFiles */
      QTimer::singleShot( 100, this, SLOT(slotPlay()) );
  }
  else
  {
   LoadPlaylist();
   
   if (playAtStartup)
     QTimer::singleShot( 100, this, SLOT(slotPlay()) );
    else
     videoWin->PlayLOGO();
  }  
}         


/******** execute remote commands (from command line or via DCOP) ***********/

void KMP::ExecuteRemoteCommand(const QString& cmd)
{
  if (cmd == "play")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    if (cPanel->isPlayButton)
      cPanel->slotPlayPause();
    return;
  }
  if (cmd == "pause")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    if (!cPanel->isPlayButton)
      cPanel->slotPlayPause();
    return;
  }
  if (cmd == "next")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    if (cPanel->isPlaying)
      slotPlayNext();
    return;  
  }    
  if (cmd == "prev")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    if (cPanel->isPlaying)
      slotPlayPrev();
    return;  
  }    
  if (cmd == "stop")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    if (cPanel->isPlaying)
      slotStop();
    return;  
  }    

  if (cmd == "volup")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    cPanel->volSlider->subtractStep();
    return;
  }  
  if (cmd == "voldown")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    cPanel->volSlider->addStep();
    return;
  }  
  if (cmd == "incpos")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    cPanel->slotPosPlus();
    return;
  }  
  if (cmd == "decpos")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    cPanel->slotPosMinus();
    return;
  }
  if (cmd == "quit")
  {
    kdDebug(555) << i18n("Execute remote command") << " '" << cmd << "'...\n";
    slotQuit();
    return;
  }
  
  kdDebug(555) << "No such command\n";
}
  
/****************/


void KMP::dragEnterEvent (QDragEnterEvent *dev)
{
  // dev->accept(true);
   dev->accept(QUriDrag::canDecode(dev) || QTextDrag::canDecode(dev));
}


void KMP::dropEvent(QDropEvent* dev)
{                                                              

  KURL::List uris;
  
  if (KURLDrag::decode(dev, uris))
  {
    playList->slotClearList();
    playList->Add(uris, NULL);
    slotPlay();
  }
  else 
  if (strcmp(dev->format(), "text/x-moz-url") == 0)    // for mozilla drops
  {
    playList->slotClearList();
    QByteArray data = dev->encodedData("text/plain");
    QString md(data);
    playList->Add(md, NULL);
    slotPlay();
  }  

}


void KMP::showEvent (QShowEvent *)
{
  if ( (videoWin->xineRunning) && (currentVisualPlugin != "none") )
   videoWin->SetVisualPlugin( currentVisualPlugin );
}  


void KMP::hideEvent (QHideEvent *)
{
  if ( (videoWin->xineRunning) && (currentVisualPlugin != "none") )
   videoWin->SetVisualPlugin( "none" );
}  


void KMP::closeEvent(QCloseEvent* cev)
{
 cev->accept();
 if (systemTray)
   hide();
  else
   slotQuit();
}


void KMP::moveEvent(QMoveEvent*)
{ 
  videoWin->GlobalPosChanged();
}



/****************************************
 *   PLAYBACK CONTROL
 ****************************************/

void KMP::playItem(PlaylistItem *item)
{
  if (!item) return;

  //Notify kaffeine to select file subtitles in slotNewChannels
  if( !item->subtitles().isEmpty() )
     stMode = fileSubtitles;
  else
     stMode = xineSubtitles;

  QString playURL = item->url();

  if (videoWin->PlayMRL(playURL, item->title(), true))
  {
    cPanel->SetPlaying();
    if (item->parent())
    {
      cTitle = dynamic_cast<PlaylistItem *>(item->parent())->title();
      cUrl = dynamic_cast<PlaylistItem *>(item->parent())->url();
      fileOpenRecent->addURL(cUrl);
    }
    else
    {
      cTitle = item->title();
      cUrl = item->url();
      fileOpenRecent->addURL(cUrl);
    }  
    if (systemTray)
    {
      systemTray->SetPlaying( item->title() );
      systemTray->slotPlayIcon();
    }  
    fsPanel->slotSetTitle( item->title() );
    fsPanel->slotSetStreamInfo( item->info() );
    fsPanel->slotSetLength( item->length() );
    setCaption( item->title() );
  }  
  else
  {
    playList->GetNext();  /* mark the next entry as current */
    if (cPanel->GetPlaying())
      slotStop();
  }    
}

 
void KMP::slotPlay()
{
  if(playList->GetCurrent())
     playItem(dynamic_cast<PlaylistItem *>(playList->GetCurrent())); 
}


void KMP::slotPlayNext()
{
  if (hasChapters)
  {
    videoWin->PlayNextChapter();
    return;
  }
  
  if(playList->GetNext())
    playItem(dynamic_cast<PlaylistItem *>(playList->GetCurrent()));
}
 

void KMP::slotPlayPrev()
{

  if (hasChapters)
  {
    videoWin->PlayPreviousChapter();
    return;
  }
  
  if(playList->GetPrevious())
    playItem(dynamic_cast<PlaylistItem *>(playList->GetCurrent()));
}


void KMP::slotEndOfPlayback()
{
  if (!cPanel->GetPlaying()) return; //end of logo, error

  if(playList->GetNext())
    playItem(dynamic_cast<PlaylistItem *>(playList->GetCurrent()));
  else
    slotStop();
}


void KMP::slotStop()
{
  stMode = xineSubtitles;
  videoWin->slotStopPlayback();
  cPanel->ResetControlPanel();
  if (systemTray)
    systemTray->slotNormalIcon();
  cTitle = QString::null;
  cUrl = QString::null;
  setCaption(i18n("Kaffeine Media Player %1").arg( VERSION ));
  tRow->slotChangeText(i18n("Ready."));
}


void KMP::slotPlayRecent(const KURL& url)
{
  kdDebug(555) << "Play recent url: " << url.prettyURL() << endl;
  playList->slotClearList();
  playList->Add(url, NULL); 
  QTimer::singleShot( 100, this, SLOT(slotPlay()) );
}


/*************************************************/

void KMP::slotGetNewTitle(const QString& title)
{
  if (!hasChapters)
    osDisplay->slotOsdShow(title);

  if (systemTray)
    systemTray->SetPlaying(title);
  fsPanel->slotSetTitle( title );
  fsPanel->slotSetStreamInfo( videoWin->GetStreamInfo() );
  fsPanel->slotSetLength( videoWin->GetLengthInfo() );
  setCaption( title );
}


void KMP::slotPlayFiles(const KURL::List& urlList)
{
  playList->slotClearList();
  playList->Add(urlList, NULL);
  slotPlay();
}


void KMP::slotClearRecent()
{
  fileOpenRecent->clear(); /* not enough, why?? */
  fileOpenRecent->setItems( QStringList() );
}
       

void KMP::slotQuit()
{
  if (videoWin->xineRunning)  
  {
    playList->slotSaveCurrentPlaylist();
    SaveConfig();
    videoWin->SaveXineConfig();
    videoWin->slotRemoveAllFilters();
  }  
   
  KApplication::exit(0);
}  


/*********************************************/


void KMP::slotOpenFile()
{
 
  KURL::List urlList;

  urlList = KFileDialog::getOpenURLs(":kaffeineMain_openFile", fileFilter + "|" + i18n("Supported Media Formats") + "\n*.*|" + i18n("All Files"), 0, i18n("Open File(s)"));

  if (!urlList.count()) return;

  playList->slotClearList();
  playList->Add(urlList, NULL);

  QTimer::singleShot( 100, this, SLOT(slotPlay()) );
}


void KMP::slotOpenUrl()
{
  KURL kurl = KURLRequesterDlg::getURL(QString::null, 0, i18n("Open URL"));

  if (!kurl.isValid()) return;

  playList->slotClearList();
  playList->Add(kurl, NULL);

  QTimer::singleShot( 100, this, SLOT(slotPlay()) );
}


void KMP::slotSaveStreamAs()
{
  QListViewItem* item = playList->GetCurrent();
  if (!item) return;

  QString saveDir = videoWin->GetStreamSaveDir();

  KURL kurl = KFileDialog::getSaveURL( saveDir + "/" + KURL( item->text(4) ).fileName(), QString::null, 0, i18n("Save Stream As..."));

  if (!kurl.isValid()) return;

  if ( saveDir != kurl.directory() )
    videoWin->SetStreamSaveDir( kurl.directory() ); /* FIXME */
  
  QString playURL = item->text(4);
  playURL = playURL + "#save:" + kurl.path();

  if (videoWin->PlayMRL(playURL, item->text(1), true))
   {
     cPanel->SetPlaying();
     cPanel->SetRecording();
     fileOpenRecent->addURL(item->text(4));
     if (item->parent())
     {
       cTitle = item->parent()->text(1);
       cUrl = item->parent()->text(4);
     }                        
     else
     {
       cTitle = item->text(1);
       cUrl = item->text(4);
     }
     if (systemTray)
     {
       systemTray->SetPlaying(cTitle);
       systemTray->slotRecordIcon();
     }  
   }
   else
   {
     slotStop();
   }
}


void KMP::slotEscape()
{
  if (viewFullscreen->isChecked())
  {
    viewFullscreen->setChecked( false );
    slotToggleFullscreen();
  }
}    
  

void KMP::slotToggleFullscreenChecked()
{
  viewFullscreen->setChecked( !viewFullscreen->isChecked() );
  slotToggleFullscreen();
}  
                                               

void KMP::slotToggleFullscreen()
{
  if (viewFullscreen->isChecked())
  {
   /** go fullscreen **/
  
    menuBar->hide();
    tRow->hide();
    cPanel->hide();

#if KDE_IS_VERSION(3,2,0)
    kdDebug(555) << "Set KWin state NET::FullScreen" << endl;
    KWin::setState( winId(), NET::FullScreen | NET::StaysOnTop );
#else
    showFullScreen();    
#endif    
        
#ifdef WINDOWMAKER
    /* Layer above most other things, like gnome panel and the windowmaker
     * dock.       
     */
   
    hide(); // unmap 
    static Atom XA_WIN_LAYER = None;
    long propvalue[1];              
    if (XA_WIN_LAYER == None)
          XA_WIN_LAYER = XInternAtom(x11Display(), "_WIN_LAYER", False);
        
    propvalue[0] = 10;     
    XChangeProperty(x11Display(), winId(), XA_WIN_LAYER,
                              XA_CARDINAL, 32, PropModeReplace, 
                              (unsigned char *)propvalue, 1);
    kdDebug(555) << "Set WIN_LAYER." << endl;
    show();
#endif     

    videoWin->StartMouseHideTimer();
  }
  else
  {
    
    if (!viewMinimalMode->isChecked())
    {
     menuBar->show();
     tRow->show();
     cPanel->show();
    }

#if KDE_IS_VERSION(3,2,0)
    KWin::clearState( winId(), NET::FullScreen | NET::StaysOnTop );
#else
    showNormal();
#endif    
   
    
    videoWin->StopMouseHideTimer();
  }
}


void KMP::slotToggleMinimalMode()
{
  if (viewFullscreen->isChecked()) return;

  if (viewMinimalMode->isChecked())
  {
    /** go to minimal mode **/
  
    menuBar->hide();
    tRow->hide();
    cPanel->hide();
  }  
  else
  {
    menuBar->show();
    tRow->show();
    cPanel->show();
  }  
}
    

void KMP::slotPlayAutoplayPlugin(const QString& plugin)
{
  QStringList mrlList;
  
  if (videoWin->GetAutoplayPluginMrl(plugin, mrlList));
  {
    playList->slotClearList();
    playList->AddAutoplayUrls(mrlList, NULL);
    slotPlay();
  }
}


void KMP::slotAddAutoplayUrls(const QString& plugin, QListViewItem* after)
{
  QStringList mrlList;

  if (videoWin->GetAutoplayPluginMrl(plugin, mrlList));
  {
    playList->AddAutoplayUrls(mrlList, after);
  }
}

    
void KMP::slotHasChapters(bool c)
{
  hasChapters = c;
}  


void KMP::slotShowPlayList()
{
  playList->show();
  playList->raise();
  if (playList->isMinimized())
    playList->showNormal();
  playList->setActiveWindow();
}


void KMP::slotShowSetup()
{
  configWin->show();
}      


void KMP::slotShowFilterConfig()
{
  filterDialog->show();
}  


void KMP::slotShowDeinterlaceDialog()
{
  DeinterlaceQuality* deinterlaceQualityDialog = new DeinterlaceQuality( 0, "deinterlace_dialog" );
  deinterlaceQualityDialog->SetQuality( currentDeinterlaceQuality );
  connect( deinterlaceQualityDialog, SIGNAL(signalConfigureDeinterlacer()), videoWin, SLOT(slotShowDeinterlaceDialog()));
  connect( deinterlaceQualityDialog, SIGNAL(signalSetDeinterlaceConfig(const QString&)), videoWin, SLOT(slotSetDeinterlaceConfig(const QString&)));
  deinterlaceQualityDialog->exec();

  currentDeinterlaceQuality = deinterlaceQualityDialog->GetQuality();
  delete deinterlaceQualityDialog;
}
  

void KMP::slotEnableAutoresize()
{
  videoWin->slotEnableAutoresize( viewAutoresize->isChecked() );
}
 

void KMP::slotEndlessMode()
{
  if (playEndlessPlay->isChecked())
    playList->SetEndless(true);
   else
    playList->SetEndless(false);
}


void KMP::slotRandomPlay()
{
  if (playRandomPlay->isChecked())
    playList->SetRandomPlay(true);
   else
    playList->SetRandomPlay(false);
}


void KMP::slotToggleBroadcastMaster()
{
  bool ok = false;

  if (broadcastMaster->isChecked())
  {
    broadcastPort = (uint)QInputDialog::getInteger( QString::null, i18n("Broadcasting Port"), broadcastPort, 0, 1000000, 1, &ok);
    if (!ok)
    {
      broadcastMaster->setChecked( false );
      return;
    }
    videoWin->SetBroadcasterPort( broadcastPort );  
  }
  else
  {
    videoWin->SetBroadcasterPort(0); /* disable */  
  }
}


/************ network broadcasting ***************/

void KMP::slotBroadcastSlave()
{

  KDialogBase* dialog = new KDialogBase( 0, "configmaster", true, QString::null, KDialogBase::Ok|KDialogBase::Cancel );
  QVBox* page = dialog->makeVBoxMainWidget();
  new QLabel(i18n("Master Address"), page);
  KLineEdit* address = new KLineEdit( broadcastMasterAddress, page);
  new QLabel(i18n("Port"), page);
  QSpinBox* port = new QSpinBox(0, 1000000, 1, page);
  port->setValue( broadcastPort );
  
  int button = dialog->exec();

  if (button == KDialogBase::Accepted)
  {
    // kdDebug(555) << "Apply master configuration" << endl;
     broadcastPort = port->value();
     broadcastMasterAddress = address->text();
  }   

  delete dialog;

  if (button == KDialogBase::Rejected) return;
  
  slotPlayRecent( QString("slave://") + broadcastMasterAddress + ":" + QString::number(broadcastPort) );
}              

/*************/


void KMP::slotGenConfig(bool loadPl, bool embed)
{
  loadPlaylistAtStartup = loadPl;             
  if (embedSystemTray != embed)
  {
    if (systemTray)
    {
      delete systemTray;
      systemTray = NULL;
    }
    else
    {
      InitSystemTray();
    }
  }
        
  embedSystemTray = embed;
}


void KMP::slotConfigXine()
{
   XineConfig* xineConfigDialog = new XineConfig(videoWin->GetXineEngine());
   xineConfigDialog->exec();
   delete xineConfigDialog;
}


void KMP::slotVideoSettings()
{
  videoSettings->show();
}  


void KMP::slotConfigureShortcuts()
{
  KKeyDialog* keyDialog = new KKeyDialog( 0 );
  keyDialog->insert( playList->GetActionCollection(), i18n("Playlist") );
  keyDialog->insert( actionCollection, i18n("Main Window") );

  int retcode = keyDialog->exec();

  if (retcode == KKeyDialog::Accepted)
  {
    keyDialog->commitChanges();
  }  
  
  delete keyDialog;
}


void KMP::slotEqualizer()
{
  equalizer->show();
}  


void KMP::slotNewXineMessage()
{
  KMessageBox::information( 0, videoWin->GetXineMessage() );
}  

/*********************Subtitles****************/
void KMP::slotSetSubtitleChannel(int selection)
{
  switch(stMode)
  {
     case fileSubtitles:
     {
       PlaylistItem *item = dynamic_cast<PlaylistItem *>(playList->GetCurrent());      
       if(item->url().contains("#subtitle:"))
         item->setUrl(item->url().section("#subtitle:",0,0));
       if(selection >= 1)
         item->setUrl(item->url()+"#subtitle:"+item->subtitles()[selection-1]);
       videoWin->slotSetFileSubtitles(item->url());
       break;
     }
     case xineSubtitles:
       videoWin->slotSetSubtitleChannel(selection);
       break;
     default:
       break;
  }
}

/*********************************************************
 *   TAKE A SCREENSHOT
 *********************************************************/

void KMP::slotScreenshot()
{
  int width, height;
  double scaleFactor;
  uchar* rgb32BitData = NULL;

  videoWin->GetScreenshot(rgb32BitData, width, height, scaleFactor);

  if (rgb32BitData == NULL)
  {
    kdError(555) << "Can't take screenshot!" << endl;
    return;
  }

  QImage shot(rgb32BitData, width, height, 32 , (uint*)&RGB_MASK , 0  , QImage::IgnoreEndian);

  kdDebug(555) << "Screenshot: width: " << width << "height: " << height << "scale factor: " << scaleFactor << endl;
 
  if (scaleFactor >= 1.0)
    width = (int)((double) width * scaleFactor);  
  else
    height = (int) ((double) height / scaleFactor);
    
  shot = shot.smoothScale(width, height);
  

/** save **/


  KFileDialog dlg(":kaffeineMain_Screenshot", i18n("*.png|PNG-File\n*.bmp|BMP-File\n*.xbm|XBM-File"),
                          0, "save screenshot", true);

  dlg.setOperationMode(KFileDialog::Saving);
  dlg.setCaption(i18n("Save screenshot as..."));
  dlg.setSelection("screenshot.png");

  ScreenshotPreview* prev = new ScreenshotPreview(shot, &dlg);
  dlg.setPreviewWidget(prev);

  dlg.exec();

  QString fileName = dlg.selectedFile();

  if (fileName.isEmpty())
  {
    delete [] rgb32BitData;
    return;
  }  

  QString type = dlg.currentFilter();
  type = (type.remove(0,2)).upper();
  kdDebug(555) << "Save screenshot as " << type << "\n";
  
 
  if  (!shot.save(fileName, type))
    kdError(555) << "Screenshot not saved successfully!" << endl;
    

  delete [] rgb32BitData;
}


/*********************************/


void KMP::slotToggleTv()
{
   /* not implemented yet */
}



/******* kbookmarkowner reimplementation ************/


void KMP::openBookmarkURL(const QString& url)
{
  playList->slotClearList();
  playList->Add(url, NULL);   // allows playlists as bookmark
  QTimer::singleShot( 100, this, SLOT(slotPlay()) );
}


QString KMP::currentTitle() const
{
   return cTitle;
}


QString KMP::currentURL() const
{
  return cUrl;
}

