
#include <xterm.h>

#if OPT_HANGUL

#include <stdlib.h>
#include <assert.h>

#include "ptyx.h"
#include "data.h"

#include "comp.h"

#define has_temp_han() (han_compose_buf[0] != 0)

Char han_compose_buf[8] = {0};
int hangul_state = 0;

int han_new_typed_row = -999;

static Composer *composer = NULL;

static void han_cursor_color_update(void);

static int
fill_return_buffer_c(char *buf, char c)
{
    if (has_temp_han()) {
	*buf++ = han_compose_buf[0];
	*buf++ = han_compose_buf[1];
	*buf = c;
	han_compose_buf[0] = 0;
	return 3;
    }
    *buf = c;
    return 1;
}

static int
fill_return_buffer_s(char *buf, char *s)
{
    int len;
    if (!s) {
	if (has_temp_han()) {
	    strcpy(buf, han_compose_buf);
	    return 2;
	}
	return 0;
    }
    len = strlen(s);
    if (has_temp_han()) {
	*buf++ = han_compose_buf[0];
	*buf++ = han_compose_buf[1];
	strcpy(buf, s);
	han_compose_buf[0] = 0;
	return len + 2;
    }
    strcpy(buf, s);
    return len;
}

void
hangul_automata_init(void)
{
    int i, n;
    char **kbd_names;
    char *dubyol = term->misc.han_dubyol_keyboard_file;
    char *sebyol = term->misc.han_sebyol_keyboard_file;
    kbd_names = composer_get_keyboard_list(KEYBOARD_DIR, &n);
    for(i=0;i<n;i++) {
	if (strcmp(kbd_names[i], dubyol) == 0) break;
    }
    if (i == n) {
	fprintf(stderr, "ι  (%s) ã  ϴ\n", dubyol);
	exit(1);
    }
    for(i=0;i<n;i++) {
	if (strcmp(kbd_names[i], sebyol) == 0) break;
    }
    if (i == n) {
	fprintf(stderr, "  (%s) ã  ϴ\n", sebyol);
	exit(1);
    }
    hangul_set_keyboard(term->screen.han_kbd);
    composer = composer_new();
    hangul_set_code(term->screen.han_code);
}

int 
hangul_automata(int c, Char *buf)
{
    int ret = 0;
    int process_ret;
    if (!composer) {
	hangul_automata_init();
    }
    if (c == -2) return hangul_automata_flush(buf);
    if (c == -1) return hangul_automata_toggle(buf);

    assert (c >= 0);

    if (!hangul_state) {
	buf[0] = c;
	return 1;
    }
    switch ((process_ret = composer_process(composer, c))) {
	case COMPOSER_NEW:
	    ret = fill_return_buffer_s(buf, NULL);
	    strcpy(han_compose_buf, composer_get_buffer(composer));
	    break;
	case COMPOSER_RAW:
	    ret = fill_return_buffer_c(buf, c);
	    han_compose_buf[0] = 0;
	    if (c == 27 && term->misc.han_esc_toggle) {
		hangul_state = 0;
		han_show_status(&term->screen, 0, 0);
		han_cursor_color_update();
	    }
	    break;
	case COMPOSER_NULL:
	    han_compose_buf[0] = 0;
	    break;
	case COMPOSER_NOT_HAN:
	    ret = fill_return_buffer_s(buf, composer_get_buffer(composer));
	    han_compose_buf[0] = 0;
	    break;
	case COMPOSER_CORRECT:
	    strcpy(han_compose_buf, composer_get_buffer(composer));
	    ret = 0;
	    break;
	case COMPOSER_ERROR:
	    Bell(0, 50);
	    return 0;
	case COMPOSER_CORRECT_NEW:
	    strcpy(buf, composer_get_buffer(composer));
	    strcpy(han_compose_buf, composer_get_buffer(composer) + 2);
	    ret = 2;
	    break;
    }
    if (ret || process_ret == COMPOSER_NULL) {
	int c = han_compose_buf[0];
	han_compose_buf[0] = 0;
	ScrnRefresh(&term->screen, 
	    term->screen.cursor_row, term->screen.cursor_col,
	    1, 2, True);
	han_compose_buf[0] = c;
	term->screen.cursor_state = OFF;
    } else {
	ShowCursor();
    }
    han_new_typed_row = term->screen.cursor_row;
    return ret;
}

int 
hangul_automata_flush(Char *buf)
{
    composer_reset(composer);
    if (has_temp_han()) {
	buf[0] = han_compose_buf[0];
	buf[1] = han_compose_buf[1];
	han_compose_buf[0] = 0;
	ShowCursor();
	return 2;
    }
    return 0;
}

static void
han_cursor_color_update(void)
{
    TScreen *screen = &term->screen;
    if (hangul_state) {
	screen->cursorcolor = screen->han_cursorcolor;
    } else {
	screen->cursorcolor = screen->eng_cursorcolor;
    }
    set_cursor_gcs(screen);
    ShowCursor();
}

int
hangul_automata_toggle(Char *buf)
{

    hangul_state = !hangul_state;
    composer_reset(composer);
    han_show_status(&term->screen, 0, 0);
    han_cursor_color_update();
    return hangul_automata_flush(buf);
}

void
hangul_set_keyboard(int keyboard)
{
    if (keyboard == 2) {
	composer_set_keymap(term->misc.han_dubyol_keyboard_file);
    } else {
	composer_set_keymap(term->misc.han_sebyol_keyboard_file);
    }
}

void
hangul_set_code(int code)
{
    if (!composer) {
	hangul_automata_init();
    }
    if (code == C_WANSUNG) {
	composer_set_output_encoding(composer, "WANSUNG");
    } else if (code == C_JOHAB) {
	composer_set_output_encoding(composer, "JOHAB");
    }
}


void 
hangul_automata_clear(void)
{
    composer_reset(composer);
    han_compose_buf[0] = 0;
}

#endif /* OPT_HANGUL */
