/*
 * MBuffer header
 *
 * Provides operations for a read-only buffer, which is usually mmap'd buffer.
 * Thus it is not null-byte terminated.
 * The operations are typically based on a line number,
 * and return the pointer of the head of the line.
 *
 * For example,
 * if the caller needs to know the byte length from 3rd line to 5th line,
 * it should call as follows,
 *  begin_pt = mbuf_goto_line(mbuf, 3);//the head of 3rd line
 *  end_pt = mbuf_goto_line(mbuf, 6); //the head of 6th line
 *  lenb = end_pt - begin_pt;
 *
 * Naming convention:
 * nl: the number of lines
 * ln: line number
 *
 * # M was mmap's m, but it lost its original meaning.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifndef __GDIFF_MBUFFER_H__
#define __GDIFF_MBUFFER_H__

#ifdef __cplusplus 
extern "C" {
#endif /* __cplusplus */

#define MBUFFER(obj)		((MBuffer*)(obj))

/* Forward declarations of types. */
typedef struct _MBuffer MBuffer;
typedef struct _MBufferPrivate MBufferPrivate;

/* 
 * Data structure definitions
 */
struct _MBuffer {
	int cur_ln;		/* current line number, starting from one */
	const char *cur_pt;	/* current point */
	int lenb;		/* length of buffer in byte used,
					   mainly used to avoid overrun */
	int nl;			/* the number of lines of buffer */

	MBufferPrivate *privat;
};


/* Global function declarations */
extern MBuffer* mbuf_new(const char *buf, int lenb, int nl, gboolean f_cache);
extern void mbuf_delete(MBuffer *mbuf);
extern void _mbuf_init(MBuffer *mbuf, const char *buf, int lenb, int nl, gboolean f_cache);
extern void _mbuf_finalize(MBuffer *mbuf);
extern const char* mbuf_goto_line(MBuffer *mbuf, int ln);
extern const char* mbuf_next_line(MBuffer *mbuf, int nl);
extern const char* mbuf_prev_line(MBuffer *mbuf, int nl);
extern int mbuf_search_string(MBuffer *mbuf, int start_ln, const char *string, int lenb);

/* Macros */
#define mbuf_goto_top(mbuf)		mbuf_goto_line(mbuf, 0)
#define mbuf_goto_bottom(mbuf)	mbuf_goto_line(mbuf, (mbuf->nl + 1))


#ifdef __cplusplus
} 
#endif /* __cplusplus */

#endif /* __GDIFF_MBUFFER_H__ */
