/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/
#pragma prototyped

#include <convert.h>

#define SMALLBUF	1000
#define NAMEBUF		100
#define EMPTY(s)	((s == 0) || (s)[0] == '\0')

#define NODE		1
#define SYNNODE		2
#define GRAPH		3
#define GXLATTR		"_gxl_"

static void write_body(Agraph_t *g, FILE *gxlFile);
static void iterate_body(Agraph_t *g);
static int Level;
static unsigned char Attrs_not_written_flag;
static Agsym_t *Tailport,*Headport;
char written_nodes[NAMEBUF][NAMEBUF];
int nodeCount = 0;
int nodeIdCounter = 1;
int graphIdCounter = 1;
int idcount = 0;
char idList[SMALLBUF][NAMEBUF];

struct namemap_t {
	char name[SMALLBUF][NAMEBUF];
	char unique_name[SMALLBUF][NAMEBUF];
	int position;
};

typedef struct namemap_t *NameMap;

NameMap nodeMap, graphMap, synNodeMap;


static void 
tabover(FILE* gxlFile)
{
	int temp;
	temp = Level;
	while (temp--) putc('\t',gxlFile);
}


static void
escapeForXML(FILE* gxlFile, char *value) {

	for(; *value != '\0'; value++) {
		if(*value == '>')
			fprintf(gxlFile, "&gt;");	
		else if(*value == '<') 
			fprintf(gxlFile, "&lt;");
		else if(*value == '&')
			fprintf(gxlFile, "&#%d;", (int)*value);
		else
			fprintf(gxlFile, "%c", *value);
	} 
}


static int
isGxlGrammar(char *name) {
	if(strncmp(name, GXLATTR, (sizeof(GXLATTR) - 1)) == 0)
		return 1;
	return -1;
}


static int
isLocatorType(char *name) {
	int len, i;
	char *loc;
	loc = "locator_";
	len = sizeof(GXLATTR) - 1;
	for(i = 0; i < len; i++) {
		if(GXLATTR[i] != *name++) {
			return -1;
		}
	}
	// also locator_
	for(i = 0; i < 8; i++) {
		if(loc[i] != *name++) {
			return -1;
		}
	}

	return 1;
}


int idexists(char *id) {
	int i;
	char *temp;

	for(i = 0; i < idcount; i++) {
		temp = idList[i];
		if(temp != NULL && strcmp(id, idList[i]) == 0)
			return 1;
	}
	return 0;
}


void addid(char *id) {
	strcpy(idList[idcount++], id);
}


char *createUniqueId(int type, char *buf) {
	switch(type) {
		case 1:
			sprintf(buf, "N_%d", nodeIdCounter++);
			if(idexists(buf)) {
				return createUniqueId(type, buf);
			}
			break;
		case 2:
			break;
		case 3:
			if(idexists(buf)) {
				return createUniqueId(type, buf);
			}
			sprintf(buf, "G_%d", graphIdCounter++);
			break;
		default:
			break;
	}
	return buf;
}


static void
addToMap(NameMap nm,char *name, char *uniqueName) {
	strcpy(nm->name[nm->position], name);
	strcpy(nm->unique_name[nm->position++], uniqueName);
}

static char
*mapLookup(NameMap nm, char *name) {
	int i;
	for(i = 0; i < nm->position; i++) {
		if(strcmp(name, nm->name[i]) == 0)
			return nm->unique_name[i];
	}
	return "";
}


static void
graph_attrs(FILE* gxlFile, Agraph_t* g) 
{
	char *val;
	char buf[NAMEBUF];
	sprintf(buf, "%srole", GXLATTR);
	val = agget(g, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		fprintf(gxlFile, " role=\"%s\"", val);
	}
	sprintf(buf, "%shypergraph", GXLATTR);
	val = agget(g, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		fprintf(gxlFile, " hypergraph=\"%s\"", val);
	}
}


static void
edge_attrs(FILE* gxlFile, Agedge_t* e) 
{
	char *val;
	char buf[NAMEBUF];
	sprintf(buf, "%sid", GXLATTR);
	val = agget(e, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		fprintf(gxlFile, " id=\"%s\"", val);
	}
	sprintf(buf, "%sfromorder", GXLATTR);
	val = agget(e, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		fprintf(gxlFile, " fromorder=\"%s\"", val);
	}
	sprintf(buf, "%stoorder", GXLATTR);
	val = agget(e, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		fprintf(gxlFile, " toorder=\"%s\"", val);
	}
}


static void
print_href(FILE* gxlFile, void* n) 
{
	char *val;
	char buf[NAMEBUF];
	sprintf(buf, "%stype", GXLATTR);
	val = agget(n, buf);
	if(val != NULL && strcmp(val, "") != 0) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<type xlink:href=\"%s\">\n", val);
		tabover(gxlFile);
		fprintf(gxlFile, "\t</type>\n");
	}
}


static void write_dict(Agraph_t *g, FILE *gxlFile, char *name, Dict_t *dict)
{
	Dict_t	*view;
	Agsym_t	*sym,*psym;

	view = dtview(dict,NIL(Dict_t*));
	for (sym = (Agsym_t*)dtfirst(dict); sym; sym = (Agsym_t*)dtnext(dict,sym)) {
		if(isGxlGrammar(sym->name) < 0) {
			if (EMPTY(sym->defval)) {	/* try to skip empty str (default) */
				if (view == NIL(Dict_t*)) continue;	/* no parent */
				psym = (Agsym_t*)dtsearch(view,sym);
				//assert(psym);
				if (EMPTY(psym->defval)) continue;	/* also empty in parent */
			}

			if(isLocatorType(sym->defval) > 0) {
				char *locatorVal;
				locatorVal = sym->defval;
				locatorVal += 13;

				tabover(gxlFile);
				fprintf(gxlFile, "\t<attr name=\"%s\">\n", sym->name);
				tabover(gxlFile);
				fprintf(gxlFile, "\t\t<locator xlink:href=\"%s\"/>\n", locatorVal);				
				tabover(gxlFile);
				fprintf(gxlFile, "\t</attr>\n");
			} else {
				tabover(gxlFile);
				fprintf(gxlFile, "\t<attr name=\"%s\" kind=\"%s\">\n", sym->name, name);
				tabover(gxlFile);
				fprintf(gxlFile, "\t\t<string>");
				escapeForXML(gxlFile, sym->defval);
				fprintf(gxlFile, "</string>\n");
				tabover(gxlFile);
				fprintf(gxlFile, "\t</attr>\n");
			}
		} else {
			// gxl attr; check for special cases like composites
			char compBuf[20];
			int len;
			len = sizeof(GXLATTR) + 9;
			sprintf(compBuf, "%scomposite_", GXLATTR);
			if(strncmp(sym->name, compBuf, len) == 0) {
				if (EMPTY(sym->defval)) {	
					if (view == NIL(Dict_t*)) continue;	
					psym = (Agsym_t*)dtsearch(view,sym);
					if (EMPTY(psym->defval)) continue;	
				}

				tabover(gxlFile);
				fprintf(gxlFile, "\t<attr name=\"%s\" kind=\"%s\">\n", ((sym->name) + len), name);
				tabover(gxlFile);
				fprintf(gxlFile, "\t\t%s\n", sym->defval);
				tabover(gxlFile);
				fprintf(gxlFile, "\t</attr>\n");
			}
		}
	}
	dtview(dict,view);			/* restore previous view */
}

static void write_dicts(Agraph_t *g, FILE *gxlFile)
{
	Agdatadict_t	*def;
	if ((def = (Agdatadict_t*)agdatadict(g))) {
		write_dict(g, gxlFile, "graph", def->dict.g);
		write_dict(g, gxlFile, "node", def->dict.n);
		write_dict(g, gxlFile, "edge", def->dict.e);
	}
}

static void write_hdr(Agraph_t *g, FILE *gxlFile, int top)
{
	char		*name,*kind, *uniqueName;
	char		buf2[NAMEBUF];
	
	Level++;
	Attrs_not_written_flag = AGATTRWF(g);

	name = agnameof(g);
	if (g->desc.directed) kind = "directed";
	else kind = "undirected";
	if (NOT(top) && agparent(g)) { 
		// this must be anonymous graph

		sprintf(buf2, "N_%s", name);

		if(idexists(buf2) > 0) {
			createUniqueId(NODE, buf2);
			addid(buf2);
			addToMap(synNodeMap, name, buf2);
		} else {
			addid(buf2);
			addToMap(synNodeMap, name, buf2);
		}

		tabover(gxlFile);
		fprintf(gxlFile, "<node id=\"%s\">\n", buf2);
		Level++;
	}
	else {
		Tailport = agattr(g,AGEDGE,"tailport",NIL(char*));
		Headport = agattr(g,AGEDGE,"headport",NIL(char*));
	}


	uniqueName = mapLookup(graphMap, name);
	tabover(gxlFile);
	fprintf(gxlFile, "<graph id=\"%s\" edgeids=\"false\" edgemode=\"%s\"", uniqueName, kind);
	graph_attrs(gxlFile, g);
	fprintf(gxlFile, ">\n");

	if(strcmp(name, uniqueName) != 0) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<attr name=\"name\">\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t\t<string>%s</string>\n", name);
		tabover(gxlFile);
		fprintf(gxlFile, "\t</attr>\n");
	}
	
	if(agisstrict(g)) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<attr name=\"strict\">\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t\t<string>true</string>\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t</attr>\n");
	}

	write_dicts(g, gxlFile);
	print_href(gxlFile, g);
	AGATTRWF(g) = NOT(AGATTRWF(g));
}

static void write_trl(Agraph_t *g, FILE *gxlFile, int top)
{
	//NOTUSED(g);
	tabover(gxlFile);
	fprintf(gxlFile, "</graph>\n");
	Level--;
	if (NOT(top) && agparent(g)) { 
		tabover(gxlFile);
		fprintf(gxlFile, "</node>\n");
		Level--;
	}
}


static void write_subgs(Agraph_t *g, FILE *gxlFile)
{
	Agraph_t	*subg;

	for (subg = agfstsubg(g); subg; subg = agnxtsubg(subg)) {
		write_hdr(subg, gxlFile, FALSE);
		write_body(subg, gxlFile);
		write_trl(subg, gxlFile, FALSE);
	}
}

static int write_edge_name(Agedge_t *e, FILE *gxlFile, int terminate)
{
	int			rv;
	char		*p;

	p = agnameof(e);
	if (NOT(EMPTY(p))) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<attr name=\"key\">\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t\t<string>%s</string>\n", p);
		tabover(gxlFile);
		fprintf(gxlFile, "\t</attr>\n");	
		rv = TRUE;
	}
	else rv = FALSE;
	return rv;
}


static void write_nondefault_attrs(void *obj, FILE *gxlFile, Dict_t *defdict)
{
	Agattr_t	*data;
	Agsym_t		*sym;
	int		cnt = 0;

	if ((AGTYPE(obj) == AGINEDGE) || (AGTYPE(obj) == AGOUTEDGE)) {
		if (write_edge_name(obj, gxlFile, FALSE)) cnt++;
	}
	data = (Agattr_t*)agattrrec(obj);
	if (data) 
	{
		for(sym = (Agsym_t*)dtfirst(defdict); sym; sym = (Agsym_t*)dtnext(defdict,sym)) 
		{
			if(isGxlGrammar(sym->name) < 0) {
				if ((AGTYPE(obj) == AGINEDGE) || (AGTYPE(obj) == AGOUTEDGE)) {
					if (Tailport && (sym->id == Tailport->id)) continue;
					if (Headport && (sym->id == Headport->id)) continue;
				}
				if (data->str[sym->id] != sym->defval) {

					if(strcmp(data->str[sym->id], "") == 0)
						continue;

					if(isLocatorType(data->str[sym->id]) > 0) {
						char *locatorVal;
						locatorVal = data->str[sym->id];
						locatorVal += 13;

						tabover(gxlFile);
						fprintf(gxlFile, "\t<attr name=\"%s\">\n", sym->name);
						tabover(gxlFile);
						fprintf(gxlFile, "\t\t<locator xlink:href=\"%s\"/>\n", locatorVal);				
						tabover(gxlFile);
						fprintf(gxlFile, "\t</attr>\n");
					} else {
						tabover(gxlFile);
						fprintf(gxlFile, "\t<attr name=\"%s\">\n", sym->name);
						tabover(gxlFile);
						fprintf(gxlFile, "\t\t<string>");
						escapeForXML(gxlFile, data->str[sym->id]);
						fprintf(gxlFile, "</string>\n");
						tabover(gxlFile);
						fprintf(gxlFile, "\t</attr>\n");
					}
				}
			} else {
				// gxl attr; check for special cases like composites
				char compBuf[20];
				int len;
				len = sizeof(GXLATTR) + 9;
				sprintf(compBuf, "%scomposite_", GXLATTR);
				if(strncmp(sym->name, compBuf, len) == 0) {
					if (data->str[sym->id] != sym->defval) {

						tabover(gxlFile);
						fprintf(gxlFile, "\t<attr name=\"%s\">\n", ((sym->name) + len));
						tabover(gxlFile);
						fprintf(gxlFile, "\t\t%s\n", data->str[sym->id]);
						tabover(gxlFile);
						fprintf(gxlFile, "\t</attr>\n");
					}
				}
			}
		}
	}
	AGATTRWF((Agobj_t*)obj) = NOT(AGATTRWF((Agobj_t*)obj));
}

static void write_nodename(Agnode_t *n, FILE *gxlFile)
{
	char		*name, *uniqueName, buf[20];

	name = agnameof(n);
	uniqueName = mapLookup(nodeMap, name);
	if (uniqueName) fprintf(gxlFile, "%s", uniqueName);
	else {
	    sprintf(buf,"_%ld_SUSPECT",AGID(n));	/* could be deadly wrong */
	    fprintf(gxlFile, "%s", buf);
	}
}

static int attrs_written(void *obj)
{
	return NOT(AGATTRWF((Agobj_t*)obj) == Attrs_not_written_flag);
}

static void write_node(Agnode_t *n, FILE *gxlFile, Dict_t *d)
{
	char *name, *uniqueName;

	name = agnameof(n);
	uniqueName = mapLookup(nodeMap, name);
	Level++;
	tabover(gxlFile);
	fprintf(gxlFile, "<node id=\"");	
	write_nodename(n, gxlFile);
	fprintf(gxlFile, "\">\n");

	print_href(gxlFile, n);

	if(strcmp(name, uniqueName) != 0) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<attr name=\"name\">\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t\t<string>%s</string>\n", name);
		tabover(gxlFile);
		fprintf(gxlFile, "\t</attr>\n");
	}

	if (NOT(attrs_written(n)))
		write_nondefault_attrs(n, gxlFile, d);
	tabover(gxlFile);
	fprintf(gxlFile, "</node>\n");
	Level--;
}


static void write_port(Agedge_t *e, FILE *gxlFile, char *name)
{
	char		*val;

	val = agget(e,name);
	if (val[0]) {
		tabover(gxlFile);
		fprintf(gxlFile, "\t<attr name=\"%s\">\n", name);
		tabover(gxlFile);
		fprintf(gxlFile, "\t\t<string>");
		escapeForXML(gxlFile, val);
		fprintf(gxlFile, "</string>\n");
		tabover(gxlFile);
		fprintf(gxlFile, "\t</attr>\n");
	}
}

static int write_edge_test(Agraph_t *g, Agedge_t *e)
{
	Agraph_t	*subg;

	/* can use agedge() because we subverted the dict compar_f */
	for (subg = agfstsubg(g); subg; subg = agnxtsubg(subg)) {
		if (agsubedge(subg,e,FALSE)) return FALSE;
	}
	return TRUE;
}

static void write_edge(Agedge_t *e, FILE *gxlFile, Dict_t *d)
{
	Agnode_t	*t,*h;

	t = AGTAIL(e);
	h = AGHEAD(e);

	Level++;
	tabover(gxlFile);
	fprintf(gxlFile, "<edge from=\"");
	write_nodename(t, gxlFile);
	fprintf(gxlFile, "\" to=\"");
	write_nodename(h, gxlFile);
	fprintf(gxlFile, "\"");
	edge_attrs(gxlFile, e);

	if (agisdirected(agraphof(t))) { 
		fprintf(gxlFile, " isdirected=\"true\">\n");
	} else {
		fprintf(gxlFile, " isdirected=\"false\">\n");
	}

	print_href(gxlFile, e);
		
	write_port(e, gxlFile, "tailport");
	write_port(e, gxlFile, "headport");
	if (NOT(attrs_written(e)))
		write_nondefault_attrs(e, gxlFile, d);
	else
		write_edge_name(e, gxlFile, TRUE);
	tabover(gxlFile);
	fprintf(gxlFile, "</edge>\n");
	Level--;
}


static int node_not_written(char *name) {
	int i;
	for(i = 0; i < nodeCount; i++) {
		if(strcmp(name, written_nodes[i]) == 0) {
			return FALSE;
		}
	}
	return TRUE;
}


static void write_body(Agraph_t *g, FILE *gxlFile)
{
	Agnode_t	*n;
	Agedge_t	*e;
	Agdatadict_t *dd;

	write_subgs(g, gxlFile);
	dd = (Agdatadict_t*)agdatadict(g);
	for (n = agfstnode(g); n; n = agnxtnode(n)) {
		char *nodename;
		nodename = agnameof(n);

		if(node_not_written(agnameof(n))) {
			strcpy(written_nodes[nodeCount++], nodename);
			write_node(n, gxlFile, dd->dict.n);
		}

		for (e = agfstout(n); e; e = agnxtout(e)) {
			if (write_edge_test(g,e))
				write_edge(e, gxlFile, dd->dict.e);
		}
	}
}


static void iterate_hdr(Agraph_t *g, int top)
{
	char		*name;
	char		*gxlId;
	char		buf[NAMEBUF], gxl_id[NAMEBUF];
	
	name = agnameof(g);
	sprintf(gxl_id, "%sid", GXLATTR);
	gxlId = agget(g, gxl_id);	

	if(gxlId && strcmp(gxlId, "") != 0) {
		if(idexists(gxlId) > 0) {
			createUniqueId(GRAPH, buf);
			addid(buf);
			addToMap(graphMap, name, buf);		
		} else {
			addid(gxlId);
			addToMap(graphMap, name, gxlId);
		}
	} else {
		if(idexists(name) > 0) {
			createUniqueId(GRAPH, buf);
			addid(buf);
			addToMap(graphMap, name, buf);
		} else {
			addid(name);
			addToMap(graphMap, name, name);
		}
	}			
}


static void iterate_subgs(Agraph_t *g)
{
	Agraph_t	*subg;

	for (subg = agfstsubg(g); subg; subg = agnxtsubg(subg)) {
		iterate_hdr(subg, FALSE);
		iterate_body(subg);
	}
}


static void iterate_body(Agraph_t *g)
{
	Agnode_t	*n;

	iterate_subgs(g);
	for (n = agfstnode(g); n; n = agnxtnode(n)) {
		char *nodename, *check, *gxlId;
		char buf[NAMEBUF], gxl_id[NAMEBUF];
		nodename = agnameof(n);

		check =	mapLookup(nodeMap, nodename);
		if(strcmp(check, "") == 0) {
			sprintf(gxl_id, "%sid", GXLATTR);
			gxlId = agget(n, gxl_id);	
			if(gxlId && strcmp(gxlId, "") != 0) {
				if(idexists(gxlId) > 0) {
					createUniqueId(NODE, buf);
					addid(buf);
					addToMap(nodeMap, nodename, buf);
				} else {
					addid(gxlId);
					addToMap(nodeMap, nodename, gxlId);
				}
			} else {
				if(idexists(nodename) > 0) {
					createUniqueId(NODE, buf);
					addid(buf);
					addToMap(nodeMap, nodename, buf);
				} else {
					addid(nodename);
					addToMap(nodeMap, nodename, nodename);
				}
			}		
		}
	}
}


void 
dot_to_gxl(Agraph_t* g, FILE *gxlFile)
{	
	nodeMap = (NameMap)malloc(sizeof(struct namemap_t));
	graphMap = (NameMap)malloc(sizeof(struct namemap_t));
	synNodeMap = (NameMap)malloc(sizeof(struct namemap_t));
	
	nodeMap->position = 0;
	graphMap->position = 0;
	synNodeMap->position = 0;

	iterate_hdr(g, TRUE);
	iterate_body(g);

	Level = 0;

	fprintf(gxlFile, "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n");
	fprintf(gxlFile, "<gxl>\n");

	write_hdr(g, gxlFile, TRUE);
	write_body(g, gxlFile);
	write_trl(g, gxlFile, TRUE);

	fprintf(gxlFile, "</gxl>\n");

	free(nodeMap);
	free(graphMap);
	free(synNodeMap);
}

