/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/
#pragma prototyped

#include	"neato.h"
#include	"pack.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

static int Pack;    /* If >= 0, layout components separately and pack together
                     * The value of Pack gives margins around graphs.
                     */
static char*  cc_pfx = "_neato_cc";

void neato_nodesize(node_t* n, boolean flip)
{
    int         w;

    w = ND_xsize(n) = POINTS(ND_width(n));
    ND_lw(n)  = ND_rw(n) = w / 2;
    ND_ht(n) = ND_ysize(n) = POINTS(ND_height(n));
}

void neato_init_node(node_t* n)
{
	common_init_node(n);
	ND_pos(n) = ALLOC(GD_ndim(n->graph),0,double);
	neato_nodesize(n,GD_left_to_right(n->graph));
}

void neato_init_edge(edge_t* e)
{
	common_init_edge(e);

	ED_factor(e) = late_double(e,E_weight,1.0,1.0);

	init_port(e->tail,e,agget(e,"tailport"));
	init_port(e->head,e,agget(e,"headport"));
}

void neato_init_node_edge(graph_t *g)
{
	node_t *n;
	edge_t *e;

    for (n = agfstnode(g); n; n = agnxtnode(g,n)) neato_init_node(n);
    for (n = agfstnode(g); n; n = agnxtnode(g,n)) {
        for (e = agfstout(g,n); e; e = agnxtout(g,e)) neato_init_edge(e);
    }
}

int init_port(node_t* n, edge_t* e, char* name)
{
	port_t	port;

	if (name == NULL) return FALSE;
	port = ND_shape(n)->portfn(n,name);
#ifdef NOTDEF
	if (n->GD_left_to_right(graph)) port.p = invflip_pt(port.p);
#endif 
	port.order = 0;
	if (e->tail == n) ED_tail_port(e) = port; else ED_head_port(e) = port;
	return TRUE;
}

void neato_cleanup_node(node_t* n)
{
	if (ND_shape(n))
		ND_shape(n)->freefn(n);
    free (ND_pos(n));
	free_label(ND_label(n));
	memset(&(n->u),0,sizeof(Agnodeinfo_t));
}

void neato_free_splines(edge_t* e)
{
	int		i;
	if (ED_spl(e)) {
		for (i = 0; i < ED_spl(e)->size; i++) free(ED_spl(e)->list[i].list);
		free(ED_spl(e)->list);
		free(ED_spl(e));
	}
	ED_spl(e) = NULL;
}

void neato_cleanup_edge(edge_t* e)
{
	neato_free_splines(e);
	free_label(ED_label(e));
	memset(&(e->u),0,sizeof(Agedgeinfo_t));
}

void neato_cleanup_graph(graph_t* g)
{
	if (Nop || (Pack < 0)) free_scan_graph(g);
	else {
		Agraph_t* mg;
		Agedge_t* me;
		Agnode_t* mn;
		Agraph_t* subg;
        int       slen = strlen (cc_pfx);

		mg = g->meta_node->graph;
		for (me = agfstout(mg,g->meta_node); me; me = agnxtout(mg,me)) {
			mn = me->head;
			subg = agusergraph(mn);
			if (strncmp(subg->name,cc_pfx,slen) == 0)
              free_scan_graph (subg);
		}
	}
	free_ugraph(g);
	free_label(GD_label(g));
	memset(&(g->u),0,sizeof(Agraphinfo_t));
}

void neato_cleanup(graph_t* g)
{
	node_t  *n;
	edge_t  *e;

	for (n = agfstnode(g); n; n = agnxtnode(g, n)) {
		for (e = agfstedge(g, n); e; e = agnxtedge(g, e, n)) {
			neato_cleanup_edge(e);
		}
		neato_cleanup_node(n);
	}
	neato_cleanup_graph(g);
}

static void
neato_init_graph (Agraph_t *g)
{
	UseRankdir = FALSE;
    graph_init(g);
    GD_drawing(g)->engine = NEATO;
	GD_ndim(g) = late_int(g,agfindattr(g,"dim"),2,2);
	Ndim = GD_ndim(g) = MIN(GD_ndim(g),MAXDIM);
    neato_init_node_edge(g);
}

void neato_layout(Agraph_t *g)
{
    int         nG;
   
	neato_init_graph(g);
	if (Nop) {
    	nG = scan_graph(g);
	    initial_positions(g, nG);
    	adjustNodes(g);
   		spline_edges(g);
	}
	else {
		char *p;
		if ((p = agget(g,"pack"))) {
		  if (sscanf(p,"%d",&Pack) != 1) {
            if ((*p == 't') || (*p == 'T')) Pack = CL_OFFSET;
            else Pack = -1;
          }
        }
        else Pack = -1;

		p = agget(g,"model");
        if (Pack >= 0) {
			graph_t*    gc;
			graph_t**   cc;
			int         n_cc;
			int         n_n;
			int         i;
			int         useCircuit;

			useCircuit = (p && (streq(p,"circuit")));
			cc = ccomps (g, &n_cc, cc_pfx);

			for (i = 0; i < n_cc; i++) {
				gc = cc[i];
				nodeInduce (gc);
    			n_n = scan_graph(gc);
				if (useCircuit) circuit_model(gc,n_n);
				else shortest_path(gc, n_n);
				initial_positions(gc, n_n);
    			diffeq_model(gc, n_n);
    			solve_model(gc, n_n);
    			final_energy(gc, n_n); 
    			adjustNodes(gc);
			}
			packGraphs (n_cc, cc, 0, Pack, 0);
  			neato_compute_bb (g);
   			spline_edges(g);
        }
        else {
    		nG = scan_graph(g);
			if (p && (streq(p,"circuit"))) {
              if (!circuit_model(g,nG)) {
                fprintf (stderr, "Warning: graph %s is disconnected. In this case, the circuit model\n", g->name);
                fprintf (stderr, "is undefined and neato is reverting to the shortest path model.\n");
                fprintf (stderr, "Alternatively, consider running neato using -Gpack=true or decomposing\n");
                fprintf (stderr, "the graph into connected components.\n");

			    shortest_path(g, nG);
              }
            }
			else shortest_path(g, nG);
    		initial_positions(g, nG);
    		diffeq_model(g, nG);
    		solve_model(g, nG);
    		final_energy(g, nG); 
    		adjustNodes(g);
   			spline_edges(g);
		}
	}
	dotneato_postprocess(g, neato_nodesize);
}   
