/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/
#pragma prototyped

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include	"render.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

static int graphviz_errors;

void dotneato_initialize(int argc, char** argv)
{
	char		*rest,c;
	int			i,nfiles;

	aginit();
	nfiles = 0;
	for (i = 1; i < argc; i++)
		if (argv[i][0] != '-') nfiles++;
	Files = N_NEW(nfiles + 1, char *);
	nfiles = 0;
	Output_lang_count = 0;
	Output_file_count = 0;
	CmdName = argv[0];
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			rest = &(argv[i][2]);
			switch (c = argv[i][1]) {
				case 'G': global_def(rest,agraphattr); break;
				case 'N': global_def(rest,agnodeattr); break;
				case 'E': global_def(rest,agedgeattr); break;
				case 'T': 
					if (Output_lang_count < MAX_PRODUCTS)
						Output_langs[Output_lang_count++] = rest[0]?rest:argv[++i];
					break;
				case 'V':
					fprintf(stderr,"%s version %s (%s)\n",
						Info[0], Info[1], Info[2]);
					exit (0);
					break;
				case 'l':
					use_library(rest[0]?rest:(*argv[i+1]!='-'?argv[++i]:NULL));
					break;
				case 'n': 
                    Nop = TRUE; 
                    if (isdigit(*rest)) Nop = atoi (rest);
                    break;
				case 'o':
					if (Output_file_count < MAX_PRODUCTS)
						Output_files[Output_file_count++] = rest[0]?rest:argv[++i];
					break;
				case 's':
                    PSinputscale = (rest[0]?atof(rest):POINTS_PER_INCH);
                    break;
				case 'v': 
                    Verbose = TRUE; 
                    if (isdigit(*rest)) Verbose = atoi (rest);
                    break;
				case 'x': Reduce = TRUE; break;
				case 'y': y_invert = TRUE; break;
				default:
					fprintf(stderr,"%s: option -%c unrecognized\n",CmdName,c);
			}
		}
		else
		Files[nfiles++] = argv[i];
	}
	/* if no -Txxx, then set default format */
        if (Output_lang_count == 0) {
                Output_lang_count++;
                Output_langs[0] = "dot";
        }
	/* if less -o than -T set remaining -o to NULL to get stdout */
	while (Output_file_count<Output_lang_count) 
		Output_files[Output_file_count++] = NULL;

	/* need to set Output_lang now based on the first -Txxx,
	 * so that CodeGen->textsize picks up an appropriate routine
	 * otherwise node sizing can be wrong for any or all of the products.
	 */
	Output_lang = lang_select(Output_langs[0]);
	/* set persistent attributes here (if not already set from command line options) */ 
	if (! (agfindattr(agprotograph()->proto->n, "label")))
		agnodeattr(NULL,"label",NODENAME_ESC);
}

void global_def(char* dcl, attrsym_t*((*dclfun)(Agraph_t*,char*,char*)))
{
	char		*p,*rhs = "true";
	if ((p = strchr(dcl,'='))) { *p++ = '\0'; rhs = p; }
	(void) dclfun(NULL,dcl,rhs);
	agoverride (1);
}

void getdoubles2pt(graph_t* g, char* name, point* result)
{
    char        *p;
    int         i;
    double      xf,yf;

    if ((p = agget(g,name))) {
        i = sscanf(p,"%lf,%lf",&xf,&yf);
        if ((i > 1) && (xf > 0) && (yf > 0)) {
            result->x = POINTS(xf);
            result->y = POINTS(yf);
        }
    }
}

void getdouble(graph_t* g, char* name, double* result)
{
    char        *p;
    double      f;

    if ((p = agget(g,name))) {
        if (sscanf(p,"%lf",&f) >= 1) *result = f;
    }
}

FILE *
next_input_file(void)
{
	static int ctr = 0;
	FILE	*rv = NULL;

	if (Files[0] == NULL) {
		if (ctr++ == 0) rv = stdin;
	}
	else {
		rv = NULL;
		while (Files[ctr]) {
			if ((rv = fopen(Files[ctr++],"r"))) break;
			else {
              fprintf(stderr,"%s: can't open %s\n",CmdName,Files[ctr-1]);
              graphviz_errors++;
            }
		}
	}
    if (rv) agsetfile (Files[0]?Files[ctr-1]:"<stdin>");
	return rv;
}

graph_t* next_input_graph(void)
{
	graph_t		*g;
	static FILE	*fp;

	if (fp == NULL) fp = next_input_file();
	g = NULL;

	while (fp != NULL) {
		if ((g = agread(fp))) break;
		fp = next_input_file();
	}
	return g;
}

void graph_init(graph_t* g)
{
	/* node_t		*n; */
	/* edge_t		*e; */

		/* initialize the graph */
		init_ugraph(g);

		/* initialize nodes */
		N_height = agfindattr(g->proto->n,"height");
		N_width = agfindattr(g->proto->n,"width");
		N_shape = agfindattr(g->proto->n,"shape");
		N_color = agfindattr(g->proto->n,"color");
		N_fillcolor = agfindattr(g->proto->n,"fillcolor");
		N_style = agfindattr(g->proto->n,"style");
		N_fontsize = agfindattr(g->proto->n,"fontsize");
		N_fontname = agfindattr(g->proto->n,"fontname");
		N_fontcolor = agfindattr(g->proto->n,"fontcolor");
		N_label = agfindattr(g->proto->n,"label");
		N_showboxes = agfindattr(g->proto->n,"showboxes");
			/* attribs for polygon shapes */
		N_sides = agfindattr(g->proto->n,"sides");
		N_peripheries = agfindattr(g->proto->n,"peripheries");
		N_skew = agfindattr(g->proto->n,"skew");
		N_orientation = agfindattr(g->proto->n,"orientation");
		N_distortion = agfindattr(g->proto->n,"distortion");
		N_fixed = agfindattr(g->proto->n,"fixedsize");
		N_layer = agfindattr(g->proto->n,"layer");
		N_group = agfindattr(g->proto->n,"group");
		N_comment = agfindattr(g->proto->n,"comment");
		N_vertices = agfindattr(g->proto->n,"vertices");
		N_z = agfindattr(g->proto->n,"z");

		/* initialize edges */
		E_weight = agfindattr(g->proto->e,"weight");
		E_color = agfindattr(g->proto->e,"color");
		E_fontsize = agfindattr(g->proto->e,"fontsize");
		E_fontname = agfindattr(g->proto->e,"fontname");
		E_fontcolor = agfindattr(g->proto->e,"fontcolor");
		E_label = agfindattr(g->proto->e,"label");
		E_label_float = agfindattr(g->proto->e,"labelfloat");
        /* vladimir */
		E_dir = agfindattr(g->proto->e,"dir");
		E_arrowhead = agfindattr(g->proto->e,"arrowhead");
		E_arrowtail = agfindattr(g->proto->e,"arrowtail");
		E_headlabel = agfindattr(g->proto->e,"headlabel");
		E_taillabel = agfindattr(g->proto->e,"taillabel");
		E_labelfontsize = agfindattr(g->proto->e,"labelfontsize");
		E_labelfontname = agfindattr(g->proto->e,"labelfontname");
		E_labelfontcolor = agfindattr(g->proto->e,"labelfontcolor");
		E_labeldistance = agfindattr(g->proto->e,"labeldistance");
		E_labelangle = agfindattr(g->proto->e,"labelangle");
        /* end vladimir */
		E_minlen = agfindattr(g->proto->e,"minlen");
		E_showboxes = agfindattr(g->proto->e,"showboxes");
		E_style = agfindattr(g->proto->e,"style");
		E_decorate = agfindattr(g->proto->e,"decorate");
		E_arrowsz = agfindattr(g->proto->e,"arrowsize");
		E_constr = agfindattr(g->proto->e,"constraint");
		E_layer = agfindattr(g->proto->e,"layer");
		E_comment = agfindattr(g->proto->e,"comment");
		E_tailclip = agfindattr(g->proto->e,"tailclip");
		E_headclip = agfindattr(g->proto->e,"headclip");
}

void init_ugraph(graph_t* g)
{
	char		*p;
	double		xf;
	static char *rankname[] = {"local","global","none",NULL};
	static int	rankcode[] =  {LOCAL, GLOBAL, NOCLUST, LOCAL};
	
	GD_drawing(g) = NEW(layout_t);

	/* set this up fairly early in case any string sizes are needed */
	if ((p = agget(g,"fontpath")) || (p = getenv("DOTFONTPATH"))) {
		/* overide GDFONTPATH in local environment if dot
		 * wants its own */
#ifdef HAVE_SETENV
		setenv("GDFONTPATH", p, 1);
#else
		static char *buf=0;

		buf=realloc(buf,strlen("GDFONTPATH=")+strlen(p)+1);
		strcpy(buf,"GDFONTPATH=");
		strcat(buf,p);
		putenv(buf);
#endif
	}

	GD_drawing(g)->quantum = late_double(g,agfindattr(g,"quantum"),0.0,0.0);
 	GD_drawing(g)->font_scale_adj = 1.0;

    /* setting rankdir=LR is only defined in dot,
     * but having it set causes shape code and others to use it. 
     * The result is confused output, so we turn it off unless requested.
     */
	if (UseRankdir)
		GD_left_to_right(g) = ((p = agget(g,"rankdir")) && streq(p,"LR"));
	else
		GD_left_to_right(g) = FALSE;
	do_graph_label(g);
	xf = late_double(g,agfindattr(g,"nodesep"),DEFAULT_NODESEP,MIN_NODESEP);
	GD_nodesep(g) = POINTS(xf);

	p = late_string(g,agfindattr(g,"ranksep"),NULL);
	if (p) {
			if (sscanf(p,"%lf",&xf) == 0) xf = DEFAULT_RANKSEP;
			else {if (xf < MIN_RANKSEP) xf = MIN_RANKSEP;}
			if (strstr(p,"equally")) GD_exact_ranksep(g) = TRUE;
	}
	else xf = DEFAULT_RANKSEP;
	GD_ranksep(g) = POINTS(xf);

	GD_showboxes(g) = late_int(g,agfindattr(g,"showboxes"),0,0);

	Epsilon = .0001 * agnnodes(g);
	getdoubles2pt(g,"size",&(GD_drawing(g)->size));
	getdoubles2pt(g,"page",&(GD_drawing(g)->page));
	getdouble(g,"epsilon",&Epsilon);
	getdouble(g,"nodesep",&Nodesep);
	getdouble(g,"nodefactor",&Nodefactor);

	GD_drawing(g)->centered = mapbool(agget(g,"center"));
	if ((p = agget(g,"rotate"))) GD_drawing(g)->landscape = (atoi(p) == 90);
	else {		/* today we learned the importance of backward compatibilty */
		if ((p = agget(g,"orientation")))
			GD_drawing(g)->landscape = ((p[0] == 'l') || (p[0] == 'L'));
	}

	p = agget(g,"clusterrank");
	CL_type = maptoken(p,rankname,rankcode);
	p = agget(g,"concentrate");
	Concentrate = mapbool(p);
	
	Nodesep = 1.0;
	Nodefactor = 1.0;
	Initial_dist = MYHUGE;
}

void free_ugraph(graph_t* g)
{
	free(GD_drawing(g));
	GD_drawing(g) = NULL;
}

void do_graph_label(graph_t* g)
{
    char    *p, *pos;
	int		pos_ix;

	/* it would be nice to allow multiple graph labels in the future */
    if ((p = agget(g,"label"))) {
        GD_label(g) = make_label(strdup(p),
		late_double(g,agfindattr(g,"fontsize"),DEFAULT_FONTSIZE,MIN_FONTSIZE),
		late_nnstring(g,agfindattr(g,"fontname"),DEFAULT_FONTNAME),
		late_nnstring(g,agfindattr(g,"fontcolor"),DEFAULT_COLOR),g);

		pos = agget(g,"labelloc");
		if (!GD_left_to_right(g)) {
			if (!pos || (pos[0] != 'b')) pos_ix = TOP_IX;
			else pos_ix = BOTTOM_IX;
			GD_border(g)[pos_ix] = cvt2pt(GD_label(g)->dimen);
		}
		else {
			/* when rotated, the labels will be restored to TOP or BOTTOM */
			if (!pos || (pos[0] != 'b')) pos_ix = RIGHT_IX;
			else pos_ix = LEFT_IX;
			GD_border(g)[pos_ix].x = GD_label(g)->dimen.y;
			GD_border(g)[pos_ix].y = GD_label(g)->dimen.x;
		}
	}
}

void dotneato_terminate(void)
{
    emit_eof();
    exit(graphviz_errors + agerrors());
}

