/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "bar_exif.h"

#include "exif.h"

#include <math.h>

#define EXIF_BAR_SIZE_INCREMENT 48
#define EXIF_BAR_ARROW_SIZE 7

static const gchar *bar_exif_key_list[] = {
	"fCamera",
	"fDateTime",
	"fShutterSpeed",
	"fAperture",
	"ExposureProgram",
	"fExposureBias",
	"fISOSpeedRating",
	"fFocalLength",
	"fSubjectDistance",
	"MeteringMode",
	"fFlash",
	"LightSource",
	"fResolution",
	"Orientation",
	"ImageDescription",
	"Copyright"
};

#define bar_exif_key_count (sizeof(bar_exif_key_list) / sizeof(gchar *))


/*
 *-------------------------------------------------------------------
 * table util
 *-------------------------------------------------------------------
 */

static GtkWidget *table_add_line(GtkWidget *table, gint x, gint y,
				 const gchar *description, const gchar *text)
{
	GtkWidget *label;
	gchar *buf;

	buf = g_strconcat((description) ? description : "fixme", ":", NULL);
	if (!text) text = "";

	label = gtk_label_new(buf);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.0);
	gtk_table_attach(GTK_TABLE(table), label,
			 x, x + 1, y, y + 1,
			 GTK_FILL, GTK_FILL,
			 2, 2);
	gtk_widget_show(label);

	label = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_table_attach(GTK_TABLE(table), label,
			 x + 1, x + 2, y, y + 1,
			 GTK_FILL, GTK_FILL,
			 2, 2);
	gtk_widget_show(label);

	g_free(buf);
	return label;
}

/*
 *-------------------------------------------------------------------
 * EXIF bar
 *-------------------------------------------------------------------
 */

typedef struct _ExifBar ExifBar;
struct _ExifBar
{
	GtkWidget *vbox;
	GtkWidget *scrolled;
	GtkWidget *table;
	GtkWidget *advanced_scrolled;
	GtkWidget *listview;
	GtkWidget **labels;

	gchar *path;
};

static gchar *bar_exif_validate_text(gchar *text)
{
	if (text && !g_utf8_validate(text, strlen(text), NULL))
		{
		gchar *tmp = text;
		text = g_convert(tmp, strlen(tmp), "UTF-8", "ISO-8859-1", NULL, NULL, NULL);
		g_free(tmp);
		}
	return text;
}

static void bar_exif_sensitive(ExifBar *eb, gint enable)
{
	gtk_widget_set_sensitive(eb->table, enable);
	if (eb->advanced_scrolled) gtk_widget_set_sensitive(eb->advanced_scrolled, enable);
}

static void bar_exif_update(ExifBar *eb)
{
	ExifData *exif;
	gint len, i;

	exif = exif_read(eb->path);

	if (!exif)
		{
		bar_exif_sensitive(eb, FALSE);
		return;
		}

	bar_exif_sensitive(eb, TRUE);

	if (GTK_WIDGET_VISIBLE(eb->scrolled))
		{
		len = bar_exif_key_count;
		for (i = 0; i < len; i++)
			{
			gchar *text;
			text = exif_get_data_as_text(exif, bar_exif_key_list[i]);
			text = bar_exif_validate_text(text);
			gtk_label_set_text(GTK_LABEL(eb->labels[i]), text);
			g_free(text);
			}
		}

	if (eb->advanced_scrolled && GTK_WIDGET_VISIBLE(eb->advanced_scrolled))
		{
		GtkListStore *store;
		GtkTreeIter iter;
		GList *work;
		
		store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(eb->listview)));
		gtk_list_store_clear(store);

		work = exif->items;
		while (work)
			{
			ExifItem *item;
			gchar *tag;
			const gchar *tag_name;
			gchar *text;
			const gchar *format;
			gchar *elements;
			const gchar *description;

			item = work->data;
			work = work->next;

			tag = g_strdup_printf("0x%04x", item->tag);
			tag_name = exif_item_get_tag_name(item);
			format = exif_item_get_format_name(item, TRUE);
			text = exif_item_get_data_as_text(item);
			text = bar_exif_validate_text(text);
			elements = g_strdup_printf("%d", item->elements);
			description = exif_item_get_description(item);
			if (!description) description = "";
			gtk_list_store_append(store, &iter);
			gtk_list_store_set(store, &iter, 0, tag, 1, tag_name, 2, text,
					   3, format, 4, elements, 5, description, -1);
			g_free(tag);
			g_free(text);
			g_free(elements);
			}
		}

	exif_free(exif);
}

static void bar_exif_clear(ExifBar *eb)
{
	gint len;
	gint i;

	if (!GTK_WIDGET_SENSITIVE(eb->labels[0])) return;

	len = bar_exif_key_count;
	for (i = 0; i < len; i++)
		{
		gtk_label_set_text(GTK_LABEL(eb->labels[i]), "");
		}

	if (eb->listview)
		{
		GtkListStore *store;

		store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(eb->listview)));
		gtk_list_store_clear(store);
		}
}

void bar_exif_set(GtkWidget *bar, const gchar *path)
{
	ExifBar *eb;

	eb = g_object_get_data(G_OBJECT(bar), "bar_exif_data");
	if (!eb) return;

	/* store this, advanced view toggle needs to reload data */
	g_free(eb->path);
	eb->path = g_strdup(path);

	bar_exif_clear(eb);
	bar_exif_update(eb);
}

static void bar_exif_add_column(GtkWidget *listview, const gchar *title, gint n)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, title);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, renderer, "text", n);
	gtk_tree_view_append_column(GTK_TREE_VIEW(listview), column);
}

static void bar_exif_advanced_build_view(ExifBar *eb)
{
	GtkListStore *store;

	if (eb->listview) return;

	store = gtk_list_store_new(6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
				      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	eb->listview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	g_object_unref(store);

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(eb->listview), TRUE);

	bar_exif_add_column(eb->listview, _("Tag"), 0);
	bar_exif_add_column(eb->listview, _("Name"), 1);
	bar_exif_add_column(eb->listview, _("Value"), 2);
	bar_exif_add_column(eb->listview, _("Format"), 3);
	bar_exif_add_column(eb->listview, _("Elements"), 4);
	bar_exif_add_column(eb->listview, _("Description"), 5);

	eb->advanced_scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(eb->advanced_scrolled),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(eb->vbox), eb->advanced_scrolled, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(eb->advanced_scrolled), eb->listview);
	gtk_widget_show(eb->listview);
}

static void bar_exif_advanced_cb(GtkWidget *widget, gpointer data)
{
	ExifBar *eb = data;
	gint advanced;

	advanced = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	if (advanced)
		{
		gtk_widget_hide(eb->scrolled);
		bar_exif_advanced_build_view(eb);
		gtk_widget_show(eb->advanced_scrolled);
		}
	else
		{
		gtk_widget_hide(eb->advanced_scrolled);
		gtk_widget_show(eb->scrolled);
		}

	bar_exif_update(eb);
}

gint bar_exif_is_advanced(GtkWidget *bar)
{
	ExifBar *eb;

	eb = g_object_get_data(G_OBJECT(bar), "bar_exif_data");
	if (!eb) return FALSE;

	return (eb->advanced_scrolled && GTK_WIDGET_VISIBLE(eb->advanced_scrolled));
}

void bar_exif_close(GtkWidget *bar)
{
	ExifBar *eb;

	eb = g_object_get_data(G_OBJECT(bar), "bar_exif_data");
	if (!eb) return;

	gtk_widget_destroy(eb->vbox);
}

static void bar_exif_size(ExifBar *eb, gint val)
{
	gint size;

	size = eb->vbox->allocation.width;
	size = CLAMP(size + val, EXIF_BAR_SIZE_INCREMENT * 2, EXIF_BAR_SIZE_INCREMENT * 16);

	gtk_widget_set_size_request(eb->vbox, size, -1);
}

static void bar_exif_larger(GtkWidget *widget, gpointer data)
{
	ExifBar *eb = data;

	bar_exif_size(eb, EXIF_BAR_SIZE_INCREMENT);
}

static void bar_exif_smaller(GtkWidget *widget, gpointer data)
{
	ExifBar *eb = data;

	bar_exif_size(eb, -EXIF_BAR_SIZE_INCREMENT);
}

static void bar_exif_destroy(GtkWidget *widget, gpointer data)
{
	ExifBar *eb = data;

	g_free(eb->labels);
	g_free(eb->path);
	g_free(eb);
}

GtkWidget *bar_exif_new(gint show_title, const gchar *path, gint advanced)
{
	ExifBar *eb;
	GtkWidget *table;
	GtkWidget *hbox;
	GtkWidget *button;
	gint len;
	gint i;

	eb = g_new0(ExifBar, 1);

	eb->labels = g_new0(GtkWidget *, bar_exif_key_count);

	eb->vbox = gtk_vbox_new(FALSE, 0);
	g_object_set_data(G_OBJECT(eb->vbox), "bar_exif_data", eb);
	g_signal_connect(G_OBJECT(eb->vbox), "destroy",
			 G_CALLBACK(bar_exif_destroy), eb);

	if (show_title)
		{
		GtkWidget *box;
		GtkWidget *label;
		GtkWidget *button;
		GtkWidget *arrow;

		box = gtk_hbox_new(FALSE, 0);

		label = gtk_label_new(_("Exif"));
		gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 0);
		gtk_widget_show(label);

		button = gtk_button_new();
		gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
		g_signal_connect(G_OBJECT(button), "clicked",
				 G_CALLBACK(bar_exif_smaller), eb);
		gtk_box_pack_end(GTK_BOX(box), button, FALSE, FALSE, 0);
		arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_NONE);
		gtk_widget_set_size_request(arrow, EXIF_BAR_ARROW_SIZE, EXIF_BAR_ARROW_SIZE);
		gtk_container_add(GTK_CONTAINER(button), arrow);
		gtk_widget_show(arrow);
		gtk_widget_show(button);

		button = gtk_button_new();
		gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
		g_signal_connect(G_OBJECT(button), "clicked",
				 G_CALLBACK(bar_exif_larger), eb);
		gtk_box_pack_end(GTK_BOX(box), button, FALSE, FALSE, 0);
		arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_NONE);
		gtk_widget_set_size_request(arrow, EXIF_BAR_ARROW_SIZE, EXIF_BAR_ARROW_SIZE);
		gtk_container_add(GTK_CONTAINER(button), arrow);
		gtk_widget_show(arrow);
		gtk_widget_show(button);

		gtk_box_pack_start(GTK_BOX(eb->vbox), box, FALSE, FALSE, 0);
		gtk_widget_show(box);
		}

	table = gtk_table_new(2, 9, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);

	eb->table = table;

	len = bar_exif_key_count;
	for (i = 0; i < len; i++)
		{
		const gchar *text;

		text = exif_get_description_by_key(bar_exif_key_list[i]);
		eb->labels[i] = table_add_line(table, 0, i, text, NULL);
		}

	eb->scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(eb->scrolled),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(eb->scrolled), table);
	gtk_widget_show(table);

	gtk_box_pack_start(GTK_BOX(eb->vbox), eb->scrolled, TRUE, TRUE, 0);
#if 0
	gtk_widget_show(eb->scrolled);
#endif

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(eb->vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = gtk_check_button_new_with_label(_("Advanced view"));
	if (advanced) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	g_signal_connect(G_OBJECT(button), "toggled",
			 G_CALLBACK(bar_exif_advanced_cb), eb);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 5);
	gtk_widget_show(button);

	eb->advanced_scrolled = NULL;
	eb->listview = NULL;

	if (advanced)
		{
		bar_exif_advanced_build_view(eb);
		gtk_widget_show(eb->advanced_scrolled);
		}
	else
		{
		gtk_widget_show(eb->scrolled);
		}

	eb->path = g_strdup(path);
	bar_exif_update(eb);

	return eb->vbox;
}

