/*
 * radauth.h
 *
 * RADIUS protocol authenticator modules for GNU Gatekeeper. 
 * H.235 based and alias based authentication schemes are supported.
 * Please see docs/radauth.txt for more details.
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: radauth.h,v $
 * Revision 1.1.2.15  2004/01/07 20:37:26  zvision
 * RRQ endpoint alias control (add/remove) through RadAuth/RadAliasAuth modules
 *
 * Revision 1.1.2.14  2003/12/21 12:20:38  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.13  2003/12/02 12:47:37  zvision
 * Q.931 Setup authenticator now returns (and documents) Q.931 cause values instead of H225_ReleaseCompleteReason
 *
 * Revision 1.1.2.12  2003/11/22 13:43:48  zvision
 * Q.931 Setup Check now accepts const callptr to prevent it from modifications
 *
 * Revision 1.1.2.11  2003/11/18 23:38:51  zvision
 * Q.931 Setup authentication optimized
 *
 * Revision 1.1.2.10  2003/09/30 11:08:16  zvision
 * Fixed non-working Q.931 authenticator due to previous Check(...) signature
 * changes. Many thanks to Oleg Ustinov!
 *
 * Revision 1.1.2.9  2003/09/24 10:19:44  zvision
 * Call duration limit for registered endpoints (through ARQ authenticators)
 * VS: ----------------------------------------------------------------------
 *
 * Revision 1.1.2.8  2003/08/25 12:18:35  zvision
 * Added h323-ivr-out attribute with an alias list (thanks Mark Lipscombe)
 *
 * Revision 1.1.2.7  2003/07/31 13:09:15  zvision
 * Added Q.931 Setup message authentication and call duration limit feature
 *
 * Revision 1.1.2.6  2003/07/07 12:02:55  zvision
 * Improved H.235 handling.
 *
 * Revision 1.1.2.5  2003/05/28 13:25:19  zvision
 * Added alias based authentication (RadAliasAuth)
 *
 * Revision 1.1.2.4  2003/05/26 23:08:18  zvision
 * New OnSend and OnReceive hooks.
 * LocalInterface config parameter Introduced.
 *
 * Revision 1.1.2.3  2003/05/13 17:48:43  zvision
 * Removed acctPort. New includeFramedIP feature.
 *
 * Revision 1.1.2.2  2003/04/29 14:56:26  zvision
 * Added H.235 capability matching
 *
 * Revision 1.1.2.1  2003/04/23 20:16:25  zvision
 * Initial revision
 *
 */
#ifndef __RADAUTH_H
#define __RADAUTH_H
#ifdef HAS_RADIUS

#include <ptlib.h>
#include "gkauth.h"
#include "radproto.h"

/**
 * Gatekeeper authenticator module for RADIUS protocol.
 * Currently it supports user authentication through
 * CATs (Cisco Access Tokens) carried inside RRQ and ARQ
 * RAS messages. If your software does not support CATs,
 * please take a look at OpenH323 H235AuthCAT authenticator class
 * - it provides an implementation for CATs.
 */
class RadAuth : public GkAuthenticator 
{
public:
	/** Create GkAuthenticator for RADIUS protocol
	*/
	RadAuth( 
		/// authenticator settings
		PConfig* cfg, 
		/// authenticator name from Gatekeeper::Auth section
		const char* authName 
		);
		
	/// Destroy the authenticator
	virtual ~RadAuth();
	
protected:		
	/** Hook for adding/modifying pdu before it is sent.
		It can be used to add custom attributes, for example.
		
		@return
		TRUE if PDU should be sent, FALSE to reject RRQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		const H225_RegistrationRequest& rrq, /// RRQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for adding/modifying pdu before it is sent.
		It can be used to add custom attributes, for example.
		
		@return
		TRUE if PDU should be sent, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		const H225_AdmissionRequest& rrq, /// ARQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject RRQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// received PDU 
		H225_RegistrationRequest& rrq, /// RRQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		This function is now DEPRECIATED - use the version with
		durationLimit parameter.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// received PDU
		const H225_AdmissionRequest& arq, /// ARQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);
		
	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// PDU received
		const H225_AdmissionRequest& arq, /// ARQ being processed
		unsigned& rejectReason, /// reject reason on return FALSE
		long& durationLimit /// call duration limit to set (-1 means no limit)
		);
		
private:
	/** Scan the array of 'aliases' for 'id' alias.
	
		@return
		TRUE if 'id' is found in the 'aliases' array.
	*/
	virtual BOOL CheckAliases( 
		/// array of aliases to be searched
		const H225_ArrayOf_AliasAddress& aliases, 
		/// alias to be searched for
		const PString& id 
		) const;
	
	
	/** Authenticate using data from RRQ RAS message.
	
		@return:
		#GkAuthenticator::Status enum# with the result of authentication.
	*/
	virtual int Check(
		/// RRQ RAS message to be authenticated
		H225_RegistrationRequest & rrq, 
		/// reference to the variable, that can be set 
		/// to custom H225_RegistrationRejectReason
		/// if the check fails
		unsigned& rejectReason
		);
		
	/** Authenticate using data from ARQ RAS message.
	
		@return:
		#GkAuthenticator::Status enum# with the result of authentication.
	*/
	virtual int Check(
		/// ARQ nessage to be authenticated
		const H225_AdmissionRequest &, 
		/// reference to the variable, that can be set 
		/// to custom H225_AdmissionRejectReason
		/// if the check fails
		unsigned& rejectReason,
		/// call duration limit to be set (-1 for no duration limit)
		long& callDurationLimit
		); 
		
//	virtual int Check(const H225_UnregistrationRequest &, unsigned& );
//	virtual int Check(const H225_GatekeeperRequest &, unsigned& );
//	virtual int Check(const H225_BandwidthRequest &, unsigned& );
//	virtual int Check(const H225_DisengageRequest &, unsigned& );
//	virtual int Check(const H225_LocationRequest &, unsigned& );
//	virtual int Check(const H225_InfoRequest &, unsigned& );

	/* No copy constructor allowed */
	RadAuth( const RadAuth& );
	/* No operator= allowed */
	RadAuth& operator=( const RadAuth& );
	
private:
	/// array of configured RADIUS server names
	PStringArray radiusServers;
	/// shared secret for gk client<->RADIUS server authorization
	PString sharedSecret;
	/// default port that will be used for sending RADIUS auth
	/// requests
	WORD authPort;
	/// base port number for UDP client socket allocation
	WORD portBase;
	/// max port number for UDP client socket allocation
	WORD portMax;
	/// timeout (ms) for a single RADIUS request
	unsigned requestTimeout;
	/// timeout (ms) for RADIUS requests IDs to be unique
	unsigned idCacheTimeout;
	/// timeout (ms) for unused sockets to be deleted
	unsigned socketDeleteTimeout;
	/// how many times to transmit a single request (1==no retransmission)
	/// to a single RADIUS server
	unsigned numRequestRetransmissions;
	/// retransmission fashion: 
	/// 	FALSE - do #numRequestRetransmissions# for server A,
	///				then do #numRequestRetransmissions# for server B, etc.
	///		TRUE - transmit request to server A, then to server B, etc.
	///				the whole procedure repeat #numRequestRetransmissions# times
	BOOL roundRobin;
	/// if TRUE Cisco VSAs are appended to the RADIUS packets
	BOOL appendCiscoAttributes;
	/// if TRUE endpoint IP is placed inside Framed-IP-Address attribute
	BOOL includeFramedIp;
	/// if TRUE an h323-ivr-out attribute will be sent with every alias
	/// found inside RRQ.m_terminalAlias
	BOOL includeTerminalAliases;
	/// Local interface RADIUS client should be bound to (multihomed hosts)
	PString localInterface;	
	/// IP address of the local interface
	PIPSocket::Address localInterfaceAddr;
	/// NAS identifier (GK name)
	PString NASIdentifier;
	/// RADIUS protocol client class associated with this authenticator
	RadiusClient* radiusClient;
	/// OID (Object Identifier) for CAT alghoritm
	static PString OID_CAT;
};

/** RADIUS Alias Authentication module.
	It authenticates endpoints/calls using non-H.235 
	attributes (alias,address,etc).
*/
class RadAliasAuth : public GkAuthenticator 
{
public:
	/** Create GkAuthenticator for RADIUS Alias authenticator
	*/
	RadAliasAuth( 
		/// authenticator settings
		PConfig* cfg, 
		/// authenticator name from Gatekeeper::Auth section
		const char* authName 
		);
		
	/// Destroy the authenticator
	virtual ~RadAliasAuth();
	
protected:		
	/** Hook for adding/modifying pdu before it is sent.
		It can be used to add custom attributes, for example.
		
		@return
		TRUE if PDU should be sent, FALSE to reject RRQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		const H225_RegistrationRequest& rrq, /// RRQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for adding/modifying pdu before it is sent.
		It can be used to add custom attributes, for example.
		
		@return
		TRUE if PDU should be sent, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		const H225_AdmissionRequest& rrq, /// ARQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for adding/modifying pdu before it is sent.
		It can be used to add custom attributes, for example.
		
		@return
		TRUE if PDU should be sent, FALSE to reject Setup
		(releaseCompleteCode can be set to indicate a particular reason).
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		const Q931& q931pdu, /// Q.931 Setup message being processed
		const H225_Setup_UUIE& setup, /// H.225 Setup message being processed
		unsigned& releaseCompleteCause, /// Q.931 cause on return FALSE
		long& durationLimit /// call duration limit to set (-1 means no limit)
		);
		
	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject RRQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// received PDU 
		H225_RegistrationRequest& rrq, /// RRQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);

	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		This function is now DEPRECIATED - use the version with
		durationLimit parameter.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// PDU received
		const H225_AdmissionRequest& arq, /// ARQ being processed
		unsigned& rejectReason /// reject reason on return FALSE
		);
		
	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject ARQ
		(rejectReason can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// PDU received
		const H225_AdmissionRequest& arq, /// ARQ being processed
		unsigned& rejectReason, /// reject reason on return FALSE
		long& durationLimit /// call duration limit to set (-1 means no limit)
		);
		
	/** Hook for processing pdu after it is received.
		It can be used to process custom attributes, for example.
		
		@return
		TRUE if PDU should be accepted, FALSE to reject Setup
		(releaseCompleteCode can be set to indicate a particular reason).
	*/
	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// PDU received
		const Q931& q931pdu, /// Q.931 Setup message being processed
		const H225_Setup_UUIE& setup, /// H.225 Setup message being processed
		unsigned& releaseCompleteCause, /// Q.931 cause on return FALSE
		long& durationLimit /// call duration limit to set (-1 means no limit)
		);
		
private:
	/** Scan the array of 'aliases' for 'id' alias.
	
		@return
		TRUE if 'id' is found in the 'aliases' array.
	*/
	virtual BOOL CheckAliases( 
		/// array of aliases to be searched
		const H225_ArrayOf_AliasAddress& aliases, 
		/// alias to be searched for
		const PString& id 
		) const;
	
	
	/** Authenticate using data from RRQ RAS message.
	
		@return:
		#GkAuthenticator::Status enum# with the result of authentication.
	*/
	virtual int Check(
		/// RRQ RAS message to be authenticated
		H225_RegistrationRequest & rrq, 
		/// reference to the variable, that can be set 
		/// to custom H225_RegistrationRejectReason
		/// if the check fails
		unsigned& rejectReason
		);
		
	/** Authenticate using data from ARQ RAS message.
	
		@return:
		#GkAuthenticator::Status enum# with the result of authentication.
	*/
	virtual int Check(
		/// ARQ message to be authenticated
		const H225_AdmissionRequest &, 
		/// reference to the variable, that can be set 
		/// to custom H225_AdmissionRejectReason
		/// if the check fails
		unsigned& rejectReason,
		/// call duration limit to be set (-1 for no duration limit)
		long& callDurationLimit
		); 
		
	/** Authenticate/Authorize Setup signalling message.
	
		@return
		e_fail - authentication failed
		e_ok - authenticated with this authenticator
		e_next - authentication could not be determined
	*/
	virtual int Check(
		/// received Q.931 Setup message
		const Q931& q931pdu, 
		/// received H.225 Setup message
		const H225_Setup_UUIE& setup, 
		/// CallRec for the call being authenticated
		const callptr& call,
		/// Q.931 cause to set, if authentication failed
		unsigned& releaseCompleteCause, 
		/// call duration limit to set (-1 for no duration limit)
		long& durationLimit
		);
		
	/* No copy constructor allowed */
	RadAliasAuth( const RadAliasAuth& );
	/* No operator= allowed */
	RadAliasAuth& operator=( const RadAliasAuth& );
	
private:
	/// array of configured RADIUS server names
	PStringArray radiusServers;
	/// shared secret for gk client<->RADIUS server authorization
	PString sharedSecret;
	/// default port that will be used for sending RADIUS auth
	/// requests
	WORD authPort;
	/// base port number for UDP client socket allocation
	WORD portBase;
	/// max port number for UDP client socket allocation
	WORD portMax;
	/// timeout (ms) for a single RADIUS request
	unsigned requestTimeout;
	/// timeout (ms) for RADIUS requests IDs to be unique
	unsigned idCacheTimeout;
	/// timeout (ms) for unused sockets to be deleted
	unsigned socketDeleteTimeout;
	/// how many times to transmit a single request (1==no retransmission)
	/// to a single RADIUS server
	unsigned numRequestRetransmissions;
	/// retransmission fashion: 
	/// 	FALSE - do #numRequestRetransmissions# for server A,
	///				then do #numRequestRetransmissions# for server B, etc.
	///		TRUE - transmit request to server A, then to server B, etc.
	///				the whole procedure repeat #numRequestRetransmissions# times
	BOOL roundRobin;
	/// if TRUE Cisco VSAs are appended to the RADIUS packets
	BOOL appendCiscoAttributes;
	/// if TRUE endpoint IP is placed inside Framed-IP-Address attribute
	BOOL includeFramedIp;
	/// if TRUE an h323-ivr-out attribute will be sent with every alias
	/// found inside RRQ.m_terminalAlias
	BOOL includeTerminalAliases;
	/// local interface RADIUS client should be bound to (multihomed hosts)
	PString localInterface;	
	/// IP address of the local interface
	PIPSocket::Address localInterfaceAddr;
	/// NAS identifier (GK name)
	PString NASIdentifier;
	/// fixed username to be send in RADIUS requests instead of alias
	PString fixedUsername;
	/// fixed password to be send in RADIUS requests instead of duplicating User-Name
	PString fixedPassword;
	/// RADIUS protocol client class associated with this authenticator
	RadiusClient* radiusClient;
};

#endif /* HAS_RADIUS */
#endif /* __RADAUTH_H */
