//////////////////////////////////////////////////////////////////
//
// WaitingARQ.cxx
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Dennis Lazreg
//
//////////////////////////////////////////////////////////////////

#if HAS_WAITARQ

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // one performance warning off
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#define snprintf _snprintf
#endif

#include "WaitingARQ.h"
#include "RasSrv.h"
#include "h323pdu.h"

const char *CTIsection = "CTI::Agents";

WaitingARQlist::WaitingARQlist()
	:
	m_active(false),
	m_sequenceNumber(1),
	m_requestTimeout(CTI_DEFAULT_TIME_OUT)
{
}

void WaitingARQlist::LoadConfig()
{
	WriteLock lock(m_listMutex);
	
	m_requestTimeout = GkConfig()->GetInteger(CTIsection, "RequestTimeout", 0);
	if( m_requestTimeout == 0 ) // for backward compatibility
		m_requestTimeout = GkConfig()->GetInteger(CTIsection, "CTI_Timeout", CTI_DEFAULT_TIME_OUT);
		
	// read the list of aliases for the virtual queue
	PString queues = GkConfig()->GetString (CTIsection, "VirtualQueueAliases" , CTI_VIRTUAL_QUEUE );
	if( queues.IsEmpty() ) // backward compatibility
		queues = GkConfig()->GetString (CTIsection, "VirtualQueue" , CTI_VIRTUAL_QUEUE );
	if( queues.IsEmpty() ) {
		m_callCenterQueues.RemoveAll();
		PTRACE(2,"GK\tVirtual Queue disabled for aliases");
	} else {
		m_callCenterQueues = queues.Tokenise(" ,;\t", FALSE);
		PTRACE(2,"GK\tVirtual Queue enabled for aliases " << queues << " and request timeout " << m_requestTimeout << " seconds");
	}

	// read the list of prefixes for the virtual queue	
	queues = GkConfig()->GetString (CTIsection, "VirtualQueuePrefixes" , "" );
	if( queues.IsEmpty() ) {
		m_callCenterPrefixes.RemoveAll();
		PTRACE(2,"GK\tVirtual Queue disabled for prefixes");
	} else {
		m_callCenterPrefixes = queues.Tokenise(" ,;\t", FALSE);
		PTRACE(2,"GK\tVirtual Queue enabled for prefixes " << queues << " and request timeout " << m_requestTimeout << " seconds");
	}
	
	// read the regular expression for the virtual queue alias match
	queues = GkConfig()->GetString (CTIsection, "VirtualQueueRegex" , "" );
	if( queues.IsEmpty() ) {
		m_callCenterRegex = PString();
		PTRACE(2,"GK\tVirtual Queue disabled for regular expression");
	} else {
		m_callCenterRegex = queues.Trim();
		PTRACE(2,"GK\tVirtual Queue enabled for regex " << m_callCenterRegex << " and request timeout " << m_requestTimeout << " seconds");
	}
	
	m_active = m_callCenterQueues.GetSize() > 0 
		|| m_callCenterPrefixes.GetSize() > 0 || !m_callCenterRegex.IsEmpty();
}

bool WaitingARQlist::IsDestinationVirtualQueue(
	const PString& destination
	) const
{
	if( !m_active )
		return false;
		
	ReadLock lock(m_listMutex);
	
	PINDEX idx;
	
	for (idx = 0; idx < m_callCenterQueues.GetSize(); ++idx)
		if ( m_callCenterQueues[idx] == destination )
			return true;
	for (idx = 0; idx < m_callCenterPrefixes.GetSize(); ++idx)
		if ( destination.Find(m_callCenterPrefixes[idx]) == 0 )
			return true;
	
	return (!m_callCenterRegex.IsEmpty())
		&& Toolkit::MatchRegex(destination,m_callCenterRegex);
}

bool WaitingARQlist::RouteToAlias( 
	const PString& agent, 
	const PString& epId, 
	const PString& callRef 
	)
{
	m_listMutex.StartRead();
	
	iterator Iter = FindByEndpointIdAndCallRef ( epId, atoi(callRef) );
	if (Iter == wArqList.end()) {
		m_listMutex.EndRead();
		PTRACE(2, "RouteToAlias: call not found");
		// TODO: signal status port
		return false;
	} else {
		H225_RasMessage &obj_rr = (*Iter)->m_ARQ;
		H225_AdmissionRequest &obj_arq	= obj_rr;

		PIPSocket::Address rx_addr = (*Iter)->m_rx_addr ;
		H323SetAliasAddress(agent, obj_arq.m_destinationInfo[0]);		 

		RasThread->DoARQ (rx_addr, obj_rr);

		m_listMutex.EndRead();
		
		m_listMutex.StartWrite();
		delete *Iter ;
		wArqList.erase(Iter);
		m_listMutex.EndWrite();
		return true;
	}
}


bool WaitingARQlist::RouteReject(
	const PString& SourceEpId, 
	const PString& CallRef 
	)
{
	m_listMutex.StartRead();
	
	iterator Iter = FindByEndpointIdAndCallRef (SourceEpId, atoi(CallRef) );
	if (Iter == wArqList.end()) {
		m_listMutex.EndRead();
		PTRACE(2, "RouteReject: call not found");
		// TODO: signal status port
		return false;
	} else {
		(*Iter)->DoARJ();
		m_listMutex.EndRead();
		
		m_listMutex.StartWrite();
		delete *Iter;
		wArqList.erase(Iter);
		m_listMutex.EndWrite();
		
		return true;
	}
}

 
bool WaitingARQlist::InsertWaitingARQ (const H225_RasMessage & obj_arq, const PIPSocket::Address & rx_addr, const PString &ccQueue, const endptr & requestingEP)
{
	WriteLock lock(m_listMutex);
	
	const H225_AdmissionRequest & obj_rr = obj_arq;

	int callRef =   obj_rr.m_callReferenceValue;

	iterator Iter = FindByCallRefNumber ( callRef );
	if (Iter == wArqList.end()) {
	  PTRACE(2, "InsertWaitingARQ in WaitingARQlist: " << (unsigned)callRef );
 		wArqList.push_back (new WaitingARQ (m_sequenceNumber++, obj_arq, rx_addr , ccQueue , requestingEP ));
		return true;
	} else {
		 PString  epId   = (*Iter)->getEndPointIdentifier () ;
		 PString  cq		 = (*Iter)->getAgentQueue () ; 
		 PTRACE(2, "InsertWaitingARQ in WaitingARQlist: Already inserted with callRef: " << callRef << " epId=" << epId << " Virtual Queue:" << cq );
		 return false ;
	}
}

bool WaitingARQlist::IsWaitingARQ (const H225_AdmissionRequest& obj_arq)
{
	if( !m_active )
		return false;
		
	ReadLock lock(m_listMutex);
	
	const int callRef = obj_arq.m_callReferenceValue;
	iterator Iter = FindByEndpointIdAndCallRef( 
		obj_arq.m_endpointIdentifier, callRef 
		);

	return Iter != wArqList.end() 
		&& ((const H225_AdmissionRequest&)((*Iter)->m_ARQ)).m_requestSeqNum.GetValue() == obj_arq.m_requestSeqNum.GetValue();
}
 
 
void WaitingARQlist::CheckWaitingARQ()
{
	if( m_active ) {
		WriteLock lock(m_listMutex);
		iterator Iter = find_if(wArqList.begin(), wArqList.end(), not1(bind2nd(mem_fun(&WaitingARQ::IsTimeOut), m_requestTimeout)));
		for_each(wArqList.begin(), Iter, mem_fun(&WaitingARQ::DoARJ) );
		for_each(wArqList.begin(), Iter, deleteobj<WaitingARQ>() );
		wArqList.erase(wArqList.begin(), Iter);
	}
}

#endif // HAS_WAITARQ
