//////////////////////////////////////////////////////////////////
//
// bookkeeping for RAS-Server in H.323 gatekeeper
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
//  990500	initial version (Xiang Ping Chen, Rajat Todi, Joe Metzger)
//  990600	ported to OpenH323 V. 1.08 (Jan Willamowius)
//  991003	switched to STL (Jan Willamowius)
//  000215	call removed from table when <=1 ep remains; marked with "towi*1" (towi)
//
//////////////////////////////////////////////////////////////////


#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // one performance warning off
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#define snprintf	_snprintf
#endif

#include <time.h>
#include <ptlib.h>
#include <h323pdu.h>
#include "ANSI.h"
#include "h323util.h"
#include "Toolkit.h"
#include "SoftPBX.h"
#include "RasSrv.h"
#include "GkClient.h"
#include "stl_supp.h"
#include "ProxyChannel.h"
#include "gk_const.h"

#define DEFAULT_CONNECT_TIMEOUT 180000

const char *CallTableSection = "CallTable";
const char *RRQFeaturesSection = "RasSrv::RRQFeatures";


EndpointRec::EndpointRec(const H225_RasMessage &completeRAS, bool Permanent)
      :	m_RasMsg(completeRAS), m_timeToLive(1), m_pollCount(2),
	m_nat(false), m_natsocket(0)
{
	m_activeCall = m_connectedCall = m_totalCall = m_usedCount = 0;
	switch (m_RasMsg.GetTag())
	{
		case H225_RasMessage::e_registrationRequest:
			SetEndpointRec((H225_RegistrationRequest &)m_RasMsg);
			PTRACE(1, "New EP|" << PrintOn(false));
			break;
		case H225_RasMessage::e_admissionRequest:
			SetEndpointRec((H225_AdmissionRequest &)m_RasMsg);
			break;
		case H225_RasMessage::e_admissionConfirm:
			SetEndpointRec((H225_AdmissionConfirm &)m_RasMsg);
			break;
		case H225_RasMessage::e_locationConfirm:
			SetEndpointRec((H225_LocationConfirm &)m_RasMsg);
			break;
		default: // should not happen
			break;
	}
	if (Permanent)
		m_timeToLive = 0;
}	

void EndpointRec::SetEndpointRec(H225_RegistrationRequest & rrq)
{
	if (rrq.m_rasAddress.GetSize() > 0)
		m_rasAddress = rrq.m_rasAddress[0];
	else
		m_rasAddress.SetTag(H225_TransportAddress::e_nonStandardAddress);
	if (rrq.m_callSignalAddress.GetSize() > 0)
		m_callSignalAddress = rrq.m_callSignalAddress[0];
	else
		m_callSignalAddress.SetTag(H225_TransportAddress::e_nonStandardAddress);
	m_endpointIdentifier = rrq.m_endpointIdentifier;
	m_terminalAliases = rrq.m_terminalAlias;
	m_terminalType = &rrq.m_terminalType;
	if (rrq.HasOptionalField(H225_RegistrationRequest::e_timeToLive))
		SetTimeToLive(rrq.m_timeToLive);
	else
		SetTimeToLive(SoftPBX::TimeToLive);
	m_fromParent = false;
}

void EndpointRec::SetEndpointRec(H225_AdmissionRequest & arq)
{
	static H225_EndpointType termType; // nouse
	// we set it to non-standard address to avoid misuse
	m_rasAddress.SetTag(H225_TransportAddress::e_nonStandardAddress);
	m_callSignalAddress = arq.m_destCallSignalAddress;
	m_terminalType = &termType;
	m_timeToLive = (SoftPBX::TimeToLive > 0) ? SoftPBX::TimeToLive : 600;
	m_fromParent = false;
}

void EndpointRec::SetEndpointRec(H225_AdmissionConfirm & acf)
{
	// there is no RAS address in ACF
	// we set it to non-standard address to avoid misuse
	m_rasAddress.SetTag(H225_TransportAddress::e_nonStandardAddress);
	m_callSignalAddress = acf.m_destCallSignalAddress;
	if (acf.HasOptionalField(H225_AdmissionConfirm::e_destinationInfo))
		m_terminalAliases = acf.m_destinationInfo;
	if (!acf.HasOptionalField(H225_AdmissionConfirm::e_destinationType))
		acf.IncludeOptionalField(H225_AdmissionConfirm::e_destinationType);
	m_terminalType = &acf.m_destinationType;
	m_timeToLive = (SoftPBX::TimeToLive > 0) ? SoftPBX::TimeToLive : 600;
	m_fromParent = true;
}

void EndpointRec::SetEndpointRec(H225_LocationConfirm & lcf)
{
	m_rasAddress = lcf.m_rasAddress;
	m_callSignalAddress = lcf.m_callSignalAddress;
	if (lcf.HasOptionalField(H225_LocationConfirm::e_destinationInfo))
		m_terminalAliases = lcf.m_destinationInfo;
	if (!lcf.HasOptionalField(H225_LocationConfirm::e_destinationType))
		lcf.IncludeOptionalField(H225_LocationConfirm::e_destinationType);
	m_terminalType = &lcf.m_destinationType;
	m_timeToLive = (SoftPBX::TimeToLive > 0) ? SoftPBX::TimeToLive : 600;
	m_fromParent = false;
}

EndpointRec::~EndpointRec()
{
	PTRACE(3, "Gk\tDelete endpoint: " << m_endpointIdentifier.GetValue() << " " << m_usedCount);
	if (m_natsocket)
		m_natsocket->SetDeletable();
}

bool EndpointRec::PrefixMatch_IncompleteAddress(const H225_ArrayOf_AliasAddress &aliases, 
                                               bool &fullMatch) const
{
        fullMatch = 0;
        int partialMatch = 0;
        PString aliasStr;
	PINDEX aliasStr_len;
	const H225_ArrayOf_AliasAddress & reg_aliases = GetAliases();
	PString reg_alias;
	// for each given alias (dialedDigits) from request message 
	for(PINDEX i = 0; i < aliases.GetSize() && !fullMatch; i++) {
//          if (aliases[i].GetTag() == H225_AliasAddress::e_dialedDigits) {
	    aliasStr = H323GetAliasAddressString(aliases[i]);
	    aliasStr_len = aliasStr.GetLength();
	    // for each alias (dialedDigits) which is stored for the endpoint in registration
	    for (PINDEX i = 0; i < reg_aliases.GetSize() && !fullMatch; i++) {
//              if (reg_aliases[i].GetTag() == H225_AliasAddress::e_dialedDigits) {
	        reg_alias = H323GetAliasAddressString(reg_aliases[i]);
                // if alias from request message is prefix to alias which is 
		//   stored in registration
	        if ((reg_alias.GetLength() >= aliasStr_len) && 
		    (aliasStr == reg_alias.Left(aliasStr_len))) {
		  // check if it is a full match 
		  if (aliasStr == reg_alias) {
		    fullMatch = 1;
  		    PTRACE(2, ANSI::DBG << "Alias " << aliasStr << " matches endpoint " 
		      << (const unsigned char *)m_endpointIdentifier.GetValue() << " (full)" << ANSI::OFF);
  		  } else {
		    partialMatch = 1;
  		    PTRACE(2, ANSI::DBG << "Alias " << aliasStr << " matches endpoint " 
		      << (const unsigned char *)m_endpointIdentifier.GetValue() << " (partial)" << ANSI::OFF);
		  }
	        }
//	      }
	    }
//	  }
	}
	return (partialMatch || fullMatch);
}

void EndpointRec::SetRasAddress(const H225_TransportAddress &a)
{
	PWaitAndSignal lock(m_usedLock);
	m_rasAddress = a;
}

void EndpointRec::SetEndpointIdentifier(const H225_EndpointIdentifier &i)
{
	PWaitAndSignal lock(m_usedLock);
	m_endpointIdentifier = i;
}
        
void EndpointRec::SetTimeToLive(int seconds)
{
	if (m_timeToLive > 0) {
		// To avoid bloated RRQ traffic, don't allow ttl < 60
		if (seconds < 60)
			seconds = 60;
		PWaitAndSignal lock(m_usedLock);
		m_timeToLive = (SoftPBX::TimeToLive > 0) ?
			std::min(SoftPBX::TimeToLive, seconds) : 0;
	}
}

void EndpointRec::SetPermanent(bool b)
{
	PWaitAndSignal lock(m_usedLock);
	m_timeToLive = (!b && SoftPBX::TimeToLive > 0) ? SoftPBX::TimeToLive : 0;
}

void EndpointRec::SetAliases(const H225_ArrayOf_AliasAddress &a)
{
	PWaitAndSignal lock(m_usedLock);
        m_terminalAliases = a;
}
        
void EndpointRec::SetEndpointType(const H225_EndpointType &t) 
{
	PWaitAndSignal lock(m_usedLock);
        *m_terminalType = t;
}

void EndpointRec::Update(const H225_RasMessage & ras_msg)
{
        if (ras_msg.GetTag() == H225_RasMessage::e_registrationRequest) {
		const H225_RegistrationRequest & rrq = ras_msg;

		// don't update rasAddress for nated endpoint
		if (!m_nat && (rrq.m_rasAddress.GetSize() >= 1))
			SetRasAddress(rrq.m_rasAddress[0]);

		if (rrq.HasOptionalField(H225_RegistrationRequest::e_timeToLive))
			SetTimeToLive(rrq.m_timeToLive);

		// H.225.0v4: ignore fields other than rasAddress, endpointIdentifier,
		// timeToLive for a lightweightRRQ
		if (!(rrq.HasOptionalField(H225_RegistrationRequest::e_keepAlive) && rrq.m_keepAlive)) {
			if (rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias)
				&& (rrq.m_terminalAlias.GetSize() >= 1))
				SetAliases(rrq.m_terminalAlias);
		}
	} else if (ras_msg.GetTag() == H225_RasMessage::e_locationConfirm) {
		const H225_LocationConfirm & lcf = ras_msg;
		SetRasAddress(lcf.m_rasAddress);
		if (lcf.HasOptionalField(H225_LocationConfirm::e_destinationInfo))
			SetAliases(lcf.m_destinationInfo);
	}
	PWaitAndSignal lock(m_usedLock);
	m_updatedTime = PTime();
	m_pollCount = 2;
}

// due to strange bug of gcc, I have to pass pointer instead of reference
bool EndpointRec::CompareAlias(const H225_ArrayOf_AliasAddress *a) const
{
	for (PINDEX i = 0; i < a->GetSize(); i++)
		for (PINDEX j = 0; j < m_terminalAliases.GetSize(); j++)
			if ((*a)[i] == m_terminalAliases[j])
				return true;
	return false;
}

bool EndpointRec::MatchAlias(
	const H225_ArrayOf_AliasAddress& aliases,
	int& matchedalias
	) const
{
	for (PINDEX i = 0; i < aliases.GetSize(); i++)
		for (PINDEX j = 0; j < m_terminalAliases.GetSize(); j++)
			if (aliases[i] == m_terminalAliases[j]) {
				matchedalias = i;
				return true;
			}
	return false;
}

EndpointRec *EndpointRec::Unregister()
{
	SendURQ(H225_UnregRequestReason::e_maintenance);
	return this;
}

EndpointRec *EndpointRec::Expired()
{
	SendURQ(H225_UnregRequestReason::e_ttlExpired);
	return this;
}

void EndpointRec::BuildLCF(H225_LocationConfirm & obj_lcf) const
{
	obj_lcf.m_callSignalAddress = GetCallSignalAddress();
	obj_lcf.m_rasAddress = GetRasAddress();
	extern const char *LRQFeaturesSection;
	if (Toolkit::AsBool(GkConfig()->GetString(LRQFeaturesSection, "IncludeDestinationInfoInLCF", "1"))) {
		obj_lcf.IncludeOptionalField(H225_LocationConfirm::e_destinationInfo);
		obj_lcf.m_destinationInfo = GetAliases();
		obj_lcf.IncludeOptionalField(H225_LocationConfirm::e_destinationType);
		obj_lcf.m_destinationType = GetEndpointType();
	}
}

PString EndpointRec::PrintOn(bool verbose) const
{
	PString msg(PString::Printf, "%s|%s|%s|%s\r\n",
		    (const unsigned char *) AsDotString(GetCallSignalAddress()),
		    (const unsigned char *) AsString(GetAliases()),
		    (const unsigned char *) AsString(GetEndpointType()),
		    (const unsigned char *) GetEndpointIdentifier().GetValue() );
	if (verbose) {
		msg += GetUpdatedTime().AsString();
		if (m_timeToLive == 0)
			msg += " (permanent)";
		PString natstring(m_nat ? m_natip.AsString() : PString());
		msg += PString(PString::Printf, " C(%d/%d/%d) %s <%d>\r\n", m_activeCall, m_connectedCall, m_totalCall, (const unsigned char *)natstring, m_usedCount);
	}
	return msg;
}

bool EndpointRec::SendURQ(H225_UnregRequestReason::Choices reason)
{
	if (GetRasAddress().GetTag() != H225_TransportAddress::e_ipAddress)
		return false;  // no valid ras address

	H225_RasMessage ras_msg;
	ras_msg.SetTag(H225_RasMessage::e_unregistrationRequest);
	H225_UnregistrationRequest & urq = ras_msg;
	urq.m_requestSeqNum.SetValue(RasThread->GetRequestSeqNum());
	urq.IncludeOptionalField(urq.e_gatekeeperIdentifier);
	urq.m_gatekeeperIdentifier.SetValue( Toolkit::GKName() );
	urq.IncludeOptionalField(urq.e_endpointIdentifier);
	urq.m_endpointIdentifier = GetEndpointIdentifier();
	urq.m_callSignalAddress.SetSize(1);
	urq.m_callSignalAddress[0] = GetCallSignalAddress();
	urq.IncludeOptionalField(H225_UnregistrationRequest::e_reason);
	urq.m_reason.SetTag(reason);

	PString msg(PString::Printf, "URQ|%s|%s|%s;\r\n", 
			(const unsigned char *) AsDotString(GetRasAddress()),
			(const unsigned char *) GetEndpointIdentifier().GetValue(),
			(const unsigned char *) urq.m_reason.GetTagName());
        GkStatus::Instance()->SignalStatus(msg);

	RasThread->ForwardRasMsg(ras_msg);
	if (reason == H225_UnregRequestReason::e_maintenance)
		RasThread->SetAlternateGK(urq);
	RasThread->SendRas(ras_msg, GetRasAddress());
	return true;
}

bool EndpointRec::SendIRQ()
{
	if (m_pollCount-- == 0 || GetRasAddress().GetTag() != H225_TransportAddress::e_ipAddress)
		return false;

	H225_RasMessage ras_msg;
	ras_msg.SetTag(H225_RasMessage::e_infoRequest);
	H225_InfoRequest & irq = ras_msg;
	irq.m_requestSeqNum.SetValue(RasThread->GetRequestSeqNum());
	irq.m_callReferenceValue.SetValue(0); // ask for each call

	PString msg(PString::Printf, "IRQ|%s|%s;\r\n", 
			(const unsigned char *) AsDotString(GetRasAddress()),
			(const unsigned char *) GetEndpointIdentifier().GetValue());
        GkStatus::Instance()->SignalStatus(msg);
	RasThread->SendRas(ras_msg, GetRasAddress());

	return true;
}

void EndpointRec::SetNATAddress(const PIPSocket::Address & ip)
{
	m_nat = true;
	m_natip = ip;

	// we keep the original private IP in signalling address,
	// because we have to use it to identify different endpoints
	// but from the same NAT box
	if (m_rasAddress.GetTag() != H225_TransportAddress::e_ipAddress)
		m_rasAddress.SetTag(H225_TransportAddress::e_ipAddress);
	H225_TransportAddress_ipAddress & rasip = m_rasAddress;
	for (int i = 0; i < 4; ++i)
		rasip.m_ip[i] = ip[i];
}

CallSignalSocket *EndpointRec::GetSocket()
{
	PWaitAndSignal lock(m_usedLock);
	CallSignalSocket *socket = m_natsocket;
	m_natsocket = 0;
	return socket;
}

void EndpointRec::SetSocket(CallSignalSocket *socket)
{
	PWaitAndSignal lock(m_usedLock);
	if (m_natsocket != socket) {
		PTRACE(3, "Q931\tAn NAT socket detected at " << socket->Name() << " for endpoint " << GetEndpointIdentifier().GetValue());
		if (m_natsocket) {
			PTRACE(1, "Warning: natsocket is overwritten by " << socket->Name());
			m_natsocket->SetDeletable();
		}
		m_natsocket = socket;
	}
}

GatewayRec::GatewayRec(const H225_RasMessage &completeRRQ, bool Permanent)
      : EndpointRec(completeRRQ, Permanent), defaultGW(false)
{
	Prefixes.reserve(8);
	GatewayRec::LoadConfig(); // static binding
}

void GatewayRec::SetAliases(const H225_ArrayOf_AliasAddress &a)
{
	EndpointRec::SetAliases(a);
	LoadConfig();
}

void GatewayRec::SetEndpointType(const H225_EndpointType &t)
{
	if (!t.HasOptionalField(H225_EndpointType::e_gateway)) {
		PTRACE(1, "RRJ: terminal type changed|" << (const unsigned char *)m_endpointIdentifier.GetValue());
		return;
	}
	EndpointRec::SetEndpointType(t);
	LoadConfig();
}

void GatewayRec::Update(const H225_RasMessage & ras_msg)
{
        if (ras_msg.GetTag() == H225_RasMessage::e_registrationRequest) {
		const H225_RegistrationRequest & rrq = ras_msg;
		if (!(rrq.HasOptionalField(H225_RegistrationRequest::e_keepAlive) && rrq.m_keepAlive))
			SetEndpointType(rrq.m_terminalType);
	} else if (ras_msg.GetTag() == H225_RasMessage::e_locationConfirm) {
		const H225_LocationConfirm & lcf = ras_msg;
		if (lcf.HasOptionalField(H225_LocationConfirm::e_destinationType))
			SetEndpointType(lcf.m_destinationType);
	}
			
	EndpointRec::Update(ras_msg);
}

void GatewayRec::AddPrefixes(const H225_ArrayOf_SupportedProtocols &protocols)
{
	for (PINDEX i = 0; i < protocols.GetSize(); ++i) {
		H225_SupportedProtocols &p = protocols[i];
		H225_ArrayOf_SupportedPrefix *supportedPrefixes = 0;
		if (p.GetTag() == H225_SupportedProtocols::e_voice) {
			H225_VoiceCaps & v = p;
			if (v.HasOptionalField(H225_VoiceCaps::e_supportedPrefixes))
				supportedPrefixes = &v.m_supportedPrefixes;
		} else if (p.GetTag() == H225_SupportedProtocols::e_h323) {
			H225_H323Caps & v = p;
			if (v.HasOptionalField(H225_H323Caps::e_supportedPrefixes))
				supportedPrefixes = &v.m_supportedPrefixes;
		} else if (p.GetTag() == H225_SupportedProtocols::e_h320) {
			H225_H320Caps & v = p;
			if (v.HasOptionalField(H225_H320Caps::e_supportedPrefixes))
				supportedPrefixes = &v.m_supportedPrefixes;
		}
		if (supportedPrefixes)
			for (PINDEX s = 0; s < supportedPrefixes->GetSize(); ++s) {
				H225_AliasAddress &a = (*supportedPrefixes)[s].m_prefix;
				if (a.GetTag() == H225_AliasAddress::e_dialedDigits)
					Prefixes.push_back((const char *)AsString(a, false));
			}
	}
}

void GatewayRec::AddPrefixes(const PString & prefixes)
{
	PStringArray p(prefixes.Tokenise(" ,;\t\n", false));
	for (PINDEX i = 0; i < p.GetSize(); ++i)
		Prefixes.push_back((const char *)p[i]);
}

void GatewayRec::SortPrefixes()
{
	// remove duplicate aliases
	sort(Prefixes.begin(), Prefixes.end(), greater<string>());
	prefix_iterator Iter = unique(Prefixes.begin(), Prefixes.end());
	Prefixes.erase(Iter, Prefixes.end());
	defaultGW = (find_if(Prefixes.begin(), Prefixes.end(), bind2nd(equal_to<string>(), "*")) != Prefixes.end());
	defaultGW = (find(Prefixes.begin(), Prefixes.end(), string("*")) != Prefixes.end());
}

bool GatewayRec::LoadConfig()
{
	PWaitAndSignal lock(m_usedLock);
	Prefixes.clear();
	if (Toolkit::AsBool(GkConfig()->GetString(RRQFeaturesSection, "AcceptGatewayPrefixes", "1")))
		if (m_terminalType->m_gateway.HasOptionalField(H225_GatewayInfo::e_protocol))
			AddPrefixes(m_terminalType->m_gateway.m_protocol);
	for (PINDEX i=0; i<m_terminalAliases.GetSize(); i++)
		AddPrefixes(GkConfig()->GetString("RasSrv::GWPrefixes", H323GetAliasAddressString(m_terminalAliases[i]), ""));
	SortPrefixes();
	return true;
}

int GatewayRec::PrefixMatch(
	const H225_ArrayOf_AliasAddress& aliases
	) const
{
	int maxlen = (defaultGW) ? 0 : -1;
	for (PINDEX i = 0; i < aliases.GetSize(); i++) {
		const unsigned tag = aliases[i].GetTag();
		if ( tag == H225_AliasAddress::e_dialedDigits
			|| tag == H225_AliasAddress::e_partyNumber
			|| tag == H225_AliasAddress::e_h323_ID) {
			
			const PString alias = H323GetAliasAddressString(aliases[i]);
			// we also allow h_323_ID aliases consisting only from digits
			if( tag == H225_AliasAddress::e_h323_ID )
				if( strspn(alias,"1234567890*#") != strlen(alias) )
					continue;
					
			const_prefix_iterator Iter = Prefixes.begin(), eIter= Prefixes.end();
			while (Iter != eIter) {
				int len = Iter->length();
				if ((maxlen < len) && (strncmp(alias, Iter->c_str(), len)==0)) {
					PTRACE(2, "Gateway " << (const unsigned char *)m_endpointIdentifier.GetValue() << " match " << Iter->c_str() );
					maxlen = len;
				}
				++Iter;
			}
		}
	}
	return maxlen;
}

int GatewayRec::PrefixMatch(
	const H225_ArrayOf_AliasAddress& aliases,
	int& matchedalias
	) const
{
	int maxlen = (defaultGW) ? 0 : -1;
	matchedalias = 0;
	
	for (PINDEX i = 0; i < aliases.GetSize(); i++) {
		const unsigned tag = aliases[i].GetTag();
		if ( tag == H225_AliasAddress::e_dialedDigits
			|| tag == H225_AliasAddress::e_partyNumber
			|| tag == H225_AliasAddress::e_h323_ID) {
			
			const PString alias = H323GetAliasAddressString(aliases[i]);
			// we also allow h_323_ID aliases consisting only from digits
			if( tag == H225_AliasAddress::e_h323_ID )
				if( strspn(alias,"1234567890*#") != strlen(alias) )
					continue;
					
			const_prefix_iterator Iter = Prefixes.begin(), eIter= Prefixes.end();
			while (Iter != eIter) {
				int len = Iter->length();
				if ((maxlen < len) && (strncmp(alias, Iter->c_str(), len)==0)) {
					PTRACE(2, "Gateway " << (const unsigned char *)m_endpointIdentifier.GetValue() << " match " << Iter->c_str() );
					maxlen = len;
					matchedalias = i;
				}
				++Iter;
			}
		}
	}
	
	// if no match has been found and this is the default gateway,
	// assume first dialedDigits or partyNumber alias match
	if( maxlen == 0 ) {
		for (PINDEX i = 0; i < aliases.GetSize(); i++)
			if (aliases[i].GetTag() == H225_AliasAddress::e_dialedDigits
				|| aliases[i].GetTag() == H225_AliasAddress::e_partyNumber) {
				matchedalias = i;
				break;
			}
	}
	return maxlen;
}

void GatewayRec::BuildLCF(H225_LocationConfirm & obj_lcf) const
{
	EndpointRec::BuildLCF(obj_lcf);
	if (PINDEX as = Prefixes.size()) {
		obj_lcf.IncludeOptionalField(H225_LocationConfirm::e_supportedProtocols);
		obj_lcf.m_supportedProtocols.SetSize(1);
		H225_SupportedProtocols &protocol = obj_lcf.m_supportedProtocols[0];
		protocol.SetTag(H225_SupportedProtocols::e_voice);
		H225_ArrayOf_SupportedPrefix & supportedPrefixes = ((H225_VoiceCaps &)protocol).m_supportedPrefixes;
		supportedPrefixes.SetSize(as);
		const_prefix_iterator Iter = Prefixes.begin();
		for (PINDEX p=0; p < as; ++p, ++Iter)
			H323SetAliasAddress(PString(Iter->c_str()), supportedPrefixes[p].m_prefix);
	}
}

PString GatewayRec::PrintOn(bool verbose) const
{
	PString msg = EndpointRec::PrintOn(verbose);
	if (verbose) {
		msg += "Prefixes: ";
		if (Prefixes.size() == 0) {
			msg += "<none>";
		} else {
			string m=Prefixes.front();
			const_prefix_iterator Iter = Prefixes.begin(), eIter= Prefixes.end();
			while (++Iter != eIter)
				m += "," + (*Iter);
			msg += m.c_str();
		}
		msg += "\r\n";
	}
	return msg;
}

OuterZoneEPRec::OuterZoneEPRec(const H225_RasMessage & completeRAS, const H225_EndpointIdentifier &epID) : EndpointRec(completeRAS, false)
{
	m_endpointIdentifier = epID;
	PTRACE(1, "New OZEP|" << PrintOn(false));
}

OuterZoneGWRec::OuterZoneGWRec(const H225_RasMessage & completeLCF, const H225_EndpointIdentifier &epID) : GatewayRec(completeLCF, false)
{
	m_endpointIdentifier = epID;

	const H225_LocationConfirm & obj_lcf = completeLCF;
	if (obj_lcf.HasOptionalField(H225_LocationConfirm::e_supportedProtocols)) {
		AddPrefixes(obj_lcf.m_supportedProtocols);
		SortPrefixes();
	}
	defaultGW = false; // don't let outer zone gateway be default
	PTRACE(1, "New OZGW|" << PrintOn(false));
}


RegistrationTable::RegistrationTable() : regSize(0)
{
	recCnt = rand()%9000 + 1000;
	ozCnt = 1000; // arbitrary chosen constant

	LoadConfig();
}

RegistrationTable::~RegistrationTable()
{
	ClearTable();
	// since the socket has been deleted, just get it away
	ForEachInContainer(RemovedList, mem_fun(&EndpointRec::GetSocket));
	DeleteObjectsInContainer<EndpointRec>(RemovedList);
}

endptr RegistrationTable::InsertRec(H225_RasMessage & ras_msg, PIPSocket::Address ip)
{
	endptr ep;
	switch (ras_msg.GetTag())
	{
		case H225_RasMessage::e_registrationRequest: {
			H225_RegistrationRequest & rrq = ras_msg;
			if (ep = FindBySignalAdr(rrq.m_callSignalAddress[0], ip))
				ep->Update(ras_msg);
			else
				ep = InternalInsertEP(ras_msg);
			break;
		}
		case H225_RasMessage::e_admissionRequest: {
			H225_AdmissionConfirm nouse;
			H225_AdmissionRequest & arq = ras_msg;
			if (arq.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) && !(ep = FindOZEPBySignalAdr(arq.m_destCallSignalAddress)))
				ep = InternalInsertOZEP(ras_msg, nouse);
			break;
		}
		case H225_RasMessage::e_admissionConfirm: {
			H225_AdmissionConfirm & acf = ras_msg;
			if (!(ep = FindOZEPBySignalAdr(acf.m_destCallSignalAddress)))
				ep = InternalInsertOZEP(ras_msg, acf);
			break;
		}
		case H225_RasMessage::e_locationConfirm: {
			H225_LocationConfirm & lcf = ras_msg;
			if ((ep = FindOZEPBySignalAdr(lcf.m_callSignalAddress)))
				ep->Update(ras_msg);
			else
				ep = InternalInsertOZEP(ras_msg, lcf);
			break;
		}
		default:
			PTRACE(1, "RegistrationTable: unable to insert " << ras_msg.GetTagName());
			break;
	}

	return ep;
}

endptr RegistrationTable::InternalInsertEP(H225_RasMessage & ras_msg)
{
	H225_RegistrationRequest & rrq = ras_msg;
	if (!rrq.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier) ||
	    !Toolkit::AsBool(GkConfig()->GetString(RRQFeaturesSection, "AcceptEndpointIdentifier", "1"))) {
		rrq.IncludeOptionalField(H225_RegistrationRequest::e_endpointIdentifier);
		endptr e = InternalFind(compose1(bind2nd(equal_to<H225_TransportAddress>(), rrq.m_callSignalAddress[0]),
			mem_fun(&EndpointRec::GetCallSignalAddress)), &RemovedList);
		if (e) // re-use the old endpoint identifier
			rrq.m_endpointIdentifier = e->GetEndpointIdentifier();
		else
			GenerateEndpointId(rrq.m_endpointIdentifier);
	}
	if (!(rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias) && (rrq.m_terminalAlias.GetSize() >= 1))) {
		rrq.IncludeOptionalField(H225_RegistrationRequest::e_terminalAlias);
		GenerateAlias(rrq.m_terminalAlias, rrq.m_endpointIdentifier);
	}

	EndpointRec *ep = rrq.m_terminalType.HasOptionalField(H225_EndpointType::e_gateway) ?
			  new GatewayRec(ras_msg) : new EndpointRec(ras_msg);
	WriteLock lock(listLock);
	EndpointList.push_back(ep);
	++regSize;
	return endptr(ep);
}

endptr RegistrationTable::InternalInsertOZEP(H225_RasMessage & ras_msg, H225_AdmissionConfirm &)
{
	H225_EndpointIdentifier epID;
	epID = "oz_" + PString(PString::Unsigned, ozCnt++) + endpointIdSuffix;
	EndpointRec *ep = new OuterZoneEPRec(ras_msg, epID);
	WriteLock lock(listLock);
	OuterZoneList.push_front(ep);
	return endptr(ep);
}

endptr RegistrationTable::InternalInsertOZEP(H225_RasMessage & ras_msg, H225_LocationConfirm & lcf)
{
	H225_EndpointIdentifier epID;
	epID = "oz_" + PString(PString::Unsigned, ozCnt++) + endpointIdSuffix;

	EndpointRec *ep;
	if (lcf.HasOptionalField(H225_LocationConfirm::e_destinationType) &&
	    lcf.m_destinationType.HasOptionalField(H225_EndpointType::e_gateway))
		ep = new OuterZoneGWRec(ras_msg, epID);
	else
		ep = new OuterZoneEPRec(ras_msg, epID);

	WriteLock lock(listLock);
	OuterZoneList.push_front(ep);
	return endptr(ep);
}

void RegistrationTable::RemoveByEndptr(const endptr & eptr)
{
	EndpointRec *ep = eptr.operator->(); // evil
	WriteLock lock(listLock);
	InternalRemove(find(EndpointList.begin(), EndpointList.end(), ep));
}

void RegistrationTable::RemoveByEndpointId(const H225_EndpointIdentifier & epId)
{
	WriteLock lock(listLock);
	InternalRemove( find_if(EndpointList.begin(), EndpointList.end(),
			compose1(bind2nd(equal_to<H225_EndpointIdentifier>(), epId),
			mem_fun(&EndpointRec::GetEndpointIdentifier)))
	);
}

void RegistrationTable::InternalRemove(iterator Iter)
{
	if (Iter == EndpointList.end()) {
	        PTRACE(1, "Warning: remove endpoint failed");
		return;
	}
	RemovedList.push_back(*Iter);
	EndpointList.erase(Iter);
	--regSize;
}

/*
template<class F> endptr RegistrationTable::InternalFind(const F & FindObject,
	const list<EndpointRec *> *List) const
{
	ReadLock lock(listLock);
	const_iterator Iter = find_if(List->begin(), List->end(), FindObject);
	return endptr((Iter != List->end()) ? *Iter : NULL);
}
*/

endptr RegistrationTable::FindByEndpointId(const H225_EndpointIdentifier & epId) const
{
	return InternalFind(compose1(bind2nd(equal_to<H225_EndpointIdentifier>(), epId),
			mem_fun(&EndpointRec::GetEndpointIdentifier)));
}

namespace { // end of anonymous namespace

class CompareSigAdr {
public:
	CompareSigAdr(const H225_TransportAddress & adr) : SigAdr(adr) {}
	bool operator()(const EndpointRec *ep) const { return ep->GetCallSignalAddress() == SigAdr; }

protected:
	const H225_TransportAddress & SigAdr;
};

class CompareSigAdrWithNAT : public CompareSigAdr {
public:
	CompareSigAdrWithNAT(const H225_TransportAddress & adr, PIPSocket::Address ip) : CompareSigAdr(adr), natip(ip) {}
	bool operator()(const EndpointRec *ep) const { return (ep->GetNATIP() == natip) && CompareSigAdr::operator()(ep); }

private:
	PIPSocket::Address natip;
};

bool operator==(const H225_TransportAddress & adr, PIPSocket::Address ip)
{
	if (ip == INADDR_ANY)
		return true;
	PIPSocket::Address ipaddr;
	return GetIPFromTransportAddr(adr, ipaddr) ? (ip == ipaddr) : false;
}

} // end of anonymous namespace

endptr RegistrationTable::FindBySignalAdr(const H225_TransportAddress & sigAd, PIPSocket::Address ip) const
{
	return (sigAd == ip) ? InternalFind(CompareSigAdr(sigAd)) : InternalFind(CompareSigAdrWithNAT(sigAd, ip));
}

endptr RegistrationTable::FindOZEPBySignalAdr(const H225_TransportAddress & sigAd) const
{
	return InternalFind(compose1(bind2nd(equal_to<H225_TransportAddress>(), sigAd),
			mem_fun(&EndpointRec::GetCallSignalAddress)), &OuterZoneList);
}

endptr RegistrationTable::FindByAliases(const H225_ArrayOf_AliasAddress & alias) const
{
	return InternalFind(bind2nd(mem_fun(&EndpointRec::CompareAlias), &alias));
}

endptr RegistrationTable::FindEndpoint(
	const H225_ArrayOf_AliasAddress& aliases, 
	bool roundrobin, 
	bool searchouterzone,
	int* matchedalias
	)
{
	endptr ep = InternalFindEP(aliases,&EndpointList,roundrobin,matchedalias);
	return (ep) ? ep : (searchouterzone 
		? InternalFindEP(aliases,&OuterZoneList,roundrobin,matchedalias) 
		: endptr(NULL));
}

endptr RegistrationTable::InternalFindEP(
	const H225_ArrayOf_AliasAddress& aliases,
	list<EndpointRec *>* eplist, 
	bool roundrobin,
	int* matchedalias
	)
{
	endptr ep(NULL);

	// if we also need to know matchedalias index, we need to do search "by hand"
	if( matchedalias ) {
		const_iterator Iter = eplist->begin(), IterLast = eplist->end();
		while (Iter != IterLast) {
			if( (*Iter)->MatchAlias(aliases,*matchedalias) ) {
				ep = endptr(*Iter);
				break;
			}
			++Iter;
		}
	} else
		ep = InternalFind(
			bind2nd(mem_fun(&EndpointRec::CompareAlias), &aliases), eplist
			);
	if (ep) {
		PTRACE(4, "Alias match for EP " << AsDotString(ep->GetCallSignalAddress()));
		return ep;
	}

	// build a list of gateways with the longest prefix match for the aliases
	int maxlen = 0;
	list<EndpointRec *> gwlist;
	listLock.StartRead();
	const_iterator Iter = eplist->begin(), IterLast = eplist->end();
	while (Iter != IterLast) {
		if ((*Iter)->IsGateway()) {
			const int len = matchedalias 
				? dynamic_cast<GatewayRec *>(*Iter)->PrefixMatch(aliases,*matchedalias)
				: dynamic_cast<GatewayRec *>(*Iter)->PrefixMatch(aliases);
			// if this prefix match is longer than the current max, 
			// clear the list and add this gateway
			if (maxlen < len) {
				gwlist.clear();
				maxlen = len;
			}
			if (maxlen == len)
				gwlist.push_back(*Iter);
		}
		++Iter;
	}
	listLock.EndRead();

	if (gwlist.size() > 0) {
		EndpointRec* gw = gwlist.front();
		if (gwlist.size() > 1 && roundrobin) {
			// NOTE: this has funny side-effect that all gateways
			// will end up at the end of the registration table
			// - that slows down gw search in the above code block
			PTRACE(5,"Applying round robin gateway selection");
			WriteLock lock(listLock);
			eplist->remove(gw);
			eplist->push_back(gw);
		}
		PTRACE(4,"Alias match for GW "<<AsDotString(gw->GetCallSignalAddress()));
		return endptr(gw);
	}
	return endptr(0);
}

void RegistrationTable::GenerateEndpointId(H225_EndpointIdentifier & NewEndpointId)
{
	NewEndpointId = PString(PString::Unsigned, ++recCnt) + endpointIdSuffix;
}


void RegistrationTable::GenerateAlias(H225_ArrayOf_AliasAddress & AliasList, const H225_EndpointIdentifier & endpointId) const
{
	AliasList.SetSize(1);
	H323SetAliasAddress(endpointId, AliasList[0]);
}

void RegistrationTable::PrintAllRegistrations(GkStatus::Client &client, BOOL verbose)
{
	PString msg("AllRegistrations\r\n");
	InternalPrint(client, verbose, &EndpointList, msg);
}

void RegistrationTable::PrintAllCached(GkStatus::Client &client, BOOL verbose)
{
	PString msg("AllCached\r\n");
	InternalPrint(client, verbose, &OuterZoneList, msg);
}

void RegistrationTable::PrintRemoved(GkStatus::Client &client, BOOL verbose)
{
	PString msg("AllRemoved\r\n");
	InternalPrint(client, verbose, &RemovedList, msg);
}

void RegistrationTable::InternalPrint(GkStatus::Client &client, BOOL verbose, list<EndpointRec *> * List, PString & msg)
{
	// copy the pointers into a temporary array to avoid large lock
	listLock.StartRead();
	const_iterator IterLast = List->end();
	unsigned k = 0, s = List->size();
	endptr *eptr = new endptr[s];
	for (const_iterator Iter = List->begin(); Iter != IterLast; ++Iter)
		eptr[k++] = endptr(*Iter);
	listLock.EndRead();
	// end of lock

	if (s > 1000) // set buffer to avoid reallocate
		msg.SetSize(s * (verbose ? 200 : 100));
	for (k = 0; k < s; k++)
		msg += "RCF|" + eptr[k]->PrintOn(verbose);
	delete [] eptr;

	msg += PString(PString::Printf, "Number of Endpoints: %u\r\n;\r\n", s);
	client.WriteString(msg);
}

void RegistrationTable::InternalStatistics(const list<EndpointRec *> *List, unsigned & s, unsigned & t, unsigned & g, unsigned & n) const
{
	ReadLock lock(listLock);
	s = List->size(), t = g = n = 0;
	const_iterator IterLast = List->end();
	for (const_iterator Iter = List->begin(); Iter != IterLast; ++Iter) {
		EndpointRec *ep = *Iter;
		++(ep->IsGateway() ? g : t);
		if (ep->IsNATed())
			++n;
	}
}

PString RegistrationTable::PrintStatistics() const
{
	unsigned es, et, eg, en;
	InternalStatistics(&EndpointList, es, et, eg, en);
	unsigned cs, ct, cg, cn; // cn is useless
	InternalStatistics(&OuterZoneList, cs, ct, cg, cn);

	return PString(PString::Printf, "-- Endpoint Statistics --\r\n"
		"Total Endpoints: %u  Terminals: %u  Gateways: %u  NATed: %u\r\n"
		"Cached Endpoints: %u  Terminals: %u  Gateways: %u\r\n",
		es, et, eg, en, cs, ct, cg);
}

void RegistrationTable::LoadConfig()
{
	endpointIdSuffix = GkConfig()->GetString("EndpointIDSuffix", "_endp");

	// Load config for each endpoint
	if (regSize > 0) {
		ReadLock lock(listLock);
		ForEachInContainer(EndpointList, mem_fun(&EndpointRec::LoadConfig));
	}

	// Load permanent endpoints
	PStringToString cfgs=GkConfig()->GetAllKeyValues("RasSrv::PermanentEndpoints");
	for (PINDEX i = 0; i < cfgs.GetSize(); ++i) {
		EndpointRec *ep;
		H225_RasMessage rrq_ras;
		rrq_ras.SetTag(H225_RasMessage::e_registrationRequest);
		H225_RegistrationRequest &rrq = rrq_ras;

		rrq.m_callSignalAddress.SetSize(1);
		GetTransportAddress(cfgs.GetKeyAt(i), GkConfig()->GetInteger("EndpointSignalPort", GK_DEF_ENDPOINT_SIGNAL_PORT), rrq.m_callSignalAddress[0]);
		endptr eptr = FindBySignalAdr(rrq.m_callSignalAddress[0]);

		// a permanent endpoint may not support RAS
		// we set an arbitrary address here
		rrq.m_rasAddress.SetSize(1);
		rrq.m_rasAddress[0] = rrq.m_callSignalAddress[0];

		rrq.IncludeOptionalField(H225_RegistrationRequest::e_endpointIdentifier);
		GenerateEndpointId(rrq.m_endpointIdentifier);

		rrq.IncludeOptionalField(rrq.e_terminalAlias);
		PStringArray sp=cfgs.GetDataAt(i).Tokenise(";", FALSE);
		PStringArray aa=sp[0].Tokenise(",", FALSE);
		PINDEX as = aa.GetSize();
		if (as > 0) {
			rrq.m_terminalAlias.SetSize(as);
			for (PINDEX p=0; p<as; p++)
				H323SetAliasAddress(aa[p], rrq.m_terminalAlias[p]);
		}
		// GatewayInfo
		if (sp.GetSize() > 1) {
			/*
			aa = sp[1].Tokenise(",", FALSE);
			as = aa.GetSize();
			if (as > 0) {
				rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_gateway);
				rrq.m_terminalType.m_gateway.IncludeOptionalField(H225_GatewayInfo::e_protocol);
				rrq.m_terminalType.m_gateway.m_protocol.SetSize(1);
				H225_SupportedProtocols &protocol=rrq.m_terminalType.m_gateway.m_protocol[0];
				protocol.SetTag(H225_SupportedProtocols::e_voice);
				((H225_VoiceCaps &)protocol).m_supportedPrefixes.SetSize(as);
				for (PINDEX p = 0; p < as; ++p)
					H323SetAliasAddress(aa[p], ((H225_VoiceCaps &)protocol).m_supportedPrefixes[p].m_prefix);
			}
			*/
			rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_gateway);
			if (eptr && !eptr->IsGateway()) {
				RemoveByEndptr(eptr);
				eptr = endptr(0);
			}
			if (eptr)
				eptr->Update(rrq_ras), ep = eptr.operator->();
			else
				ep = new GatewayRec(rrq_ras, true);
			GatewayRec *gw = dynamic_cast<GatewayRec *>(ep);
			gw->AddPrefixes(sp[1]);
			gw->SortPrefixes();
		} else {
			rrq.m_terminalType.IncludeOptionalField(H225_EndpointType::e_terminal);
			if (eptr && eptr->IsGateway()) {
				RemoveByEndptr(eptr);
				eptr = endptr(0);
			}
			if (eptr)
				eptr->Update(rrq_ras);
			else
				ep = new EndpointRec(rrq_ras, true);
		}
		if (!eptr) {
			PTRACE(2, "Add permanent endpoint " << AsDotString(rrq.m_callSignalAddress[0]));
			WriteLock lock(listLock);
			EndpointList.push_back(ep);
			++regSize;
		}
	}
}

void RegistrationTable::ClearTable()
{
	WriteLock lock(listLock);
	// Unregister all endpoints, and move the records into RemovedList
	transform(EndpointList.begin(), EndpointList.end(),
		back_inserter(RemovedList), mem_fun(&EndpointRec::Unregister));
	EndpointList.clear();
	regSize = 0;
	copy(OuterZoneList.begin(), OuterZoneList.end(), back_inserter(RemovedList));
	OuterZoneList.clear();
}

void RegistrationTable::CheckEndpoints()
{
	PTime now;
	WriteLock lock(listLock);

	iterator Iter = EndpointList.begin(), eIter = EndpointList.end();
	while (Iter != eIter) {
		iterator i = Iter++;
		EndpointRec *ep = *i;
		if (!ep->IsUpdated(&now) && !ep->SendIRQ()) {
			SoftPBX::DisconnectEndpoint(endptr(ep));
			ep->Expired();
			RemovedList.push_back(ep);
			EndpointList.erase(i);
			--regSize;
			PTRACE(2, "Endpoint " << ep->GetEndpointIdentifier().GetValue() << " expired.");
		}
	}

	Iter = partition(OuterZoneList.begin(), OuterZoneList.end(),
		bind2nd(mem_fun(&EndpointRec::IsUpdated), &now));
#if PTRACING
	if (ptrdiff_t s = distance(Iter, OuterZoneList.end()))
		PTRACE(2, s << " outerzone endpoint(s) expired.");
#endif
	copy(Iter, OuterZoneList.end(), back_inserter(RemovedList));
	OuterZoneList.erase(Iter, OuterZoneList.end());

	// Cleanup unused EndpointRec in RemovedList
	Iter = partition(RemovedList.begin(), RemovedList.end(), mem_fun(&EndpointRec::IsUsed));
	for_each(Iter, RemovedList.end(), deleteobj<EndpointRec>());
	RemovedList.erase(Iter, RemovedList.end());
}

CallRec::CallRec(
	const H225_AdmissionRequest& arq,
	/// an endpoint that originated the ARQ
	const endptr& requestingep,
	/// called endpoint (if it is a registered endpoint)
	const endptr& calledep,
	/// bandwidth consumed by this call
	int bandwidth,
	/// whether to route H.245 channel (even if false, later we can decide to route H.245)
	bool routeH245,
	/// if the call will use gatekeeper routed signalling mode
	bool gkRoutedSignalling
	)
	: 
	m_callIdentifier(arq.m_callIdentifier), 
	m_conferenceIdentifier(arq.m_conferenceID),
	m_srcInfo(AsString(arq.m_srcInfo)), 
	m_bandWidth(bandwidth), m_CallNumber(0),
	m_Called(calledep),
	m_crv(arq.m_callReferenceValue.GetValue() & 0x7fffu),
	m_callingSocket(0), m_calledSocket(0),
	m_usedCount(0), m_nattype(none), m_h245Routed(routeH245),
	m_registered(false), m_forwarded(false),
	m_setupTime(0), m_disconnectTime(0),
	m_disconnectCause(0), m_srcSignalAddr(NULL), m_destSignalAddr(NULL)
#ifdef HAS_ACCT
	, m_acctEventsLogged(0)
#endif
{
	if( arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo) )
		m_destInfo = AsString(arq.m_destinationInfo);
	else
		m_destInfo = "unknown";

	CallTable* const ctable = CallTable::Instance();
	
	m_timer = m_creationTime = time(NULL);
	m_durationLimit = ctable->GetDefaultDurationLimit();
#ifdef HAS_ACCT
	m_lastAcctUpdateTime = m_creationTime;
#endif

	if( !gkRoutedSignalling ) {
		m_connectTime = m_creationTime;
		m_timeout = m_durationLimit;
	} else {
		m_connectTime = 0;
		m_timeout = ctable->GetConnectTimeout() / 1000;
	}
	
	if( (!arq.m_answerCall) && requestingep ) {
		m_Calling = requestingep;
		if( m_Calling->IsNATed() ) {
			m_nattype |= callingParty;
			m_h245Routed = true;
		}
		m_Calling->AddCall();
		if( m_connectTime )
			m_Calling->AddConnectedCall();
			
		if( m_Calling->GetCallSignalAddress().IsValid() )
			m_srcSignalAddr = (H225_TransportAddress*)
				(m_Calling->GetCallSignalAddress().Clone());
	} else {
		if( (!gkRoutedSignalling) 
			&& arq.HasOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress) )
			m_srcSignalAddr = 
				(H225_TransportAddress*)(arq.m_srcCallSignalAddress.Clone());
	}

	if( m_Called ) {
		if( m_Called->IsFromParent() )
			m_registered = true;
		if( m_Called->IsNATed() ) {
			m_nattype |= calledParty;
			m_h245Routed = true;
		}
		m_Called->AddCall();
		if( m_connectTime )
			m_Called->AddConnectedCall();
			
		if( m_Called->GetCallSignalAddress().IsValid() )
			m_destSignalAddr = (H225_TransportAddress*)
				(m_Called->GetCallSignalAddress().Clone());
	}
	
	if( m_destSignalAddr == NULL 
		&& arq.HasOptionalField(H225_AdmissionRequest::e_destCallSignalAddress) )
		m_destSignalAddr = (H225_TransportAddress*)
			(arq.m_destCallSignalAddress.Clone());
}

CallRec::CallRec(
	/// Q.931 Setup message received from the calling endpoint
	const Q931& q931pdu,
	/// Setup UUIE received from the calling endpoint
	const H225_Setup_UUIE& setup,
	/// timestamp for Setup message reception
	const time_t setupTime,
	/// signalling channel from the calling endpoint
	CallSignalSocket* callingSocket,
	/// a called endpoint (if it is a registered endpoint)
	endptr& calledEP,
	/// bandwidth consumed by this call
	int bandwidth,
	/// whether to route H.245 channel (even if false, later we can decide to route H.245)
	bool routeH245
#ifdef HAS_ACCT
	/// Calling-Station-Id
	, const PString& callingStationId,
	/// Called-Station-Id
	const PString& calledStationId
#endif
	)
	: 
	m_callIdentifier(setup.m_callIdentifier), 
	m_conferenceIdentifier(setup.m_conferenceID),
	m_bandWidth(bandwidth), m_CallNumber(0),
	m_Called(calledEP),
	m_crv(q931pdu.GetCallReference() & 0x7fffu),
	m_callingSocket(callingSocket), m_calledSocket(0),
	m_usedCount(0), m_nattype(none), m_h245Routed(routeH245),
	m_registered(false), m_forwarded(false),
	m_timeout(0),
	m_setupTime(setupTime), m_connectTime(0), m_disconnectTime(0),
	m_disconnectCause(0), m_srcSignalAddr(NULL), m_destSignalAddr(NULL)
#ifdef HAS_ACCT
	, m_callingStationId((const char*)callingStationId),
	m_calledStationId((const char*)calledStationId),
	m_acctEventsLogged(0)
#endif
{
	if( setup.HasOptionalField(H225_Setup_UUIE::e_sourceAddress) )
		m_srcInfo = AsString(setup.m_sourceAddress);

	// check if the destCallSignalAddress is a signaling address 
	// of the called party or the gk
	bool useDestCallSignalAddress = false;
	if( callingSocket && setup.HasOptionalField(H225_Setup_UUIE::e_destCallSignalAddress) ) {
		PIPSocket::Address sigaddr1,sigaddr2;
		WORD sigport1 = 0, sigport2;
		
		callingSocket->GetLocalAddress(sigaddr1,sigport1);
		
		useDestCallSignalAddress = !(GetIPAndPortFromTransportAddr(setup.m_destCallSignalAddress,sigaddr2,sigport2)
			&& sigaddr1.IsValid() && sigaddr2.IsValid() 
			&& (DWORD)sigaddr1 == (DWORD)sigaddr2 && sigport1 == sigport2);
	}
	
	if( m_Called && useDestCallSignalAddress
		&& m_Called->GetCallSignalAddress() == setup.m_destCallSignalAddress )
		m_destInfo = AsDotString(setup.m_destCallSignalAddress);
	else if( setup.HasOptionalField(H225_Setup_UUIE::e_destinationAddress) )
		m_destInfo = AsString(setup.m_destinationAddress);
	else if( useDestCallSignalAddress )
		m_destInfo = AsDotString(setup.m_destCallSignalAddress);

	if( m_Called ) {
		m_registered = m_Called->IsFromParent();
		if( m_Called->IsNATed() ) {
			m_nattype |= calledParty;
			m_h245Routed = true;
		}
		m_Called->AddCall();
		
		if( m_Called->GetCallSignalAddress().IsValid() )
			m_destSignalAddr = (H225_TransportAddress*)
				(m_Called->GetCallSignalAddress().Clone());
	}

	m_timer = m_creationTime = time(NULL);
	if( m_creationTime > m_setupTime )
		m_creationTime = m_setupTime;

	CallTable* const ctable = CallTable::Instance();
	m_timeout = ctable->GetConnectTimeout() / 1000;
	m_durationLimit = ctable->GetDefaultDurationLimit();
#ifdef HAS_ACCT
	m_lastAcctUpdateTime = m_creationTime;
#endif

	if( setup.HasOptionalField(H225_Setup_UUIE::e_sourceCallSignalAddress) )
		m_srcSignalAddr = 
			(H225_TransportAddress*)(setup.m_sourceCallSignalAddress.Clone());
	else if( m_callingSocket ) {
		m_srcSignalAddr = new H225_TransportAddress(H225_TransportAddress::e_ipAddress);
		
		PIPSocket::Address addr;
		WORD port = 0;
		m_callingSocket->GetPeerAddress(addr,port);
		
		H225_TransportAddress_ipAddress& ipport = *m_srcSignalAddr;
		ipport.m_ip[0] = addr[0];
		ipport.m_ip[1] = addr[1];
		ipport.m_ip[2] = addr[2];
		ipport.m_ip[3] = addr[3];
		ipport.m_port = port;
	}
	
	if( m_destSignalAddr == NULL && useDestCallSignalAddress )
		m_destSignalAddr = (H225_TransportAddress*)
			(setup.m_destCallSignalAddress.Clone());
}

CallRec::~CallRec()
{
	PTRACE(3, "Gk\tDelete Call No. " << m_CallNumber);
	delete m_srcSignalAddr;
	delete m_destSignalAddr;
}

/*
void CallRec::SetCalling(const endptr & NewCalling, unsigned crv)
{
	PWaitAndSignal lock(m_usedLock);
	if (m_Calling != NewCalling) {
		if (m_Calling)
			m_Calling->RemoveCall();
		m_Calling = NewCalling;
		if (crv)
			m_callingCRV = crv;
		if (m_Calling) {
			m_Calling->AddCall();
			
			m_nattype &= ~callingParty;
			if (m_Calling->IsNATed()) {
				m_nattype |= callingParty;
				m_h245Routed = true;
			}
			if (m_Calling->GetCallSignalAddress().IsValid()) {
				delete m_srcSignalAddr;
				m_srcSignalAddr = (H225_TransportAddress*)
					(m_Calling->GetCallSignalAddress().Clone());
			}
		}
	}
}
*/

void CallRec::InternalSetCalled(
	const endptr& newCalled
	)
{
	if (m_Called != newCalled) {
		if (m_Called)
			m_Called->RemoveCall();
		
		m_Called = newCalled;
		
		if (m_Called) {
			m_Called->AddCall();
			
			if (m_Called->IsFromParent())
				SetRegistered(true);
			
			m_nattype &= ~calledParty;
			if (m_Called->IsNATed()) {
				m_nattype |= calledParty;
				m_h245Routed = true;
			}
			if( m_Called->GetCallSignalAddress().IsValid() ) {
				delete m_destSignalAddr;
				m_destSignalAddr = (H225_TransportAddress*)
					(m_Called->GetCallSignalAddress().Clone());
			}
		}
	}
}

void CallRec::SetForward(CallSignalSocket *socket, const endptr & forwarded, const PString & forwarder, const PString & altDestInfo)
{
	PWaitAndSignal lock(m_usedLock);
	
	m_forwarded = true;
	m_Forwarder = (socket == m_calledSocket) ? m_Called : endptr(0);
	if (!forwarder)
		m_srcInfo += "=" + forwarder;
	m_destInfo = altDestInfo;
	m_nattype &= ~calledParty;
	// FIXME: how about m_registered and m_h245Routed?
	InternalSetCalled(forwarded);
}

void CallRec::SetConnected()
{
	SetConnectTime(time(NULL));
	
	if (m_Calling)
		m_Calling->AddConnectedCall();
	if (m_Called)
		m_Called->AddConnectedCall();
}

void CallRec::SetDurationLimit(long seconds)
{
	PWaitAndSignal lock(m_usedLock);
	// allow only to restrict duration limit
	const long sec = (m_durationLimit && seconds) 
		? PMIN(m_durationLimit,seconds) : PMAX(m_durationLimit,seconds);
	m_durationLimit = sec;
	if (IsConnected())
		m_timeout = sec;
}

/*
void CallRec::InternalSetEP(endptr & ep, const endptr & nep)
{
	if (ep != nep) {
		if (ep)
			ep->RemoveCall();
		m_usedLock.Wait();
		ep = nep;
		m_usedLock.Signal();
		if (ep)
			ep->AddCall();
	}
}
*/

void CallRec::RemoveAll()
{
	if (IsRegistered()) {
		H225_RasMessage ras_msg;
		BuildDRQ(ras_msg, H225_DisengageReason::e_normalDrop);
		RasThread->GetGkClient()->SendDRQ(ras_msg);
	}
	if (m_Calling)
		m_Calling->RemoveCall();
	if (m_Called)
		m_Called->RemoveCall();
}

void CallRec::RemoveSocket()
{
	if (m_sockLock.WillBlock()) // locked by SendReleaseComplete()?
		return; // avoid deadlock

	PWaitAndSignal lock(m_sockLock);
	if (m_callingSocket) {
		m_callingSocket->SetDeletable();
		m_callingSocket = 0;
	}

	// m_calledSocket may be an invalid pointer
//	if (m_calledSocket) {
//		m_calledSocket->SetDeletable();
		m_calledSocket = 0;
//	}
}

int CallRec::CountEndpoints() const
{
	PWaitAndSignal lock(m_usedLock);
	int result = 0;
	if (m_Calling)
		++result;
	if (m_Called)
		++result;
	return result;
}

void CallRec::Disconnect(bool force)
{
	if ((force || Toolkit::AsBool(GkConfig()->GetString(RoutedSec, "DropCallsByReleaseComplete", "0"))) && (m_callingSocket || m_calledSocket))
		SendReleaseComplete();
	else
		SendDRQ();

	PTRACE(2, "Gk\tDisconnect Call No. " << m_CallNumber);
}

void CallRec::SendReleaseComplete(const H225_CallTerminationCause *cause)
{
	m_sockLock.Wait();
	if (m_callingSocket) {
		PTRACE(4, "Sending ReleaseComplete to calling party ...");
		m_callingSocket->SendReleaseComplete(cause);
	}
	if (m_calledSocket) {
		PTRACE(4, "Sending ReleaseComplete to called party ...");
		m_calledSocket->SendReleaseComplete(cause);
	}
	m_sockLock.Signal();

	RemoveSocket();
}

void CallRec::BuildDRQ(H225_RasMessage & ras_msg, unsigned reason)
{
	ras_msg.SetTag(H225_RasMessage::e_disengageRequest);
	H225_DisengageRequest & drq = ras_msg;
	drq.m_requestSeqNum.SetValue(RasThread->GetRequestSeqNum());
	drq.m_disengageReason.SetTag(reason);
	drq.m_conferenceID = m_conferenceIdentifier;
	drq.IncludeOptionalField(H225_DisengageRequest::e_callIdentifier);
	drq.m_callIdentifier = m_callIdentifier;
	drq.m_callReferenceValue = m_Calling ? m_crv : (m_crv | 0x8000u);
}

void CallRec::SendDRQ()
{
	H225_RasMessage ras_msg;
	BuildDRQ(ras_msg, H225_DisengageReason::e_forcedDrop);
	H225_DisengageRequest & drq = ras_msg;
	drq.IncludeOptionalField(H225_DisengageRequest::e_gatekeeperIdentifier);
	drq.m_gatekeeperIdentifier = Toolkit::GKName();

// Warning: For an outer zone endpoint, the endpoint identifier may not correct
	if (m_Called) {
		drq.m_endpointIdentifier = m_Called->GetEndpointIdentifier();
		RasThread->SendRas(ras_msg, m_Called->GetRasAddress());
	}
	if (m_Calling) {
		drq.m_endpointIdentifier = m_Calling->GetEndpointIdentifier();
		drq.m_callReferenceValue = m_crv;
		RasThread->SendRas(ras_msg, m_Calling->GetRasAddress());
	}
}

namespace { // end of anonymous namespace

PString GetEPString(const endptr & ep, const CallSignalSocket *socket)
{
	if (ep) {
		return PString(PString::Printf, "%s|%s",
			(const char *)AsDotString(ep->GetCallSignalAddress()),
			(const char *)ep->GetEndpointIdentifier().GetValue());
	}
	return socket ? socket->Name() + "| " : PString(" | ");
}

PString GetEPString(const endptr & ep, const H225_TransportAddress* sigaddr)
{
	if (ep) {
		return PString(PString::Printf, "%s|%s",
			(const char *)AsDotString(ep->GetCallSignalAddress()),
			(const char *)ep->GetEndpointIdentifier().GetValue());
	}
	return sigaddr ? AsDotString(*sigaddr) + "| " : PString(" | ");
}

} // end of anonymous namespace

PString CallRec::GenerateCDR() const
{
	PString timeString;
	
	const time_t eTime = m_disconnectTime ? m_disconnectTime : time(NULL);
	const PTime endTime(eTime);
	
	if (m_connectTime != 0) {
		// if connected, reported duration should be at least 1 second
		const PTime startTime(m_connectTime);
		timeString = PString(PString::Printf, "%ld|%s|%s",
			(eTime > m_connectTime)?(eTime - m_connectTime):1,
			(const char *)startTime.AsString(),
			(const char *)endTime.AsString()
		);
	} else {
		timeString = "0|unconnected|" + endTime.AsString();
	}

	return PString(PString::Printf, "CDR|%d|%s|%s|%s|%s|%s|%s|%s;",
		m_CallNumber,
		(const char *)AsString(m_callIdentifier.m_guid),
		(const char *)timeString,
		(const char *)GetEPString(m_Forwarder ? m_Forwarder : m_Calling, m_srcSignalAddr),
		(const char *)GetEPString(m_Called, m_destSignalAddr),
		(const char *)m_destInfo,
		(const char *)m_srcInfo,
		(const char *)Toolkit::Instance()->GKName()
	);
}

PString CallRec::PrintOn(bool verbose) const
{
	int timer = time(NULL) - m_timer;
	int left = (m_timeout > 0 ) ? m_timeout - timer : 0;
	PString result(PString::Printf,
		"Call No. %d | CallID %s | %d | %d\r\nDial %s\r\nACF|%s|%d|%s|%s|false;\r\nACF|%s|%d|%s|%s|true;\r\n",
		m_CallNumber, (const char *)AsString(m_callIdentifier.m_guid), timer, left,
		(const char *)m_destInfo,
		(const char *)GetEPString(m_Calling, m_srcSignalAddr), 
		m_crv,
		(const char*)m_destInfo,(const char*)m_srcInfo,
		(const char *)GetEPString(m_Called, m_destSignalAddr), 
		m_crv | 0x8000u,
		(const char*)m_destInfo,
		(const char*)m_srcInfo
	);
	if (verbose) {
		result += PString(PString::Printf, "# %s|%s|%d|%s <%d>\r\n",
				(m_Calling) ? (const char *)AsString(m_Calling->GetAliases()) : "?",
				(m_Called) ? (const char *)AsString(m_Called->GetAliases()) : "?",
				m_bandWidth,
				m_connectTime ? (const char *)PTime(m_connectTime).AsString() : "unconnected",
				m_usedCount
			  );
	}

	return result;
}

void CallRec::SetSetupTime(time_t tm)
{
	PWaitAndSignal lock(m_usedLock);
	if( m_setupTime == 0 )
		m_setupTime = tm;
	if( m_connectTime == 0 )
		m_timer = tm;
	// can be the case, because CallRec is usually created
	// after Setup message has been received
	if( m_creationTime > m_setupTime )
		m_creationTime = m_setupTime;
}

void CallRec::SetConnectTime(time_t tm)
{
	PWaitAndSignal lock(m_usedLock);
	if( m_connectTime == 0 ) {
		m_timer = m_connectTime = tm;
		m_timeout = m_durationLimit;
		if( m_disconnectTime && m_disconnectTime <= m_connectTime )
			m_disconnectTime = m_connectTime+1;
	}
	// can be the case for direct signalling mode, 
	// because CallRec is usually created after ARQ message 
	// has been received
	if( m_creationTime > m_connectTime )
		m_creationTime = m_connectTime;
}

void CallRec::SetDisconnectTime(time_t tm)
{
	PWaitAndSignal lock(m_usedLock);
	if( m_disconnectTime == 0 )
		m_disconnectTime = (m_connectTime && m_connectTime >= tm)
			? (m_connectTime + 1) : tm;
}

long CallRec::GetDuration() const
{
	PWaitAndSignal lock(m_usedLock);
	if( m_connectTime ) {
		if( m_disconnectTime )
			return (m_disconnectTime > m_connectTime) 
				? (m_disconnectTime-m_connectTime) : 1;
		else
			return (long)time(NULL) - m_connectTime;
	} else
		return 0;
}

bool CallRec::GetSrcSignalAddr(
	/// IPv4 address to be returned
	PIPSocket::Address& addr, 
	/// port number to be returned
	WORD& port 
	) const
{
	PWaitAndSignal lock(m_usedLock);
	return m_srcSignalAddr 
		&& GetIPAndPortFromTransportAddr(*m_srcSignalAddr,addr,port);
}

bool CallRec::GetSrcSignalAddr( 
	/// H.225 transport address to be returned
	H225_TransportAddress& taddr 
	) const
{
	PWaitAndSignal lock(m_usedLock);
	if( m_srcSignalAddr && m_srcSignalAddr->IsValid() ) {
		taddr = *m_srcSignalAddr;
		return true;
	} else
		return false;
}

bool CallRec::SetSrcSignalAddr(
	/// the new H.225 transport address to be set
	const H225_TransportAddress& taddr
	)
{
	if( taddr.IsValid() ) {
		PWaitAndSignal lock(m_usedLock);
		delete m_srcSignalAddr;
		m_srcSignalAddr = (H225_TransportAddress*)(taddr.Clone());
		return true;
	} else
		return false;
}

bool CallRec::SetSrcSignalAddr(
	/// the new IPv4 address to be set
	const PIPSocket::Address& addr,
	/// the new port number to be set
	const WORD port
	)
{
	if( addr.IsValid() ) {
		PWaitAndSignal lock(m_usedLock);
		
		delete m_srcSignalAddr;
		m_srcSignalAddr = new H225_TransportAddress(H225_TransportAddress::e_ipAddress);
		H225_TransportAddress_ipAddress& ipport = *m_srcSignalAddr;
		ipport.m_ip[0] = addr[0];
		ipport.m_ip[1] = addr[1];
		ipport.m_ip[2] = addr[2];
		ipport.m_ip[3] = addr[3];
		ipport.m_port = port;
		return true;
	} else
		return false;
}

bool CallRec::GetDestSignalAddr(
	/// IPv4 address to be returned
	PIPSocket::Address& addr, 
	/// port number to be returned
	WORD& port 
	) const
{
	PWaitAndSignal lock(m_usedLock);
	return m_destSignalAddr 
		&& GetIPAndPortFromTransportAddr(*m_destSignalAddr,addr,port);
}

bool CallRec::GetDestSignalAddr( 
	/// H.225 transport address to be returned
	H225_TransportAddress& taddr 
	) const
{
	PWaitAndSignal lock(m_usedLock);
	if( m_destSignalAddr && m_destSignalAddr->IsValid() ) {
		taddr = *m_destSignalAddr;
		return true;
	} else
		return false;
}

bool CallRec::SetDestSignalAddr(
	/// the new H.225 transport address to be set
	const H225_TransportAddress& taddr
	)
{
	if( taddr.IsValid() ) {
		PWaitAndSignal lock(m_usedLock);
		delete m_destSignalAddr;
		m_destSignalAddr = (H225_TransportAddress*)(taddr.Clone());
		return true;
	} else
		return false;
}

bool CallRec::SetDestSignalAddr(
	/// the new IPv4 address to be set
	const PIPSocket::Address& addr,
	/// the new port number to be set
	const WORD port
	)
{
	if( addr.IsValid() ) {
		PWaitAndSignal lock(m_usedLock);
		
		delete m_destSignalAddr;
		m_destSignalAddr = new H225_TransportAddress(H225_TransportAddress::e_ipAddress);
		H225_TransportAddress_ipAddress& ipport = *m_destSignalAddr;
		ipport.m_ip[0] = addr[0];
		ipport.m_ip[1] = addr[1];
		ipport.m_ip[2] = addr[2];
		ipport.m_ip[3] = addr[3];
		ipport.m_port = port;
		return true;
	} else
		return false;
}

#ifdef HAS_ACCT
PString CallRec::GetCallingStationId()
{
	PWaitAndSignal lock(m_usedLock);
	return m_callingStationId;
}

void CallRec::SetCallingStationId( const PString& id )
{
	PWaitAndSignal lock(m_usedLock);
	m_callingStationId = id;
}

PString CallRec::GetCalledStationId()
{
	PWaitAndSignal lock(m_usedLock);
	return m_calledStationId;
}

void CallRec::SetCalledStationId( const PString& id )
{
	PWaitAndSignal lock(m_usedLock);
	m_calledStationId = id;
}

PString CallRec::GetAcctSessionId()
{
	PWaitAndSignal lock(m_usedLock);
	return m_acctSessionId;
}

void CallRec::SetAcctSessionId( const PString& id )
{
	PWaitAndSignal lock(m_usedLock);
	m_acctSessionId = id;
}

#endif

CallTable::CallTable() : m_CallNumber(0), m_capacity(-1)
{
	LoadConfig();
	m_CallCount = m_successCall = m_neighborCall = m_parentCall = m_activeCall = 0;
}

CallTable::~CallTable()
{
	ForwardedCallList.clear();
        for_each(CallList.begin(), CallList.end(),
		bind1st(mem_fun(&CallTable::InternalRemovePtr), this));
	DeleteObjectsInContainer<CallRec>(RemovedList);
}

void CallTable::LoadConfig()
{
	ForwardedCallList.clear();
	m_genNBCDR = Toolkit::AsBool(GkConfig()->GetString(CallTableSection, "GenerateNBCDR", "1"));
	m_genUCCDR = Toolkit::AsBool(GkConfig()->GetString(CallTableSection, "GenerateUCCDR", "0"));
	SetTotalBandWidth(GkConfig()->GetInteger("TotalBandwidth", m_capacity));
	// this timeout is stored in CallTable, not in CallRec
	// because it is constant for all CallRecs and should not add another
	// 4 bytes to each CallRec unnecessary
	m_connectTimeout = PMAX(
		GkConfig()->GetInteger(RoutedSec, "ConnectTimeout",DEFAULT_CONNECT_TIMEOUT),
		5000
		);
	m_defaultDurationLimit = GkConfig()->GetInteger(
		CallTableSection, "DefaultCallDurationLimit", 0
		);
	// backward compatibility - check DefaultCallTimeout
	if (m_defaultDurationLimit == 0)
		m_defaultDurationLimit = GkConfig()->GetInteger(
			CallTableSection, "DefaultCallTimeout", 0
			);
#ifdef HAS_ACCT
	m_acctUpdateInterval = GkConfig()->GetInteger(CallTableSection, "AcctUpdateInterval", 0);
	if( m_acctUpdateInterval != 0 )
		m_acctUpdateInterval = PMAX(m_acctUpdateInterval,10);
#endif
}

void CallTable::Insert(CallRec * NewRec)
{
	NewRec->SetCallNumber(++m_CallNumber);
	WriteLock lock(listLock);
	CallList.push_back(NewRec);
	++m_CallCount, ++m_activeCall;
	PTRACE(2, "CallTable::Insert(CALL) Call No. " << m_CallNumber << ", total sessions : " << m_activeCall);
}

void CallTable::SetTotalBandWidth(int bw)
{
	if ((m_capacity = bw) >= 0) {
		int used = 0;
		WriteLock lock(listLock);
		iterator Iter = CallList.begin(), eIter = CallList.end();
		while (Iter != eIter)
			used += (*Iter++)->GetBandWidth();
		if (bw > used)
			m_capacity -= used;
		else
			m_capacity = 0;
	}
}

bool CallTable::GetAdmission(int bw)
{
	if (m_capacity < 0)
		return true;
	if (m_capacity < bw)
		return false;

	m_capacity -= bw;
	PTRACE(2, "GK\tAvailable Bandwidth " << m_capacity);
	return true;
}

bool CallTable::GetAdmission(int bw, const callptr & call)
{
	return GetAdmission(bw - call->GetBandWidth());
}

callptr CallTable::FindCallRec(const H225_CallIdentifier & CallId) const
{
	return InternalFind(bind2nd(mem_fun(&CallRec::CompareCallId), &CallId));
}

callptr CallTable::FindCallRec(const H225_CallReferenceValue & CallRef) const
{
	return InternalFind(bind2nd(mem_fun(&CallRec::CompareCRV), CallRef.GetValue()));
}

callptr CallTable::FindCallRec(PINDEX CallNumber) const
{
	return InternalFind(bind2nd(mem_fun(&CallRec::CompareCallNumber), CallNumber));
}

callptr CallTable::FindCallRec(const endptr & ep) const
{
	return InternalFind(bind2nd(mem_fun(&CallRec::CompareEndpoint), &ep));
}

callptr CallTable::FindBySignalAdr(const H225_TransportAddress & SignalAdr) const
{
	return InternalFind(bind2nd(mem_fun(&CallRec::CompareSigAdr), &SignalAdr));
}

void CallTable::ClearTable()
{
	WriteLock lock(listLock);
	iterator Iter = CallList.begin(), eIter = CallList.end();
	while (Iter != eIter) {
		iterator i = Iter++;
		(*i)->Disconnect();
		InternalRemove(i);
	}
}

void CallTable::CheckCalls()
{
#ifdef HAS_ACCT
	long acctinterval = m_acctUpdateInterval;
	if( acctinterval > 0 )
		ConfigReloadMutex.StartRead();
#endif
	const time_t now = time(NULL);
	WriteLock lock(listLock);
	iterator Iter = CallList.begin(), eIter = CallList.end();
	while (Iter != eIter) {
		iterator i = Iter++;
		if ((*i)->IsTimeout(now)) {
			(*i)->Disconnect();
			InternalRemove(i);
		} 
#ifdef HAS_ACCT
		else if( acctinterval > 0 && (*i)->IsConnected() ) {
			if( (now - (*i)->GetLastAcctUpdateTime()) >= acctinterval ) {
				GkAcctLoggers* acct = Toolkit::Instance()->GetAcctList();
				if( acct != NULL ) {
					callptr cptr(*i);
					if( !acct->LogAcctEvent( GkAcctLogger::AcctUpdate, cptr, now ) )
						PTRACE(3,"GKACCT\tAcctUpdate event log failed for call no. "
							<<(*i)->GetCallNumber()
							);
				}
			}
		}
#endif
	}
#ifdef HAS_ACCT
	if( acctinterval > 0 )
		ConfigReloadMutex.EndRead();
#endif

	Iter = partition(RemovedList.begin(), RemovedList.end(), mem_fun(&CallRec::IsUsed));
	for_each(Iter, RemovedList.end(), deleteobj<CallRec>());
	RemovedList.erase(Iter, RemovedList.end());
}

void CallTable::RemoveCall(const H225_DisengageRequest & obj_drq, const endptr & ep)
{
	callptr call = obj_drq.HasOptionalField(H225_DisengageRequest::e_callIdentifier) ? FindCallRec(obj_drq.m_callIdentifier) : FindCallRec(obj_drq.m_callReferenceValue.GetValue());
	if (call) {
		if (ep == call->GetForwarder())
			return;

		if (ep != call->GetCallingParty() && ep != call->GetCalledParty()) {
			PTRACE(3, "GK\tWarning: CallRec doesn't belong to the requesting endpoint!");
			return;
		}
		if (Toolkit::AsBool(GkConfig()->GetString(RoutedSec, "SendReleaseCompleteOnDRQ", "0")))
			call->SendReleaseComplete(obj_drq.HasOptionalField(H225_DisengageRequest::e_terminationCause) ? &obj_drq.m_terminationCause : 0);
		RemoveCall(call);
	}
}

void CallTable::RemoveCall(const callptr & call)
{
	if (call)
		InternalRemovePtr(call.operator->());
}

bool CallTable::InternalRemovePtr(CallRec *call)
{
	PTRACE(6, ANSI::PIN << "GK\tRemoving callptr:" << AsString(call->GetCallIdentifier().m_guid) << "...\n" << ANSI::OFF);
	WriteLock lock(listLock);
	InternalRemove(find(CallList.begin(), CallList.end(), call));
	return true; // useless, workaround for VC
}

void CallTable::InternalRemove(const H225_CallIdentifier & CallId)
{
	PTRACE(5, ANSI::PIN << "GK\tRemoving CallId:" << AsString(CallId.m_guid) << "...\n" << ANSI::OFF);
	WriteLock lock(listLock);
	InternalRemove(
		find_if(CallList.begin(), CallList.end(),
		bind2nd(mem_fun(&CallRec::CompareCallId), &CallId))
	);
}

void CallTable::InternalRemove(unsigned CallRef)
{
	PTRACE(5, ANSI::PIN << "GK\tRemoving CallRef:" << CallRef << "...\n" << ANSI::OFF);
	WriteLock lock(listLock);
	InternalRemove(
		find_if(CallList.begin(), CallList.end(),
		bind2nd(mem_fun(&CallRec::CompareCRV), CallRef))
	);
}

void CallTable::InternalRemove(iterator Iter)
{
	if (Iter == CallList.end()) {
		return;
	}

	CallRec *call = *Iter;
	
	call->SetDisconnectTime(time(NULL));
	
	if ((m_genNBCDR || call->GetCallingAddress()) && (m_genUCCDR || call->IsConnected())) {
		PString cdrString(call->GenerateCDR() + "\r\n");
		GkStatus::Instance()->SignalStatus(cdrString, 1);
		PTRACE(1, cdrString);
#if PTRACING
	} else {
		if (!call->IsConnected())
			PTRACE(2, "CDR\tignore not connected call");
		else	
			PTRACE(2, "CDR\tignore caller from neighbor");
#endif
	}

#ifdef HAS_ACCT
	{
		ReadLock cfgLock(ConfigReloadMutex);
				
		GkAcctLoggers* acct = Toolkit::Instance()->GetAcctList();
		if( acct ) {
			if( call->GetAcctEventsLogged() & GkAcctLogger::AcctStop )
				PTRACE(5,"GKACCT\tAcct-Stop already logged for call no "
					<<call->GetCallNumber()
					);
			else {
				call->SetAcctEventsLogged(GkAcctLogger::AcctStop);
				callptr cptr(call);
				if( !acct->LogAcctEvent( GkAcctLogger::AcctStop, cptr ) )
					PTRACE(3,"GKACCT\tAcctStop log failed for call no. "
						<<call->GetCallNumber()
						);
			}
		}
	}
#endif

	--m_activeCall;
	if (call->IsConnected())
		++m_successCall;
	if (!call->GetCallingParty())
		++(call->IsRegistered() ? m_parentCall : m_neighborCall);

	call->RemoveAll();
	call->RemoveSocket();
	if (m_capacity >= 0)
		m_capacity += call->GetBandWidth();

	RemovedList.push_back(call);
	CallList.erase(Iter);
}

void CallTable::InternalStatistics(unsigned & n, unsigned & act, unsigned & nb, unsigned & np, PString & msg, BOOL verbose) const
{
	ReadLock lock(listLock);
	n = m_activeCall, act = nb = np = 0;
	const_iterator eIter = CallList.end();
	for (const_iterator Iter = CallList.begin(); Iter != eIter; ++Iter) {
		CallRec *call = *Iter;
		if (call->IsConnected())
			++act;
		if (!call->GetCallingParty())
			++(call->IsRegistered() ? np : nb);
		if (!msg)
			msg += call->PrintOn(verbose);
	}
}

void CallTable::PrintCurrentCalls(GkStatus::Client & client, BOOL verbose) const
{
	PString msg = "CurrentCalls\r\n";
	unsigned n, act, nb, np;
	InternalStatistics(n, act, nb, np, msg, verbose);
	
	PString bandstr;
	if (m_capacity >= 0)
		bandstr = PString(PString::Printf, "\r\nAvailable Bandwidth: %u", m_capacity);
	msg += PString(PString::Printf, "Number of Calls: %u Active: %u From Neighbor: %u From Parent: %u%s\r\n;\r\n", n, act, nb, np, (const char *)bandstr);
	client.WriteString(msg);
}

PString CallTable::PrintStatistics() const
{
	PString dumb;
	unsigned n, act, nb, np;
	InternalStatistics(n, act, nb, np, dumb, FALSE);

	return PString(PString::Printf, "-- Call Statistics --\r\n"
		"Current Calls: %u Active: %u From Neighbor: %u From Parent: %u\r\n"
		"Total Calls: %u  Successful: %u  From Neighbor: %u  From Parent: %u\r\n",
		n, act, nb, np,
		m_CallCount, m_successCall, m_neighborCall, m_parentCall);
}

/*
void CallTable::AddForwardedCall(const callptr & call)
{
	WriteLock lock(flistLock);
	ForwardedCallList.push_front(call);
	PTRACE(3, "GK\tMark call " << call->GetCallNumber() << " as forwarded, size " << ForwardedCallList.size());
	if (ForwardedCallList.size() > 100) // todo: be an option
		ForwardedCallList.pop_back();
}

endptr CallTable::IsForwardedCall(const callptr & call)
{
	endptr ep;
	ReadLock lock(flistLock);
	for (std::list<callptr>::iterator b = ForwardedCallList.begin(), e = ForwardedCallList.end(); b != e; ++b)
		if ((*b)->GetCallIdentifier() == call->GetCallIdentifier()) {
			ep = (*b)->GetCalledParty();
			PTRACE_IF(3, ep, "GK\tForwarded call from " << ep->GetEndpointIdentifier().GetValue() << " detected, size " << ForwardedCallList.size());
			ForwardedCallList.erase(b);
			break;
		}
	return ep;
}
*/
