/* config.c - 2000/06/21 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2003  Jerome Couderc <j.couderc@ifrance.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>

#include "setting.h"
#include "prefs.h"
#include "bar.h"
#include "easytag.h"
#include "charset.h"
#include "scan.h"
#include "misc.h"
#include "cddb.h"
#include "i18n.h"


/***************
 * Declaration *
 ***************/

/*
 * Nota :
 *  - no trailing slashes on directory name to avoid problem with
 *    NetBSD's mkdir(2).
 */

// Base directory created into home dir 
gchar *EASYTAG_DIR                                 = ".easytag";
// File for configuration
gchar *CONFIG_FILE                                 = ".easytag/easytagrc";
// File of masks for tag scanner
gchar *SCAN_TAG_MASKS_FILE                         = ".easytag/scan_tag.mask";
// File of masks for rename file scanner
gchar *RENAME_FILE_MASKS_FILE                      = ".easytag/rename_file.mask";
// File for history of DefaultPathToMp3 combobox
gchar *DEFAULT_PATH_TO_MP3_HISTORY_FILE            = ".easytag/default_path_to_mp3.history";
// File for history of DefaultComment combobox
gchar *DEFAULT_TAG_COMMENT_HISTORY_FILE            = ".easytag/default_tag_comment.history";
// File for history of BrowserEntry combobox
gchar *PATH_ENTRY_HISTORY_FILE                     = ".easytag/browser_path.history";
// File for history of PlayListNameEntry combobox
gchar *PLAY_LIST_NAME_HISTORY_FILE                 = ".easytag/play_list_name.history";
// File for history of run program combobox for directories
gchar *RUN_PROGRAM_WITH_DIRECTORY_HISTORY_FILE     = ".easytag/run_program_with_directory.history";
// File for history of run program combobox for files
gchar *RUN_PROGRAM_WITH_FILE_HISTORY_FILE          = ".easytag/run_program_with_file.history";
// File for history of run player combobox
gchar *AUDIO_FILE_PLAYER_HISTORY_FILE              = ".easytag/audio_file_player.history";
// File for history of search string combobox
gchar *SEARCH_FILE_HISTORY_FILE                    = ".easytag/search_file.history";
// File for history of FileToLoad combobox
gchar *FILE_TO_LOAD_HISTORY_FILE                   = ".easytag/file_to_load.history";
// File for history of PlayListContentMaskEntry combobox
gchar *PLAYLIST_CONTENT_HISTORY_FILE               = ".easytag/playlist_content.history";
// File for history of CddbSearchStringEntry combobox
gchar *CDDB_SEARCH_STRING_HISTORY_FILE             = ".easytag/cddb_search_string.history";
// File for history of CddbSearchStringInResultEntry combobox
gchar *CDDB_SEARCH_STRING_IN_RESULT_HISTORY_FILE   = ".easytag/cddb_search_string_in_result.history";



/**************
 * Prototypes *
 **************/



/********************
 * Config Variables *
 ********************/
tConfigVariable Config_Variables[] =
{
    {"load_on_startup",                CV_TYPE_BOOL,    &LOAD_ON_STARTUP            },
    {"default_path_to_mp3",            CV_TYPE_STRING,  &DEFAULT_PATH_TO_MP3        },
    {"browser_line_style",             CV_TYPE_BOOL,    &BROWSER_LINE_STYLE         },
    {"browser_expander_style",         CV_TYPE_BOOL,    &BROWSER_EXPANDER_STYLE     },
    {"write_id3v1_tag",                CV_TYPE_BOOL,    &WRITE_ID3V1_TAG            },
    {"write_id3v2_tag",                CV_TYPE_BOOL,    &WRITE_ID3V2_TAG            },
    {"strip_tag_when_empty_fields",    CV_TYPE_BOOL,    &STRIP_TAG_WHEN_EMPTY_FIELDS},
    {"convert_old_id3v2_tag_version",  CV_TYPE_BOOL,    &CONVERT_OLD_ID3V2_TAG_VERSION},
    {"browse_subdir",                  CV_TYPE_BOOL,    &BROWSE_SUBDIR              },
    {"open_selected_browser_node",     CV_TYPE_BOOL,    &OPEN_SELECTED_BROWSER_NODE },
    {"keep_tree_browser_in_memory",    CV_TYPE_BOOL,    &KEEP_TREE_BROWSER_IN_MEMORY},
    {"save_config_before_exit",        CV_TYPE_BOOL,    &SAVE_CONFIG_BEFORE_EXIT    },

    {"set_main_window_size",           CV_TYPE_BOOL,    &SET_MAIN_WINDOW_SIZE       },
    {"main_window_height",             CV_TYPE_INT,     &MAIN_WINDOW_HEIGHT         },
    {"main_window_width",              CV_TYPE_INT,     &MAIN_WINDOW_WIDTH          },
    {"set_pane_handle_position1",      CV_TYPE_INT,     &SET_PANE_HANDLE_POSITION1  },
    {"pane_handle_position1",          CV_TYPE_INT,     &PANE_HANDLE_POSITION1      },
    {"set_pane_handle_position2",      CV_TYPE_INT,     &SET_PANE_HANDLE_POSITION2  },
    {"pane_handle_position2",          CV_TYPE_INT,     &PANE_HANDLE_POSITION2      },
    {"show_header_infos",              CV_TYPE_BOOL,    &SHOW_HEADER_INFO           },
    {"use_colored_tooltips",           CV_TYPE_BOOL,    &USE_COLORED_TOOLTIPS       },

    {"replace_illegal_character_in_filename",CV_TYPE_BOOL,    &REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME},
    {"date_auto_completion",                 CV_TYPE_BOOL,    &DATE_AUTO_COMPLETION                 },
    {"number_track_formated",                CV_TYPE_BOOL,    &NUMBER_TRACK_FORMATED                },
    {"number_track_formated_spin_button",    CV_TYPE_INT,     &NUMBER_TRACK_FORMATED_SPIN_BUTTON    },
    {"ogg_tag_write_xmms_comment",           CV_TYPE_BOOL,    &OGG_TAG_WRITE_XMMS_COMMENT           },
    {"sorting_file_mode",                    CV_TYPE_BOOL,    &SORTING_FILE_MODE                    },
    {"sorting_file_case_sensitive",          CV_TYPE_BOOL,    &SORTING_FILE_CASE_SENSITIVE          },

    {"use_character_set_translation",        CV_TYPE_BOOL,    &USE_CHARACTER_SET_TRANSLATION        },
    {"file_character_set",                   CV_TYPE_STRING,  &FILE_CHARACTER_SET                   },
    {"user_character_set",                   CV_TYPE_STRING,  &USER_CHARACTER_SET                   },

    {"message_box_position_none",            CV_TYPE_BOOL,    &MESSAGE_BOX_POSITION_NONE            },
    {"message_box_position_center",          CV_TYPE_BOOL,    &MESSAGE_BOX_POSITION_CENTER          },
    {"message_box_position_mouse",           CV_TYPE_BOOL,    &MESSAGE_BOX_POSITION_MOUSE           },

    {"audio_file_player",                    CV_TYPE_STRING,  &AUDIO_FILE_PLAYER                    },

    {"convert_underscore_and_p20_into_space",CV_TYPE_BOOL,    &CONVERT_UNDERSCORE_AND_P20_INTO_SPACE},
    {"convert_space_into_underscore",        CV_TYPE_BOOL,    &CONVERT_SPACE_INTO_UNDERSCORE        },
    {"overwrite_tag_field",                  CV_TYPE_BOOL,    &OVERWRITE_TAG_FIELD                  },
    {"set_default_comment",                  CV_TYPE_BOOL,    &SET_DEFAULT_COMMENT                  },
    {"default_comment",                      CV_TYPE_STRING,  &DEFAULT_COMMENT                      },
    {"crc32_comment",                        CV_TYPE_BOOL,    &SET_CRC32_COMMENT                    },
    {"open_scanner_window_on_startup",       CV_TYPE_BOOL,    &OPEN_SCANNER_WINDOW_ON_STARTUP       },
    {"scanner_window_on_top",                CV_TYPE_BOOL,    &SCANNER_WINDOW_ON_TOP                },
    {"set_scanner_window_position",          CV_TYPE_BOOL,    &SET_SCANNER_WINDOW_POSITION          },
    {"scanner_window_x",                     CV_TYPE_INT,     &SCANNER_WINDOW_X                     },
    {"scanner_window_y",                     CV_TYPE_INT,     &SCANNER_WINDOW_Y                     },

    {"confirm_before_exit",                  CV_TYPE_BOOL,    &CONFIRM_BEFORE_EXIT                  },
    {"confirm_write_tag",                    CV_TYPE_BOOL,    &CONFIRM_WRITE_TAG                    },
    {"confirm_rename_file",                  CV_TYPE_BOOL,    &CONFIRM_RENAME_FILE                  },
    {"confirm_write_playlist",               CV_TYPE_BOOL,    &CONFIRM_WRITE_PLAYLIST               },
    {"confirm_delete_file",                  CV_TYPE_BOOL,    &CONFIRM_DELETE_FILE                  },
    {"options_notebook_page",                CV_TYPE_INT,     &OPTIONS_NOTEBOOK_PAGE                },
    {"process_filename_field",               CV_TYPE_BOOL,    &PROCESS_FILENAME_FIELD               },
    {"process_title_field",                  CV_TYPE_BOOL,    &PROCESS_TITLE_FIELD                  },
    {"process_artist_field",                 CV_TYPE_BOOL,    &PROCESS_ARTIST_FIELD                 },
    {"process_album_field",                  CV_TYPE_BOOL,    &PROCESS_ALBUM_FIELD                  },
    {"process_genre_field",                  CV_TYPE_BOOL,    &PROCESS_GENRE_FIELD                  },
    {"process_comment_field",                CV_TYPE_BOOL,    &PROCESS_COMMENT_FIELD                },
    {"pf_convert_into_space",                CV_TYPE_BOOL,    &PF_CONVERT_INTO_SPACE                },
    {"pf_convert_space",                     CV_TYPE_BOOL,    &PF_CONVERT_SPACE                     },
    {"pf_convert_all_uppercase",             CV_TYPE_BOOL,    &PF_CONVERT_ALL_UPPERCASE             },
    {"pf_convert_all_downcase",              CV_TYPE_BOOL,    &PF_CONVERT_ALL_DOWNCASE              },
    {"pf_convert_first_letter_uppercase",    CV_TYPE_BOOL,    &PF_CONVERT_FIRST_LETTER_UPPERCASE    },
    {"pf_convert_first_letters_uppercase",   CV_TYPE_BOOL,    &PF_CONVERT_FIRST_LETTERS_UPPERCASE   },
    {"pf_remove_space",                      CV_TYPE_BOOL,    &PF_REMOVE_SPACE                      },
    {"pf_insert_space",                      CV_TYPE_BOOL,    &PF_INSERT_SPACE                      },
    {"pf_only_one_space",                    CV_TYPE_BOOL,    &PF_ONLY_ONE_SPACE                    },

    {"playlist_name",                        CV_TYPE_STRING,  &PLAYLIST_NAME                        },
    {"playlist_use_mask_name",               CV_TYPE_BOOL,    &PLAYLIST_USE_MASK_NAME               },
    {"playlist_use_dir_name",                CV_TYPE_BOOL,    &PLAYLIST_USE_DIR_NAME                },
    {"playlist_full_path",                   CV_TYPE_BOOL,    &PLAYLIST_FULL_PATH                   },
    {"playlist_relative_path",               CV_TYPE_BOOL,    &PLAYLIST_RELATIVE_PATH               },
    {"playlist_create_in_parent_dir",        CV_TYPE_BOOL,    &PLAYLIST_CREATE_IN_PARENT_DIR        },
    {"playlist_use_dos_separator",           CV_TYPE_BOOL,    &PLAYLIST_USE_DOS_SEPARATOR           },
    {"playlist_content_none",                CV_TYPE_BOOL,    &PLAYLIST_CONTENT_NONE                },
    {"playlist_content_filename",            CV_TYPE_BOOL,    &PLAYLIST_CONTENT_FILENAME            },
    {"playlist_content_mask",                CV_TYPE_BOOL,    &PLAYLIST_CONTENT_MASK                },
    {"playlist_content_mask_value",          CV_TYPE_STRING,  &PLAYLIST_CONTENT_MASK_VALUE          },

    {"load_file_run_scanner",                CV_TYPE_BOOL,    &LOAD_FILE_RUN_SCANNER                },

    {"cddb_server_port",                     CV_TYPE_BOOL,    &CDDB_SERVER_PORT                     },
    {"cddb_server_cgi_path",                 CV_TYPE_STRING,  &CDDB_SERVER_CGI_PATH                 },
    {"cddb_use_proxy",                       CV_TYPE_BOOL,    &CDDB_USE_PROXY                       },
    {"cddb_proxy_name",                      CV_TYPE_STRING,  &CDDB_PROXY_NAME                      },
    {"cddb_proxy_port",                      CV_TYPE_BOOL,    &CDDB_PROXY_PORT                      },
//    {"cddb_proxy_user_name",                 CV_TYPE_STRING,  &CDDB_PROXY_USER_NAME                 },
//    {"cddb_proxy_user_password",             CV_TYPE_STRING,  &CDDB_PROXY_USER_PASSWORD             },
    {"set_cddb_window_size",                 CV_TYPE_BOOL,    &SET_CDDB_WINDOW_SIZE                 },
    {"cddb_window_height",                   CV_TYPE_BOOL,    &CDDB_WINDOW_HEIGHT                   },
    {"cddb_window_width",                    CV_TYPE_BOOL,    &CDDB_WINDOW_WIDTH                    },
    {"set_cddb_pane_handle_position",        CV_TYPE_BOOL,    &SET_CDDB_PANE_HANDLE_POSITION        },
    {"cddb_pane_handle_position",            CV_TYPE_BOOL,    &CDDB_PANE_HANDLE_POSITION            },

    {"cddb_search_in_all_fields",            CV_TYPE_BOOL,    &CDDB_SEARCH_IN_ALL_FIELDS            },
    {"cddb_search_in_artist_field",          CV_TYPE_BOOL,    &CDDB_SEARCH_IN_ARTIST_FIELD          },
    {"cddb_search_in_title_field",           CV_TYPE_BOOL,    &CDDB_SEARCH_IN_TITLE_FIELD           },
    {"cddb_search_in_track_name_field",      CV_TYPE_BOOL,    &CDDB_SEARCH_IN_TRACK_NAME_FIELD      },
    {"cddb_search_in_other_field",           CV_TYPE_BOOL,    &CDDB_SEARCH_IN_OTHER_FIELD           },
    {"cddb_show_categories",                 CV_TYPE_BOOL,    &CDDB_SHOW_CATEGORIES                 },

    {"cddb_search_in_all_categories",        CV_TYPE_BOOL,    &CDDB_SEARCH_IN_ALL_CATEGORIES        },
    {"cddb_search_in_blues_categories",      CV_TYPE_BOOL,    &CDDB_SEARCH_IN_BLUES_CATEGORY        },
    {"cddb_search_in_classical_categories",  CV_TYPE_BOOL,    &CDDB_SEARCH_IN_CLASSICAL_CATEGORY    },
    {"cddb_search_in_country_categories",    CV_TYPE_BOOL,    &CDDB_SEARCH_IN_COUNTRY_CATEGORY      },
    {"cddb_search_in_folk_categories",       CV_TYPE_BOOL,    &CDDB_SEARCH_IN_FOLK_CATEGORY         },
    {"cddb_search_in_jazz_categories",       CV_TYPE_BOOL,    &CDDB_SEARCH_IN_JAZZ_CATEGORY         },
    {"cddb_search_in_misc_categories",       CV_TYPE_BOOL,    &CDDB_SEARCH_IN_MISC_CATEGORY         },
    {"cddb_search_in_newage_categories",     CV_TYPE_BOOL,    &CDDB_SEARCH_IN_NEWAGE_CATEGORY       },
    {"cddb_search_in_reggae_categories",     CV_TYPE_BOOL,    &CDDB_SEARCH_IN_REGGAE_CATEGORY       },
    {"cddb_search_in_rock_categories",       CV_TYPE_BOOL,    &CDDB_SEARCH_IN_ROCK_CATEGORY         },
    {"cddb_search_in_soundtrack_categories", CV_TYPE_BOOL,    &CDDB_SEARCH_IN_SOUNDTRACK_CATEGORY   },

    {"cddb_set_to_all_fields",               CV_TYPE_BOOL,    &CDDB_SET_TO_ALL_FIELDS               },
    {"cddb_set_to_title",                    CV_TYPE_BOOL,    &CDDB_SET_TO_TITLE                    },
    {"cddb_set_to_artist",                   CV_TYPE_BOOL,    &CDDB_SET_TO_ARTIST                   },
    {"cddb_set_to_album",                    CV_TYPE_BOOL,    &CDDB_SET_TO_ALBUM                    },
    {"cddb_set_to_year",                     CV_TYPE_BOOL,    &CDDB_SET_TO_YEAR                     },
    {"cddb_set_to_track",                    CV_TYPE_BOOL,    &CDDB_SET_TO_TRACK                    },
    {"cddb_set_to_track_total",              CV_TYPE_BOOL,    &CDDB_SET_TO_TRACK_TOTAL              },
    {"cddb_set_to_genre",                    CV_TYPE_BOOL,    &CDDB_SET_TO_GENRE                    },
    {"cddb_set_to_file_name",                CV_TYPE_BOOL,    &CDDB_SET_TO_FILE_NAME                },
    
    {"cddb_run_scanner",                     CV_TYPE_BOOL,    &CDDB_RUN_SCANNER                     }

};




/*************
 * Functions *
 *************/

/*
 * Define and Load default values into config variables
 */
void Init_Config_Variables (void)
{
    /*
     * Common
     */
    LOAD_ON_STARTUP               = 1;
    DEFAULT_PATH_TO_MP3           = g_strdup(HOME_VARIABLE);
    BROWSER_LINE_STYLE            = GTK_CTREE_LINES_NONE;
    BROWSER_EXPANDER_STYLE        = GTK_CTREE_EXPANDER_TRIANGLE;
    WRITE_ID3V1_TAG               = 1;
    WRITE_ID3V2_TAG               = 1;
    STRIP_TAG_WHEN_EMPTY_FIELDS   = 1;
    CONVERT_OLD_ID3V2_TAG_VERSION = 1;
    BROWSE_SUBDIR                 = 1;
    OPEN_SELECTED_BROWSER_NODE    = 1;
    KEEP_TREE_BROWSER_IN_MEMORY   = 0;
    SAVE_CONFIG_BEFORE_EXIT       = 0;

    /*
     * User Interface
     */
    SET_MAIN_WINDOW_SIZE      = 1;
    MAIN_WINDOW_HEIGHT        = -1;
    MAIN_WINDOW_WIDTH         = 800;
    SET_PANE_HANDLE_POSITION1 = 1;
    PANE_HANDLE_POSITION1     = 490;
    SET_PANE_HANDLE_POSITION2 = 1;
    PANE_HANDLE_POSITION2     = 230;
    SHOW_HEADER_INFO          = 1;
    USE_COLORED_TOOLTIPS      = 0;

    /*
     * Scanner
     */
    CONVERT_UNDERSCORE_AND_P20_INTO_SPACE = 1;
    CONVERT_SPACE_INTO_UNDERSCORE         = 0;
    OVERWRITE_TAG_FIELD                   = 0;
    SET_DEFAULT_COMMENT                   = 0;
    DEFAULT_COMMENT                       = g_strdup("Tagged with EasyTAG");
    SET_CRC32_COMMENT                     = 0;
    OPEN_SCANNER_WINDOW_ON_STARTUP        = 1;
    SCANNER_WINDOW_ON_TOP                 = 1;
    SET_SCANNER_WINDOW_POSITION           = 0;
    SCANNER_WINDOW_X                      = 11;
    SCANNER_WINDOW_Y                      = 155;

    /*
     * Misc
     */
    REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME = 1;
    DATE_AUTO_COMPLETION                   = 1;
    NUMBER_TRACK_FORMATED                  = 1;
    NUMBER_TRACK_FORMATED_SPIN_BUTTON      = 2;
    OGG_TAG_WRITE_XMMS_COMMENT             = 1;
    SORTING_FILE_MODE                      = SORTING_BY_ASCENDING_FILENAME;
    SORTING_FILE_CASE_SENSITIVE            = 0;
    USE_CHARACTER_SET_TRANSLATION          = 0;
    FILE_CHARACTER_SET                     = g_strdup("UTF-8");
    USER_CHARACTER_SET                     = g_strdup("ISO-8859-15");

    MESSAGE_BOX_POSITION_NONE              = 0;
    MESSAGE_BOX_POSITION_CENTER            = 1;
    MESSAGE_BOX_POSITION_MOUSE             = 0;

    AUDIO_FILE_PLAYER                      = g_strdup("xmms -p");

    /*
     * Confirmation
     */
    CONFIRM_BEFORE_EXIT    = 1;
    CONFIRM_WRITE_TAG      = 1;
    CONFIRM_RENAME_FILE    = 1;
    CONFIRM_DELETE_FILE    = 1;
    CONFIRM_WRITE_PLAYLIST = 1;

    /*
     * Scanner window
     */
    PROCESS_FILENAME_FIELD             = 0;
    PROCESS_TITLE_FIELD                = 1;
    PROCESS_ARTIST_FIELD               = 1;
    PROCESS_ALBUM_FIELD                = 1;
    PROCESS_GENRE_FIELD                = 1;
    PROCESS_COMMENT_FIELD              = 1;
    PF_CONVERT_INTO_SPACE              = 1;
    PF_CONVERT_SPACE                   = 0;
    PF_CONVERT_ALL_UPPERCASE           = 0;
    PF_CONVERT_ALL_DOWNCASE            = 0;
    PF_CONVERT_FIRST_LETTER_UPPERCASE  = 0;
    PF_CONVERT_FIRST_LETTERS_UPPERCASE = 1;
    PF_REMOVE_SPACE                    = 0;
    PF_INSERT_SPACE                    = 0;
    PF_ONLY_ONE_SPACE                  = 1;

    /*
     * Playlist window
     */
    PLAYLIST_NAME                   = g_strdup("playlist_%a_-_%t");
    PLAYLIST_USE_MASK_NAME          = 0;
    PLAYLIST_USE_DIR_NAME           = 1;
    PLAYLIST_FULL_PATH              = 0;
    PLAYLIST_RELATIVE_PATH          = 1;
    PLAYLIST_CREATE_IN_PARENT_DIR   = 0;
    PLAYLIST_USE_DOS_SEPARATOR      = 0;
    PLAYLIST_CONTENT_NONE           = 0;
    PLAYLIST_CONTENT_FILENAME       = 1;
    PLAYLIST_CONTENT_MASK           = 0;
    PLAYLIST_CONTENT_MASK_VALUE     = g_strdup("%n/%l - %a - %b - %t");

    /*
     * Load File window
     */
    LOAD_FILE_RUN_SCANNER     = 0;
    
    /*
     * CDDB
     */
    CDDB_SERVER_NAME     = g_strdup("freedb.freedb.org");
    CDDB_SERVER_PORT     = 80;
    CDDB_SERVER_CGI_PATH = g_strdup("/~cddb/cddb.cgi");
    CDDB_USE_PROXY       = 0;
    CDDB_PROXY_NAME      = g_strdup("localhost");
    CDDB_PROXY_PORT      = 8080;
    //CDDB_PROXY_USER_NAME          = NULL;
    //CDDB_PROXY_USER_PASSWORD      = NULL;
    SET_CDDB_WINDOW_SIZE          =  0;
    CDDB_WINDOW_HEIGHT            = -1;
    CDDB_WINDOW_WIDTH             = -1;
    SET_CDDB_PANE_HANDLE_POSITION =  0;
    CDDB_PANE_HANDLE_POSITION     = -1;

    CDDB_SEARCH_IN_ALL_FIELDS           = 0;
    CDDB_SEARCH_IN_ARTIST_FIELD         = 1;
    CDDB_SEARCH_IN_TITLE_FIELD          = 1;
    CDDB_SEARCH_IN_TRACK_NAME_FIELD     = 0;
    CDDB_SEARCH_IN_OTHER_FIELD          = 0;
    CDDB_SHOW_CATEGORIES                = 0;

    CDDB_SEARCH_IN_ALL_CATEGORIES       = 1;
    CDDB_SEARCH_IN_BLUES_CATEGORY       = 0;
    CDDB_SEARCH_IN_CLASSICAL_CATEGORY   = 0;
    CDDB_SEARCH_IN_COUNTRY_CATEGORY     = 0;
    CDDB_SEARCH_IN_FOLK_CATEGORY        = 0;
    CDDB_SEARCH_IN_JAZZ_CATEGORY        = 0;
    CDDB_SEARCH_IN_MISC_CATEGORY        = 1;
    CDDB_SEARCH_IN_NEWAGE_CATEGORY      = 1;
    CDDB_SEARCH_IN_REGGAE_CATEGORY      = 0;
    CDDB_SEARCH_IN_ROCK_CATEGORY        = 1;
    CDDB_SEARCH_IN_SOUNDTRACK_CATEGORY  = 0;

    CDDB_SET_TO_ALL_FIELDS  = 1;
    CDDB_SET_TO_TITLE       = 1;
    CDDB_SET_TO_ARTIST      = 0;
    CDDB_SET_TO_ALBUM       = 0;
    CDDB_SET_TO_YEAR        = 0;
    CDDB_SET_TO_TRACK       = 1;
    CDDB_SET_TO_TRACK_TOTAL = 1;
    CDDB_SET_TO_GENRE       = 0;
    CDDB_SET_TO_FILE_NAME   = 1;
    
    CDDB_RUN_SCANNER        = 0;
     
    /*
     * Other parameters
     */
    OPTIONS_NOTEBOOK_PAGE = 0;
}


/*
 * Put changes in config variables
 */
void Apply_Changes (void)
{
    if (OptionsWindow)
    {
        /*
         * Changes to config file
         */
        /* Common */
        LOAD_ON_STARTUP               = GTK_TOGGLE_BUTTON(LoadOnStartup)->active;
        DEFAULT_PATH_TO_MP3           = g_strdup(gtk_entry_get_text_1(DefaultPathToMp3));
        WRITE_ID3V1_TAG               = GTK_TOGGLE_BUTTON(WriteId3v1Tag)->active;
        WRITE_ID3V2_TAG               = GTK_TOGGLE_BUTTON(WriteId3v2Tag)->active;
        STRIP_TAG_WHEN_EMPTY_FIELDS   = GTK_TOGGLE_BUTTON(StripTagWhenEmptyFields)->active;
        CONVERT_OLD_ID3V2_TAG_VERSION = GTK_TOGGLE_BUTTON(ConvertOldId3v2TagVersion)->active;
        BROWSE_SUBDIR                 = GTK_TOGGLE_BUTTON(BrowseSubdir)->active;
        OPEN_SELECTED_BROWSER_NODE    = GTK_TOGGLE_BUTTON(OpenSelectedBrowserNode)->active;
        KEEP_TREE_BROWSER_IN_MEMORY   = GTK_TOGGLE_BUTTON(KeepTreeBrowserInMemory)->active;
        SAVE_CONFIG_BEFORE_EXIT       = GTK_TOGGLE_BUTTON(SaveConfigBeforeExit)->active;

        /* User interface */
        SET_MAIN_WINDOW_SIZE          = GTK_TOGGLE_BUTTON(SetMainWindowSize)->active;
        MAIN_WINDOW_HEIGHT            = atoi(gtk_entry_get_text_1(MainWindowHeight));
        MAIN_WINDOW_WIDTH             = atoi(gtk_entry_get_text_1(MainWindowWidth));
        SET_PANE_HANDLE_POSITION1     = GTK_TOGGLE_BUTTON(SetPaneHandlePosition1)->active;
        PANE_HANDLE_POSITION1         = atoi(gtk_entry_get_text_1(PaneHandlePosition1));
        SET_PANE_HANDLE_POSITION2     = GTK_TOGGLE_BUTTON(SetPaneHandlePosition2)->active;
        PANE_HANDLE_POSITION2         = atoi(gtk_entry_get_text_1(PaneHandlePosition2));
        SHOW_HEADER_INFO              = GTK_TOGGLE_BUTTON(ShowHeaderInfos)->active;
        USE_COLORED_TOOLTIPS          = GTK_TOGGLE_BUTTON(UseColoredTooltips)->active;

        /* Misc */
        REPLACE_ILLEGAL_CHARACTERS_IN_FILENAME = GTK_TOGGLE_BUTTON(ReplaceIllegalCharactersInFilename)->active;
        DATE_AUTO_COMPLETION                   = GTK_TOGGLE_BUTTON(DateAutoCompletion)->active;
        NUMBER_TRACK_FORMATED                  = GTK_TOGGLE_BUTTON(NumberTrackFormated)->active;
        NUMBER_TRACK_FORMATED_SPIN_BUTTON      = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(NumberTrackFormatedSpinButton));
        OGG_TAG_WRITE_XMMS_COMMENT             = GTK_TOGGLE_BUTTON(OggTagWriteXmmsComment)->active;
        SORTING_FILE_CASE_SENSITIVE            = GTK_TOGGLE_BUTTON(SortingFileCaseSensitive)->active;
        USE_CHARACTER_SET_TRANSLATION          = GTK_TOGGLE_BUTTON(UseCharacterSetTranslation)->active;
        FILE_CHARACTER_SET                     = Charset_Get_Name_From_Title(gtk_entry_get_text_1(FileCharacterSetEntry));
        USER_CHARACTER_SET                     = Charset_Get_Name_From_Title(gtk_entry_get_text_1(UserCharacterSetEntry));

        MESSAGE_BOX_POSITION_NONE   = GTK_TOGGLE_BUTTON(MessageBoxPositionNone)->active;
        MESSAGE_BOX_POSITION_CENTER = GTK_TOGGLE_BUTTON(MessageBoxPositionCenter)->active;
        MESSAGE_BOX_POSITION_MOUSE  = GTK_TOGGLE_BUTTON(MessageBoxPositionMouse)->active;

        AUDIO_FILE_PLAYER           = g_strdup(gtk_entry_get_text_1(FilePlayerEntry));

        /* Scanner */
        CONVERT_UNDERSCORE_AND_P20_INTO_SPACE = GTK_TOGGLE_BUTTON(ConvertUnderscoreAndP20IntoSpace)->active;
        CONVERT_SPACE_INTO_UNDERSCORE         = GTK_TOGGLE_BUTTON(ConvertSpaceIntoUnderscore)->active;

        OVERWRITE_TAG_FIELD = GTK_TOGGLE_BUTTON(OverwriteTagField)->active;
        SET_DEFAULT_COMMENT = GTK_TOGGLE_BUTTON(SetDefaultComment)->active;
        DEFAULT_COMMENT     = g_strdup(gtk_entry_get_text_1(DefaultComment));
        SET_CRC32_COMMENT   = GTK_TOGGLE_BUTTON(Crc32Comment)->active;

        OPEN_SCANNER_WINDOW_ON_STARTUP = GTK_TOGGLE_BUTTON(OpenScannerWindowOnStartup)->active;
        SCANNER_WINDOW_ON_TOP          = GTK_TOGGLE_BUTTON(ScannerWindowOnTop)->active;
        SET_SCANNER_WINDOW_POSITION    = GTK_TOGGLE_BUTTON(SetScannerWindowPosition)->active;
        SCANNER_WINDOW_X               = atoi(gtk_entry_get_text_1(ScannerWindowX));
        SCANNER_WINDOW_Y               = atoi(gtk_entry_get_text_1(ScannerWindowY));

        /* CDDB */
        CDDB_SERVER_NAME     = g_strdup(gtk_entry_get_text_1(CddbServerName));
        CDDB_SERVER_PORT     = atoi(gtk_entry_get_text_1(CddbServerPort));
        CDDB_SERVER_CGI_PATH = g_strdup(gtk_entry_get_text_1(CddbServerCgiPath));
        CDDB_USE_PROXY       = GTK_TOGGLE_BUTTON(CddbUseProxy)->active;
        CDDB_PROXY_NAME      = g_strdup(gtk_entry_get_text_1(CddbProxyName));
        CDDB_PROXY_PORT      = atoi(gtk_entry_get_text_1(CddbProxyPort));
//        CDDB_PROXY_USER_NAME          = g_strdup(gtk_entry_get_text_1(CddbProxyUserName));
//        CDDB_PROXY_USER_PASSWORD      = g_strdup(gtk_entry_get_text_1(CddbProxyUserPassword));
        SET_CDDB_WINDOW_SIZE          = GTK_TOGGLE_BUTTON(SetCddbWindowSize)->active;
        CDDB_WINDOW_HEIGHT            = atoi(gtk_entry_get_text_1(CddbWindowHeight));
        CDDB_WINDOW_WIDTH             = atoi(gtk_entry_get_text_1(CddbWindowWidth));
        SET_CDDB_PANE_HANDLE_POSITION = GTK_TOGGLE_BUTTON(SetCddbPaneHandlePosition)->active;
        CDDB_PANE_HANDLE_POSITION     = atoi(gtk_entry_get_text_1(CddbPaneHandlePosition));

        /* Confirmation */
        CONFIRM_BEFORE_EXIT    = GTK_TOGGLE_BUTTON(ConfirmBeforeExit)->active;
        CONFIRM_WRITE_TAG      = GTK_TOGGLE_BUTTON(ConfirmWriteTag)->active;
        CONFIRM_RENAME_FILE    = GTK_TOGGLE_BUTTON(ConfirmRenameFile)->active;
        CONFIRM_DELETE_FILE    = GTK_TOGGLE_BUTTON(ConfirmDeleteFile)->active;
        CONFIRM_WRITE_PLAYLIST = GTK_TOGGLE_BUTTON(ConfirmWritePlayList)->active;

        /* Get the last visible notebook page */
        OPTIONS_NOTEBOOK_PAGE = gtk_notebook_get_current_page(GTK_NOTEBOOK(OptionsNoteBook));

        /* Parameters and variables of Scanner Window are in "scan.c" file */
    }

    /*
     * Changes in user interface
     */
    if (SHOW_HEADER_INFO) gtk_widget_show_all(HeaderInfosTable);
    else                  gtk_widget_hide_all(HeaderInfosTable);

    /* Update state of check-menu-item into main/popup menu to browse subdirs */
    Check_Menu_Item_Update_Browse_Subdir();

/*    if (SET_WINDOW_SIZE) gdk_window_resize(MainWindow->window,WINDOW_HEIGHT,WINDOW_WIDTH);
 *    else                 gdk_window_resize(MainWindow->window,0,0);
 */
    /* Reload if number of character changed for track list */
    //Load_Track_List_To_UI();

    if (ScannerWindow)
    {
        gtk_widget_hide(ScannerWindow);
        if (SCANNER_WINDOW_ON_TOP)
            gtk_window_set_transient_for(GTK_WINDOW(ScannerWindow),GTK_WINDOW(MainWindow));
        else
            gtk_window_set_transient_for(GTK_WINDOW(ScannerWindow),NULL);
        gtk_widget_show(ScannerWindow);
        Scan_Set_Scanner_Window_Init_Position();
    }
        
    if (WritePlaylistWindow)
    {
        /* Playlist */
        /* Also set in the function 'Playlist_Write_Button_Pressed' */
        PLAYLIST_NAME                 = g_strdup(gtk_entry_get_text_1(PlayListNameEntry));
        PLAYLIST_USE_MASK_NAME        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_mask_name));
        PLAYLIST_USE_DIR_NAME         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_dir_name));
        PLAYLIST_FULL_PATH            = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_full_path));
        PLAYLIST_RELATIVE_PATH        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_relative_path));
        PLAYLIST_CREATE_IN_PARENT_DIR = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_create_in_parent_dir));
        PLAYLIST_USE_DOS_SEPARATOR    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_use_dos_separator));
        PLAYLIST_CONTENT_NONE         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_none));
        PLAYLIST_CONTENT_FILENAME     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_filename));
        PLAYLIST_CONTENT_MASK         = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(playlist_content_mask));
        PLAYLIST_CONTENT_MASK_VALUE   = g_strdup(gtk_entry_get_text_1(PlayListContentMaskEntry));
    }

    if (LoadFilenameWindow)
    {
        LOAD_FILE_RUN_SCANNER = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(LoadFileRunScanner));
    }

    if (CddbWindow)
    {
        CDDB_SEARCH_IN_ALL_FIELDS       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllFields));
        CDDB_SEARCH_IN_ARTIST_FIELD     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInArtistField));
        CDDB_SEARCH_IN_TITLE_FIELD      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTitleField));
        CDDB_SEARCH_IN_TRACK_NAME_FIELD = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInTrackNameField));
        CDDB_SEARCH_IN_OTHER_FIELD      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInOtherField));
        CDDB_SHOW_CATEGORIES            = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbShowCategoriesButton));

        CDDB_SEARCH_IN_ALL_CATEGORIES      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInAllCategories));
        CDDB_SEARCH_IN_BLUES_CATEGORY      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInBluesCategory));
        CDDB_SEARCH_IN_CLASSICAL_CATEGORY  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInClassicalCategory));
        CDDB_SEARCH_IN_COUNTRY_CATEGORY    = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInCountryCategory));
        CDDB_SEARCH_IN_FOLK_CATEGORY       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInFolkCategory));
        CDDB_SEARCH_IN_JAZZ_CATEGORY       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInJazzCategory));
        CDDB_SEARCH_IN_MISC_CATEGORY       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInMiscCategory));
        CDDB_SEARCH_IN_NEWAGE_CATEGORY     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInNewageCategory));
        CDDB_SEARCH_IN_REGGAE_CATEGORY     = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInReggaeCategory));
        CDDB_SEARCH_IN_ROCK_CATEGORY       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInRockCategory));
        CDDB_SEARCH_IN_SOUNDTRACK_CATEGORY = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSearchInSoundtrackCategory));

        CDDB_SET_TO_ALL_FIELDS  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAllFields));
        CDDB_SET_TO_TITLE       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTitle));
        CDDB_SET_TO_ARTIST      = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToArtist));
        CDDB_SET_TO_ALBUM       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToAlbum));
        CDDB_SET_TO_YEAR        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToYear));
        CDDB_SET_TO_TRACK       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrack));
        CDDB_SET_TO_TRACK_TOTAL = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToTrackTotal));
        CDDB_SET_TO_GENRE       = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToGenre));
        CDDB_SET_TO_FILE_NAME   = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbSetToFileName));

        CDDB_RUN_SCANNER        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(CddbRunScanner));
    }

    /* Config part of scanner window */
    Scan_Apply_Changes();
}

void Save_Changes (void)
{
    Apply_Changes();
    Save_Config_To_File();
}


void Save_Config_Now (void)
{
    Save_Changes();
    Statusbar_Message(_("Configuration saved"),TRUE);
}


/*
 * Write the config file
 */
void Save_Config_To_File (void)
{
    gchar *file_path = NULL;
    FILE *file;

    /* The file to write */
    if (!HOME_VARIABLE) return;
    file_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            CONFIG_FILE,NULL);
    
    if ( Create_Easytag_Directory()==0 || (file=fopen(file_path,"w+"))==0 )
    {
        g_print(_("ERROR: Can't write config file: %s (%s)\n\a"),file_path,g_strerror(errno));
    }else
    {
        gint ConfigVarListLen = sizeof(Config_Variables)/sizeof(tConfigVariable);
        gint i;
        gchar *data = NULL;
        for (i=0; i<ConfigVarListLen; i++)
        {
            switch(Config_Variables[i].type)
            {
                case CV_TYPE_INT:
                case CV_TYPE_BOOL:
                {
                    data = g_strdup_printf("%s=%i\n",Config_Variables[i].name,
                                                     *(int *)Config_Variables[i].var);
                    fwrite(data,strlen(data),1,file);
                    //g_print("# var:%d\n",*(int *)Config_Variables[i].var);
                    g_free(data);
                    break;
                }
                case CV_TYPE_STRING:
                {
                    /* Doesn't write datum if empty */
                    if ( (*(char **)Config_Variables[i].var)==NULL ) break;
                    
                    data = g_strdup_printf("%s=%s\n",Config_Variables[i].name,
                                                     *(char **)Config_Variables[i].var);
                    fwrite(data,strlen(data),1,file);
                    //g_print("# var:'%s' => %s",*(char **)Config_Variables[i].var,data);
                    g_free(data);
                    break;
                }
                default:
                {
                    g_print("ERROR: Can't save: type of config variable not supported "
                            "for '%s'!\n\a",Config_Variables[i].name);
                    break;
                }
            }
        }
        fclose(file);
    }
    if (file_path) g_free(file_path);

    //Display_Config();
}


/*
 * Parse lines read (line as <var_description>=<value>) and load the values 
 * into the corresponding config variables.
 */
void Set_Config (gchar *line)
{
    gchar *var_descriptor;
    gchar *var_value;
    gint ConfigVarListLen;
    gint i;

    if (*line=='\n' || *line=='#') return;

    /* Cut string */
    var_descriptor = (gchar*)strtok(line,"=");
    var_value      = (gchar*)strtok(NULL,"=");
    //g_print("\nstr1:'%s',\t str2:'%s'",var_descriptor,var_value);

    ConfigVarListLen = sizeof(Config_Variables)/sizeof(tConfigVariable);
    for (i=0; i<ConfigVarListLen; i++)
    {
        if (!strcmp(Config_Variables[i].name,var_descriptor))
        {
            switch(Config_Variables[i].type)
            {
                case CV_TYPE_INT:
                case CV_TYPE_BOOL:
                {
                          *(int *)Config_Variables[i].var = strtol(var_value, NULL, 10);
                    break;
                }
                case CV_TYPE_STRING:
                {
                    if (!var_value)
                    {
                        *(char **)Config_Variables[i].var = NULL;
                        //g_print("\nConfig File Warning: Field of '%s' has no value!\n",var_descriptor);
                    } else
                    {
                        if ( *(char **)Config_Variables[i].var != NULL )
                            g_free(*(char **)Config_Variables[i].var);
                        *(char **)Config_Variables[i].var = g_malloc(strlen(var_value)+1);
                            strcpy( *(char **)Config_Variables[i].var,var_value );
                    }
                    break;
                }
                default:
                {
                    g_print("ERROR: Can't read: type of config variable not supported "
                            "for '%s'!\n\a",Config_Variables[i].name);
                    break;
                }
            }
        }
    }
}


/*
 * Read config from config file
 */
void Read_Config (void)
{
    gchar *file_path = NULL;
    FILE *file;
    gchar buffer[MAX_STRING_LEN];

    /* The file to read */
    if (!HOME_VARIABLE) return;
    file_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            CONFIG_FILE,NULL);
    
    if ( (file=fopen(file_path,"r"))==0 )
    {
        g_print(_("\nCan't open configuration file '%s' (%s)\n"),file_path,g_strerror(errno));
        g_print(_("Loading default configuration...\n"));
    }else
    {
        while(fgets(buffer,sizeof(buffer),file))
        {
            if (buffer[strlen(buffer)-1]=='\n')
                buffer[strlen(buffer)-1]='\0';
            Set_Config(buffer);
        }
        fclose(file);
    }
    if (file_path) g_free(file_path);
}


/*
 * Display values in config variables
 * For debuging only!
 */
void Display_Config (void)
{
    gchar *file_path = NULL;
    FILE *file;

    /* The file to write */
    if (!HOME_VARIABLE) return;
    file_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            CONFIG_FILE,NULL);
    
    if ( (file=fopen(file_path,"r"))==0 )
    {
        g_print("Can't open configuration file '%s' (%s)\n",file_path,g_strerror(errno));
    }else
    {
        gint ConfigVarListLen = sizeof(Config_Variables)/sizeof(tConfigVariable);
        gint i;
        
        g_print("\n## Current Config ##");
        for (i=0; i<ConfigVarListLen; i++)
        {
            switch(Config_Variables[i].type)
            {
                case CV_TYPE_INT:
                case CV_TYPE_BOOL:
                {
                    g_print("\n%d: %s=%d",i,Config_Variables[i].name,
                                     *(int*)Config_Variables[i].var);
                    break;
                }
                case CV_TYPE_STRING:
                {
                    g_print("\n%d: %s=%s",i,Config_Variables[i].name,
                                     *(char**)Config_Variables[i].var);
                    break;
                }
                default:
                {
                    g_print("NOT IMPLEMENTED (Save_Config)!! \n\a");
                    break;
                }
            }
        }
        g_print("\n## End Current Config ##\n");
        fclose(file);
    }
    if (file_path) g_free(file_path);
}




/*
 * Create the main directory with empty history files
 */
void Setting_Create_Files (void)
{
    gchar *home_path = NULL;
    gchar *file_path = NULL;
    FILE  *file;
    
    /* The file to write */
    if (!HOME_VARIABLE)
        return;
        
    if ( Create_Easytag_Directory()==FALSE )
        return;
    
    home_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            NULL);
    
    file_path = g_strconcat(home_path,CONFIG_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,SCAN_TAG_MASKS_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,RENAME_FILE_MASKS_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,DEFAULT_PATH_TO_MP3_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,DEFAULT_TAG_COMMENT_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,PATH_ENTRY_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);

    file_path = g_strconcat(home_path,PLAY_LIST_NAME_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);

    file_path = g_strconcat(home_path,RUN_PROGRAM_WITH_DIRECTORY_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,RUN_PROGRAM_WITH_FILE_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,AUDIO_FILE_PLAYER_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,SEARCH_FILE_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,FILE_TO_LOAD_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,PLAYLIST_CONTENT_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,CDDB_SEARCH_STRING_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    file_path = g_strconcat(home_path,CDDB_SEARCH_STRING_IN_RESULT_HISTORY_FILE,NULL);
    if ( (file=fopen(file_path,"a+")) == 0 )
        fclose(file);
    g_free(file_path);
    
    g_free(home_path);
}



/*
 * Save a glist list into file
 * Note: if list==NULL, then the file is empty (so the default list will be load at the next reading)
 */
void Save_Glist_To_File (GList *list, gchar *filename)
{
    gchar *file_path = NULL;
    FILE *file;
    gchar *data = NULL;
    GList *tmplist = NULL;

    /* The file to write */
    if (!HOME_VARIABLE) return;
    file_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            filename,NULL);
    
    if ( Create_Easytag_Directory()==0 || (file=fopen(file_path,"w+"))==0 )
    {
        g_print(_("ERROR: Can't write list to file: %s (%s)\n\a"),file_path,g_strerror(errno));
    }else
    {
        tmplist = g_list_first (list);
        while (tmplist)
        {
            /*g_print("Writing data:'%s'\n",(gchar *)tmplist->data);*/
            data = g_strdup_printf("%s\n",(gchar *)tmplist->data);
            if (data)
            {
                fwrite(data,strlen(data),1,file);
                g_free(data);
            }
            tmplist = g_list_next (tmplist);
        }
        fclose(file);
    }
    if (file_path) g_free(file_path);
}


/*
 * Load a glist from file passed in parameter
 */
GList *Create_Glist_From_File (gchar *filename)
{
    GList *list = NULL;
    gchar *file_path = NULL;
    FILE *file;
    gchar buffer[MAX_STRING_LEN];

    /* The file to write */
    if (!filename || !HOME_VARIABLE) return NULL;
    file_path = g_strconcat(HOME_VARIABLE,
                            HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                            filename,NULL);
    
    if ( (file=fopen(file_path,"r"))==0 )
    {
        g_print(_("Can't open file '%s' (%s)\n"),file_path,g_strerror(errno));
    }else
    {
        gchar *data = NULL;

        while(fgets(buffer,sizeof(buffer),file))
        {
            if (buffer[strlen(buffer)-1]=='\n')
                buffer[strlen(buffer)-1]='\0';
            data = g_strdup(buffer);
            if (data && strlen(data)>0)
                list = g_list_append(list,data);
        }
        fclose(file);
    }
    if (file_path) g_free(file_path);
    return list;
}


/*
 * Functions for writing and reading list of 'Fill Tag' masks
 */
GList *Load_Scan_Tag_Masks_List (void)
{
    return Create_Glist_From_File(SCAN_TAG_MASKS_FILE);
}
void Save_Scan_Tag_Masks_List (GList *list)
{
    Save_Glist_To_File(list,SCAN_TAG_MASKS_FILE);
}


/*
 * Functions for writing and reading list of 'Rename File' masks
 */
GList *Load_Rename_File_Masks_List (void)
{
    return Create_Glist_From_File(RENAME_FILE_MASKS_FILE);
}
void Save_Rename_File_Masks_List (GList *list)
{
    Save_Glist_To_File(list,RENAME_FILE_MASKS_FILE);
}



/*
 * Functions for writing and reading list of 'DefaultPathToMp3' combobox
 */
GList *Load_Default_Path_To_MP3_List (void)
{
    return Create_Glist_From_File(DEFAULT_PATH_TO_MP3_HISTORY_FILE);
}
void Save_Default_Path_To_MP3_List (GList *list)
{
    Save_Glist_To_File(list,DEFAULT_PATH_TO_MP3_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of 'DefaultComment' combobox
 */
GList *Load_Default_Tag_Comment_Text_List (void)
{
    return Create_Glist_From_File(DEFAULT_TAG_COMMENT_HISTORY_FILE);
}
void Save_Default_Tag_Comment_Text_List (GList *list)
{
    Save_Glist_To_File(list,DEFAULT_TAG_COMMENT_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of 'BrowserEntry' combobox
 */
GList *Load_Path_Entry_List (void)
{
    return Create_Glist_From_File(PATH_ENTRY_HISTORY_FILE);
}
void Save_Path_Entry_List (GList *list)
{
    Save_Glist_To_File(list,PATH_ENTRY_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of 'PlayListNameEntry' combobox
 */
GList *Load_Play_List_Name_List (void)
{
    return Create_Glist_From_File(PLAY_LIST_NAME_HISTORY_FILE);
}
void Save_Play_List_Name_List (GList *list)
{
    Save_Glist_To_File(list,PLAY_LIST_NAME_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox to run program (tree browser)
 */
GList *Load_Run_Program_With_Directory_List (void)
{
    return Create_Glist_From_File(RUN_PROGRAM_WITH_DIRECTORY_HISTORY_FILE);
}
void Save_Run_Program_With_Directory_List (GList *list)
{
    Save_Glist_To_File(list,RUN_PROGRAM_WITH_DIRECTORY_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox to run program (file browser)
 */
GList *Load_Run_Program_With_File_List (void)
{
    return Create_Glist_From_File(RUN_PROGRAM_WITH_FILE_HISTORY_FILE);
}
void Save_Run_Program_With_File_List (GList *list)
{
    Save_Glist_To_File(list,RUN_PROGRAM_WITH_FILE_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox to run file audio player
 */
GList *Load_Audio_File_Player_List (void)
{
    return Create_Glist_From_File(AUDIO_FILE_PLAYER_HISTORY_FILE);
}
void Save_Audio_File_Player_List (GList *list)
{
    Save_Glist_To_File(list,AUDIO_FILE_PLAYER_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox to search a string into file (tag or filename)
 */
GList *Load_Search_File_List (void)
{
    return Create_Glist_From_File(SEARCH_FILE_HISTORY_FILE);
}
void Save_Search_File_List (GList *list)
{
    Save_Glist_To_File(list,SEARCH_FILE_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox of path of file to load to rename files
 */
GList *Load_File_To_Load_List (void)
{
    return Create_Glist_From_File(FILE_TO_LOAD_HISTORY_FILE);
}
void Save_File_To_Load_List (GList *list)
{
    Save_Glist_To_File(list,FILE_TO_LOAD_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox of playlist content
 */
GList *Load_Playlist_Content_Mask_List (void)
{
    return Create_Glist_From_File(PLAYLIST_CONTENT_HISTORY_FILE);
}
void Save_Playlist_Content_Mask_List (GList *list)
{
    Save_Glist_To_File(list,PLAYLIST_CONTENT_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox of cddb search string
 */
GList *Load_Cddb_Search_String_List (void)
{
    return Create_Glist_From_File(CDDB_SEARCH_STRING_HISTORY_FILE);
}
void Save_Cddb_Search_String_List (GList *list)
{
    Save_Glist_To_File(list,CDDB_SEARCH_STRING_HISTORY_FILE);
}

/*
 * Functions for writing and reading list of combobox of cddb search string in result list
 */
GList *Load_Cddb_Search_String_In_Result_List (void)
{
    return Create_Glist_From_File(CDDB_SEARCH_STRING_IN_RESULT_HISTORY_FILE);
}
void Save_Cddb_Search_String_In_Result_List (GList *list)
{
    Save_Glist_To_File(list,CDDB_SEARCH_STRING_IN_RESULT_HISTORY_FILE);
}





/*
 * Create the directory used by EasyTAG to store files for each user.
 * If the directory already exists, does nothing and returns 1.
 * If unable to create the directory, returns 0.
 */
gboolean Create_Easytag_Directory (void)
{
    gchar *easytag_path = NULL;
    DIR *dir;

    if (!HOME_VARIABLE)
    {
        g_print(_("ERROR: The environment variable HOME is not defined!\n\n"));
        return FALSE;
    }

    /* Directory to create (if doesn't exists) with absolute path
     * Note for NetBSD : avoid passing a trailing slash to mkdir() */
    easytag_path = g_strconcat(HOME_VARIABLE,
                               HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!='/'?"/":"",
                               EASYTAG_DIR,
                               //EASYTAG_DIR[strlen(EASYTAG_DIR)-1]!='/'?"/":"",
                               NULL);

    if ( (dir=opendir(easytag_path)) == NULL )
    {
        if ( (mkdir(easytag_path,S_IRWXU|S_IXGRP|S_IRGRP)) == -1)
        {
            g_print(_("ERROR: Can't create directory '%s' (%s)!\n"),easytag_path,g_strerror(errno));
            return FALSE;          
        }
    }else
    {
        closedir(dir);
    }
    
    if (easytag_path)
        g_free(easytag_path);

    return TRUE;
}
