/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiWord
 * Copyright (C) 2003 Francis James Franklin <fjf@alinameridon.com>
 * Copyright (C) 2003 AbiSource, Inc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */


#ifndef PD_ABIWORD_2_H
#define PD_ABIWORD_2_H

/* pre-emptive dismissal; ut_types.h is needed by just about everything,
 * so even if it's commented out in-file that's still a lot of work for
 * the preprocessor to do...
 */
#ifndef UT_TYPES_H
#include "ut_types.h"
#endif

#include "ut_string_class.h"
#include "pl_Listener.h"

class UT_ByteBuf;
class PD_Document;

#define XMLNS_URL_AbiWord_2 "http://www.abisource.com/2003/"

#define DTD_PUBLIC_AbiWord_2 "-//ABISOURCE//DTD AWML 2.0 Strict//EN"
#define DTD_SYSTEM_AbiWord_2 "http://www.abisource.com/2003/awml.mod"

class ABI_EXPORT PD_AbiWord_2 : public PL_Listener
{
public:
	enum ElementType
	{
		et_a,
		et_abiword,
		et_br,
		et_bookmark,
		et_cbr,
		et_cell,
		et_col,
		et_d,
		et_data,
		et_endnote,
		et_endnotes,
		et_field,
		et_footnote,
		et_footnotes,
		et_ignoredwords,
		et_image,
		et_iw,
		et_l,
		et_lists,
		et_m,
		et_metadata,
		et_noteanchor,
		et_notelink,
		et_p,
		et_pagesize,
		et_pbr,
		et_r,
		et_revisions,
		et_row,
		et_s,
		et_section,
		et_span,
		et_styles,
		et_table,
		et_vcell,	// not a DTD element; this is used for structural purposes
		et_CDATA	// okay, not technically an element, but let's treat it as one
	};
	enum TextDirection
	{
		dir_default = 0, // i.e., no direction specified
		dir_ltr,
		dir_rtl
	};

	class ABI_EXPORT Writer
	{
	public:
		virtual Writer () { }

		virtual bool A2_write (const UT_UTF8String & str) = 0;
	};

	/* CSS Properties
	 */

	class ABI_EXPORT StyleMap
	{
	public:
		static StyleMap * map (const char * style);
	private:
		StyleMap ();
	public:
		~StyleMap ();

		/* append "style" attribute to cache
		 */
		bool append (UT_UTF8String & cache, bool strip);

	private:
		UT_StringPtrMap *	m_map;
	};

	/* Attributes
	 */

	class ABI_EXPORT Attr_Core
	{
	public:
		/* responsibility for "style" passes here; require "new_id" & "style" to be non-NULL
		 */
		static Attr_Core * attr_core (const char * old_id, const char * new_id,
									  const char * style_class, PropertyMap * style);
	private:
		/* responsibility for "old_id", "style_class" & "style" passes here 
		 */
		Attr_Core (UT_UTF8String * old_id, const char * new_id,
				   UT_UTF8String * style_class, PropertyMap * style);
	public:
		~Attr_Core ();

		/* append attributes to cache
		 */
		bool append (UT_UTF8String & cache, bool strip);

		const UT_UTF8String * old_id () const { return m_old_id; } // may be NULL
		const UT_UTF8String & new_id () const { return m_new_id; }

		const UT_UTF8String * style_class () const { return m_style_class; } // may be NULL

		PropertyMap & style () const { return *m_style; }

	private:
		UT_UTF8String *	m_old_id;
		UT_UTF8String	m_new_id;
		UT_UTF8String *	m_style_class;
		PropertyMap *	m_style;
	};
	class ABI_EXPORT Attr_I18N
	{
	public:
		static Attr_I18N * attr_i18n (const char * xml_lang, const char * dir);
	private:
		/* responsibility for "xml_lang" passes here
		 */
		Attr_I18N (UT_UTF8String * xml_lang, TextDirection dir);
	public:
		~Attr_I18N ();

		/* append attributes to cache
		 */
		bool append (UT_UTF8String & cache, bool strip);

		const UT_UTF8String * xml_lang () const { return m_xml_lang; } // may be NULL

		TextDirection dir () const { return m_dir; } // may be dir_default ( == 0)

	private:
		UT_UTF8String *	m_xml_lang;
		TextDirection	m_dir;
	};

	/* Elements
	 */

	class ABI_EXPORT Element
	{
	protected:
		Element (ElementType et);
	public:
		virtual ~Element ();

		ElementType type () const { return m_et; }

		Element * operator[] (UT_uint32 i) const
		{
			return (i < m_count) ? m_element[i] : 0;
		}
		UT_uint32 count () const { return m_count; }

	protected:
		/* tells each child element to write itself
		 */
		virtual bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		bool ins (UT_uint32 i, Element * element); // responsibility for element deletion passes here
	public:
		/* returns false if element is not a child; if detach is true then the
		 * element is removed from the list of children but not deleted
		 */
		bool del (UT_uint32 i, Element * element, bool detach = false);

		virtual bool swap (UT_uint32 i, UT_uint32 j); // returns false if either element is not a child

		bool index (UT_uint32 & i, Element * element); // returns false if element is not a child

	private:
		ElementType		m_et;

		Element **		m_element;

		UT_uint32		m_count;
		UT_uint32		m_max;
	};
	class ABI_EXPORT Stack
	{
	public:
		Stack ();

		~Stack ();

		bool push (Element * element);
		void pop ();

		bool popTo (ElementType et);
		bool popTo (ElementType et1, ElementType et2);

		void clear ();

		Element * top () const
		{
			return m_count ? m_element[m_count-1] : 0;
		}

		Element * operator[] (UT_uint32 i) const
		{
			return (i < m_count) ? m_element[i] : 0;
		}
		UT_uint32 count () const { return m_count; }

	private:
		Element **		m_element;

		UT_uint32		m_count;
		UT_uint32		m_max;
	};
	class ABI_EXPORT El_M : public Element
	{
	public:
		El_M (const char * key, const char * value);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & key () const { return m_key; }

	private:
		UT_UTF8String	m_key;
	public:
		UT_UTF8String	m_value;
	};
	class ABI_EXPORT El_MetaData : public Element
	{
	public:
		El_MetaData ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_M * insert (const char * key, const char * value);

		El_M * lookup_key (const char * key);
	};
	class ABI_EXPORT El_R : public Element
	{
	public:
		El_R (const char * old_id, const char * new_id, const char * reviewer);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	public:
		UT_UTF8String	m_reviewer;
	};
	class ABI_EXPORT El_Revisions : public Element
	{
	public:
		El_Revisions ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_R * insert (const char * old_id, const char * new_id, const char * reviewer);

		El_R * lookup_old_id (const char * old_id);
		El_R * lookup_new_id (const char * new_id);
	};
	class ABI_EXPORT El_S : public Element
	{
	public:
		El_S (const char * name, const char * type, const char * style,
			  const char * based_on, const char * followed_by);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & name () const { return m_name; }

	private:
		UT_UTF8String	m_name;
	public:
		UT_UTF8String	m_type;
		UT_UTF8String	m_style;
		UT_UTF8String	m_based_on;
		UT_UTF8String	m_followed_by;
	};
	class ABI_EXPORT El_Styles : public Element
	{
	public:
		El_Styles ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_S * insert (const char * name, const char * type,
					   const char * style,
					   const char * based_on, const char * followed_by);

		El_S * lookup_name (const char * name);
	};
	class ABI_EXPORT El_IW : public Element
	{
	public:
		El_IW (const char * word);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & word () const { return m_word; }

	private:
		UT_UTF8String	m_word;
	};
	class ABI_EXPORT El_IgnoredWords : public Element
	{
	public:
		El_IgnoredWords ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_IW * insert (const char * word);

		El_IW * lookup_word (const char * word);
	};
	class ABI_EXPORT El_L : public Element
	{
	public:
		El_L (const char * old_id, const char * new_id, const char * old_parent_id,
			  const char * type, const char * start_value,
			  const char * list_decimal, const char * list_delim);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	public:
		UT_UTF8String	m_old_parent_id;
		UT_UTF8String	m_new_parent_id; // warning: this needs to be determined
		UT_UTF8String	m_type;
		UT_UTF8String	m_start_value;
		UT_UTF8String	m_list_decimal;
		UT_UTF8String	m_list_delim;
	};
	class ABI_EXPORT El_Lists : public Element
	{
	public:
		El_Lists ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_L * insert (const char * old_id, const char * new_id,
					   const char * old_parent_id,
					   const char * type, const char * start_value,
					   const char * list_decimal, const char * list_delim);

		El_L * lookup_old_id (const char * old_id);
		El_L * lookup_new_id (const char * new_id);

		bool sync (); // determine new_parent_id for each child
	};
	class ABI_EXPORT El_PageSize : public Element
	{
	public:
		El_PageSize (const char * page_type, const char * page_scale,
					 const char * width, const char * height,
					 const char * orientation, const char * units);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & page_type ()   const { return m_page_type; }

	private:
		UT_UTF8String	m_page_type;
	public:
		UT_UTF8String	m_page_scale;
		UT_UTF8String	m_width;
		UT_UTF8String	m_height;
		UT_UTF8String	m_orientation;
		UT_UTF8String	m_units;
	};
	class ABI_EXPORT El_Br : public Element
	{
	public:
		El_Br ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);
	};
	class ABI_EXPORT El_CBr : public Element
	{
	public:
		El_CBr ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);
	};
	class ABI_EXPORT El_PBr : public Element
	{
	public:
		El_PBr ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);
	};
	class ABI_EXPORT El_BookMark : public Element
	{
	public:
		El_BookMark (const char * old_id, const char * new_id);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	};
	class ABI_EXPORT El_Image : public Element
	{
	public:
		El_Image (const char * old_href, const char * new_href, const char * style);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		UT_UTF8String	m_old_href;
		UT_UTF8String	m_new_href; // warning: this needs to be determined
		UT_UTF8String	m_style;
	};
	class ABI_EXPORT El_CDATA : public Element
	{
	public:
		El_CDATA (const UT_UTF8String & cdata);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		UT_UTF8String	m_cdata;
	};
	class ABI_EXPORT El_Span : public Element
	{
	public:
		El_Span (const char * style_class, const char * style);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_CDATA * cdata) { return Element::ins (i, cdata); }

		UT_UTF8String	m_style_class;
		UT_UTF8String	m_style;
	};
	class ABI_EXPORT El_Field : public Element
	{
	public:
		El_Field (const char * style_class, const char * style, const char * type);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		bool ins (UT_uint32 i, El_CDATA * cdata) { return Element::ins (i, cdata); }
		bool ins (UT_uint32 i, El_Span  *  span) { return Element::ins (i,  span); }

		UT_UTF8String	m_style_class;
		UT_UTF8String	m_style;
		UT_UTF8String	m_type;
	};
	class ABI_EXPORT El_NoteAnchor : public Element
	{
	public:
		El_NoteAnchor (const char * style_class, const char * style);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		bool ins (UT_uint32 i, El_CDATA * cdata) { return Element::ins (i, cdata); }
		bool ins (UT_uint32 i, El_Span  *  span) { return Element::ins (i,  span); }

		UT_UTF8String	m_style_class;
		UT_UTF8String	m_style;
	};
	class ABI_EXPORT El_NoteLink : public Element
	{
	public:
		El_NoteLink (const char * old_href,
					 const char * style_class, const char * style, const char * type);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		bool ins (UT_uint32 i, El_CDATA * cdata) { return Element::ins (i, cdata); }
		bool ins (UT_uint32 i, El_Span  *  span) { return Element::ins (i,  span); }

		UT_UTF8String	m_old_href;
		UT_UTF8String	m_new_href; // warning: this needs to be determined
		UT_UTF8String	m_style_class;
		UT_UTF8String	m_style;
		UT_UTF8String	m_type;
	};
	class ABI_EXPORT El_A : public Element
	{
	public:
		El_A (const char * old_href);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_CDATA    *    cdata) { return Element::ins (i,    cdata); }
		bool ins (UT_uint32 i, El_Br       *       br) { return Element::ins (i,       br); }
		bool ins (UT_uint32 i, El_CBr      *      cbr) { return Element::ins (i,      cbr); }
		bool ins (UT_uint32 i, El_PBr      *      pbr) { return Element::ins (i,      pbr); }
		bool ins (UT_uint32 i, El_Field    *    field) { return Element::ins (i,    field); }
		bool ins (UT_uint32 i, El_Image    *    image) { return Element::ins (i,    image); }
		bool ins (UT_uint32 i, El_BookMark * bookmark) { return Element::ins (i, bookmark); }
		bool ins (UT_uint32 i, El_Span     *     span) { return Element::ins (i,     span); }
		bool ins (UT_uint32 i, El_NoteLink * notelink) { return Element::ins (i, notelink); }

		UT_UTF8String	m_old_href;
		UT_UTF8String	m_new_href; // warning: this needs to be determined
	};
	class ABI_EXPORT El_P : public Element
	{
	public:
		El_P (const char * style_class, const char * style,
			  const char * old_list_id, const char * level);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_CDATA    *    cdata) { return Element::ins (i,    cdata); }
		bool ins (UT_uint32 i, El_Br       *       br) { return Element::ins (i,       br); }
		bool ins (UT_uint32 i, El_CBr      *      cbr) { return Element::ins (i,      cbr); }
		bool ins (UT_uint32 i, El_PBr      *      pbr) { return Element::ins (i,      pbr); }
		bool ins (UT_uint32 i, El_Field    *    field) { return Element::ins (i,    field); }
		bool ins (UT_uint32 i, El_Image    *    image) { return Element::ins (i,    image); }
		bool ins (UT_uint32 i, El_BookMark * bookmark) { return Element::ins (i, bookmark); }
		bool ins (UT_uint32 i, El_Span     *     span) { return Element::ins (i,     span); }
		bool ins (UT_uint32 i, El_A        *        a) { return Element::ins (i,        a); }
		bool ins (UT_uint32 i, El_NoteLink * notelink) { return Element::ins (i, notelink); }

		UT_UTF8String	m_style_class;
		UT_UTF8String	m_style;
		UT_UTF8String	m_old_list_id;
		UT_UTF8String	m_new_list_id; // warning: this needs to be determined
		UT_UTF8String	m_level;
	};

	class El_Table;

	class ABI_EXPORT El_Cell : public Element // requires at least one child <p>
	{
	private:
		El_Cell (const char * style, UT_uint32 rowspan, UT_uint32 colspan);
	public:
		static El_Cell * cell (const char * style, UT_uint32 rowspan, UT_uint32 colspan);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_P     *     p) { return Element::ins (i,     p); }
		bool ins (UT_uint32 i, El_Table * table) { return Element::ins (i, table); }

		UT_UTF8String	m_style;
		UT_UTF8String	m_rowspan;
		UT_UTF8String	m_colspan;
	};
	class ABI_EXPORT El_VCell : public Element // not a DTD element; this is used for structural purposes
	{
	public:
		El_VCell ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip); // no output

		bool	m_virtual; // initially false
	};
	class ABI_EXPORT El_Col : public Element
	{
	public:
		El_Col (const char * style);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		UT_UTF8String	m_style;
	};
	class ABI_EXPORT El_Row : public Element
	{
	private:
		El_Row (const char * style);
	public:
		/* The restraints on rows are greater; the static constructor will
		 * create a matrix of cols/vcells
		 */
		static El_Row * row (const char * style, UT_uint32 col_count);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* replace the vcell at i with the specified cell
		 */
		bool set (UT_uint32 i, El_Cell * Cell); // ownership of Cell passes here

	public:
		UT_UTF8String	m_style;
	};
	class ABI_EXPORT El_Table : public Element
	{
	private:
		El_Table (const char * new_id, const char * style,
				  UT_uint32 row_count, UT_uint32 col_count);
	public:
		/* The restraints on tables are greater; the static constructor will
		 * create a matrix of cols/rows/vcells
		 */
		static El_Table * table (const char * new_id, const char * style,
								 UT_uint32 row_count, UT_uint32 col_count);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* create a new row at the bottom of the table
		 */
		El_Row * add_row (const char * style);

		/* helper system for adding cells sequentially to a table
		 */
		bool auto_end (); // call at end

		/* begin a new table row
		 */
		El_Row * auto_row (const char * style);

		/* add a cell to the current row
		 */
		El_Cell * auto_cell (const char * style, UT_uint32 rowspan, UT_uint32 colspan);

		const UT_UTF8String & new_id () const { return m_new_id; }

		El_Row * row (UT_uint32 i) const;
		{
			return (i < m_row_count) ? (*this)[m_col_count+i] : 0;
		}
		UT_uint32 row_count () const { return m_row_count; }

		El_Col * col (UT_uint32 i) const
		{
			return (i < m_col_count) ? (*this)[i] : 0;
		}
		UT_uint32 col_count () const { return m_col_count; }

	private:
		UT_UTF8String	m_new_id;
		UT_UTF8String	m_rows;
		UT_UTF8String	m_cols;
	public:
		UT_UTF8String	m_style;
	private:
		UT_uint32		m_row_count;
		UT_uint32		m_col_count;

		bool			m_auto;
		UT_uint32		m_auto_row;
		UT_uint32		m_auto_col;
	};
	class ABI_EXPORT El_Section : public Element
	{
	public:
		El_Section (const char * old_id, const char * new_id, const char * type,
					const char * style,
					const char * old_header_id, const char * old_footer_id,
					const char * columns, const char * column_gap);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_P     *     p) { return Element::ins (i,     p); }
		bool ins (UT_uint32 i, El_Table * table) { return Element::ins (i, table); }

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	public:
		UT_UTF8String	m_type;
		UT_UTF8String	m_style;
		UT_UTF8String	m_old_header_id;
		UT_UTF8String	m_new_header_id; // warning: this needs to be determined
		UT_UTF8String	m_old_footer_id;
		UT_UTF8String	m_new_footer_id; // warning: this needs to be determined
		UT_UTF8String	m_columns;
		UT_UTF8String	m_column_gap;
	};
	class ABI_EXPORT El_EndNote : public Element
	{
	public:
		El_EndNote (const char * old_id, const char * new_id);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_P * p) { return Element::ins (i, p); }

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	public:
		El_NoteAnchor	m_note_anchor;
	};
	class ABI_EXPORT El_EndNotes : public Element
	{
	public:
		El_EndNotes ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_EndNote * insert (const char * old_id, const char * new_id);

		El_EndNote * lookup_old_id (const char * old_id);
		El_EndNote * lookup_new_id (const char * new_id);
	};
	class ABI_EXPORT El_FootNote : public Element
	{
	public:
		El_FootNote (const char * old_id, const char * new_id);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		/* responsibility for element deletion passes here
		 */
		bool ins (UT_uint32 i, El_P * p) { return Element::ins (i, p); }

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String	m_old_id;
		UT_UTF8String	m_new_id;
	public:
		El_NoteAnchor	m_note_anchor;
	};
	class ABI_EXPORT El_FootNotes : public Element
	{
	public:
		El_FootNotes ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_FootNote * insert (const char * old_id, const char * new_id);

		El_FootNote * lookup_old_id (const char * old_id);
		El_FootNote * lookup_new_id (const char * new_id);
	};
	class ABI_EXPORT El_D : public Element
	{
	public:
		El_D (const char * old_id, const char * new_id, const char * type);

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		const UT_UTF8String & old_id () const { return m_old_id; }
		const UT_UTF8String & new_id () const { return m_new_id; }

	private:
		UT_UTF8String		m_old_id;
		UT_UTF8String		m_new_id;
	public:
		UT_UTF8String		m_type;

		const UT_ByteBuf *	m_data;
	};
	class ABI_EXPORT El_Data : public Element
	{
	public:
		El_Data ();

		bool write (Writer * writer, UT_UTF8String & cache, bool strip);

		El_D * insert (const char * old_id, const char * new_id, const char * type);

		El_D * lookup_old_id (const char * old_id);
		El_D * lookup_new_id (const char * new_id);
	};
	class ABI_EXPORT El_AbiWord : public Element
	{
	public:
		El_AbiWord (const char * fileformat, const char * template,
					const char * styles, const char * version, const char * style);

		/* These are optional sections, children of El_AbiWord, occuring in a specific order.
		 * If necessary, these methods create the section and insert it into the correct place.
		 */
		El_MetaData *		getMetaData ();
		El_Revisions *		getRevisions ();
		El_Styles *			getStyles ();
		El_IgnoredWords *	getIgnoredWords ();
		El_Lists *			getLists ();
		El_FootNotes *		getFootNotes ();
		El_EndNotes *		getEndNotes ();
		El_Data *			getData ();

		/* returns false if either element is not a child, or if child element-types differ
		 */
		bool swap (UT_uint32 i, UT_uint32 j);

		bool append (EL_PageSize * pagesize); // responsibility for element deletion passes here
		bool append (EL_Section  *  section);

		bool ins (UT_uint32 i, EL_PageSize * pagesize); // responsibility for element deletion passes here
		bool ins (UT_uint32 i, EL_Section  *  section);

		EL_PageSize * insert (const char * page_type, const char * page_scale,
							  const char * width, const char * height,
							  const char * orientation, const char * units);

		EL_PageSize * lookup_page_type (const char * page_type);

		EL_Section * insert (const char * old_id, const char * new_id, const char * type,
							 const char * style, const char * header_id, const char * footer_id,
							 const char * columns, const char * column_gap);

		EL_Section * lookup_old_id (const char * old_id);
		EL_Section * lookup_new_id (const char * new_id);

		bool write (Writer * writer, UT_UTF8String & cache,
					bool include_meta_data, bool embed_object_data, bool strip = true);

		UT_UTF8String	m_fileformat;
		UT_UTF8String	m_template;
		UT_UTF8String	m_styles;
		UT_UTF8String	m_version;
		UT_UTF8String	m_style;
	};

	PD_AbiWord_2 (PD_Document * doc);

	~PD_AbiWord_2 ();

	/* ==== easy tree construction ====
	 */

	/* must be called at start
	 */
	bool abiword (const char * _fileformat, const char * _template,
				  const char * _styles, const char * _version, const char * _style);

	/* must be called at end
	 */
	void abiword ();

	/* child of: <p>
	 */
	bool a (const char * _href);
	void a () { m_stack.pop (); } // end

	/* child of: <a> | <p>
	 */
	bool bookmark (const char * _id);

	/* child of: <a> | <p>
	 */
	bool br ();

	/* child of: <a> | <p>
	 */
	bool cbr ();

	/* child of: <a> | <field> | <notelink> | <p> | <span>
	 */
	bool CDATA (const UT_UTF8String & text);

	/* child of: <table>
	 */
	bool cell (const char * _style, const char * _colspan, const char * _rowspan);
	void cell () { m_stack.pop (); } // end

	/* child of: <table>
	 * 
	 * where column=1.."columns" (the table element)
	 */
	bool col (UT_uint32 column, const char * _style);

	/* unrestricted
	 */
	El_D * d (const char * _id, const char * _type);

	/* unrestricted
	 */
	bool endnote (const char * _id);
	void endnote () { m_stack.pop (); } // end

	/* child of: <a> | <p>
	 */
	bool field (const char * _class, const char * _style, const char * _type);
	void field () { m_stack.pop (); } // end

	/* unrestricted
	 */
	bool footnote (const char * _id);
	void footnote () { m_stack.pop (); } // end

	/* child of: <a> | <p>
	 */
	bool image (const char * _href, const char * _style);

	/* unrestricted
	 */
	bool iw (const char * _word);

	/* unrestricted
	 */
	bool l (const char * _id, const char * _parent_id, const char * _type,
			const char * _start_value,
			const char * _list_decimal, const char * _list_delim);

	/* unrestricted
	 */
	bool m (const char * _key, const char * _value);

	/* child of: <endnote> | <footnote>
	 */
	bool noteanchor (const char * _class, const char * _style);
	void noteanchor () { m_stack.pop (); } // end

	/* child of: <a> | <p>
	 */
	bool notelink (const char * _href, const char * _class, const char * _style, const char * _type);
	void notelink () { m_stack.pop (); } // end

	/* child of: <cell> | <section>
	 */
	bool p (const char * _class, const char * _style,
			const char * _list_id, const char * _level);
	void p () { m_stack.pop (); } // end

	/* unrestricted
	 */
	bool pagesize (const char * _page_type, const char * _page_scale,
				   const char * _width, const char * _height,
				   const char * _orientation, const char * _units);

	/* child of: <a> | <p>
	 */
	bool pbr ();

	/* unrestricted
	 */
	bool r (const char * _id, const char * _reviewer);

	/* child of: <table>
	 */
	bool row (const char * _style);
	void row () { m_stack.pop (); } // end

	/* unrestricted
	 */
	bool s (const char * _name, const char * _type, const char * _style,
			const char * _based_on, const char * _followed_by);

	/* cannot be a child
	 */
	bool section (const char * _id, const char * _type, const char * _style,
				  const char * _header_id, const char * _footer_id,
				  const char * _columns, const char * _column_gap);
	void section () { m_stack.pop (); } // end

	/* child of: <a> | <field> | <notelink> | <p>
	 */
	bool span (const char * _class, const char * _style);
	void span () { m_stack.pop (); } // end

	/* child of: <cell> | <section>
	 */
	bool table (const char * _style, UT_uint32 rows, UT_uint32 cols);
	void table () { m_stack.pop (); } // end

	/* ^^^^ easy tree construction ^^^^
	 */

	/* import
	 */
	bool appendToDocument ();
	bool insertInDocument (); // for adding to open document

	/* export
	 */
	bool writeDocument (Writer * writer, bool include_meta_data, bool embed_object_data, bool strip = true);

private:
	PD_Document *	m_document;
	El_AbiWord *	m_tree;

	Stack			m_stack;

	const char * NewID () const { return m_document->m_XML_ID.next (); }

public:
	/* PL_Listener implementation
	 */

	bool	populate (PL_StruxFmtHandle sfh, const PX_ChangeRecord * pcr);

	bool	populateStrux (PL_StruxDocHandle sdh, const PX_ChangeRecord * pcr, 
						   PL_StruxFmtHandle * psfh);

	/* empty method
	 */
	bool	change (PL_StruxFmtHandle sfh, const PX_ChangeRecord * pcr);

	/* empty method
	 */
	bool	insertStrux (PL_StruxFmtHandle sfh, const PX_ChangeRecord * pcr,
						 PL_StruxDocHandle sdhNew,
						 PL_ListenerId lid,
						 void (* pfnBindHandles) (PL_StruxDocHandle sdhNew,
												  PL_ListenerId lid,
												  PL_StruxFmtHandle sfhNew));

	/* empty method
	 */
	bool	signal (UT_uint32 iSignal);

private:
	bool	CDATA (const UT_UCSChar * buffer, UT_uint32 length);

	bool	bookmark (PT_AttrPropIndex api);
	bool	cell (PT_AttrPropIndex api);
	bool	endnote (PT_AttrPropIndex api);
	bool	footnote (PT_AttrPropIndex api);
	bool	field (PT_AttrPropIndex api);
	bool	image (PT_AttrPropIndex api);
	bool	p (PT_AttrPropIndex api);
	bool	section (PT_AttrPropIndex api);
	bool	span (PT_AttrPropIndex api);
	bool	table (PT_AttrPropIndex api);

	UT_UTF8String styleFromProps (const PP_AttrProp * pAP);
};

#endif /* PD_ABIWORD_2_H */
