/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#include "base.h"
#include "situa.h"
#include "verts.h"
#include <time.h>

// GP: clean

/*****************************************************************

    S   i   t   u   a   t   i   o   n   O   b   j 

*****************************************************************/

SituationObj::SituationObj(Processor* proc_) : proc ( proc_ )
{
    clear();
    logfile = errwfile = NULL;
    openDefaultFiles();
}

SituationObj::~SituationObj()
{
    if (logfile)
        stdclose(logfile);
    if (errwfile)
        stdclose(errwfile);
    history.freeall(FALSE);
}

Str SituationObj::timeStr()
{
    time_t currtime;
    time(&currtime);
    return asctime(localtime(&currtime));
}

void SituationObj::pushCurrent()
{
    SitData *p = new SitData;
    p -> currV = currV;
    p -> currLine = currLine;
    p -> currFile = currFile;
    history.append(p);
}

void SituationObj::popCurrent()
{
    SitData *p = history.last();
    currV = p -> currV;
    currLine = p -> currLine;
    currFile = p -> currFile;
    history.freelast(FALSE);
}

void SituationObj::pushCurrV(Vertex *v)
{
    pushCurrent();
    setCurrV(v);
}

void SituationObj::setCurrV(Vertex *v)
{
    currV = v;
    currLine = v -> lineno;
}

void SituationObj::setCurrLine(int lno)
{
    currLine = lno;
}

void SituationObj::setCurrFile(Str& fname)
{
    currFile = fname;
};

eFlag SituationObj::openDefaultFiles()
{
    E( msgOutputFile((char *) "stderr", NULL) );
    return OK;
}

eFlag SituationObj::closeFiles()
{
    if (logfile)
        stdclose(logfile);
    logfile = NULL;
    if (errwfile)
        stdclose(errwfile);
    errwfile = NULL;
    return OK;
}

eFlag SituationObj::eraseLog(char *newLogFile)
{
    if (logfile)
        stdclose(logfile);
    logfile = NULL;
    if (newLogFile)
    {
        if (!(logfile = stdopen(newLogFile,"w")))
          Err1(this, E_FILE_OPEN, newLogFile);
    }
    return OK;
}

eFlag SituationObj::msgOutputFile(char *_errwfn, char *_logfn)
{
    E( closeFiles() );
    if (_logfn)
    {
        if (!(logfile = stdopen(_logfn,"a")))
          Err1(this, E_FILE_OPEN, _logfn);
    }
    if (_errwfn)
    {
        if (!(errwfile = stdopen(_errwfn,"w")))
          Err1(this, E_FILE_OPEN, _errwfn);
    }
    return OK;
}

// constructMsgFields
// called to transform a List of Str's to a NULL-terminated array
// of char*'s. Stores just POINTERS so make sure the strings don't change.
// Dispose of the return value using delete[].

char **constructMsgFields(PList<DStr*>& strings)
{
    int len = strings.number();
    char **p = new char*[len + 1];
    p[len] = NULL;
    int i;
    for (i = 0; i < len; i++)
        p[i] = (char*)(*strings[i]);
    return p;
}

void SituationObj::message(MsgType type, MsgCode code, 
                         const Str& arg1, const Str& arg2)
{
    char buf[512];
    PList<DStr*> out;
    void *messengerUD;
    MessageHandler *messenger = proc -> getMessageHandler(&messengerUD);
    if (messenger)
    {
        out.append(new DStr("msgtype:"));
        switch(type)
        {
        case MT_ERROR: *(out[0]) += "error"; break;
        case MT_WARN: *(out[0]) += "warning"; break;
        case MT_LOG: *(out[0]) += "log"; break;
        };
    }
    if (type != MT_LOG)
    {
        sprintf(buf,"code:%d",code);
        out.append(new DStr(buf));
    }
    if (messenger)
        out.append(new DStr("module:Sablotron"));
    if (!currFile.isEmpty())
    {
        sprintf(buf,"URI:%s",(char*) currFile);
        out.append(new DStr(buf));
    }
    if (currLine && type != MT_LOG)
    {
        sprintf(buf,"line:%d",currLine);
        out.append(new DStr(buf));
    }
    if (currV && type != MT_LOG)
    {
        DStr nameStr;
        currV -> speak(nameStr, SM_NAME);
        sprintf(buf,"node:%s%s'%s'",
            vertexTypeNames[currV -> vt & VT_BASE],
            (currV -> vt == VT_VERTEX ? "" : " "),
            (char *) nameStr);
        out.append(new DStr(buf));
    }
    SabMsg *p = GetMessage(code);

    if (p -> text[0])
    {
        DStr msgText = messenger ? (char*)"msg:" : (char*)"";
        sprintf(buf,p -> text,(char*)(Str&)arg1,(char*)(Str&)arg2);
        msgText += buf;
        out.append(new DStr(msgText));
    }

    if (messenger)
    {
        // construct the message fields
        char **msgFields = constructMsgFields(out);
        MH_ERROR externalCode = 
            messenger -> makeCode(messengerUD, proc,
                type == MT_ERROR ? 1 : 0, 
                                MH_FACILITY_SABLOTRON, (unsigned short)code);
        
        // FIXME: casting to MH_LEVEL -- necessary?
        switch(type)
        {
        case MT_ERROR:
            messenger -> error(messengerUD, proc,
                externalCode, (MH_LEVEL) MH_LEVEL_ERROR, msgFields); break;
        case MT_WARN:
            messenger -> log(messengerUD, proc,
                externalCode, (MH_LEVEL) MH_LEVEL_WARN, msgFields); break;
        case MT_LOG:
            messenger -> log(messengerUD, proc,
                externalCode, (MH_LEVEL) MH_LEVEL_INFO, msgFields); break;
        }
        delete[] msgFields;
        // note that the strings pointed at by msgFields members are deleted
        // upon destruction of 'out'
    }
    else
    {
        // construct the message
        DStr fullout;
        if (type != MT_LOG)
        {
            fullout = GetMessage((MsgCode)(MSG_ERROR + type)) -> text;
            fullout += " ";
            int outnum = out.number();
            for (int j = 0; j < outnum; j++)
            {
                if (j < outnum - 1)
                    fullout += "[";
                fullout += *out[j];
                if (j < outnum - 1)
                    fullout += "] ";
                if (j == outnum-2)
                    fullout += "\n  ";
            }
        }
        else
        {
            if (out.number())
                fullout = *(out.last());
        }


        // display the message yourself
        FILE *thefile = (type == MT_LOG ? logfile : errwfile);
        if (thefile)
            fprintf(thefile,"%s\n",(char*) fullout);
    };

    // in any case dispose of the temp string array
    out.freeall(FALSE);
};


void SituationObj::error(MsgCode code, 
                         const Str& arg1, const Str& arg2)
{
    message(MT_ERROR, code, arg1, arg2);
    // only log the error if we are using our own reporter
    if (!proc -> getMessageHandler(NULL))
        message(MT_LOG, code, arg1, arg2);
    pending = code;
}

void SituationObj::warning(MsgCode code, 
                         const Str& arg1, const Str& arg2)
{
    message(MT_WARN, code, arg1, arg2);
    // only log the warning if we are using our own reporter
    if (!proc -> getMessageHandler(NULL))
        message(MT_LOG, code, arg1, arg2);
}

void SituationObj::logmsg(MsgCode code, 
                         const Str& arg1, const Str& arg2)
{
    message(MT_LOG, code, arg1, arg2);
}

Bool SituationObj::isError()
{
    return (Bool) (pending != E_OK);
};

int SituationObj::getError() const
{
    return pending;
}

void SituationObj::clearError()
{
    pending = E_OK;
}

void SituationObj::clear()
{
    currV = NULL;
    currLine = 0;
    currFile.empty();
    clearError();
}
