/*************************************************************************
 *
 *  $RCSfile: itempath.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 17:03:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define INCL_DOSERRORS
#include <svpm.h>

#include <string.h>
#include <folder.hxx>
#include <itempath.hxx>
#include <longname.hxx>

#define PDATA ((PItemIDPathData) pData)


static const char szAbsURL[] = "file://";
static const char szBinStr[] = "/binary:";
static const char szBinMark[] = "/#";

static const char szSlash[] = "/";

static const char cSlash = '/', cBackSlash = '\\';
static const char cPipe	 = '|', cColon = ':';
static const char cBra	 = '<';

static struct entry
{
	const char * pszOffice;
	const char * pszWPS;
} entries[] =
{
	VIRTUAL_DESKTOP_URL,	  "<WP_DESKTOP>" ,
	VIRTUAL_NETWORK_URL,	  "<WP_NETWORK>",
	VIRTUAL_BOOKMARKS_URL,	  "<WP_COOLURLSFOLDER>",
	VIRTUAL_COMMON_PROGS_URL, "<WP_PROGRAMSFOLDER>",
	VIRTUAL_PRINTERS_URL,	  "<WP_PRINTERSFOLDER>",
	VIRTUAL_CONTROLS_URL,	  "<WP_CONFIG>",
	VIRTUAL_PROGRAMS_URL,	  "<WP_PROGRAMS>",
	NULL,		 NULL
};


static void Verminator( String& rString )
{

	// for UNC notation, ignore double-back-slashes at the beginning
	int nPos = 1;
	while( ( nPos = rString.SearchAndReplace( "\\\\", "\\", nPos ) )
		   != STRING_NOTFOUND )
		;
	if( rString.Len() > 1 )
		rString.EraseTrailingChars( cBackSlash );
}

/*************************************************************************
 *
 *	Data handling methods
 *
 *************************************************************************
 */

inline PItemIDPathData copyData( PItemIDPathData pData )
{
	PItemIDPathData pCopy = new ItemIDPathData;

	if( pCopy )
	{
		pCopy->nRefCount  = 1;
		pCopy->nImplFlags = pData->nImplFlags;
		pCopy->sPath	  = pData->sPath;
		pCopy->sLocation  = pData->sLocation;
		memcpy(&pCopy->Cache, &pData->Cache, sizeof(ItemIDPathData::Cache));
	}

	return pCopy;
}

inline BOOL equalFlags(const PItemIDPathData pData1, const PItemIDPathData pData2)
{
	return ((pData1->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE) ==
			(pData2->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE));
}


/*************************************************************************
 *
 * Reference counting methods
 *
 *************************************************************************
 */

inline PItemIDPathData incRefCount( PItemIDPathData pData )
{
	// maximum of references reached ?
	if( pData->nRefCount == (USHORT) -1)
		return copyData(pData);

	pData->nRefCount++;
	return pData;
}

inline void decRefCount( PItemIDPathData pData )
{
	if( --(pData->nRefCount) == 0 )
		delete pData;
}

/******************************************************************************
 *
 * Constructors and Destructor
 *
 ******************************************************************************
 */

ItemIDPath::ItemIDPath() : pData( NULL )
{
}

ItemIDPath::~ItemIDPath()
{
	if(pData)
		decRefCount(PDATA);
}

ItemIDPath::ItemIDPath( SpecialFolderID eFolder )
{
	pData = new ItemIDPathData;
	if(!pData) return;

	PDATA->nRefCount  = 1;
	PDATA->nImplFlags = ITEM_IMPL_FLAG_IS_ABSOLUTE;

	switch(eFolder)
	{
	case FOLDER_ROOT:
	case FOLDER_WORKPLACE:
	case FOLDER_DRIVES:
		break;
	case FOLDER_DESKTOP:
		PDATA->sPath = entries[0].pszWPS;
		break;
	case FOLDER_NETWORK:
		PDATA->sPath = entries[1].pszWPS;
		break;
	case FOLDER_BOOKMARKS:
		PDATA->sPath = entries[2].pszWPS;
		break;
	case FOLDER_COMMON_PROGS:
		PDATA->sPath = entries[3].pszWPS;
		break;
	case FOLDER_PRINTERS:
		PDATA->sPath = entries[4].pszWPS;
		break;
	case FOLDER_CONTROLS:
		PDATA->sPath = entries[5].pszWPS;
		break;
	case FOLDER_PROGRAMS:
		PDATA->sPath = entries[6].pszWPS;
		break;
	default:
		delete pData;
	}

    // query the real location later
    if(eFolder == FOLDER_PROGRAMS)
        PDATA->sLocation = entries[0].pszWPS;
    else
        PDATA->sLocation = PDATA->sPath.Upper();
}


ItemIDPath::ItemIDPath( void* pNewData, size_t nBytes ) : pData( NULL )
{
	if( pNewData && nBytes == sizeof( ItemIDPathData ))
		pData = incRefCount( (PItemIDPathData) pNewData );
}


ItemIDPath::ItemIDPath( const ItemIDPath& rPath ) : pData( NULL )
{
	if( rPath.pData )
		pData = incRefCount( (PItemIDPathData) rPath.pData );
}

ItemIDPath::ItemIDPath( const String& rDescription, BOOL bIsFileDesc )
{
	String aPath = rDescription;
	USHORT nIndex;

	// create private data
	pData = new ItemIDPathData;
	if( !pData ) return;

	PDATA->nRefCount = 1;
	PDATA->nImplFlags = ITEM_IMPL_FLAG_IS_ABSOLUTE;

	// if View-URL use everything after '#'
	nIndex = aPath.Search( '#' );
	if ( nIndex != STRING_NOTFOUND )
		aPath.Erase( 0, nIndex + 1 );

	// remove protocol prefix
	nIndex = sizeof( szAbsURL ) - 1;
	if( aPath.ICompare( szAbsURL, nIndex ) == COMPARE_EQUAL )
		aPath.Erase( 0, nIndex );

	// replace slashes with backslashes
	nIndex = 0;
	while(( nIndex = aPath.SearchAndReplace( cSlash, cBackSlash, nIndex ) )
		   != STRING_NOTFOUND )
		;

	// replace pipes with colons
	nIndex = 0;
	while(( nIndex = aPath.SearchAndReplace( cPipe, cColon, nIndex ) )
		   != STRING_NOTFOUND )
		;

	// remove leading \ if not unc notation
	if( aPath.GetChar(0) == cBackSlash )
	{
		if( aPath.GetChar(1) != cBackSlash )
			aPath.Erase( 0, 1);
	}

	Verminator( aPath );

    String aToken;
    USHORT nLast;
    BOOL   bIsVirtual = FALSE;

    for(nIndex = 0; nIndex != STRING_NOTFOUND;)
    {
        USHORT nActual = nIndex;

        aToken = aPath.GetToken( 0, '\\', nIndex);
        nLast  = aToken.Len() - 1;

        // search matching brackets
        if(aToken.GetChar(0) == *VIRTUAL_URL_OPENBRACKET &&
           aToken.GetChar(nLast) == *VIRTUAL_URL_CLOSEBRACKET)
        {
            PM_BYTE nEntry = 0;

            if(!bIsVirtual)
                bIsVirtual = TRUE;

            // replace special tokens
            while(entries[nEntry].pszOffice)
            {
                if(aToken.Compare(entries[nEntry].pszOffice) == COMPARE_EQUAL)
                {
                    aPath.SetToken( 0, '\\', String(entries[nEntry].pszWPS), nActual);
                    break;
                }

                nEntry++;
            }

            // replace brackets and upper case token when not replaced completly
            if(!entries[nEntry].pszOffice)
            {
                aToken[0]     = '<';
                aToken[nLast] = '>';
                aToken.ToUpper();

                aPath.SetToken( 0, '\\', aToken, nActual);
            }
        }
    }
                    
	PDATA->sPath = aPath;

	// for special objects, save filesystem location
    if(bIsVirtual)
    {
        String aTmpStr = aPath;

        nIndex = 0;
        while((nIndex = aTmpStr.SearchAndReplace("<WP_PROGRAMS>", "<WP_DESKTOP>", nIndex ))
              != STRING_NOTFOUND )
            ;

        // query the real location later
        PDATA->sLocation = WinFolderImpl::GetValidObject(aTmpStr);
	}
}

/******************************************************************************
 *
 * operators
 *
 ******************************************************************************
 */

ItemIDPath& ItemIDPath::operator=( const ItemIDPath& rPath )
{
	if( pData )
		decRefCount( (PItemIDPathData) pData );

	if( rPath.pData )
		pData = incRefCount( (PItemIDPathData) rPath.pData );
	else
		pData = NULL;

	return *this;
}

ItemIDPath& ItemIDPath::operator=( const String& rPath )
{
	ItemIDPath aTmp( rPath );

	return *this = aTmp;
}


ItemIDPath& ItemIDPath::operator+=( const ItemIDPath& rPath )
{
	PItemIDPathData pPathData = (PItemIDPathData) rPath.pData;

	if(pPathData)
	{
		// check if rPath is not absolute
		if( pData && !(pPathData->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE) )
		{
			PItemIDPathData pTmp;

			// copy data for writing
			pTmp = copyData( PDATA );
			decRefCount( PDATA );
			pData = pTmp;

            // if child is virtual, take it location
            if(WinFolderImpl::isVirtual(pPathData->sPath))
            {
                PDATA->sLocation = pPathData->sLocation;
            }

            // if *this is virtual, convert to real path
            else if(WinFolderImpl::isVirtual(PDATA->sPath))
            {
                PDATA->sPath = GetHostNotationPath();
                PDATA->sLocation.Erase();
            }

            // concat locations if set (FAT)
            else if(PDATA->sLocation.Len())
            {
                if(pPathData->sLocation.Len())
                {
                    PDATA->sLocation = pPathData->sLocation;
                }
                else
                    // have to search again here
                    PDATA->sLocation.Erase();
            }
                
			// append backslash if not root
			if(PDATA->sPath.Len() != 0)
			{
				// append to binary string
				PDATA->sPath += cBackSlash;
			}
			PDATA->sPath += pPathData->sPath;
		}
		else
			// actual id-path is empty or rPath is absolute
			*this = rPath;
	}

	return *this;
}

ItemIDPath ItemIDPath::operator+( const ItemIDPath& rPath ) const
{
	ItemIDPath aSum( *this );
	aSum += rPath;
	return aSum;
}

int ItemIDPath::operator==(const ItemIDPath& rPath) const
{
	int nRet = 0;

	// if the data pointers are equal, the ID paths must be equal
	if(pData == rPath.pData)
	{
		nRet++;
	}
	else if(pData && rPath.pData)
	{
		String aTmp1 = GetHostNotationPath();
		String aTmp2 = rPath.GetHostNotationPath();

		if(aTmp1.Len() == 0 && aTmp2.Len() == 0)
		{
			aTmp1 = WinFolderImpl::GetValidObject(PDATA->sPath).Upper();
			aTmp2 = WinFolderImpl::GetValidObject(((PItemIDPathData) rPath.pData)->sPath).Upper();
		}
		
		if(aTmp1 == aTmp2)
			nRet = equalFlags(PDATA,(PItemIDPathData) rPath.pData);
	}

	return nRet;
}

int ItemIDPath::operator!=( const ItemIDPath& rPath ) const
{
	return ! operator==( rPath );
}


ItemIDPath ItemIDPath::operator[]( int nIndex ) const
{
	ItemIDPath aTmp (*this);

	if( pData )
	{
		Verminator( PDATA->sPath );
		
		if( nIndex == 0 && (PDATA->sPath).GetChar(0) == cBackSlash )
			return ItemIDPath(FOLDER_ROOT);

		aTmp = PDATA->sPath.GetToken( nIndex, cBackSlash );
		((PItemIDPathData) aTmp.pData)->nImplFlags &= ~ITEM_IMPL_FLAG_IS_ABSOLUTE;
	}
	
	return aTmp;
}


int ItemIDPath::GetTokenCount() const
{
	int nTokens = 0;
	
	if( pData )
		nTokens = PDATA->sPath.GetTokenCount( cBackSlash );

	return nTokens;
}


ItemIDPath ItemIDPath::GetToken( int nIndex ) const
{
	return operator[](nIndex);
}

/******************************************************************************
 *
 * File URL Descriptions
 *
 ******************************************************************************
 */

String ItemIDPath::GetFileDescription() const
{
	String aRet = GetHostNotationPath();

	if(pData && aRet.Len() > 0)
	{
		// replace backslashes with slashes
		USHORT nIndex = 0;
		while( ( nIndex = aRet.SearchAndReplace( cBackSlash, cSlash, nIndex ) )
			   != STRING_NOTFOUND )
			;

		// replace colons with pipes
		nIndex = 0;
		while( ( nIndex = aRet.SearchAndReplace( cColon, cPipe, nIndex ) )
			   != STRING_NOTFOUND )
			;

		if(PDATA->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE)
		{
			// if not UNC notation, insert leading slash
			if( aRet.GetChar(0) != cSlash )
				aRet = szSlash + aRet;
			
			aRet = String(szAbsURL) + aRet;
		}
	}

	return aRet;
}

String ItemIDPath::GetBinaryDescription() const
{
	String aRet;

	if(pData)
	{
		aRet = PDATA->sPath;

		// replace wps specific ids
		PM_BYTE nEntry = 0;
		USHORT nIndex;

        // filter c:\\<WP_DESKTOP>
		nIndex = 0;
        nIndex = aRet.Search(":\\<", nIndex);
        if(nIndex != STRING_NOTFOUND)
        {
            aRet.Erase(nIndex - 1, 3);
        }

		while(entries[nEntry].pszOffice)
		{
			// replace office specific ids
			nIndex = 0;
			while((nIndex = aRet.SearchAndReplace(entries[nEntry].pszWPS, entries[nEntry].pszOffice, nIndex ))
				  != STRING_NOTFOUND )
				;

			nEntry++;
		}

        // replace '<' with '{'
		nIndex = 0;
		while((nIndex = aRet.SearchAndReplace('<', *VIRTUAL_URL_OPENBRACKET, nIndex))
			   != STRING_NOTFOUND )
            ;

        // replace '>' with '}'
		nIndex = 0;
		while((nIndex = aRet.SearchAndReplace('>', *VIRTUAL_URL_CLOSEBRACKET, nIndex))
			   != STRING_NOTFOUND )
            ;

		// replace backslashes with slashes
		nIndex = 0;
		while((nIndex = aRet.SearchAndReplace(cBackSlash, cSlash, nIndex))
			   != STRING_NOTFOUND )
			;

		// replace colons with pipes
		nIndex = 0;
		while((nIndex = aRet.SearchAndReplace(cColon, cPipe, nIndex))
			   != STRING_NOTFOUND )
			;

		if(PDATA->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE)
		{
			// if not UNC notation, insert leading slash
			if( aRet.GetChar(0) != cSlash )
				aRet = szSlash + aRet;

			aRet = String( szAbsURL ) + aRet;
		}
	}
	return aRet;
}

String ItemIDPath::GetHostNotationPath() const
{
	String aRet;

	if(pData)
	{
		aRet = PDATA->sPath;

        if(WinFolderImpl::isVirtual(aRet))
		{
			// location still object id ?
			if(PDATA->sLocation.GetChar(0) == cBra)
			{
				CHAR   szBuffer[CCHMAXPATHCOMP];

				// query object location
				if(WinFolderImpl::isAvailable())
				{
					PDATA->sLocation = WinFolderImpl::queryLocation((PSZ) PDATA->sLocation.GetStr());
				}
				else
				{
					PDATA->sLocation = String();
				}
			}

			aRet = PDATA->sLocation;
		}

		// replace long filenames if necessary
		else if(aRet.Len() > 3)
		{
			String aFileSystemName;

			if(DrivesFolderImpl::GetFileSystemName(aRet, aFileSystemName) &&
				aFileSystemName == "FAT")
			{
				if(PDATA->sLocation.Len() == 0)
				{
					CHAR	szPath[ CCHMAXPATHCOMP ];

					getValidPath(aRet, szPath, CCHMAXPATHCOMP);
					PDATA->sLocation = szPath;
				}

				aRet = PDATA->sLocation;
			}
		}
    }

    // append delimiter when drive root
    if(aRet.Len() == 2 && aRet.GetChar(1) == ':')
        aRet += '\\';

	return aRet;
}


/******************************************************************************
 *
 * direct data access methods
 *
 ******************************************************************************
 */


void ItemIDPath::SetData( void* pNewData, size_t nBytes )
{
	if( pData )
		decRefCount((PItemIDPathData) pData);
	
	if( nBytes == sizeof(ItemIDPathData))
	{
		pData = incRefCount( (PItemIDPathData) pNewData );
	}
}

const void* ItemIDPath::GetDataPtr() const
{
	return pData;
}

size_t ItemIDPath::GetDataSize() const
{
	return sizeof(ItemIDPathData);
}

BOOL ItemIDPath::Split(ItemIDPath &rParent, ItemIDPath &rChild) const
{
	if(pData)
	{
		USHORT nTokens = PDATA->sPath.GetTokenCount(cBackSlash);

		if(nTokens > 1)
		{
			String aPath = PDATA->sPath;

			// remove last token
			aPath.SetToken(nTokens - 1, cBackSlash, String( "" ));
			// delete trailing '\\'
			Verminator(aPath);

			// set up parent - invalidate cache
			rParent.pData = copyData(PDATA);
			PItemIDPathData pParentData = (PItemIDPathData) rParent.pData;
			pParentData->nImplFlags &= ~ITEM_IMPL_FLAG_IS_CACHED;
			pParentData->sPath = aPath;

			// if special object path, replace location
            if(WinFolderImpl::isVirtual(pParentData->sPath))
                pParentData->sLocation = WinFolderImpl::GetValidObject(pParentData->sPath).Upper();

            // if location set, file resides on FAT
            else if(pParentData->sLocation.Len())
            {
                // remove last token
                pParentData->sLocation.SetToken(nTokens - 1, cBackSlash, String( "" ));
                Verminator(pParentData->sLocation);
            }

			// setup child - save cache data
			rChild.pData = copyData(PDATA);
			PItemIDPathData pChildData	= (PItemIDPathData) rChild.pData;
			pChildData->nImplFlags &= ~ITEM_IMPL_FLAG_IS_ABSOLUTE;
			pChildData->sPath = PDATA->sPath.GetToken(nTokens - 1, cBackSlash);

			// copy location from absolut item
			pChildData->sLocation = PDATA->sLocation;

			return TRUE;
		}

		// all other absolute URLs without workplace have the parent root
		if(PDATA->sPath.Len() == 0)
		{
			return FALSE;
		}

		if(PDATA->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE)
		{
			rParent = ItemIDPath(FOLDER_ROOT);

			// child is exact copy but relative
			rChild.pData = copyData(PDATA);
			((PItemIDPathData)rChild.pData)->nImplFlags &= ~ITEM_IMPL_FLAG_IS_ABSOLUTE;

			return TRUE;
		}
	}
		
	return FALSE;
}
