/*************************************************************************
 *
 *  $RCSfile: simpleheap.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: jb $ $Date: 2002/03/28 08:34:59 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "datalock.hxx"
#include "heapmanager.hxx"

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _OSL_MUTEX_HXX_
#include <osl/mutex.hxx>
#endif
#ifndef _OSL_INTERLOCK_H_
#include <osl/interlck.h>
#endif

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef INCLUDED_MAP
#include <map>
#define INCLUDED_MAP
#endif

// -----------------------------------------------------------------------------	
namespace configmgr
{
// -----------------------------------------------------------------------------	
    namespace memory
    {
    // -------------------------------------------------------------------------
        extern HeapManager & localHeap();
    // -------------------------------------------------------------------------
        struct OrderByHeapId
        {
            typedef HeapManager::HeapIdParam HeapIdParam;

            bool operator()(HeapIdParam lhs, HeapIdParam rhs) const 
            {
                return lhs.compareTo(rhs) < 0;
            }
        };
    // -------------------------------------------------------------------------
        struct HeapSet
        {
            typedef HeapManager::HeapIdParam    HeapIdParam;
            typedef HeapManager::HeapId         HeapId;
            typedef HeapManager::Handle         Heap;

            typedef std::map<HeapId,Heap,OrderByHeapId> Map;

            HeapSet(HeapManager & _mgr) : m_mgr(_mgr) {}

            Heap add(Heap _aHeap) { return m_heaps[m_mgr.id(_aHeap)] = _aHeap; }
            void remove(Heap _aHeap) { m_heaps.erase(m_mgr.id(_aHeap)); }

            Heap heap(HeapIdParam _anId) const 
            {
                Map::const_iterator it = m_heaps.find(_anId);
                return it != m_heaps.end() ? it->second : NULL;
            }
        private:
            HeapManager &   m_mgr;
            Map             m_heaps;
        };
    // -------------------------------------------------------------------------
        class SimpleHeapManagerBase : public HeapManager
        {
            static void * ptr(Address _address) { return reinterpret_cast<void*>(_address); }
            static Address address(void * _ptr) { return reinterpret_cast<Address>(_ptr); }

            virtual void *  base(Handle _aHeap);
        protected: 
            SimpleHeapManagerBase() : m_heaps(*this) {}

            Handle  addHeap(Handle _aHeap)      { return m_heaps.add(_aHeap);   }
            void    removeHeap(Handle _aHeap)   { m_heaps.remove(_aHeap);       }

            Handle  findHeap( HeapId _anId )    
            { 
                osl::MutexGuard aGuard(m_mutex); 
                return m_heaps.heap(_anId);  
            }

            osl::Mutex & getHeapListMutex()     { return m_mutex; }
        protected:      
            Address allocateMemory(Size _sz);
            Address reallocateMemory(Address _p, Size _sz);
            void    deallocateMemory(Address _p);

        protected:      
            virtual Lock *  createHeapLock(Handle _aHeap);
            virtual void    destroyHeapLock(Handle _aHeap, Lock * _pLock);

        private:
            osl::Mutex  m_mutex;
            HeapSet     m_heaps;
        };
    // -------------------------------------------------------------------------
        class SimpleCheckingHeapManager : public SimpleHeapManagerBase
        {
            struct HeapData
            {
                HeapId      id;
                
                oslInterlockedCount  refs;

                Size        requested;
                Size        allocated;

                Size available() const { return requested-allocated; }

                typedef std::map<Address, Size> Map;
                Map allocations;
            };

            static HeapData * heap(Handle _handle) { return reinterpret_cast<HeapData*>(_handle); }
            static Handle handle(HeapData * _data) { return reinterpret_cast<Handle>(_data); }

            static void makeNewId(HeapData * _data);
       public:      
            virtual Handle createHeap(Size _sInitialSize);
            virtual Handle attachHeap(HeapIdParam  _anId, Size & _rInitialSize);
            virtual Size   growHeap(Handle _aHeap, Size _sNewSize);
            virtual void   destroyHeap(Handle _aHeap);

            virtual HeapId  id(Handle _aHeap);

            virtual Address allocate(Handle _aHeap, Size _sNewSize);
            virtual Address reallocate(Handle _aHeap, Address _aOldAddress, Size _sNewSize);
            virtual void    deallocate(Handle _aHeap, Address _aAddress);
            virtual bool    didAllocate(Handle _aHeap, Address _aAddress);
        };
    // -------------------------------------------------------------------------
        class SimpleHeapLock : public DataLock
        {
            osl::Mutex m_mutex;
        public:   
            virtual void acquireReadAccess() { m_mutex.acquire(); }
            virtual void releaseReadAccess() { m_mutex.release(); }

            virtual void acquireWriteAccess() { m_mutex.acquire(); }     
            virtual void releaseWriteAccess() { m_mutex.release(); }
        };
    // -------------------------------------------------------------------------
    // -------------------------------------------------------------------------
        HeapManager & cacheHeap()
        {
            static SimpleCheckingHeapManager aInstance;
            return aInstance;
        }
    // -------------------------------------------------------------------------
        HeapManager & localHeap()
        {
            static SimpleCheckingHeapManager aInstance;
            return aInstance;
        }
    // -------------------------------------------------------------------------
    // -------------------------------------------------------------------------

        HeapManager::Address SimpleHeapManagerBase::allocateMemory(Size _sz)
        {
            void * p = ::rtl_allocateMemory(_sz);

            OSL_ENSURE(p,"Could not allocate: std::malloc failed");

            return address(p);
        }
    // -------------------------------------------------------------------------

        HeapManager::Address SimpleHeapManagerBase::reallocateMemory(Address _p, Size _sz)
        {
            void * pOld = ptr(_p);

            void * pNew = ::rtl_reallocateMemory(pOld,_sz);

            OSL_ENSURE(pNew,"Could not reallocate: std::realloc failed");

            return address(pNew);
        }
    // -------------------------------------------------------------------------

        void SimpleHeapManagerBase::deallocateMemory(Address _p)
        {
            void * p = ptr(_p);

            ::rtl_freeMemory(p); 
        }
    // -------------------------------------------------------------------------

        void *  SimpleHeapManagerBase::base(Handle _aHeap)
        {
            return NULL;
        }
    // -------------------------------------------------------------------------

        DataLock *  SimpleHeapManagerBase::createHeapLock(Handle )
        {
            return new SimpleHeapLock();
        }
    // -------------------------------------------------------------------------

        void SimpleHeapManagerBase::destroyHeapLock(Handle , DataLock * _pLock)
        {
            if (SimpleHeapLock* pMyLock = /*dynamic_*/ static_cast<SimpleHeapLock*>(_pLock))
            {
                delete pMyLock;
            }
        }
    // -------------------------------------------------------------------------
    // -------------------------------------------------------------------------

        void SimpleCheckingHeapManager::makeNewId(HeapData * _data)
        {
            OSL_PRECOND( _data, "Internal Error: NULL data");

            sal_Int64 nData = reinterpret_cast<sal_Int64>(_data);

            HeapId aNewId = HeapId::valueOf(nData,16);

            _data->id = aNewId;
        }
    // -------------------------------------------------------------------------

        HeapManager::HeapId SimpleCheckingHeapManager::id(Handle _aHeap)
        {
            if (_aHeap == 0) return HeapId();

            HeapData * data = heap(_aHeap);

            return data->id;
        }
    // -------------------------------------------------------------------------

        HeapManager::Handle SimpleCheckingHeapManager::createHeap(Size _sInitialSize)
        {
            // need to have 4-byte pointers for this implementation
            OSL_ASSERT(sizeof(void *) == sizeof(Address));

            HeapData * pHeap = new HeapData();

            pHeap->refs      = 1;

            pHeap->requested = _sInitialSize;
            pHeap->allocated = 0;

            osl::MutexGuard aGuard( getHeapListMutex() );

            makeNewId( pHeap );

            OSL_ENSURE( ! findHeap(pHeap->id), "Internal Error: New heap id already registered");

            return addHeap(handle(pHeap) );
        }
    // -------------------------------------------------------------------------

        HeapManager::Handle SimpleCheckingHeapManager::attachHeap(HeapIdParam  _anId, Size & _rInitialSize)
        {
            osl::MutexGuard aGuard( getHeapListMutex() );

            Handle aHeap = findHeap(_anId);

            OSL_ENSURE( aHeap, "Error: No such Heap");

            if (aHeap != 0)
            {
                HeapData * data = heap(aHeap);

                OSL_ENSURE( data->id == _anId, "Internal error: Inconsistent heap id");

                _rInitialSize = data->requested;
             
                ++data->refs;
            }
            else
                _rInitialSize = 0;

            return aHeap;
        }
    // -------------------------------------------------------------------------

        HeapManager::Size   SimpleCheckingHeapManager::growHeap(Handle _aHeap, Size _sNewSize)
        {
            OSL_PRECOND( _aHeap, "Error: NULL data");

            HeapData * data = heap(_aHeap);

            OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

            if (data->requested < _sNewSize) data->requested = _sNewSize; 

            return data->requested;
        }
    // -------------------------------------------------------------------------

        void   SimpleCheckingHeapManager::destroyHeap(Handle _aHeap)
        {
            OSL_PRECOND( _aHeap, "Error: NULL data");

            HeapData * data = heap(_aHeap);

            osl::MutexGuard aGuard( getHeapListMutex() );

            if (0 == --data->refs)
            {
                OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

    //            OSL_ENSURE(data->allocations.empty(), "Memory leaks found");
                for (HeapData::Map::iterator it = data->allocations.begin();
                     it != data->allocations.end();
                     ++it)
                {
                    deallocateMemory(it->first);
                    data->allocated -= it->second;

                }
                OSL_ENSURE(data->allocated == 0, "Memory tracking error detected");

                removeHeap(_aHeap);

                delete data;
            }
        }
    // -------------------------------------------------------------------------

        bool SimpleCheckingHeapManager::didAllocate(Handle _aHeap, Address _aAddress)
        {
            OSL_PRECOND( _aHeap, "Error: NULL data");
            OSL_PRECOND( _aAddress, "Warning: checking NULL address - will fail");

            HeapData * data = heap(_aHeap);

            OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

            /* Condition is:
                    allocationAddress <= _aAddress &&
                    _aAddress <= allocationAddress + allocationSize
               (<= to allow past-the-end pointer)
            */

            HeapData::Map::iterator it = data->allocations.upper_bound(_aAddress);

            if (it == data->allocations.begin()) 
                return false;

            --it; // Now it is the last iterator that is less than or equal to _aAddress

            OSL_ASSERT( it->first <= _aAddress ); // first part of condition handled by map

            return _aAddress <= it->first + it->second;
        }
    // -------------------------------------------------------------------------

        HeapManager::Address SimpleCheckingHeapManager::allocate(Handle _aHeap, Size _nSize)
        {
            OSL_PRECOND( _aHeap, "Error: NULL data");

            HeapData * data = heap(_aHeap);

            OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

            if (_nSize > data->available()) 
                return 0;

            Address p = allocateMemory(_nSize);
            if (p)
            {
                data->allocations[p] = _nSize;
                data->allocated += _nSize;
            }
            return p;
        }
    // -------------------------------------------------------------------------

        HeapManager::Address SimpleCheckingHeapManager::reallocate(Handle _aHeap, Address _aOld, Size _nNewSize)
        {
            if (_aOld == 0) return allocate(_aHeap,_nNewSize);

            OSL_PRECOND( _aHeap, "Error: NULL data");

            HeapData * data = heap(_aHeap);

            OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

            OSL_ENSURE( data->allocations.count(_aOld), "Error: pointer being reallocated is not from this data");

            Size nOldSize = data->allocations[_aOld];

            if (nOldSize >= _nNewSize) 
                return _aOld;

            if (_nNewSize-nOldSize > data->available()) 
                return 0;

            Address aNew = reallocateMemory(_aOld,_nNewSize);
            if (aNew)
            {
                if (aNew != _aOld) data->allocations.erase(_aOld);

                data->allocations[aNew] = _nNewSize;
                data->allocated += _nNewSize - nOldSize;
            }

            return aNew;
        }
    // -------------------------------------------------------------------------

        void SimpleCheckingHeapManager::deallocate(Handle _aHeap, Address _p)
        {
            OSL_PRECOND( _aHeap, "Error: NULL data");
            OSL_PRECOND( _p, "Error: Trying to free NULL address");

            HeapData * data = heap(_aHeap);

            OSL_ENSURE( findHeap(data->id) == _aHeap, "Internal error: Inconsistent heap id");

            OSL_ENSURE( data->allocations.count(_p), "Error: pointer being freed is not from this data");

            Size freed = data->allocations[_p];
            data->allocated -= freed;
            data->allocations.erase(_p);

            deallocateMemory(_p);
        }
    // -------------------------------------------------------------------------
    // -------------------------------------------------------------------------
    }
// -----------------------------------------------------------------------------	
} // namespace configmgr


