/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef COMMON_SINGLETON_H
#define COMMON_SINGLETON_H

#include "common/noncopyable.h"

namespace Common {

/**
 * @defgroup common_singleton Singleton
 * @ingroup common
 *
 * @brief API for managing singletons.
 *
 * @{
 */

/**
 * Generic template base class for implementing the singleton design pattern.
 */
template<class T>
class Singleton : NonCopyable {
private:
	Singleton(const Singleton&);
	Singleton &operator=(const Singleton &);

	/**
	 * The default object factory used by the template class Singleton.
	 * By specialising this template function, one can make a singleton use a
	 * custom object factory. For example, to support encapsulation, your
	 * singleton class might be pure virtual (or "abstract" in Java terminology),
	 * and you specialise makeInstance to return an instance of a subclass.
	 */
	//template<class T>
#if defined(__WINS__)
//FIXME verify if __WINS__ needs this still
public:
#endif
	static T *makeInstance() {
		return new T();
	}

	static void destroyInstance() {
		delete _singleton;
		_singleton = 0;
	}


public:
	static bool hasInstance() {
		return _singleton != 0;
	}

	static T& instance() {
		// TODO: We aren't thread safe. For now we ignore it since the
		// only thing using this singleton template is the config manager,
		// and that is first instantiated long before any threads.
		// TODO: We don't leak, but the destruction order is nevertheless
		// semi-random. If we use multiple singletons, the destruction
		// order might become an issue. There are various approaches
		// to solve that problem, but for now this is sufficient
		if (!_singleton)
			_singleton = T::makeInstance();
		return *_singleton;
	}

	static void destroy() {
		T::destroyInstance();
	}
protected:
	Singleton()		{ }
	virtual ~Singleton()	{ }

	typedef T	SingletonBaseType;

	static T *_singleton;
};

/**
 * Note that you need to use this macro from the Common namespace.
 *
 * This is because C++ requires initial explicit specialization
 * to be placed in the same namespace as the template.
 */
#define DECLARE_SINGLETON(T) \
	template<> T *Singleton<T>::_singleton = 0

/** @} */

} // End of namespace Common

#endif
