/*  ---------------------------------------------------------------
    xhkeys. Jul 2002
    Copyright (C) 2002,  Michael Glickman  <wmalms@yahoo.com>
    License: GPL
    --------------------------------------------------------------- */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "xhkeys_conf.h"

#define _GNU_SOURCE
#include <getopt.h>

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif

#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#include "xhkeys.h"

typedef enum {
#ifdef PLUGIN_SUPPORT
  OP_PLUGINS,
#endif
  OP_KEYCODES,
  OP_MOUSEINFO,
  OP_LIST,
  OP_HELP,
  OP_MODIFY,
  OP_DELETE,
} OperationTypeEnum;  

// system
extern char *optarg;
extern int optind, opterr, optopt;

static OperationTypeEnum opType;

static unsigned int curScan = ~0;
static unsigned int curModif = ~0;

static unsigned short listOption;   // 0 -ask, 1-by code, 2 - all

const char *resFileName = NULL;

Display *dpy;
Window wnd;
Bool isChild = False;
int screenNo;
Atom WMStateAtom;
int count = 1;
#ifdef PLUGIN_SUPPORT
char *pluginName = NULL;
#endif
static Bool listGlobal;

extern int logMode;		// 0 - none, 1 - error, 2 - all
extern Bool logConsole;

static void showHelp(void) {
  static const char *helpText[] = {
	"",
	"xhkconf ver. " VERSION_TEXT ", compiled " __DATE__ " " __TIME__,
	"A configurator for xhkeys.",
	"(C) 2002, Michael Glickman <wmalms@yahoo.com>",
	"Web: www.geocities.com/wmalms",
	"",
	"Modes:",
	"\t-k --keys        show key scan codes", 
	"\t-m --mouse       show mouse button event details", 
	"\t-e --edit        add/edit assignment (default)", 
	"\t-l --list        display assignment", 
	"\t-d --delete      delete assignment", 
#ifdef PLUGIN_SUPPORT
	"\t-p --plugins     add/edit plugins", 
#endif
	"\t-h --help        this page", 
	"",
	"Parameters:",
	"\t-f --file <name> configuration file name", 
	"\t                 (default ~/.XHkeys)",
	"\t-s --scan k[.s]  scan_code.shift_state",
	"\t                 (e.g. 176.1)",
	"\t-c --count       maximum number of keys",
	"\t                 (0-unlimited, default 1)",
	"\t-g --global      list global assignments",
	"\t                 (with --list only)",
	"\t-a --all         list all assignments",
	"\t                 (with --list only)",
	"", 
	"See xhkeys.html for details", 
	"", NULL };
	
  const char *helpLine;
  int i;
  
  for (i=0; (helpLine=helpText[i])!= NULL; i++)
	puts(helpLine);
	
}

static Bool parseScanModif(const char *par)
{
  char *parcopy, *strp, s = '\0';
  unsigned int scan, modif = 0;
  static const char *seps = ".,"; 
  static const char *seps1 = " \t"; 
  Bool result = False;

  parcopy = strdup(par);
  if (parcopy == NULL) return False;

  scan = (unsigned int)strtoul(parcopy, &strp, 0);
  s = *strp++;
  if (strchr(seps1, s) != NULL) s = '\0';
  	
  if (s != '\0') {
	if (strchr(seps, s) == NULL)
  	  goto TheEnd;
	else {  
  	  modif = (unsigned int)strtoul(strp, &strp, 0);
	  s = *strp;	
	  if (s != '\0' &&
	      strchr(seps1, s) == NULL) goto TheEnd;
	}	
  }
  	
  curScan = scan;
  curModif = modif;
  listOption = 1;  		// By Code
  result = True;
  
  
TheEnd:
  free(parcopy);  
  return result;
}

static Bool ProcessCommandLine(int argc, char * const argv[])
{

  int result;
  
  static struct option long_opts[] =
  {
   { "all", no_argument, NULL, 'a'},
   { "file", required_argument, NULL, 'f'},
   { "scan", required_argument, NULL, 's'},
   { "count", required_argument, NULL, 'c'},
   { "delete", no_argument, NULL, 'd'},
   { "edit", no_argument, NULL, 'e'},
#ifdef PLUGIN_SUPPORT
   { "plugin", optional_argument, NULL, 'p'},
   { "plugins", optional_argument, NULL, 'p'},
#endif
   { "help", no_argument, NULL, 'h'},
   { "keys", no_argument, NULL, 'k'},
   { "mouse", no_argument, NULL, 'm'},
   { "global", no_argument, NULL, 'g'},
   { "delete", no_argument, NULL, 'd'},
   { "list", no_argument, NULL, 'l'},
   {  NULL, 0, NULL,  0} 
  };

  resFileName = NULL;
  opType = OP_MODIFY;
  curScan = ~0;
  listOption = 0;
  curModif = 0;
  count = 1;
  listGlobal = False;
  logMode = 1;		// Log all
  logConsole = True;	// Log to console
#ifdef PLUGIN_SUPPORT
  pluginName = NULL;
#endif
  
  while ((result = getopt_long(argc, argv, "ac:def:gp::hklms:", long_opts, NULL)) != -1) {
	switch(result) 
	{
	  case 'c':
	  {
		char *next;
		int cnt = (int) strtol(optarg, &next, 0);
		if (*next == '\0' || *next == ' ') count = cnt;
	  }		
		break;

	  case 'd':
		opType = OP_DELETE;
		break;	
				
	  case 'e':
		opType = OP_MODIFY;
	   if (optarg != NULL) 
		  parseScanModif(optarg);
		break;	

	  case 'f':
		resFileName = optarg;
		break;	

	  case 'g':
		listGlobal = True;
		break;	

	  case 'h':
		opType = OP_HELP;
		break;	

	  case 'k':
		opType = OP_KEYCODES;
		break;	

	  case 'l':
		opType = OP_LIST;
		break;	

	  case 'a':
		listOption = 2;
		break;

	  case 's':
		if (parseScanModif(optarg) == False) {
		  fprintf (stderr, "Invalid scan code + shift combination\n");
		  return False;		
		}
		break;

	  case 'm':
		opType = OP_MOUSEINFO;
		break;	
	    
	  // Process 'p' as invalid, if PLUGIN_SUPPORT is disabled!    
	  case 'p':
#ifdef PLUGIN_SUPPORT
		opType = OP_PLUGINS;
		if (optarg != NULL && pluginName == NULL)
		    pluginName = strdup(optarg);
		return True;
#endif
	  case '?':
		return False;	
	}
  
  }
  
  return True;

}

void xhkc_terminate(void)
{
#ifdef PLUGIN_SUPPORT
  if (pluginName != NULL) {
    free(pluginName); pluginName = NULL;
  }
#endif

  FreeResources();
  if (dpy != NULL) XCloseDisplay(dpy);
}


int main(int argc, char * const argv[])
{
  char *errMsg = NULL;
  Bool success = False;
  int getFlags;

  if (!ProcessCommandLine(argc, argv)) {
	showHelp();
	return 16;
  }
  
  if (opType == OP_HELP) {
	showHelp();
	return 0;
  }

  dpy = XOpenDisplay(NULL);
  if (dpy == NULL) {
	errMsg = "Can't open display";
	goto OutOfHere;
  }  

  screenNo = XDefaultScreen(dpy);    
  
  wnd = DefaultRootWindow(dpy);
  if (wnd == None) {
	errMsg = "No root window";
	goto OutOfHere;
  }  

  WMStateAtom = XInternAtom(dpy, "WM_STATE", True);

    switch(opType)
    {
#ifdef PLUGIN_SUPPORT
	case OP_PLUGINS:
	    getFlags = 2;		// Plugins only
	    break;

#endif
	case OP_KEYCODES:
	case OP_MOUSEINFO:
	    getFlags = 0;		// None
	    break;
	    
	default:    
	    getFlags = 3;		// Boith
    }	    
	    
  ProcessResources(resFileName, opType==OP_LIST && listGlobal,
                                 getFlags, &errMsg);
	
  switch(opType) {
	case OP_KEYCODES:
	  success = showScanCodes(&errMsg);
	  break;

	case OP_MOUSEINFO:
	  success = showMouseEventInfos(&errMsg);
	  break;

	case OP_LIST:
	  switch(listOption)
	  {
		case 0:
		  success = displayCommand(&errMsg);
		  break;
		  
		case 1:
		  success = displayCommandByCode(curScan, curModif, &errMsg);
		  break;
		  	  
		case 2:
	  	  success = displayAllCommands(&errMsg);
		  break;
	  }
	  break;

	case OP_MODIFY:
	  switch(listOption)
	  {
		case 0:
		  success = editCommand(&errMsg);
		  break;
		  
		case 1:
		  success = editCommandByCode(curScan, curModif, &errMsg);
		  break;
		  	  
	  }
	  break;

	case OP_DELETE:
	  switch(listOption)
	  {
		case 0:
		  success = deleteCommand(&errMsg);
		  break;
		  
		case 1:
		  success = deleteCommandByCode(curScan, curModif, &errMsg);
		  break;
		  	  
	  }
	  break;

#ifdef PLUGIN_SUPPORT
	case OP_PLUGINS:
	    if (pluginName == NULL)
		success = configurePlugins(&errMsg);
	    else	
		success = editPluginByName(pluginName, &errMsg);
	    // No break is necessary	
#endif
  	
  	default:		// OP_MODIFY
	  break;
  }	  
				  

OutOfHere:
  xhkc_terminate();  
  if (success == False) {
	if (errMsg)	fprintf(stderr, "%s\n", errMsg);
	return 16;
  }	

  return 0;
}


