//
// Syd: rock-solid application kernel
// src/kernel/link.rs: symlink(2) and symlinkat(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsFd;

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, unistd::symlinkat, NixPath};

use crate::{
    fs::FsFlags,
    hook::{RemoteProcess, SysArg, UNotifyEventRequest},
    kernel::sandbox_path,
    sandbox::Capability,
};

pub(crate) fn sys_symlink(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        // SAFETY: No checking of the target is done.
        // This is consistent with the system call.
        let arg = SysArg {
            path: Some(1),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            ..Default::default()
        };
        syscall_symlink_handler(request, arg)
    })
}

pub(crate) fn sys_symlinkat(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        // SAFETY: No checking of the target is done.
        // This is consistent with the system call.
        let arg = SysArg {
            dirfd: Some(1),
            path: Some(2),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::MISS_LAST,
            ..Default::default()
        };
        syscall_symlink_handler(request, arg)
    })
}

/// A helper function to handle symlink{,at} syscalls.
fn syscall_symlink_handler(
    request: UNotifyEventRequest,
    arg: SysArg,
) -> Result<ScmpNotifResp, Errno> {
    let req = request.scmpreq;

    let process = RemoteProcess::new(request.scmpreq.pid());

    // SAFETY: Read remote path, process will be validated
    // using the seccomp request ID before use.
    let target = unsafe { process.remote_path(req.data.args[0]) }?;
    if !request.is_valid() {
        return Err(Errno::ESRCH);
    }

    // SAFETY: symlink() returns ENOENT if target is an empty string.
    if target.is_empty() {
        return Err(Errno::ENOENT);
    }

    // Read remote path.
    let sandbox = request.get_sandbox();
    let (path, _) = request.read_path(&sandbox, arg, false)?;

    // Check for access.
    let hide = sandbox.enabled(Capability::CAP_STAT);
    sandbox_path(
        Some(&request),
        &sandbox,
        request.scmpreq.pid(), // Unused when request.is_some()
        path.abs(),
        Capability::CAP_SYMLINK,
        hide,
        "symlink",
    )?;
    drop(sandbox); // release the read-lock.

    // All done, call underlying system call.
    symlinkat(
        &target,
        path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
        path.base,
    )
    .map(|_| request.return_syscall(0))
}
