//
// Syd: rock-solid application kernel
// src/kernel/chroot.rs: chroot(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    hook::{SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
};

pub(crate) fn sys_chroot(request: UNotifyEventRequest) -> ScmpNotifResp {
    let argv = &[SysArg {
        path: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(request, "chroot", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read lock.

        // SAFETY: SysArg has one element.
        #[allow(clippy::disallowed_methods)]
        let path = path_args.0.as_ref().unwrap();

        if path.abs().is_rootfs() {
            // chroot("/") is a no-op.
            return Ok(request.return_syscall(0));
        } else if let Some(typ) = path.typ.as_ref() {
            if !typ.is_dir() {
                return Ok(request.fail_syscall(Errno::ENOTDIR));
            }
        } else {
            return Ok(request.fail_syscall(Errno::ENOENT));
        }

        // Acquire a write lock and chroot the sandbox.
        let mut sandbox = request.get_mut_sandbox();
        sandbox.chroot();
        drop(sandbox); // release the write-lock.

        Ok(request.return_syscall(0))
    })
}
