/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl	 
 * 
 *	FILE: "cpptclInt.h"
 *								   created:	11/5/96 {8:46:19 pm}
 *								   last update: 24/12/97 {10:20:09 am}
 *	 Author:  Vince	Darley 
 *	 E-mail:  darley@fas.harvard.edu
 *	   mail:  Divison of Applied Sciences, Harvard University
 *			  Cambridge	MA 02138
 *	    www:  <http://www.fas.harvard.edu/~darley/>
 * ===================================================================
 * Copyright (c) 1997  Vince Darley
 *  
 * See the file "license.terms" for information on usage and 
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * ===================================================================
 * ###################################################################
 */

#ifndef _Cpptcl_CpptclInternal_
#define _Cpptcl_CpptclInternal_

#include <assert.h>

// Can't use Stubs because of global constructors
#define USE_TCL_STUBS

// Pull in Tcl header file
#include <tcl.h>

#include "cpptclCompilers.h"

#ifndef DLL_IMPORT_EXPORT
#    define DLL_IMPORT_EXPORT DLL_EXPORT
#endif

#ifdef __GNUC__ // GCC 2.7.2
	#define NO_EXCEPTION_HANDLING
	#ifndef STATIC_LINKING_ONLY
	// if we may be linking dynamically, we need this flag
	#define NO_STATIC_INITIALISERS
	#endif
#endif

#define CPPTCL_USE_SCOPED_OBJECTS


// We replace tcl_obj with this efficient Tcl_Obj wrapper
class tcl_obj;

// Forward class declarations
struct Tcl_Interp;
class tcl_args;

//@Section: Cpptcl library
//@Man: The Cpptcl API
//@{
/// The Cpptcl C init procedure for Tcl loading
extern "C" int Cpptcl_Init(Tcl_Interp*);
/// The C++ equivalent to 'Tcl_PkgInitProc' which will be used by your pkgs.
typedef int Cpptcl_PackageInitProc(tcl_obj&);
/// The Cpptcl C++ init procedure
extern Cpptcl_PackageInitProc Cpptcl_Init;
/// A utility procedure for use in your C init procedure
/** 
 * You can use this	as follows:
 *	   extern "C" int Mypkg_Init(Tcl_Interp*);
 * 
 *	   int Mypkg_Init(Tcl_Interp* interp) {	
 *		   return Cpptcl_PkgInit(interp, Mypkg_Cppinit);
 *	   }
 *	   
 *	   Cpptcl_PackageInitProc Mypkg_Cppinit;
 *	   int Mypkg_Cppinit(tcl_obj& interp) {
 *		   ...
 *	   }
 *	   
 * but normally just do:
 * 
 *	   int Cpptcl_InitFunction(Mypkg_Init) {
 *		   ...
 *	   }
 *
 */	
extern int Cpptcl_PkgInit(Tcl_Interp* interp, Cpptcl_PackageInitProc cpx_iproc);
/// Wrap an interpreter into a tcl_obj.  
/** 
 * Your	code will normally always use 'Cpptcl_PkgInit' and thereafter
 * use tcl_objs only.  So you won't normally need this function call.
 */
extern tcl_obj& cpptcl_create_stream(Tcl_Interp*);
/// Version
#define CPPTCL_VERSION "2.0"
/// Special code to make argument handling work well
#define CPPTCL_NOT_HANDLED 37

/// Simplifies object declaration (can't use with multiple inheritance)
/** 
 * Declare an object using this macro.  It relies upon objects
 * obeying the Cpptcl RTTI system of using a static	'_type'	field.
 * If you need multiple-inheritance, you will have to call the metaobject
 * directly.  See 'cpptcl_metaobject' for details.	Also if	you	wish
 * to have the object's	Tcl	creation command be	different from its '_type'
 * then	you	must call directly.
 */
#define Cpptcl_Object(obj,parent) \
	tcl_base::metaobject->new_object_type(obj::_type,parent ::_type)
/// Used like this 'Cpptcl_Object(obj,parent).Cpptcl_ContainedIn(container_type);
#define Cpptcl_ContainedIn(cl) \
	container(cl::_type)
//@}
/// Does everything necessary to interface your package to Tcl and Cpptcl
#define Cpptcl_InitFunction(pkg_init) \
	pkg_init (tcl_obj& interp); \
	extern "C" Tcl_PackageInitProc pkg_init; \
	extern "C" DLL_EXPORT int pkg_init(Tcl_Interp* interp) { \
		return Cpptcl_PkgInit(interp, pkg_init); \
	} \
	CPPTCL_PLATFORM_ENTRY; \
	DLL_EXPORT \
	int pkg_init(tcl_obj& interp)
	
#define Cpptcl_InitNoEntryFunction(pkg_init) \
	pkg_init (tcl_obj& interp); \
	extern "C" Tcl_PackageInitProc pkg_init; \
	extern "C" DLL_EXPORT int pkg_init(Tcl_Interp* interp) { \
		return Cpptcl_PkgInit(interp, pkg_init); \
	} \
	DLL_EXPORT \
	int pkg_init(tcl_obj& interp)
	
//@Man: For inside the class definition
//@{
/// abstract base class without members
#define Cpptcl_BaseClass(cl) \
	virtual const meta_object& meta_info(void) const {return _type;} \
	virtual cpx_type type(void) const {return _type.type;} \
	static DLL_IMPORT_EXPORT const meta_object _type
/// abstract class with members
#define Cpptcl_AbstractClass(cl) \
	virtual const meta_object& meta_info(void) const {return _type;} \
	virtual cpx_type type(void) const {return _type.type;} \
	friend class meta__members<cl>; \
	static DLL_IMPORT_EXPORT const meta_type<cl> _type
/// non-abstract class with members
#define Cpptcl_Class(cl) \
	virtual const meta_object& meta_info(void) const {return _type;} \
	virtual cpx_type type(void) const {return _type.type;} \
	friend class meta__members<cl>; \
	static DLL_IMPORT_EXPORT const meta_createable_type<cl> _type
///
//@}

//@Man: For the beginning of the source file
//@{
/// declare members
#ifdef __MWWERKS__
#define Cpptcl_Members(cl) \
  cpp_member_info * meta__members<cl>::_member_info = 0; \
  static const cpp_member_info meta__members ## cl ## _member_info[] 
/// instantiate non-abstract class with members
#define Cpptcl_IClass(cl,name,parent) \
  static meta_members<cl> cl ## _members(meta__members ## cl ## _member_info,sizeof(meta__members ## cl ## _member_info)); \
  DLL_EXPORT  \
  const meta_createable_type<cl> cl::_type(name,&cl ## _members)
/// instantiate non-abstract class with no members
#define Cpptcl_IClassNoMembers(cl,name,parent) \
  DLL_EXPORT  \
  const meta_createable_type<cl> cl::_type(name,0)
/// instantiate abstract base class without members
#define Cpptcl_IBaseClass(cl,name,parent) \
  DLL_EXPORT  \
  const meta_object cl::_type(name)
/// instantiate abstract class with members
#define Cpptcl_IAbstractClass(cl,name,parent) \
  static meta_members<cl> cl ## _members(meta__members ## cl ## _member_info,sizeof(meta__members ## cl ## _member_info)); \
  DLL_EXPORT  \
  const meta_type<cl> cl::_type(name,&cl ## _members)
#else
#define Cpptcl_Members(cl) \
  const cpp_member_info meta__members<cl>::_member_info[]
/// instantiate non-abstract class with members
#define Cpptcl_IClass(cl,name,parent) \
  static meta_members<cl> cl ## _members; \
  DLL_EXPORT  \
  const meta_createable_type<cl> cl::_type(name,&cl ## _members)
/// instantiate non-abstract class with no members
#define Cpptcl_IClassNoMembers(cl,name,parent) \
  DLL_EXPORT  \
  const meta_createable_type<cl> cl::_type(name,0)
/// instantiate abstract base class without members
#define Cpptcl_IBaseClass(cl,name,parent) \
  DLL_EXPORT  \
  const meta_object cl::_type(name)
/// instantiate abstract class with members
#define Cpptcl_IAbstractClass(cl,name,parent) \
  static meta_members<cl> cl ## _members; \
  DLL_EXPORT  \
  const meta_type<cl> cl::_type(name,&cl ## _members)
#endif
//@}

//@Section: Cpptcl library
//@Man: Cpptcl RTTI 
//@{
// Perhaps I should simplify these.

/// Is type 'a' descended from type 'b'?
/** 
 * Is type 'a' descended from type 'b'?
 * 'a' is a	character string
 * 'b' is the name of a	C++	class
 * The return value	is a bool
 */
#define maycast(a,b) tcl_interaction::metaobject->is_of_type(a,b::_type.type)
/// Now 'b' is also a character string.
/** 
 * Now 'b' is also a character string.
 */
#define namecast(a,b) tcl_interaction::metaobject->is_of_type(a,b)
/// Is object 'a' descended from a type 'b'.
/** 
 * Is object 'a' descended from	a type 'b'.
 * 'a' is a	pointer	to an actual object	in existence
 * 'b' is the name of a	C++	class
 * The return value	is either a	cast object	or zero	if the cast	failed.
 * The user	should check the return	value before using.
 */
#define ckcast(a,b) (namecast(a->type(),b::_type.type) ? (b*)a : (b*) 0)

/// Do a cast.
/** 
 * Do a	cast.
 * As with ckcast, except this is used where we	expect a cast to work, and wish
 * to dump if it fails.	i.e. we're not going to	check if it	worked.	If debugging
 * is off, then	this just does an ordinary cast.
 */
#ifndef SLOW_DEBUG
/// this is the default
#define cast(a,b) ((b*)a)
#else
/// this is a _lot_ slower but checks type dynamically
#define cast(a,b) (namecast(a->type(),b::_type.type) ? (b*)a : (b*) 0)
#endif
//@}

// ------------------------------------------------------------------
//  Everything below is for debugging and/or compilation
// ------------------------------------------------------------------

#define DO_NOTHING ;

#ifdef DEBUG_PRINT
	#define PRINT(var); cout << #var << " = " << var << endl;
#else
	#define PRINT(var);
#endif



#endif
