/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains font-related routines.
 */

#include "types.h"
#include "devtab.h"
#include "font.h"
#include "tracedef.h"

#include "arith.p"
#include "heap.p"
#include "raster.p"
#include "strng.p"

/*
 *	Routine Load_All_Fonts loads the fonts that were
 *	referenced in the DVI file:
 */

int Load_All_Fonts (Device_Ptr, Use_TFM, DVI_Fraction, Job_Magnification,
		    Load_Func)
struct Device_Table *Device_Ptr;
unsigned long Use_TFM, Job_Magnification;
struct Ratio *DVI_Fraction;
int (*Load_Func)();
{
	auto   struct Font_Definition *Font_Ptr;
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned long Cksum;
	auto   int Index, Err_Count, Load_Status;
	auto   char *Font_Area, *Prefix;
	static char Temp_Str[32];
	extern long Convert();
	extern int Load_TFM(), strcmp(), strlen();
	extern char TeX_TFM_Dir[];
	msgcode DVIOUT_CKSUMMISMATCH, DVIOUT_USEUNDEFCHAR;
/*
 *	Initialize:
 */
	Char_Count = 0;
	Max_Char_Size = 0;
	Err_Count = 0;
/*
 *	For each font...
 */
	for (Font_Ptr = Font_Def_Head; Font_Ptr != 0; Font_Ptr = Font_Ptr->Link) {
		Undef_Char_Count = 0;
/*
 *	Determine if the font is 'Native'. The font name must
 *	begin with one of a set of special prefixes to be
 *	identified as a font native to the output device:
 */
		for (Index = 0; (Prefix = Device_Ptr->Native_Font_Prefix[Index]) != 0; Index++) {
			stringcpy_m (Temp_Str, Font_Ptr->Name_Ptr, strlen (Prefix) + 1);
			if (strcmp (Temp_Str, Prefix) == 0) {
				Font_Ptr->Flags |= NATIVE_FONT;
				break;
			}
		}
/*
 *	Call the routine to load the font from either a pixel file or the
 *	TFM file. If a pixel file error occurs and the user has OK'd the
 *	use of TFM files, load the font from the TFM file instead (this
 *	will leave 'holes' in the output, of course):
 */
		Font_Area = (*Font_Ptr->Area_Ptr == '\0') ? TeX_TFM_Dir : Font_Ptr->Area_Ptr;
		Cksum = Font_Ptr->Checksum;
		if ((Font_Ptr->Flags & NATIVE_FONT) != 0 ||
		    ((Load_Status = (*Load_Func) (Device_Ptr->Default_Font_Directory,
						  &Device_Ptr->Device_Resolution,
						  Font_Ptr)) == 0 && Use_TFM != 0))
			Load_Status = Load_TFM (Font_Area, &Device_Ptr->Device_Resolution, Font_Ptr);
		if (Load_Status == 0) {
			Err_Count++;
			continue;
		}
		if (Font_Ptr->Checksum != 0 && Cksum != 0 && Font_Ptr->Checksum != Cksum)
			Message (DVIOUT_CKSUMMISMATCH, Font_Ptr->Name_Ptr, 1);
		if (Undef_Char_Count > 0)
			Message (DVIOUT_USEUNDEFCHAR, Font_Ptr->Name_Ptr, 1);
/*
 *	If the font is the MARK font, pick up the font's design size
 *	and at size now:
 */
		if ((Font_Ptr->Flags & MARK_FONT) != 0 && Font_Ptr->Design_Size == 0) {
			Font_Ptr->Design_Size = XN_Div_D_T_M (Font_Ptr->Font_Design_Size,
							      XN_Div_D_T_M (1587500, DVI_Fraction->Denominator,
									    DVI_Fraction->Numerator),
							      473628672);
			Font_Ptr->At_Size = XN_Div_D_T_M (XN_Div_D_T_M (Font_Ptr->Design_Size,
									Font_Ptr->Magnification,
									1000),
							  1000, Job_Magnification);
		}
		fold_m (Font_Ptr->Font_Coding);
		fold_m (Font_Ptr->Font_Family);
/*
 *	Accumulate statistics about the font; map TeX font character
 *	codes to native character codes if the font is native;
 *	compute character width in DVI units and in 1/1024 pixels;
 *	convert characters of mirror-image fonts;
 */
		for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
		if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0) {
			Char_Count++;
			Char_Ptr->DVI_Width = XN_Div_D_T_M (Char_Ptr->DVI_Width, Font_Ptr->At_Size, 1<<20);
			Char_Ptr->H_Escapement = Convert (Char_Ptr->DVI_Width);
			Char_Ptr->V_Escapement = 0;
			if ((Font_Ptr->Flags & MIRROR_FONT) != 0) {
				Char_Ptr->H_Escapement = -Char_Ptr->H_Escapement;
				Char_Ptr->X_Origin = (short) Char_Ptr->Pixel_Width - Char_Ptr->X_Origin;
				Reflect_Raster_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
						  Char_Ptr->Pixel_Array);
			}
		}
/*
 *	Set max sizes; do some /tracing:
 */
		Set_Pixel_Bounds (Font_Ptr);
		if (Font_Ptr->Max_Unpacked > Max_Char_Size)
			Max_Char_Size = Font_Ptr->Max_Unpacked;
		if (trace_font)
			Show_Font (Font_Ptr);
		if (trace_char || trace_raster) {
			for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
			if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0) {
				printf ("Character code %lu:\n", Char_Ptr->Character_Code);
				if (trace_char) {
					printf ("  Raster size %u by %u origined at (%d,%d)\n",
						Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
						Char_Ptr->X_Origin, Char_Ptr->Y_Origin);
					printf ("  Width in DVI units: %ld\n", Char_Ptr->DVI_Width);
					printf ("  Character escapement (1/1024 pixels): h = %ld; v = %ld\n",
						Char_Ptr->H_Escapement, Char_Ptr->V_Escapement);
				}
				if (trace_raster) {
					printf ("  Pictorial raster description:\n");
					Display_Unpacked_Raster (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
								 Char_Ptr->X_Origin, Char_Ptr->Y_Origin,
								 Char_Ptr->Pixel_Array);
				}
			}
		}
	}
	if (trace_font) {
		printf ("Total number of characters in all fonts is %u\n", Char_Count);
		printf ("Maximum character size (width times height) is %u\n", Max_Char_Size);
	}
	return ((Err_Count == 0) ? 1 : 0);
}

Show_Font (Font_Ptr)
struct Font_Definition *Font_Ptr;
{
	static char *PXL_Type[] = { "TFM", "PXL", "PK", "GF", "Internal" };

	printf ("Font name: %s\n", Font_Ptr->Name_Ptr);
	printf ("  Font number: %ld  Font area: \"%s\"  Checksum: %lu\n",
		Font_Ptr->Font_Number, Font_Ptr->Area_Ptr, Font_Ptr->Checksum);
	printf ("  Font design size (fix_word): %lu used at magnification %lu\n",
		Font_Ptr->Font_Design_Size, Font_Ptr->Magnification);
	printf ("  Design size: %lu; At size: %lu\n", Font_Ptr->Design_Size, Font_Ptr->At_Size);
	printf ("  Maximum character raster array sizes: %lu (unpacked), %lu (packed), %lu (compressed)\n",
		Font_Ptr->Max_Unpacked, Font_Ptr->Max_Packed, Font_Ptr->Max_Compressed);
	printf ("  Maximum character raster width and height: %u by %u\n", Font_Ptr->Max_Width,
		Font_Ptr->Max_Height);
	if (Font_Ptr->Flags != 0) {
		printf ("  Flags:");
		if ((Font_Ptr->Flags & NATIVE_FONT) != 0)
			printf (" native");
		if ((Font_Ptr->Flags & MIRROR_FONT) != 0)
			printf (" mirror");
		if ((Font_Ptr->Flags & MARK_FONT) != 0)
			printf (" mark");
		if ((Font_Ptr->Flags & TEMP_FONT) != 0)
			printf (" temporary");
		printf ("\n");
	}
	printf ("  %s file format is %lu\n", PXL_Type[Font_Ptr->Pixel_Id>>ID_V_TYPE],
		Font_Ptr->Pixel_Id & ID_M_FORMAT);
}

int Check_Undefined_Char (Font_Ptr, Low_Index, High_Index)
struct Font_Definition *Font_Ptr;
unsigned int Low_Index, High_Index;
{
	auto   struct Font_Definition *F_Ptr;
	auto   unsigned int Index;
	auto   int Count;

	F_Ptr = Font_Ptr;
	Count = 0;
	for (Index = Low_Index; Index < High_Index; Index++)
	if (F_Ptr->Font_Directory[Index] != 0) {
		F_Ptr->Font_Directory[Index] = 0;
		Count++;
	}
	return (Count);
}

/*
 *	Routine Download_Fonts downloads the characters used in the fonts
 *	used in order of decreasing frequency.
 */

Download_Fonts (Download_F, Resolution)
int (*Download_F)();
struct Ratio *Resolution;
{
	auto   struct Char_Definition *Char_Ptr, **Def_Vector;
	auto   struct Font_Definition *Font_Ptr;
	auto   struct Encoding *Enc_Ptr;
	auto   pointer Font_Heap, Char_Heap;
	auto   unsigned long Freq, Kndex;
	auto   unsigned int Index, Count;
	extern unsigned char *Mem_Alloc();
/*
 *	First, scan through all the fonts to determine the most frequently
 *	used:
 */
	Font_Heap = Initialize_Heap_M (Font_Count, -1);
	for (Font_Ptr = Font_Def_Head; Font_Ptr != 0; Font_Ptr = Font_Ptr->Link) {
		Freq = 0;
		for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
		if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0)
			Freq += Char_Ptr->Driver_Id;
		if (Freq > 0)
			Insert_Heap_Entry_M (Font_Heap, Freq, Font_Ptr);
	}
/*
 *	Now, extracting font definitions from the heap in order of
 *	decreasing frequency, determine the characters used from the
 *	font in order of decreasing frequency; call the download_font
 *	routine with only these characters and fonts:
 */
	while (Extract_Heap_Entry_M (Font_Heap, &Freq, &Kndex) != 0) {
		Font_Ptr = (struct Font_Definition *) Kndex;
		Char_Heap = Initialize_Heap_M (Font_Ptr->Char_Dir_Count, -1);
		Count = 0;
		for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
		if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0 &&
		    Char_Ptr->Driver_Id != 0) {
			Insert_Heap_Entry_M (Char_Heap, Char_Ptr->Driver_Id, Index);
			Char_Ptr->Driver_Id = 0;
			Count++;
		}
		Def_Vector = (struct Char_Definition **) Mem_Alloc (Count * sizeof (struct Char_Definition *));
		Count = 0;
		while (Extract_Heap_Entry_M (Char_Heap, &Freq, &Kndex) != 0)
			Def_Vector[Count++] = Font_Ptr->Font_Directory[Kndex];
		Dissolve_Heap_M (Char_Heap);
/*
 *	Download the font to the device. The device driver is responsible
 *	for setting the encoding vector, if it is something other that the
 *	identity encoding. Then map each character code into the code
 *	that the output device will recognize as the position within the
 *	font that each character occupies.
 */
		(*Download_F) (Font_Ptr, Def_Vector, Count);
		Mem_Free (Def_Vector);
		if ((Font_Ptr->Flags & NATIVE_FONT) != 0) {
			Enc_Ptr = Font_Ptr->Font_Encoding;
			for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
			if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0)
				Char_Ptr->Character_Code =
				    (Char_Ptr->Character_Code >= Enc_Ptr->Encoding_Length) ?
					Enc_Ptr->Undef_Code :
					Enc_Ptr->Encoding_Map[Char_Ptr->Character_Code];
		}
	}
	Dissolve_Heap_M (Font_Heap);
}

/*
 *	Because of the effects of repeated rounding, sometimes the
 *	computation to determine the font magnification number
 *	used in pixel file names is off by one. These next two
 *	routines compute what the number SHOULD be, in order to
 *	properly identify the pixel file for a particular magnification.
 */

int Get_Font_Mag (Scale, Resolution, Approx_Mag)
struct Ratio *Resolution;
int Scale, Approx_Mag;
{
	auto   int Index, Mag, Mag_Value, Temp;
	static struct Ratio Scale_Values[] = {
		{ 1, 1 },			/* \magstep 0 */
		{ 1095445115, 1000000000 },	/* \magstep 0.5 */
		{ 12, 10 },			/* \magstep 1 */
		{ 1314534138, 1000000000 },	/* \magstep 0.5 scaled 1.2 */
		{ 144, 100 },			/* \magstep 2 */
		{ 1577440966, 1000000000 },	/* \magstep 0.5 scaled 1.44 */
		{ 1728, 1000 },			/* \magstep 3 */
		{ 1892929159, 1000000000 },	/* \magstep 0.5 scaled 1.728 */
		{ 2, 1 },			/* 2000 */
		{ 20736, 10000 },		/* \magstep 4 */
		{ 227151499, 100000000 },	/* \magstep 0.5 scaled 2.0736 */
		{ 248832, 100000 },		/* \magstep 5 */
		{ 272581799, 100000000 },	/* \magstep 0.5 scaled 2.48832 */
		{ 2985984, 1000000 },		/* \magstep 6 */
		{ 3, 1 },			/* 3000 */
		{ 327098159, 100000000 },	/* \magstep 0.5 scaled 2.985984 */
		{ 35831808, 10000000 },		/* \magstep 7 */
		{ 392517812, 100000000 },	/* \magstep 0.5 scaled 3.5831808 */
		{ 4, 1 },			/* 4000 */
		{ 429981696, 100000000 },	/* \magstep 8 */
		{ 471021353, 100000000 },	/* \magstep 0.5 scaled 4.29981696 */
		{ 5, 1 },			/* 5000 */
		{ 6, 1 },			/* 6000 */
		{ 7, 1 },			/* 7000 */
		{ 8, 1 }			/* 8000 */
	};

	Mag_Value = Approx_Mag;
	for (Index = 0; Index < arraysize (Scale_Values); Index++) {
		Mag = XN_Div_D_R_M (XN_Div_D_R_M (Scale * Resolution->Numerator * Resolution->Denominator,
						  Scale_Values[Index].Numerator,
						  Scale_Values[Index].Denominator),
				    1, Resolution->Denominator * Resolution->Denominator);
		if ((Temp = Mag - Mag_Value) < 0)
			Temp = -Temp;
		if (Temp < 4) {
			Mag_Value = Mag;
			break;
		} else if (Mag > Mag_Value)
			break;
	}
	return (Mag_Value);
}

/*
 *	Routine Clip_Character determines if a character is within a
 *	specified clip bounds.
 */

int Clip_Character (X, Y, Char_Ptr, Left, Right, Top, Bottom)
int X, Y, Left, Right, Top, Bottom;
struct Char_Definition *Char_Ptr;
{
	auto   int XX, YY;

	return (((XX = X - (int) Char_Ptr->X_Origin) >= Left &&
		 XX + (int) Char_Ptr->Pixel_Width < Right &&
		 (YY = Y - (int) Char_Ptr->Y_Origin) >= Top &&
		 YY + (int) Char_Ptr->Pixel_Height < Bottom) ? 1 : 0);
}

/*
 *	Routine String_Width determines the DVI width of a
 *	string of native characters:
 */

long String_Width (Font_Ptr, String, Space_Ratio)
struct Font_Definition *Font_Ptr;
char *String;
struct Ratio *Space_Ratio;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   long Width, Space_Width;
	auto   int Count, Index;
	auto   char c;
	extern unsigned char XOrd[];
	extern int strlen();

	Space_Width = XN_Div_D_T_M (Font_Ptr->Design_Size, Space_Ratio->Numerator, Space_Ratio->Denominator);
	Count = strlen (String);
	Width = 0;
	for (Index = 0; Index < Count; Index++) {
		if ((c = String[Index]) == ' ')		/* Fonts don't have space characters */
			Width += Space_Width;
		else if ((Char_Ptr = Font_Ptr->Font_Directory[XOrd[c]]) != 0)
			Width += Char_Ptr->DVI_Width;
	}
	return (Width);
}

/*
 *	Routine Set_Pixel_Bounds determines the maximum width, height
 *	and combination width * height for a font.
 */

Set_Pixel_Bounds (Font_Ptr)
struct Font_Definition *Font_Ptr;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned long Max_Packed_Size, Max_Unpacked_Size, Max_Compressed_Size, Size;
	auto   unsigned int Index;
	auto   unsigned short Max_Wid, Max_Hgt, Packed_Width;

	Max_Packed_Size = Max_Unpacked_Size = Max_Compressed_Size = 0;
	Max_Wid = Max_Hgt = 0;
	for (Index = 0; Index < Font_Ptr->Char_Dir_Count; Index++)
	if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0) {
		if (Char_Ptr->Pixel_Width > Max_Wid)
			Max_Wid = Char_Ptr->Pixel_Width;
		if (Char_Ptr->Pixel_Height > Max_Hgt)
			Max_Hgt = Char_Ptr->Pixel_Height;
		if ((Size = Char_Ptr->Pixel_Width * Char_Ptr->Pixel_Height) > Max_Unpacked_Size)
			Max_Unpacked_Size = Size;
		if ((Size = ((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height) > Max_Packed_Size)
			Max_Packed_Size = Size;
		if ((Size = Compressed_Size_Func_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height)) > Max_Compressed_Size)
			Max_Compressed_Size = Size;
	}
	Font_Ptr->Max_Packed = Max_Packed_Size;
	Font_Ptr->Max_Unpacked = Max_Unpacked_Size;
	Font_Ptr->Max_Compressed = Max_Compressed_Size;
	Font_Ptr->Max_Width = Max_Wid;
	Font_Ptr->Max_Height = Max_Hgt;
}
