(* assign - Routines to assign and deassign a channel to a terminal line
   (frequently an auto-dial modem) from a pool of such lines. These routines
   will optionally wait for a line to become available. LAT lines are supported
   and handled properly. Logical names with multiple translations pointing at
   multiple devices are also handled properly. Recursive definitions of this
   type up to 10 levels deep are allowed. In the case of a spooled device
   the avoid_spooling flag controls whether or not these routines assign
   to the real device and bypass spooling. *)

(* Written by Ned Freed, 18-Oct-1988
   Copyright (c) 1988 PMDF Project *)

[inherit ('sys$library:starlet.pen')] module lib_assign (output);

const
  maxsearchdepth = LNM$C_MAXDEPTH;
  debugging = false;

  (* LAT rejection codes -- no standard symbolic names yet *)
  LAT__UNKNOWN = 0;
  LAT__SHUTDOWN = 2;
  LAT__INSRES = 5;
  LAT__INUSE = 6;
  LAT__NOSUCHSERV = 7;
  LAT__SERVDIS = 8;
  LAT__SERVNOTOFF = 9;
  LAT__NOSUCHPORT = 10;
  LAT__IMMACCREJ = 13;
  LAT__ACCDENIED = 14;
  LAT__CORRREQ = 15;
  LAT__FUNNOTSUP = 16;
  LAT__SESSNOTSTART = 17;
  LAT__QUENTDEL = 18;
  LAT__BADPARM = 19;

type
  $uword = [word] 0..65535;
  $word = [word] -32768..32767;
  $qwad = [quad, unsafe] array [1..4] of $word;

  (* A string descriptor type used to handle string parameters in the
     most general possible way. *)

  longest_possible_string = packed array [1..65535] of char;
  string_descriptor = record
                        length : [word] 0..65535;
                        dtype, dclass : [byte] 0..255;
                        address : ^longest_possible_string;
                      end;

  string = varying [256] of char;

  item = record
           len  : [word] 0..65535;
           code : [word] 0..65535;
           addr : [long] integer;
           rlen : [long] integer;
         end; (* item record *)

[asynchronous, unbound, external (LIB$SCOPY_DXDX)]
function copy_string (var    src_string : string_descriptor;
                      %descr dst_string : string) : integer; extern;

[asynchronous, unbound, external (LIB$WAIT)]
function LIB$WAIT (seconds : real) : integer; extern;

[global] function lib_assign (var devnam : string_descriptor; var chan : $uword;
  var mbxnam : string_descriptor; wait_periods, avoid_spooling : integer) :
  integer;

label
  10, 999;

var
  vms_status, depth, possibles : integer; inner_devnam, inner_mbxnam : string;
  mbxnam_present : boolean; items : array [1..4] of item;

  function trial_assign (current_devnam : string) : integer;

  label
    777, 888, 999;

  var
    vms_status, dvi_class, dvi_spl : integer;
    alloc_devnam, dvi_devnam : string; iosb : [volatile] $qwad;

  begin (* trial_assign *)
    if debugging then writeln (' trial_assign (', current_devnam, ')');
    if odd (avoid_spooling) then begin
      vms_status := $ALLOC (devnam := current_devnam,
                            phybuf := alloc_devnam.body,
                            phylen := alloc_devnam.length);
      if debugging then writeln ('  $ALLOC status = ', vms_status:0);
      if not odd (vms_status) then begin
        if vms_status = SS$_DEVALLOC then possibles := succ (possibles);
        goto 999;
      end;
      if debugging then writeln ('  $ALLOC returned device: ', alloc_devnam);
    end else alloc_devnam := current_devnam;
    if mbxnam_present then
      vms_status := $ASSIGN (devnam := alloc_devnam, chan := chan,
                             mbxnam := inner_mbxnam)
    else vms_status := $ASSIGN (devnam := alloc_devnam, chan := chan);
    if debugging then writeln ('  $ASSIGN status = ', vms_status:0);
    if not odd (vms_status) then begin
      if vms_status = SS$_DEVALLOC then possibles := succ (possibles);
      goto 888;
    end;
    with items[1] do begin
      len := 4; code := DVI$_SPL + DVI$M_SECONDARY;
      addr := iaddress (dvi_spl); rlen := 0;
    end; (* with *)
    with items[2] do begin
      len := 0; code := 0; addr := 0; rlen := 0;
    end; (* with *)
    dvi_spl := 0;
    vms_status := $GETDVIW (chan := chan, iosb := iosb, itmlst := items);
    if debugging then writeln ('  dvi_spl = ', dvi_spl:0);
    with items[1] do begin
      len := 4; code := DVI$_DEVCLASS + dvi_spl;
      addr := iaddress (dvi_class); rlen := 0;
    end; (* with *)
    with items[2] do begin
      len := 256; code := DVI$_DEVNAM + dvi_spl;
      addr := iaddress (dvi_devnam.body); rlen := iaddress (dvi_devnam.length);
    end; (* with *)
    with items[3] do begin
      len := 0; code := 0; addr := 0; rlen := 0;
    end; (* with *)
    vms_status := $GETDVIW (chan := chan, iosb := iosb, itmlst := items);
    if debugging then begin
      writeln ('  $GETDVIW status = ', vms_status:0);
      writeln ('  $GETDVIW IOSB status = ', iosb[1]:0);
    end;
    if odd (vms_status) then vms_status := iosb[1];
    if not odd (vms_status) then goto 777;
    if debugging then writeln ('  dvi_devnam = ', dvi_devnam);
    if debugging then writeln ('  dvi_class = ', dvi_class:0);
    if (dvi_spl = 0) or odd (avoid_spooling) then begin
      if dvi_class <> DC$_TERM then begin
        vms_status := SS$_IVDEVNAM; goto 777;
      end;
      if index (dvi_devnam, 'LT') > 0 then begin
        vms_status := $QIO (chan := chan, iosb := IOSB,
                            func := IO$_TTY_PORT + IO$M_LT_CONNECT);
        if debugging then writeln ('  $QIO status = ', vms_status:0);
        if vms_status = SS$_DEVACTIVE then begin
          if debugging then writeln ('  Device active, repeating $QIO');
          LIB$WAIT (5.0); 
          vms_status := $QIO (chan := chan, iosb := IOSB,
                              func := IO$_TTY_PORT + IO$M_LT_CONNECT);
          if debugging then writeln ('  $QIO status = ', vms_status:0);
        end;
        if vms_status = SS$_DEVACTIVE then possibles := succ (possibles);
        if not odd (vms_status) then goto 777;
        vms_status := $SYNCH (iosb := iosb);
        if debugging then writeln ('  $SYNCH status = ', vms_status:0);
        if not odd (vms_status) then goto 777;
        vms_status := iosb[1];
        if debugging then writeln ('  $QIO IOSB status = ', vms_status:0);
        if vms_status = SS$_ABORT then begin
          if debugging then writeln ('  LAT rejection code = ', iosb[2]:0);
          case iosb[2] of
            (* temporary errors *)
            LAT__INUSE, LAT__IMMACCREJ, LAT__CORRREQ, LAT__SESSNOTSTART,
            LAT__QUENTDEL : possibles := succ (possibles);
            (* permanent errors *)
            LAT__UNKNOWN, LAT__SHUTDOWN, LAT__INSRES, LAT__NOSUCHSERV,
            LAT__SERVDIS, LAT__SERVNOTOFF, LAT__NOSUCHPORT, LAT__ACCDENIED,
            LAT__FUNNOTSUP, LAT__BADPARM : ;
            (* unknowns are permanent *)
            otherwise ;
          end; (* case *)
        end;
        if not odd (vms_status) then goto 777;
      end;
    end;
    goto 999;
  777:
    if debugging then writeln ('  deassigning device');
    $DASSGN (chan := chan);
  888:
    if odd (avoid_spooling) then begin
      if debugging then writeln ('  deallocating device');
      $DALLOC (devnam := alloc_devnam);
    end;
  999:
    trial_assign := vms_status;
    if debugging then writeln ('  trial_assign status = ', vms_status:0);
  end; (* trial_assign *)

  function inner_assign (current_devnam : string) : integer;

  label
    10;

  var
    lnm_max_index, lnm_index, lnm_attributes : unsigned;
    lnm_string : string;
    vms_status : integer;

  begin (* inner_assign *)
    if debugging then writeln (' inner_assign (', current_devnam, ')');
    (* Increment depth count *)
    depth := succ (depth);
    if debugging then writeln ('  depth now = ', depth:0);
    (* Remove any trailing colons *)
    if length (current_devnam) > 0 then
      if current_devnam[length (current_devnam)] = ':' then begin
        current_devnam := substr (current_devnam,
                                  1, pred (length (current_devnam)));
        if debugging then
          writeln ('  trailing colon removed: ', current_devnam);
      end;
    (* Start by determining what sort of name we have been given *)
    with items[1] do begin
      len := 4; code := LNM$_MAX_INDEX;
      addr := iaddress (lnm_max_index); rlen := 0;
    end;      
    with items[2] do begin
      len := 0; code := 0; addr := 0; rlen := 0;
    end;      
    vms_status := $TRNLNM (attr := LNM$M_CASE_BLIND, tabnam := 'LNM$FILE_DEV',
                           lognam := current_devnam, itmlst := items);
    if odd (vms_status) then begin
      if debugging then writeln ('  lnm_max_index = ', lnm_max_index:0);
      for lnm_index := 0 to lnm_max_index do begin
        with items[1] do begin
          len := 4; code := LNM$_INDEX;
          addr := iaddress (lnm_index); rlen := 0;
        end;      
        with items[2] do begin
          len := 4; code := LNM$_ATTRIBUTES;
          addr := iaddress (lnm_attributes); rlen := 0;
        end;      
        with items[3] do begin
          len := 256; code := LNM$_STRING;
          addr := iaddress (lnm_string.body);
          rlen := iaddress (lnm_string.length);
        end;      
        with items[4] do begin
          len := 0; code := 0; addr := 0; rlen := 0;
        end;      
        vms_status := $TRNLNM (attr := LNM$M_CASE_BLIND,
                               tabnam := 'LNM$FILE_DEV',
                               lognam := current_devnam, itmlst := items);
        if odd (vms_status) then begin
          if debugging then writeln (  '  index #', lnm_index:0,
                                     ' translation = ', lnm_string);
          if uand (lnm_attributes, lnm$m_exists) <> 0 then begin
            if (uand (lnm_attributes, lnm$m_terminal) <> 0) or
               (depth >= maxsearchdepth) then
              vms_status := trial_assign (lnm_string)
            else vms_status := inner_assign (lnm_string);
            if odd (vms_status) then goto 10;
          end;
        end;
      end; (* for *)
      vms_status := SS$_NOSUCHDEV;
    end else vms_status := trial_assign (current_devnam);
  10:
    (* Decrement depth count *)
    depth := pred (depth);
    if debugging then writeln ('  depth now = ', depth:0);
    inner_assign := vms_status;
    if debugging then writeln ('  inner_assign status = ', vms_status:0);
  end; (* inner_assign *)

begin (* lib_assign *)
  if debugging then writeln ('lib_assign called.');
  vms_status := copy_string (devnam, inner_devnam);
  if not odd (vms_status) then goto 999;
  if debugging then writeln (' devnam = ', inner_devnam);
  mbxnam_present := iaddress (mbxnam) <> 0;
  if mbxnam_present then begin
    vms_status := copy_string (mbxnam, inner_mbxnam);
    if not odd (vms_status) then goto 999;
    if inner_mbxnam = '' then mbxnam_present := false;
  end;
  if debugging then begin
    if mbxnam_present then writeln (' mbxnam = ', inner_mbxnam)
    else writeln (' no mbxnam provided');
    writeln (' wait_periods = ', wait_periods:0);
  end;
10:
  depth := 0; possibles := 0;
  vms_status := inner_assign (inner_devnam);
  if debugging then writeln (' possibles = ', possibles:0);
  if (not odd (vms_status)) and
     (wait_periods <> 0) and (possibles > 0) then begin
    if debugging then writeln (' inner_assign failed, waiting and looping');
    vms_status := LIB$WAIT (30.0); 
    if not odd (vms_status) then goto 999;
    if wait_periods > 0 then wait_periods := pred (wait_periods);
    if debugging then writeln (' wait_periods = ', wait_periods:0);
    goto 10;
  end;
999:
  if debugging then writeln (' lib_assign status = ', vms_status);
  lib_assign := vms_status;
end; (* lib_assign *)  

[global] function lib_dassgn (chan : $uword;
  avoid_spooling : integer) : integer;

var
  vms_status, dvi_spl : integer; items : array [1..2] of item;
  iosb : [volatile] $qwad; dvi_devnam : string;

begin (* lib_dassgn *)
  if debugging then writeln ('lib_dassgn called.');
  with items[1] do begin
    len := 4; code := DVI$_SPL + DVI$M_SECONDARY;
    addr := iaddress (dvi_spl); rlen := 0;
  end; (* with *)
  with items[2] do begin
    len := 0; code := 0; addr := 0; rlen := 0;
  end; (* with *)
  dvi_spl := 0;
  vms_status := $GETDVIW (chan := chan, iosb := iosb, itmlst := items);
  if debugging then writeln ('  dvi_spl = ', dvi_spl:0);
  with items[1] do begin
    len := 256; code := DVI$_DEVNAM + dvi_spl;
    addr := iaddress (dvi_devnam.body); rlen := iaddress (dvi_devnam.length);
  end; (* with *)
  with items[2] do begin
    len := 0; code := 0; addr := 0; rlen := 0;
  end; (* with *)
  vms_status := $GETDVIW (chan := chan, iosb := iosb, itmlst := items);
  if debugging then begin
    writeln ('  $GETDVIW status = ', vms_status:0);
    writeln ('  $GETDVIW IOSB status = ', iosb[1]:0);
  end;
  if odd (vms_status) then vms_status := iosb[1];
  if debugging then writeln ('  dvi_devnam = ', dvi_devnam);
  if odd (vms_status) then begin
    if (dvi_spl = 0) or odd (avoid_spooling) then
      if index (dvi_devnam, 'LT') > 0 then begin
        vms_status := $QIOW (chan := chan, iosb := IOSB,
                             func := IO$_TTY_PORT + IO$M_LT_DISCON);
        if debugging then writeln ('  $QIO status = ', vms_status:0);
        if debugging then writeln ('  $QIO IOSB status = ', vms_status:0);
      end;
    vms_status := $DASSGN (chan := chan);
    if odd (avoid_spooling) then $DALLOC (devnam := dvi_devnam);
  end else vms_status := $DASSGN (chan := chan);
  lib_dassgn := vms_status;
end; (* lib_dassgn *)

end. (* module assign *)
