/* -*-C-*- */
#include "texsgi.h"
#include "externs.h"
#include "proto.h"
char *getenv(char *s);

void
  dbgopen(fp, fname, openmode)
FILE           *fp;  /* file pointer */
char           *fname;  /* file name */
char           *openmode;  /* open mode flags */
{
  if (DBGOPT(DBG_OKAY_OPEN) && (fp != (FILE *) NULL)) {
    (void) fprintf(stderr, "%%Open [%s] mode [%s]--[OK]\n", fname, openmode);
  }
  if (DBGOPT(DBG_FAIL_OPEN) && (fp == (FILE *) NULL)) {
    (void) fprintf(stderr, "%%Open [%s] mode [%s]--[FAILED]\n", fname, openmode);
  }
}

void
  fontfile(filelist, font_path, font_name, magnification)
char           *filelist[MAXFORMATS];  /* output filename list */
char           *font_path;  /* host font file pathname */
char           *font_name;  /* TeX font_name */
int             magnification;  /* magnification value */
{
  
  /*
   * Given a TeX font_name and a magnification value, construct a list
   * of system-dependent  host  filenames,  returning  them  in  the
   * first argument.  The files are not guaranteed to exist.  The font
   * names by default contains names for the PK,  GF font files, but
   * the selection, and search order, may be changed at run time by
   * defining a value for the environment  variable FONTLIST.  See
   * initglob()  for details.
   */
  
  register int    m;  /* index into filelist[] */
  register INT16  k;  /* loop index */
  float           fltdpi;  /* dots/inch */
  int             dpi;  /* dots/inch for filename */
  
  /*
   * We need to convert an old-style ROUNDED integer magnification
   * based on 1000 units = 200 dpi to an actual rounded dpi value.
   * 
   * We have magnification = round(real_mag) = trunc(real_mag + 0.5) which
   * implies float(magnification) - 0.5 <= real_mag <
   * float(magnification) + 0.5
   * 
   * We want dpi = round(real_mag/5.0) which implies float(dpi) - 0.5 <=
   * real_mag/5.0 < float(dpi) + 0.5 or 5.0*float(dpi) - 2.5 <=
   * real_mag < 5.0*float(dpi) + 2.5
   * 
   * We can combine the two to conclude that 5.0*float(dpi) - 2.5 <
   * float(magnification) + 0.5 or 5.0*float(dpi) - 3 <
   * float(magnification) which gives the STRICT inequality float(dpi)
   * < (float(magnification) + 3.0)/5.0
   */
  
  fltdpi = (((float) magnification) + 3.0) / 5.0;
  dpi = (int) fltdpi;
  if (fltdpi == (float) dpi)
    dpi--;    /* enforce inequality */
#ifdef DEBUG1
  fprintf(stderr, "fontfile: fltdpi: %f , dpi: %d\n", fltdpi, dpi);
#endif
  
  for (k = 0; k < MAXFORMATS; ++k)  /* loop over possible file
				     * types */
    *filelist[k] = '\0';  /* Initially, all filenames are empty */
  
  m = 0;      /* index in filelist[] */
  for (k = 0; k < MAXFORMATS; ++k) {  /* loop over possible file
				       * types */
    
    if (k == gf_index) {
      (void) sprintf(filelist[m++], "%s%s.%dgf",
		     font_path, font_name, dpi);
#ifdef DEBUG1
      (void) fprintf(stderr, "In fontfile: Path = %s\n", filelist[m - 1]);
#endif
#if 0
      (void) sprintf(filelist[m++], "%s%d/%s.gf",
		     font_path, dpi, font_name);
#ifdef DEBUG1
      (void) fprintf(stderr, "In fontfile: Path = %s\n", filelist[m - 1]);
#endif
      (void) sprintf(filelist[m++], "%s%s.%dgf",
		     font_path, font_name, dpi);
#ifdef DEBUG1
      (void) fprintf(stderr, "In fontfile: Path = %s\n", filelist[m - 1]);
#endif
#endif
    } else if (k == pk_index) {
      (void) sprintf(filelist[m++], "%s%s.%dpk",
		     font_path, font_name, dpi);
#ifdef DEBUG1
      (void) fprintf(stderr, "In fontfile: Path = %s\n", filelist[m - 1]);
#endif
#if 0
      (void) sprintf(filelist[m++], "%s%d/%s.pk",
		     font_path, dpi, font_name);
#ifdef DEBUG1
      (void) fprintf(stderr, "In fontfile: Path = %s\n", filelist[m - 1]);
#endif
#endif
    }
  }
  
}

BOOLEAN
  fontsub(subfont, submag, TeXfont, TeXmag)
char           *subfont;/* new fontname returned */
int            *submag;  /* new magnification returned (0 if entire
			  * family) */
     char           *TeXfont;/* TeX requested fontname */
     int             TeXmag;  /* TeX requested magnification value */
     
     /***********************************************************************
       
       Given a  TeX  fontname  and  a  magnification  value,  search  the  font
       substitution file (if  any) for  the first pattern  defining a  possible
       replacement, and return  their values  in subfont[] and  submag, with  a
       function value of TRUE.   If no substitution  is possible, subfont[]  is
       set to a null string, submag to 0, and the function value to FALSE.
       
       The substitution file is searched for under the names
       
       (1) subfile (by default, or from command line)
       (2) subname subext
       (3) subpath subname subext
       
       allowing the user to have document-file specific substitutions, TeX-wide
       private substitutions, and system TeX-wide substitutions.  For  example,
       these   files    might    be   "sample.sub",    "texfonts.sub",    and
       "texinputs:texfonts.sub" for a document "sample.tex".
       
       Font substitution lines have the form:
       
       % comment
       oldname.oldmag  ->  subname.submag  % comment
       oldname oldmag  ->  subname submag  % comment
       oldname   ->  subname    % comment
       
       Examples are:
       
       % These provide replacements for some LaTeX invisible fonts:
       iamr10 1500  ->  amr10 1500  % comment
       iamr10.1500  ->  amr10.1500  % comment
       iamssb8    ->  amssb8    % comment
       
       These are  easily readable  by sscanf().   The first  two forms  request
       substitution of a  particular font  and magnification.   The third  form
       substitutes an entire font  family; the closest available  magnification
       to the required one will be  used.  Any dots in the non-comment  portion
       will be converted  to spaces, and  therefore, cannot be  part of a  name
       field.
       
       The first  matching substitution  will  be selected,  so  magnification-
       specific  substitutions   should   be  given  first,   before   family
       substitutions.
       
       Comments are introduced by percent and continue to end-of-line, just  as
       for TeX.   One  whitespace character  is  equivalent to  any  amount  of
       whitespace.  Whitespace and comments are optional.
       ***********************************************************************/
     
{
  static FILE    *subfp = (FILE *) NULL;  /* file pointer for sub file */
  static BOOLEAN  have_file = TRUE;  /* memory of open success */
  char            line[MAXSTR];  /* for sub file lines */
  char           *pline;  /* pointer to line[] */
  static char     fname[MAXFNAME];  /* sub file name */
  char            oldfont[MAXFNAME];  /* working storage for font
				       * names */
  int             oldmag;  /* font magnification */
  int             k;  /* sscanf() result */
  int             line_number;  /* for error messages */
  
  if (!have_file) {  /* no file available */
    *subfont = '\0';
    *submag = 0;
    return (FALSE);
  } else if (subfp == (FILE *) NULL) {  /* happens only first time  */
    (void) strcpy(fname, subfile);  /* try document specific: */
    subfp = fopen(fname, "r");  /* e.g. foo.sub (from
				 * foo.dvi) */
    DEBUG_OPEN(subfp, fname, "r");
    if (subfp == (FILE *) NULL) {
      (void) strcpy(fname, subname);  /* try path specific: */
      (void) strcat(fname, subext);  /* e.g. texfonts.sub */
      subfp = fopen(fname, "r");
      DEBUG_OPEN(subfp, fname, "r");
    }
    if (subfp == (FILE *) NULL) {
      (void) strcpy(fname, subpath);  /* try system specific: */
      (void) strcat(fname, subname);  /* e.g.
				       * texinputs:texfonts.sub
				       *  */
      (void) strcat(fname, subext);
      subfp = fopen(fname, "r");
      DEBUG_OPEN(subfp, fname, "r");
    }
    if (subfp == (FILE *) NULL) {  /* could not open any
				    * substitution file */
      have_file = FALSE;
      *subfont = '\0';
      *submag = 0;
      return (FALSE);
    }
  }
  line_number = 0;
  (void) rewind(subfp);
  while (fgets(line, MAXSTR, subfp) != (char *) NULL) {
    line_number++;  /* count lines */
    
    pline = &line[0];  /* truncate at comment start or
			* end-of-line */
    while ((*pline) && (*pline != '%') &&
           (*pline != '\r') && (*pline != '\n')) {
      if (*pline == '.')  /* convert dots to whitespace
			   * for sscanf() */
        *pline = ' ';
      ++pline;
    }
    *pline = '\0';  /* replace terminator by NUL */
    
    if ((k = (int) sscanf(line, " %s %d -> %s %d", oldfont, &oldmag,
			  subfont, submag)) == 4) {
      if ((TeXmag == oldmag) && (strcm2(oldfont, TeXfont) == 0))
        return (TRUE);
    } else if ((k = (int) sscanf(line, " %s -> %s", oldfont, subfont)) == 2) {
      *submag = 0;
      if (strcm2(oldfont, TeXfont) == 0)
        return (TRUE);
    } else if (k != EOF) {  /* EOF means no fields, so have empty
			     * line */
      (void) sprintf(message,
		     "fontsub():  Bad font substitution at line %d in file \
[%s]: [%s]",
               line_number, fname, line);
      (void) warning(message);
    }
  }
  *subfont = '\0';
  *submag = 0;
  return (FALSE);
}

void
getbytes(fp, cp, n)    /* get n bytes from file fp */
  register FILE  *fp;  /* file pointer   */
  register char  *cp;  /* character pointer */
  register BYTE   n;  /* number of bytes */

{
  while (n--)
    *cp++ = (char) getc(fp);
}

void
  getfntdf()
{
  
  /***********************************************************************
    read the font definitions as they are in the postamble of the DVI  file.
    Note that the font directory is not yet loaded.
    ***********************************************************************/
  
  register BYTE   byte;
  
  while (((byte = (BYTE) nosignex(dvifp, (BYTE) 1)) >= FNT_DEF1)
         && (byte <= FNT_DEF4))
    readfont((INT32) nosignex(dvifp, (BYTE) (byte - FNT_DEF1 + 1)));
  if (byte != POST_POST)
    (void) fatal("getfntdf():  POST_POST missing after fontdefs");
}

void
  getpgtab(lastpageptr)
long            lastpageptr;

{
  register long   p;
  register INT16  i, k;
  
  (void) fseek(dvifp, lastpageptr, 0);
  p = lastpageptr;
  
  for (k = 0; (p != (-1)) && (k < MAXPAGE); ++k) {
    page_ptr[MAXPAGE - 1 - k] = p;
    (void) fseek(dvifp, (long) p, 0);
    
    if ((BYTE) nosignex(dvifp, (BYTE) 1) != BOP)
      (void) fatal(
		   "getpgtab():  Invalid BOP (beginning-of-page) back chain");
    
    for (i = 0; i <= 9; ++i)
      (void) nosignex(dvifp, (BYTE) 4);  /* discard
					  * count0..count9 */
    p = (long) signex(dvifp, (BYTE) 4);
  }
  page_count = k;
  if (k >= MAXPAGE)
    (void) warning("getpgtab():  Page table full...rebuild driver with \
larger MAXPAGE");
  else      /* move pointer table to front of array */
    for (k = 0; k < page_count; ++k)
      page_ptr[k] = page_ptr[MAXPAGE - page_count + k];
}

float
  inch(s)
char           *s;

/***********************************************************************
  Convert a value field in s[] of the form
  
  ------    ------------------------------
  value      units implied
  ------    ------------------------------
  #.##bp    big point (1in = 72bp)
  #.##cc    cicero (1cc = 12dd)
  #.##cm    centimeter
  #.##dd    didot point (1157dd = 1238pt)
  #.##in    inch
  #.##mm    millimeter (10mm = 1cm)
  #.##pc    pica (1pc = 12pt)
  #.##pt    point (72.27pt = 1in)
  #.##sp    scaled point (65536sp = 1pt)
  ------    ------------------------------
  
  to inches, returning it as the function value.  A leading plus or  minus
  sign is optional.  The letter case of the dimension name is ignored.  No
  space is permitted between the number and the dimension.
  ***********************************************************************/

{
  BOOLEAN         neg;
  register char  *tc;
  register double flnum, pten;
  
  flnum = 0.0;
  tc = s;
  neg = FALSE;
  
  if (*tc == '-') {
    ++tc;
    neg = TRUE;
  } else if (*tc == '+') {
    ++tc;
    neg = FALSE;
  }
  /*
   * we do not use sprintf() here because we want to be able to easily
   * find the dimension string
   */
  
  for (; isdigit(*tc); ++tc)  /* collect integer part */
    flnum = flnum * 10.0 + (double) ((*tc) - '0');
  
  if (*tc == '.')    /* collect fractional part */
    for ((pten = 10.0, ++tc); isdigit(*tc); (pten *= 10.0, ++tc))
      flnum += ((double) ((*tc) - '0')) / pten;
  
  if (strcm2(tc, "in") == 0)  /* inches */
    ;
  else if (strcm2(tc, "cm") == 0)  /* centimeters */
    flnum /= 2.54;
  else if (strcm2(tc, "pt") == 0)  /* points */
    flnum /= 72.27;
  else if (strcm2(tc, "bp") == 0)  /* big points */
    flnum /= 72.0;
  else if (strcm2(tc, "cc") == 0)  /* cicero */
    flnum *= 12.0 * (1238.0 / 1157.0) / 72.27;
  else if (strcm2(tc, "dd") == 0)  /* didot points */
    flnum *= (1238.0 / 1157.0) / 72.27;
  else if (strcm2(tc, "mm") == 0)  /* millimeters */
    flnum /= 25.4;
  else if (strcm2(tc, "pc") == 0)  /* pica */
    flnum *= 12.0 / 72.27;
  else if (strcm2(tc, "sp") == 0)  /* scaled points */
    flnum /= (65536.0 * 72.27);
  else
    /* unknown units -- error  */
    /* -- fix due to Mikko Kokkonen (mikko@retina.hut.fi)*/
    ;
  if (neg)
    flnum = -flnum;
  return ((float) flnum);
}

void
  fatal(msg)      /* issue a fatal error message */
char           *msg;  /* message string */
{
  if (g_dolog && (g_logfp == (FILE *) NULL) && g_logname[0]) {
    g_logfp = fopen(g_logname, "w+");
    DEBUG_OPEN(g_logfp, g_logname, "w+");
    if (g_logfp == (FILE *) NULL) {
      (void) fprintf(stderr, "Cannot open log file [%s]\n", g_logname);
    } else {
      (void) fprintf(stderr, "[Log file [%s] created]\n", g_logname);
    }
  }
  if (g_dolog && (g_logfp != (FILE *) NULL) && g_logname[0]) {
    NL(g_logfp);
    
    (void) fputs(g_progname, g_logfp);
    (void) fputs(": FATAL--", g_logfp);
    (void) fputs(msg, g_logfp);
    NL(g_logfp);
    (void) fprintf(g_logfp, "Current TeX page counters: [%s]\n", tctos());
  }
  rsetterm();
  
  NL(stderr);
  
  (void) fputs(g_progname, stderr);
  (void) fputs(": FATAL--", stderr);
  (void) fputs(msg, stderr);
  NL(stderr);
  (void) fprintf(stderr, "Current TeX page counters: [%s]\n\n", tctos());
  
  g_errenc = 1;    /* set global fatal exit code */
  alldone();
}

void
  initglob()
{        /* initialize global variables */
  register INT16  i;  /* loop index */
  register char  *tcp;  /* temporary character pointer */
  
  /***********************************************************************
    Set up masks such that rightones[k]  has 1-bits at the right end  of
    the word from k ..   HOST_WORD_SIZE-1, where bits are numbered  from
    left (high-order) to right (lower-order), 0 .. HOST_WORD_SIZE-1.
    
    img_mask[k] has  a 1-bit  in  position k,  bits numbered  as  above.
    power[k] is  1  <<  k,  and gpower[k]  is  2**k-1  (i.e.  1-bits  in
    low-order k positions).  These 3 require only 32 entries each  since
    they deal with 32-bit words from the PK and GF font files.
    
    These are set  at run-time, rather  than compile time  to avoid  (a)
    problems with C  preprocessors which sometimes  do not handle  large
    constants correctly, and (b) host byte-order dependence.
    ***********************************************************************/
  
  rightones[HOST_WORD_SIZE - 1] = 1;
  for (i = HOST_WORD_SIZE - 2; (i >= 0); --i)
    rightones[i] = (rightones[i + 1] << 1) | 1;
  
  img_mask[31] = 1;
  for (i = 30; i >= 0; --i)
    img_mask[i] = img_mask[i + 1] << 1;
  
  power[0] = 1;
  for (i = 1; i <= 31; ++i)
    power[i] = power[i - 1] << 1;
  
  gpower[0] = 0;    /* NB: we have gpower[0..32], not [0..31] */
  for (i = 1; i <= 32; ++i)
    gpower[i] = power[i - 1] | gpower[i - 1];
  
  debug_code = 0;
  runmag = STDMAG;  /* default runtime magnification */
  
  npage = 0;
  copies = 1;    /* number of copies of each page */
  topmargin = 1.0;  /* DVI driver standard default */
  leftmargin = 1.0;  /* DVI driver standard default */
  
  subfile[0] = '\0';
  
#if    VIRTUAL_FONTS
  virt_font = FALSE;
#endif
  
  /*
   * Copy default file fields into global  variables, then replace by
   * any runtime environment  variable definitions.  We need not  do
   * this for TOPS-20   and  VAX  VMS,  since   SUBPATH and  FONTPATH
   * are  already initialized to logical  names  which   the  operating
   * system   will translate at file open time.
   */
  
  (void) strcpy(helpcmd, DVIHELP);
  (void) strcpy(subpath, SUBPATH);
  (void) strcpy(subname, SUBNAME);
  (void) strcpy(subext, SUBEXT);
  (void) strcpy(fontpath, FONTPATH);
  (void) strcpy(fontlist, FONTLIST);
  
  if ((tcp = getenv("DVIHELP")) != (char *) NULL)
    (void) strcpy(helpcmd, tcp);
  
  if ((tcp = getenv(TEXINPUTS)) != (char *) NULL)
    (void) strcpy(subpath, tcp);
  if ((tcp = getenv(TEXFONTS)) != (char *) NULL)
    (void) strcpy(fontpath, tcp);
  
  if ((tcp = getenv("FONTLIST")) != (char *) NULL)
    (void) strcpy(fontlist, tcp);
}

void
  option(optstr)      /* process command-line option */
char           *optstr;  /* option string (e.g. "-m1500") */
{
  register UNSIGN32 k;  /* magnification */
  register int    value;  /* converted digit string value */
  register int    m;  /* loop index */
  char           *p;  /* pointer into optstr */
  int             p1, p2, p3;  /* temp values for sscanf() */
  double          fltmag;  /* floating-point mag value */
  
  typedef struct {
    char           *envname;
    char           *envvar;
  }               envstruct;
  
  static envstruct envlist[] =
    {      /* names and addresses of environment vars
	    * (alphabetical order) */
      "DVIHELP", helpcmd,
      "FONTLIST", fontlist,
      "TEXFONTS", fontpath,
      "TEXINPUTS", subpath,
    };
  
  if (*optstr != '-')
    return;    /* return if this is not an option */
  
  switch (*(optstr + 1)) {
    
  case 'a':    /* A selects virtual font caching */
  case 'A':
    
#if    VIRTUAL_FONTS
    virt_font = TRUE;
#endif
    break;
    
  case 'b':    /* b selects backwards printing order */
  case 'B':
    backwards = TRUE;
    break;
    
    
  case 'c':    /* c selects number of copies */
  case 'C':
    copies = (UNSIGN16) atoi(optstr + 2);
    copies = MAX(1, MIN(copies, 256));  /* enforce reasonable
					 * limits */
    break;
    
    
  case 'd':    /* d selects debug output */
  case 'D':
    debug_code |= (BYTE) atoi(optstr + 2);
    break;
    
    
  case 'e':    /* e specifies ``environment'' variable
		* definition */
  case 'E':    /* We ignore letter case since these come
		* from */
    /* the command line */
    if (!(p = strrchr(optstr, '='))) {
      (void) usage(stderr);
      (void) sprintf(message,
		     "option():  Bad switch [%s]--expected -eENVNAME=value",
		     optstr);
      (void) fatal(message);
    }
    *p = '\0';  /* clobber "=" by string terminator */
    for (m = 0; m < sizeof(envlist) / sizeof(envlist[0]); ++m) {
      if (strcm2(optstr + 2, envlist[m].envname) == 0) {
        (void) strcpy(envlist[m].envvar, p + 1);
        return;
      }
    }
    (void) usage(stderr);
    
    (void) sprintf(message, "option():  Bad switch [%s]--expected one of:",
		   optstr);
    for (m = 0; m < sizeof(envlist) / sizeof(envlist[0]); ++m) {
      (void) strcat(message, " ");
      (void) strcat(message, envlist[m].envname);
    }
    (void) fatal(message);
    break;
    
    
  case 'f':    /* f specifies font substitution file */
  case 'F':
    (void) strcpy(subfile, optstr + 2);
    break;
    
    
  case 'l':    /* l prohibits logging of errors */
  case 'L':
    g_dolog = FALSE;
    break;
    
    
  case 'm':    /* m selects alternate magnification */
  case 'M':
    /*
     * Collect 2*value initially.  Then if value is small, assume
     * user specified magstep value; magstep k means 1.2**k,
     * where k is integral, or half-integral.  Otherwise, assume
     * user has specified pxl file magnification (1000 == 200dpi,
     * 1500 == 300dpi, etc.).
     */
    
    
    if (strchr(optstr + 2, '.') != (char *) NULL)
      fltmag = (double) (2.0 * atof(optstr + 2));
    else
      fltmag = (double) (2 * atoi(optstr + 2));
    if ((0.0 != fltmag) &&
        (-30.0 <= fltmag) && (fltmag <= 30.0)) {  /* magstep 15 is limit */
      if (fltmag < 0.0)
        runmag = (UNSIGN32) (0.5 +
			     (1.0 / pow((double) sqrt(1.2), -fltmag)) * (double) STDMAG);
      else
        runmag = (UNSIGN32) (0.5 + pow((double) sqrt(1.2), fltmag) *
			     (double) STDMAG);
    } else
      runmag = (UNSIGN32) (0.5 + fltmag / 2.0);
    
    k = MAGSIZE(actfact(runmag));  /* rounded magnification */
    if (k != runmag) {
      (void) sprintf(message,
		     "option():  Requested magnification %d not available.",
		     (int) runmag);
      (void) warning(message);
      runmag = k;
      k = (UNSIGN32) MAX(1, MIN(mag_index, (magtablesize - 1)));
      (void) sprintf(message,
		     "  Magnification reset to nearest legal value %d in \
family ..%d..%d..%d..",
		     (int) runmag, (int) MAGSIZE(mag_table[k - 1]),
		     (int) MAGSIZE(mag_table[k]),
		     (int) MAGSIZE(mag_table[k + 1]));
      (void) warning(message);
    }
    break;
    
    
  case 'o':
  case 'O':    /* o selects output page range (-o# or -o#:#
		* or -o#:#:#) */
    p1 = p2 = p3 = 0;
    value = (int) sscanf(optstr + 2, "%d:%d:%d", &p1, &p2, &p3);
    page_begin[npage] = p1;
    page_end[npage] = p2;
    page_step[npage] = p3;
    switch (value) {
    case 1:
      optstr++;  /* point past "-" */
      
      do {  /* skip over digit string */
        optstr++;
      }
      while (isdigit(*optstr) || (*optstr == '-') || (*optstr == '+'));
      
      if (*optstr) {  /* trash follows number */
        (void) usage(stderr);
        (void) sprintf(message,
		       "option():  %s is not a legal page number switch", optstr);
        (void) fatal(message);
      } else  /* had expected end-of-string */
        page_end[npage] = (INT16) page_begin[npage];
      page_step[npage] = 1;
      break;
      
    case 2:  /* supply default step */
      page_step[npage] = 1;
      /* FALL THROUGH to case 3 for order check */
      
    case 3:  /* check for positive step */
      page_step[npage] = ABS(page_step[npage]);
      if (page_step[npage] == 0)
        page_step[npage] = 1;
      break;
      
    default:  /* illegal value */
      (void) usage(stderr);
      (void) sprintf(message,
		     "option():  %s is not a legal page number switch", optstr);
      (void) fatal(message);
    }
    npage++;
    break;
    
    
  case 'p':    /* p prohibits pre-font loading */
  case 'P':
    preload = FALSE;
    break;
    
    
  case 'q':    /* q inhibits status display */
  case 'Q':
    quiet = TRUE;
    break;
    
    
  case 's':    /* specify a startup file */
  case 'S':
      read_startup_file(optstr + 2);
    break;
    
    
  case 'v':    /* report messages etc. */
  case 'V':
    quiet = FALSE;
    break;
    
    
  case 'x':
  case 'X':
    leftmargin = inch(optstr + 2);
    break;
    
    
  case 'y':
  case 'Y':
    topmargin = inch(optstr + 2);
    break;
    
#if    BSD42
  case 'z':
  case 'Z':
    spool_output = TRUE;  /* offer to send output to spooler */
    break;
#endif        /* BSD42 */
    
  default:
    (void) usage(stderr);
    (void) sprintf(message, "option():  %s is not a legal switch", optstr);
    (void) fatal(message);
  }
}




void
  reldfont(tfontptr)    /* load (or reload) font parameters */
struct font_entry *tfontptr;
{
  register UNSIGN16 the_char;  /* loop index */
  int             err_code;
  register struct char_entry *tcharptr;  /* temporary char_entry
					  * pointer */
  
  tfontptr->font_mag = 
    (UNSIGN32) ((actfact(MAGSIZE((float) tfontptr->s / (float)tfontptr->d)) *
		 ((float) runmag / (float) STDMAG) *
				    
#if    USEGLOBALMAG
		 actfact(mag) *
#endif
				    
		 (float) RESOLUTION * 5.0) + 0.5);
  
  tfontptr->designsize = (UNSIGN32) 0L;
  tfontptr->hppp = (UNSIGN32) 0L;
  tfontptr->vppp = (UNSIGN32) 0L;
  tfontptr->min_m = (INT32) 0L;
  tfontptr->max_m = (INT32) 0L;
  tfontptr->min_n = (INT32) 0L;
  tfontptr->max_n = (INT32) 0L;
  
  for (the_char = FIRSTPXLCHAR; the_char <= LASTPXLCHAR; the_char++) {
    tcharptr = &(tfontptr->ch[the_char]);
    tcharptr->dx = (INT32) 0L;
    tcharptr->dy = (INT32) 0L;
    tcharptr->hp = (COORDINATE) 0;
    tcharptr->fontrp = -1L;
    tcharptr->pxlw = (UNSIGN16) 0;
    tcharptr->rasters = (UNSIGN32 *) NULL;
    tcharptr->refcount = 0;
    tcharptr->tfmw = 0L;
    tcharptr->wp = (COORDINATE) 0;
    tcharptr->xoffp = (COORDINATE) 0;
    tcharptr->yoffp = (COORDINATE) 0;
  }
  
  if (tfontptr != pfontptr)
    (void) openfont(tfontptr->n);
  
  if (fontfp == (FILE *) NULL)  /* have empty font with zero metrics */
    return;
  
  for (;;) {    /* fake one-trip loop test for font
		 * types PK, GF, in order of preference */
    (void) rewind(fontfp);  /* position to beginning-of-file */
    if (((BYTE) nosignex(fontfp, (BYTE) 1) == (BYTE) PKPRE) &&
        ((BYTE) nosignex(fontfp, (BYTE) 1) == (BYTE) PKID)) {
      tfontptr->font_type = (BYTE) FT_PK;
      tfontptr->charxx = (void (*) ()) charpk;
      err_code = readpk();
      break;
    }
    (void) rewind(fontfp);  /* position to beginning-of-file */
    if (((BYTE) nosignex(fontfp, (BYTE) 1) == (BYTE) GFPRE) &&
        ((BYTE) nosignex(fontfp, (BYTE) 1) == (BYTE) GFID)) {
      tfontptr->font_type = (BYTE) FT_GF;
      tfontptr->charxx = (void (*) ()) chargf;
      err_code = readgf();
      break;
    }
    err_code = (int) EOF;
    break;
  }      /* end one-trip loop */
  
  if (err_code) {
    (void) sprintf(message,
		   "reldfont():  Font file [%s] is not a valid GF, PK file",
		   tfontptr->name);
    (void) fatal(message);
  }
}

void
  setfntnm(k)
register INT32  k;

/***********************************************************************
  This routine is used to specify the  font to be used in printing  future
  characters.
  ***********************************************************************/

{
  register struct font_entry *p;
  
  p = hfontptr;
  while ((p != (struct font_entry *) NULL) && (p->k != k))
    p = p->next;
  if (p == (struct font_entry *) NULL) {
    (void) sprintf(message, "setfntnm():  font %ld undefined", k);
    (void) fatal(message);
  } else
    fontptr = p;
  
}

void
  special(s)      /* process TeX \special{} string in s[] */
register char  *s;
{
  (void) sprintf(message,
		 "special():  TeX \\special{%s} not implemented in this driver", s);
  (void) warning(message);
}

char           *
  strchr(s, c)      /* return address of c in s[], or (char*)NULL
		     * if not found. */
register char  *s;  /* c may be NUL; terminator of s[] is
		     * included in search. */
register int    c;
{
  while ((*s) && ((*s) != c))
    ++s;
  
  if ((*s) == c)
    return (s);
  else
    return ((char *) NULL);
}


/***********************************************************************
  Compare strings (ignoring case), and return:
  s1>s2:  >0
  s1==s2:  0
  s1<s2:  <0
  ***********************************************************************/

/*
 * toupper() is supposed to work for all letters, but PCC-20 does it
 * incorrectly if the argument is not already lowercase; this definition
 * fixes that.
 */

#define UC(c) (islower(c) ? toupper(c) : c)

int
  strcm2(s1, s2)
register char  *s1, *s2;

{
  while ((*s1) && (UC(*s1) == UC(*s2))) {
    s1++;
    s2++;
  }
  return ((int) (UC(*s1) - UC(*s2)));
}
#undef UC

/*
 * toupper() is supposed to work for all letters, but PCC-20 does it
 * incorrectly if the argument is not already lowercase; this definition
 * fixes that.
 */

#define UC(c) (islower(c) ? toupper(c) : c)

int
  strid2(string, substring)  /* Return index (0,1,...) of substring in
			      * string */
char            string[];  /* or -1 if not found.  Letter case
			    * is IGNORED. */
char            substring[];
{
  register int    k;  /* loop index */
  register int    limit;  /* loop limit */
  register char  *s;
  register char  *sub;
  
  limit = (int) strlen(string) - (int) strlen(substring);
  
  for (k = 0; k <= limit; ++k) {  /* simple (and slow) linear search */
    s = &string[k];
    
    for (sub = &substring[0]; (UC(*s) == UC(*sub)) && (*sub); (++s, ++sub))
      /* NO-OP */ ;
    
    if (*sub == '\0')  /* then all characters match */
      return (k);  /* success -- match at index k */
  }
  
  return (-1);    /* failure */
}

char           *
  strrchr(s, c)      /* return address of last occurrence of c in
		      * s[], */
/* or (char*)NULL if not found.  c may be NUL; */
/* terminator of s[] is included in search. */
register char  *s;
register int    c;
{
  register char  *t;
  
  t = (char *) NULL;
  for (;;) {    /* loop forever */
    if (*s == c)
      t = s;  /* remember last match position */
    if (!*s)
      break;  /* exit when NULL reached */
    ++s;    /* advance to next character */
  }
  return (t);
}

void
  unloadfonts()
{        /* mark all fonts as not loaded and set
	  * no current fonts */
  INT16           k;
  
  for (fontptr = hfontptr; fontptr != (struct font_entry *) NULL;
       fontptr = fontptr->next) {
    for (k = 0; k < NPXLCHARS; ++k)
      fontptr->ch[k].isloaded = FALSE;
    
    if (fontptr->font_file_id != (FILE *) NULL) {
      (void) fclose(fontptr->font_file_id);
      fontptr->font_file_id = (FILE *) NULL;
    }
  }
  
  fontfp = (FILE *) NULL;  /* no current font file */
  for (; nopen > 0; --nopen) {  /* clear font file cache */
    font_files[nopen].font_id = (FILE *) NULL;
    font_files[nopen].use_count = (INT16) 0;
  }
  
  /*
   * NB: It is important here that the loop index be global; the
   * relation of fontptr to pfontptr is used by openfont() to decide
   * whether the font file is already open.
   */
  for (fontptr = hfontptr; fontptr != (struct font_entry *) NULL;
       fontptr = fontptr->next) {
    pfontptr = (struct font_entry *) (NULL);  /* so reldfont() calls
					       * openfont() */
    (void) reldfont(fontptr);  /* get new font metrics */
  }
}

void
  usage(fp)      /* print usage message to fp and return */
FILE           *fp;
{
  (void) fprintf(fp, "[TeX82 DVI Translator Version %s]\n", VERSION_NO);
  (void) fprintf(fp, "[%s]\n", DEVICE_ID);
  
  (void) sprintf(message,
		 "Usage: %s {-a} {-b} {-c#} {-d#} {-eENVNAME=value} {-ffontsubfile} \
{-l} {-m#} {-o#:#:#} {-o#:#} {-o#} {-p} {-x#{units}} {-y#{units}} dvifile(s)\n",
           g_progname);

  (void) fprintf(fp, message);

  (void) fprintf(fp,
           "For documentation on this program, try the operating command(s):\n");

  (void) fprintf(fp, helpcmd);
}

void
warning(msg)      /* issue a warning */
  char           *msg;  /* message string  */
{
  
  if (quiet)    /* no warnings in quiet mode */
    return;
  
  if (g_dolog && (g_logfp == (FILE *) NULL) && g_logname[0]) {
    g_logfp = fopen(g_logname, "w+");
    DEBUG_OPEN(g_logfp, g_logname, "w+");
    if (g_logfp != (FILE *) NULL) {
      (void) fprintf(stderr, "[Log file [%s] created]\n", g_logname);
    }
  }
  if (g_dolog && (g_logfp != (FILE *) NULL) && g_logname[0]) {
    NL(g_logfp);
    
    (void) fputs(msg, g_logfp);
    NL(g_logfp);
    (void) fprintf(g_logfp, "Current TeX page counters: [%s]\n", tctos());
  }
  if (g_dolog && (g_logfp == (FILE *) NULL) && g_logname[0]) {
    (void) sprintf(message, "Cannot open log file [%s]\n", g_logname);
    fatal(message);
  }
  (void) fputs(msg, stderr);
  NL(stderr);
  if (g_logname[0]) {
    (void) fprintf(stderr, "Current TeX page counters: [%s]\n", tctos());
  }
}

void
  gotint()
{
  ;
}


