(* VDU routines for a VIS550 terminal. *)

#include 'globals.h';
#include 'screenio.h';
#include 'vdu.h';
#include 'tek4010vdu.h';

(******************************************************************************)

PROCEDURE StartText;

(* We are about to draw text in dialogue region. *)

BEGIN
TEK4010StartText;
END; (* StartText *)

(******************************************************************************)

PROCEDURE MoveAbs (row, col : INTEGER);

(* Move cursor to given screen position. *)

BEGIN
WriteChar(ESC); WriteChar('[');
WriteInt(row);
WriteChar(';');
WriteInt(col);
WriteChar('H');
END; (* MoveAbs *)

(******************************************************************************)

PROCEDURE MoveToTextLine (line : INTEGER);

(* Move current position to start of given line. *)

BEGIN
WriteChar(CAN);
MoveAbs(line,1);
END; (* MoveToTextLine *)

(******************************************************************************)

PROCEDURE ClearTextLine (line : INTEGER);

(* Erase given line; note that DVItoVDU does not assume anything about the
   current position at the end of this routine.
*)

BEGIN
WriteChar(CAN);
MoveAbs(line,1);
WriteChar(ESC);
WriteString('[K');   (* erase to end of line *)
END; (* ClearTextLine *)

(******************************************************************************)

PROCEDURE ClearScreen;

BEGIN
WriteChar(CAN);
WriteChar(ESC);
WriteString('[2J');   (* erase all alphanumerics *)
TEK4010ClearScreen;
END; (* ClearScreen *)

(******************************************************************************)

PROCEDURE StartGraphics;

(* We are about to draw in window region. *)

BEGIN
TEK4010StartGraphics;
END; (* StartGraphics *)

(******************************************************************************)

PROCEDURE LoadFont (fontname : string;
                    fontsize : INTEGER;
                    mag, hscale, vscale : REAL);

BEGIN
TEK4010LoadFont(fontname,fontsize,mag,hscale,vscale);
END; (* LoadFont *)

(******************************************************************************)

PROCEDURE ShowChar (screenh, screenv : INTEGER;
                    ch : CHAR);

(* The TEK4010 Terse mode characters on the VIS550 need to be dragged down
   so that any descenders will be below the baseline represented by screenv.
*)

BEGIN
screenv := screenv + dragdown;
IF screenv > 779 THEN
   TEK4010ShowChar(screenh, 779, ch)   (* drag down as far as possible *)
ELSE
   TEK4010ShowChar(screenh, screenv, ch);
END; (* ShowChar *)

(******************************************************************************)

PROCEDURE ShowRectangle (screenh, screenv,          (* top left pixel *)
                         width, height : INTEGER;   (* size of rectangle *)
                         ch : CHAR);                (* black pixel *)

(* Display the given rectangle. *)

VAR pos : INTEGER;

BEGIN
IF height = 1 THEN BEGIN            (* show row vector *)
   pos := 779 - screenv;
   WriteChar(GS);
   SendXY(screenh,pos);             (* move cursor to start of row *)
   SendXY(screenh+width-1,pos);     (* draw vector to end of row *)
END
ELSE IF width = 1 THEN BEGIN        (* show column vector *)
   pos := 779 - screenv;
   WriteChar(GS);
   SendXY(screenh,pos);             (* move cursor to start of column *)
   SendXY(screenh,pos-height+1);    (* draw vector to end of column *)
END
ELSE BEGIN
   (* assume height and width > 1; draw and fill rectangle *)
   pos := 779 - (screenv+height-1);
   WriteChar(ESC);         WriteChar('/');
   WriteInt(screenh);      WriteChar(';');   (* left *)
   WriteInt(pos);          WriteChar(';');   (* bottom *)
   WriteInt(width-1);      WriteChar(';');
   WriteInt(height+1);     WriteChar('y');
   (* Note that there are a few problems with this command:
      - we need to subtract 1 from width.  While this prevents exceeding the
        right edge (reason unknown), it causes missing pixel columns.
      - we need to ADD 1 to height to avoid missing pixel rows.
      - the smallest rectangle drawn is 2 by 2.
      - the new cursor position is undefined.
      These funnies are outweighed by the improved efficiency in drawing large
      rectangles.
   *)
   havesentxy := FALSE;   (* need to re-synch cursor position *)
END;
END; (* ShowRectangle *)

(******************************************************************************)

PROCEDURE ResetVDU;

BEGIN
WriteChar(CAN);
END; (* ResetVDU *)

(******************************************************************************)

PROCEDURE InitVDU;

(* The dialog region will be the top 4 text lines in VT100 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 33 in VT100 mode.
*)

BEGIN
InitTEK4010VDU;
DVIstatusl    := 1;      (* DVItoVDU assumes top text line = 1 *)
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := 33;     (* also number of text lines on screen *)
(* The above values assume the VIS550 is in VT100 mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
windowv  := 92;          (* approx. height in TEK4010 pixels of 4 text lines
                            i.e. ~ 4 * 780/34 *)
windowh  := 0;
windowht := 780-windowv;
windowwd := 1024;

WriteChar(GS);
WriteChar(ESC);
WriteChar('@');          (* solid fill for rectangular draw and fill *)
WriteChar(CAN);
END; (* InitVDU *)
